//
// Copyright (C) 2011 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

#include "update_engine/payload_generator/payload_signer.h"

#include <endian.h>

#include <memory>
#include <utility>

#include <base/logging.h>
#include <base/strings/string_number_conversions.h>
#include <base/strings/string_split.h>
#include <base/strings/string_util.h>
#include <brillo/data_encoding.h>
#include <openssl/err.h>
#include <openssl/pem.h>

#include "update_engine/common/constants.h"
#include "update_engine/common/hash_calculator.h"
#include "update_engine/common/subprocess.h"
#include "update_engine/common/utils.h"
#include "update_engine/payload_consumer/delta_performer.h"
#include "update_engine/payload_consumer/payload_constants.h"
#include "update_engine/payload_consumer/payload_metadata.h"
#include "update_engine/payload_consumer/payload_verifier.h"
#include "update_engine/payload_generator/delta_diff_generator.h"
#include "update_engine/payload_generator/payload_file.h"
#include "update_engine/update_metadata.pb.h"

using std::string;
using std::vector;

namespace chromeos_update_engine {

namespace {
// Given raw |signatures|, packs them into a protobuf and serializes it into a
// string. Returns true on success, false otherwise.
bool ConvertSignaturesToProtobuf(const vector<brillo::Blob>& signatures,
                                 const vector<size_t>& padded_signature_sizes,
                                 string* out_serialized_signature) {
  TEST_AND_RETURN_FALSE(signatures.size() == padded_signature_sizes.size());
  // Pack it into a protobuf
  Signatures out_message;
  for (size_t i = 0; i < signatures.size(); i++) {
    const auto& signature = signatures[i];
    const auto& padded_signature_size = padded_signature_sizes[i];
    TEST_AND_RETURN_FALSE(padded_signature_size >= signature.size());
    Signatures::Signature* sig_message = out_message.add_signatures();
    // Skip assigning the same version number because we don't need to be
    // compatible with old major version 1 client anymore.

    // TODO(Xunchang) don't need to set the unpadded_signature_size field for
    // RSA key signed signatures.
    sig_message->set_unpadded_signature_size(signature.size());
    brillo::Blob padded_signature = signature;
    padded_signature.insert(
        padded_signature.end(), padded_signature_size - signature.size(), 0);
    sig_message->set_data(padded_signature.data(), padded_signature.size());
  }

  // Serialize protobuf
  TEST_AND_RETURN_FALSE(
      out_message.SerializeToString(out_serialized_signature));
  LOG(INFO) << "Signature blob size: " << out_serialized_signature->size();
  return true;
}

// Given an unsigned payload under |payload_path| and the |payload_signature|
// and |metadata_signature| generates an updated payload that includes the
// signatures. It populates |out_metadata_size| with the size of the final
// manifest after adding the fake signature operation, and
// |out_signatures_offset| with the expected offset for the new blob, and
// |out_metadata_signature_size| which will be size of |metadata_signature|
// if the payload major version supports metadata signature, 0 otherwise.
// Returns true on success, false otherwise.
bool AddSignatureBlobToPayload(const string& payload_path,
                               const string& payload_signature,
                               const string& metadata_signature,
                               brillo::Blob* out_payload,
                               uint64_t* out_metadata_size,
                               uint32_t* out_metadata_signature_size,
                               uint64_t* out_signatures_offset) {
  uint64_t manifest_offset = 20;
  const int kProtobufSizeOffset = 12;

  brillo::Blob payload;
  TEST_AND_RETURN_FALSE(utils::ReadFile(payload_path, &payload));
  PayloadMetadata payload_metadata;
  TEST_AND_RETURN_FALSE(payload_metadata.ParsePayloadHeader(payload));
  uint64_t metadata_size = payload_metadata.GetMetadataSize();
  uint32_t metadata_signature_size =
      payload_metadata.GetMetadataSignatureSize();
  // Write metadata signature size in header.
  uint32_t metadata_signature_size_be = htobe32(metadata_signature.size());
  memcpy(payload.data() + manifest_offset,
         &metadata_signature_size_be,
         sizeof(metadata_signature_size_be));
  manifest_offset += sizeof(metadata_signature_size_be);
  // Replace metadata signature.
  payload.erase(payload.begin() + metadata_size,
                payload.begin() + metadata_size + metadata_signature_size);
  payload.insert(payload.begin() + metadata_size,
                 metadata_signature.begin(),
                 metadata_signature.end());
  metadata_signature_size = metadata_signature.size();
  LOG(INFO) << "Metadata signature size: " << metadata_signature_size;

  DeltaArchiveManifest manifest;
  TEST_AND_RETURN_FALSE(payload_metadata.GetManifest(payload, &manifest));

  // Is there already a signature op in place?
  if (manifest.has_signatures_size()) {
    // The signature op is tied to the size of the signature blob, but not it's
    // contents. We don't allow the manifest to change if there is already an op
    // present, because that might invalidate previously generated
    // hashes/signatures.
    if (manifest.signatures_size() != payload_signature.size()) {
      LOG(ERROR) << "Attempt to insert different signature sized blob. "
                 << "(current:" << manifest.signatures_size()
                 << "new:" << payload_signature.size() << ")";
      return false;
    }

    LOG(INFO) << "Matching signature sizes already present.";
  } else {
    // Updates the manifest to include the signature operation.
    PayloadSigner::AddSignatureToManifest(
        payload.size() - metadata_size - metadata_signature_size,
        payload_signature.size(),
        &manifest);

    // Updates the payload to include the new manifest.
    string serialized_manifest;
    TEST_AND_RETURN_FALSE(manifest.AppendToString(&serialized_manifest));
    LOG(INFO) << "Updated protobuf size: " << serialized_manifest.size();
    payload.erase(payload.begin() + manifest_offset,
                  payload.begin() + metadata_size);
    payload.insert(payload.begin() + manifest_offset,
                   serialized_manifest.begin(),
                   serialized_manifest.end());

    // Updates the protobuf size.
    uint64_t size_be = htobe64(serialized_manifest.size());
    memcpy(&payload[kProtobufSizeOffset], &size_be, sizeof(size_be));
    metadata_size = serialized_manifest.size() + manifest_offset;

    LOG(INFO) << "Updated payload size: " << payload.size();
    LOG(INFO) << "Updated metadata size: " << metadata_size;
  }
  uint64_t signatures_offset =
      metadata_size + metadata_signature_size + manifest.signatures_offset();
  LOG(INFO) << "Signature Blob Offset: " << signatures_offset;
  payload.resize(signatures_offset);
  payload.insert(payload.begin() + signatures_offset,
                 payload_signature.begin(),
                 payload_signature.end());

  *out_payload = std::move(payload);
  *out_metadata_size = metadata_size;
  *out_metadata_signature_size = metadata_signature_size;
  *out_signatures_offset = signatures_offset;
  return true;
}

// Given a |payload| with correct signature op and metadata signature size in
// header and |metadata_size|, |metadata_signature_size|, |signatures_offset|,
// calculate hash for payload and metadata, save it to |out_hash_data| and
// |out_metadata_hash|.
bool CalculateHashFromPayload(const brillo::Blob& payload,
                              const uint64_t metadata_size,
                              const uint32_t metadata_signature_size,
                              const uint64_t signatures_offset,
                              brillo::Blob* out_hash_data,
                              brillo::Blob* out_metadata_hash) {
  if (out_metadata_hash) {
    // Calculates the hash on the manifest.
    TEST_AND_RETURN_FALSE(HashCalculator::RawHashOfBytes(
        payload.data(), metadata_size, out_metadata_hash));
  }
  if (out_hash_data) {
    // Calculates the hash on the updated payload. Note that we skip metadata
    // signature and payload signature.
    HashCalculator calc;
    TEST_AND_RETURN_FALSE(calc.Update(payload.data(), metadata_size));
    TEST_AND_RETURN_FALSE(signatures_offset >=
                          metadata_size + metadata_signature_size);
    TEST_AND_RETURN_FALSE(calc.Update(
        payload.data() + metadata_size + metadata_signature_size,
        signatures_offset - metadata_size - metadata_signature_size));
    TEST_AND_RETURN_FALSE(calc.Finalize());
    *out_hash_data = calc.raw_hash();
  }
  return true;
}

std::unique_ptr<EVP_PKEY, decltype(&EVP_PKEY_free)> CreatePrivateKeyFromPath(
    const string& private_key_path) {
  FILE* fprikey = fopen(private_key_path.c_str(), "rb");
  if (!fprikey) {
    PLOG(ERROR) << "Failed to read " << private_key_path;
    return {nullptr, nullptr};
  }

  auto private_key = std::unique_ptr<EVP_PKEY, decltype(&EVP_PKEY_free)>(
      PEM_read_PrivateKey(fprikey, nullptr, nullptr, nullptr), EVP_PKEY_free);
  fclose(fprikey);
  return private_key;
}

}  // namespace

bool PayloadSigner::GetMaximumSignatureSize(const string& private_key_path,
                                            size_t* signature_size) {
  *signature_size = 0;
  auto private_key = CreatePrivateKeyFromPath(private_key_path);
  if (!private_key) {
    LOG(ERROR) << "Failed to create private key from " << private_key_path;
    return false;
  }

  *signature_size = EVP_PKEY_size(private_key.get());
  return true;
}

void PayloadSigner::AddSignatureToManifest(uint64_t signature_blob_offset,
                                           uint64_t signature_blob_length,
                                           DeltaArchiveManifest* manifest) {
  LOG(INFO) << "Making room for signature in file";
  manifest->set_signatures_offset(signature_blob_offset);
  manifest->set_signatures_size(signature_blob_length);
}

bool PayloadSigner::VerifySignedPayload(const string& payload_path,
                                        const string& public_key_path) {
  brillo::Blob payload;
  TEST_AND_RETURN_FALSE(utils::ReadFile(payload_path, &payload));
  PayloadMetadata payload_metadata;
  TEST_AND_RETURN_FALSE(payload_metadata.ParsePayloadHeader(payload));
  DeltaArchiveManifest manifest;
  TEST_AND_RETURN_FALSE(payload_metadata.GetManifest(payload, &manifest));
  TEST_AND_RETURN_FALSE(manifest.has_signatures_offset() &&
                        manifest.has_signatures_size());
  uint64_t metadata_size = payload_metadata.GetMetadataSize();
  uint32_t metadata_signature_size =
      payload_metadata.GetMetadataSignatureSize();
  uint64_t signatures_offset =
      metadata_size + metadata_signature_size + manifest.signatures_offset();
  CHECK_EQ(payload.size(), signatures_offset + manifest.signatures_size());
  brillo::Blob payload_hash, metadata_hash;
  TEST_AND_RETURN_FALSE(CalculateHashFromPayload(payload,
                                                 metadata_size,
                                                 metadata_signature_size,
                                                 signatures_offset,
                                                 &payload_hash,
                                                 &metadata_hash));
  string signature(payload.begin() + signatures_offset, payload.end());
  string public_key;
  TEST_AND_RETURN_FALSE(utils::ReadFile(public_key_path, &public_key));
  TEST_AND_RETURN_FALSE(payload_hash.size() == kSHA256Size);

  auto payload_verifier = PayloadVerifier::CreateInstance(public_key);
  TEST_AND_RETURN_FALSE(payload_verifier != nullptr);

  TEST_AND_RETURN_FALSE(
      payload_verifier->VerifySignature(signature, payload_hash));
  if (metadata_signature_size) {
    signature.assign(payload.begin() + metadata_size,
                     payload.begin() + metadata_size + metadata_signature_size);
    TEST_AND_RETURN_FALSE(metadata_hash.size() == kSHA256Size);
    TEST_AND_RETURN_FALSE(
        payload_verifier->VerifySignature(signature, metadata_hash));
  }
  return true;
}

bool PayloadSigner::SignHash(const brillo::Blob& hash,
                             const string& private_key_path,
                             brillo::Blob* out_signature) {
  LOG(INFO) << "Signing hash with private key: " << private_key_path;
  // We expect unpadded SHA256 hash coming in
  TEST_AND_RETURN_FALSE(hash.size() == kSHA256Size);
  // The code below executes the equivalent of:
  //
  // openssl rsautl -raw -sign -inkey |private_key_path|
  //   -in |padded_hash| -out |out_signature|

  auto private_key = CreatePrivateKeyFromPath(private_key_path);
  if (!private_key) {
    LOG(ERROR) << "Failed to create private key from " << private_key_path;
    return false;
  }

  int key_type = EVP_PKEY_id(private_key.get());
  brillo::Blob signature;
  if (key_type == EVP_PKEY_RSA) {
    RSA* rsa = EVP_PKEY_get0_RSA(private_key.get());
    TEST_AND_RETURN_FALSE(rsa != nullptr);

    brillo::Blob padded_hash = hash;
    PayloadVerifier::PadRSASHA256Hash(&padded_hash, RSA_size(rsa));

    signature.resize(RSA_size(rsa));
    ssize_t signature_size = RSA_private_encrypt(padded_hash.size(),
                                                 padded_hash.data(),
                                                 signature.data(),
                                                 rsa,
                                                 RSA_NO_PADDING);
    if (signature_size < 0) {
      LOG(ERROR) << "Signing hash failed: "
                 << ERR_error_string(ERR_get_error(), nullptr);
      return false;
    }
    TEST_AND_RETURN_FALSE(static_cast<size_t>(signature_size) ==
                          signature.size());
  } else if (key_type == EVP_PKEY_EC) {
    EC_KEY* ec_key = EVP_PKEY_get0_EC_KEY(private_key.get());
    TEST_AND_RETURN_FALSE(ec_key != nullptr);

    signature.resize(ECDSA_size(ec_key));
    unsigned int signature_size;
    if (ECDSA_sign(0,
                   hash.data(),
                   hash.size(),
                   signature.data(),
                   &signature_size,
                   ec_key) != 1) {
      LOG(ERROR) << "Signing hash failed: "
                 << ERR_error_string(ERR_get_error(), nullptr);
      return false;
    }

    // NIST P-256
    LOG(ERROR) << "signature max size " << signature.size() << " size "
               << signature_size;
    TEST_AND_RETURN_FALSE(signature.size() >= signature_size);
    signature.resize(signature_size);
  } else {
    LOG(ERROR) << "key_type " << key_type << " isn't supported for signing";
    return false;
  }
  out_signature->swap(signature);
  return true;
}

bool PayloadSigner::SignHashWithKeys(const brillo::Blob& hash_data,
                                     const vector<string>& private_key_paths,
                                     string* out_serialized_signature) {
  vector<brillo::Blob> signatures;
  vector<size_t> padded_signature_sizes;
  for (const string& path : private_key_paths) {
    brillo::Blob signature;
    TEST_AND_RETURN_FALSE(SignHash(hash_data, path, &signature));
    signatures.push_back(signature);

    size_t padded_signature_size;
    TEST_AND_RETURN_FALSE(
        GetMaximumSignatureSize(path, &padded_signature_size));
    padded_signature_sizes.push_back(padded_signature_size);
  }
  TEST_AND_RETURN_FALSE(ConvertSignaturesToProtobuf(
      signatures, padded_signature_sizes, out_serialized_signature));
  return true;
}

bool PayloadSigner::SignPayload(const string& unsigned_payload_path,
                                const vector<string>& private_key_paths,
                                const uint64_t metadata_size,
                                const uint32_t metadata_signature_size,
                                const uint64_t signatures_offset,
                                string* out_serialized_signature) {
  brillo::Blob payload;
  TEST_AND_RETURN_FALSE(utils::ReadFile(unsigned_payload_path, &payload));
  brillo::Blob hash_data;
  TEST_AND_RETURN_FALSE(CalculateHashFromPayload(payload,
                                                 metadata_size,
                                                 metadata_signature_size,
                                                 signatures_offset,
                                                 &hash_data,
                                                 nullptr));
  TEST_AND_RETURN_FALSE(
      SignHashWithKeys(hash_data, private_key_paths, out_serialized_signature));
  return true;
}

bool PayloadSigner::SignatureBlobLength(const vector<string>& private_key_paths,
                                        uint64_t* out_length) {
  DCHECK(out_length);
  brillo::Blob hash_blob;
  TEST_AND_RETURN_FALSE(HashCalculator::RawHashOfData({'x'}, &hash_blob));
  string sig_blob;
  TEST_AND_RETURN_FALSE(
      SignHashWithKeys(hash_blob, private_key_paths, &sig_blob));
  *out_length = sig_blob.size();
  return true;
}

bool PayloadSigner::HashPayloadForSigning(const string& payload_path,
                                          const vector<size_t>& signature_sizes,
                                          brillo::Blob* out_payload_hash_data,
                                          brillo::Blob* out_metadata_hash) {
  // Create a signature blob with signatures filled with 0.
  // Will be used for both payload signature and metadata signature.
  vector<brillo::Blob> signatures;
  for (int signature_size : signature_sizes) {
    signatures.emplace_back(signature_size, 0);
  }
  string signature;
  TEST_AND_RETURN_FALSE(
      ConvertSignaturesToProtobuf(signatures, signature_sizes, &signature));

  brillo::Blob payload;
  uint64_t metadata_size, signatures_offset;
  uint32_t metadata_signature_size;
  // Prepare payload for hashing.
  TEST_AND_RETURN_FALSE(AddSignatureBlobToPayload(payload_path,
                                                  signature,
                                                  signature,
                                                  &payload,
                                                  &metadata_size,
                                                  &metadata_signature_size,
                                                  &signatures_offset));
  TEST_AND_RETURN_FALSE(CalculateHashFromPayload(payload,
                                                 metadata_size,
                                                 metadata_signature_size,
                                                 signatures_offset,
                                                 out_payload_hash_data,
                                                 out_metadata_hash));
  return true;
}

bool PayloadSigner::AddSignatureToPayload(
    const string& payload_path,
    const vector<size_t>& padded_signature_sizes,
    const vector<brillo::Blob>& payload_signatures,
    const vector<brillo::Blob>& metadata_signatures,
    const string& signed_payload_path,
    uint64_t* out_metadata_size) {
  // TODO(petkov): Reduce memory usage -- the payload is manipulated in memory.

  // Loads the payload and adds the signature op to it.
  string payload_signature, metadata_signature;
  TEST_AND_RETURN_FALSE(ConvertSignaturesToProtobuf(
      payload_signatures, padded_signature_sizes, &payload_signature));
  if (!metadata_signatures.empty()) {
    TEST_AND_RETURN_FALSE(ConvertSignaturesToProtobuf(
        metadata_signatures, padded_signature_sizes, &metadata_signature));
  }
  brillo::Blob payload;
  uint64_t signatures_offset;
  uint32_t metadata_signature_size;
  TEST_AND_RETURN_FALSE(AddSignatureBlobToPayload(payload_path,
                                                  payload_signature,
                                                  metadata_signature,
                                                  &payload,
                                                  out_metadata_size,
                                                  &metadata_signature_size,
                                                  &signatures_offset));

  LOG(INFO) << "Signed payload size: " << payload.size();
  TEST_AND_RETURN_FALSE(utils::WriteFile(
      signed_payload_path.c_str(), payload.data(), payload.size()));
  return true;
}

bool PayloadSigner::GetMetadataSignature(const void* const metadata,
                                         size_t metadata_size,
                                         const string& private_key_path,
                                         string* out_signature) {
  // Calculates the hash on the updated payload. Note that the payload includes
  // the signature op but doesn't include the signature blob at the end.
  brillo::Blob metadata_hash;
  TEST_AND_RETURN_FALSE(
      HashCalculator::RawHashOfBytes(metadata, metadata_size, &metadata_hash));

  brillo::Blob signature;
  TEST_AND_RETURN_FALSE(SignHash(metadata_hash, private_key_path, &signature));

  *out_signature = brillo::data_encoding::Base64Encode(signature);
  return true;
}

}  // namespace chromeos_update_engine
