#!/usr/bin/python -B

# Copyright 2018 The Android Open Source Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# Notice:
# - verify_eccdata_strict.py: Verify data which is generated by this
# version of this toolset.
# - verify_eccdata_compatibility.py: Verify data which is generated by any
# newer version of this tool set for ensuring backward compatibility.

import sys
import argparse
import protobuf_ecc_data_pb2

parser = argparse.ArgumentParser()
parser.add_argument("--input", required=True)
parser.add_argument("--strict", action="store_true")
args = parser.parse_args()

all_ecc_info = protobuf_ecc_data_pb2.AllInfo()

with open(args.input, "rb") as ecc_data_source_file:
  all_ecc_info.ParseFromString(ecc_data_source_file.read())

if (args.strict):
  print("Verify in strict mode")

assert all_ecc_info.HasField("revision")
assert all_ecc_info.revision > 0
assert len(all_ecc_info.countries) > 0

loaded_iso = []
for country_info in all_ecc_info.countries:
  assert country_info.HasField("iso_code")
  assert len(country_info.iso_code) > 0
  assert country_info.iso_code == country_info.iso_code.strip().upper()
  assert country_info.iso_code not in loaded_iso
  loaded_iso.append(country_info.iso_code)
  assert country_info.HasField("ecc_fallback")
  assert len(country_info.ecc_fallback) > 0

  if len(country_info.eccs) > 0:
    loaded_phone_number = []
    for ecc_info in country_info.eccs:
      assert ecc_info.HasField("phone_number")
      phone_number = ecc_info.phone_number.strip()
      assert len(phone_number) > 0
      assert phone_number not in loaded_phone_number
      loaded_phone_number.append(phone_number)

      if (args.strict):
        assert len(ecc_info.types) > 0
        loaded_types = []
        for ecc_type in ecc_info.types:
          assert ecc_type == protobuf_ecc_data_pb2.EccInfo.POLICE or \
              ecc_type == protobuf_ecc_data_pb2.EccInfo.AMBULANCE or \
              ecc_type == protobuf_ecc_data_pb2.EccInfo.FIRE
          assert ecc_type not in loaded_types
          loaded_types.append(ecc_type)
      else:
        # For forward compatibility, ecc_info.types could be null if a phone
        # number contains only new types which is not defined now. Just leave
        # a warning message for this case.
        if len(ecc_info.types) == 0:
          print("WARNING: No recognizable type for " + \
              country_info.iso_code + " - " + ecc_info.phone_number)
        else:
          loaded_types = []
          for ecc_type in ecc_info.types:
            assert ecc_type not in loaded_types
            loaded_types.append(ecc_type)
  elif (args.strict):
    print("Warning: Empty ecc list for country " + country_info.iso_code)
