// Copyright 2023, The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

//! Tests HMAC with the test cases in [RFC 4231] Section 4
//!
//! [RFC 4231]: https://datatracker.ietf.org/doc/html/rfc4231

use bssl_avf::{hmac_sha256, Result};

#[test]
fn rfc4231_test_case_1() -> Result<()> {
    const KEY: &[u8; 20] = &[0x0b; 20];
    const DATA: &[u8] = b"Hi There";
    const HMAC_SHA256: [u8; 32] = [
        0xb0, 0x34, 0x4c, 0x61, 0xd8, 0xdb, 0x38, 0x53, 0x5c, 0xa8, 0xaf, 0xce, 0xaf, 0x0b, 0xf1,
        0x2b, 0x88, 0x1d, 0xc2, 0x00, 0xc9, 0x83, 0x3d, 0xa7, 0x26, 0xe9, 0x37, 0x6c, 0x2e, 0x32,
        0xcf, 0xf7,
    ];
    assert_eq!(HMAC_SHA256, hmac_sha256(KEY, DATA)?);
    Ok(())
}

#[test]
fn rfc4231_test_case_2() -> Result<()> {
    const KEY: &[u8] = b"Jefe";
    const DATA: &[u8] = b"what do ya want for nothing?";
    const HMAC_SHA256: [u8; 32] = [
        0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e, 0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75,
        0xc7, 0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83, 0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec,
        0x38, 0x43,
    ];
    assert_eq!(HMAC_SHA256, hmac_sha256(KEY, DATA)?);
    Ok(())
}

#[test]
fn rfc4231_test_case_3() -> Result<()> {
    const KEY: &[u8; 20] = &[0xaa; 20];
    const DATA: &[u8; 50] = &[0xdd; 50];
    const HMAC_SHA256: [u8; 32] = [
        0x77, 0x3e, 0xa9, 0x1e, 0x36, 0x80, 0x0e, 0x46, 0x85, 0x4d, 0xb8, 0xeb, 0xd0, 0x91, 0x81,
        0xa7, 0x29, 0x59, 0x09, 0x8b, 0x3e, 0xf8, 0xc1, 0x22, 0xd9, 0x63, 0x55, 0x14, 0xce, 0xd5,
        0x65, 0xfe,
    ];
    assert_eq!(HMAC_SHA256, hmac_sha256(KEY, DATA)?);
    Ok(())
}

#[test]
fn rfc4231_test_case_4() -> Result<()> {
    const KEY: &[u8; 25] = &[
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    ];
    const DATA: &[u8; 50] = &[0xcd; 50];
    const HMAC_SHA256: [u8; 32] = [
        0x82, 0x55, 0x8a, 0x38, 0x9a, 0x44, 0x3c, 0x0e, 0xa4, 0xcc, 0x81, 0x98, 0x99, 0xf2, 0x08,
        0x3a, 0x85, 0xf0, 0xfa, 0xa3, 0xe5, 0x78, 0xf8, 0x07, 0x7a, 0x2e, 0x3f, 0xf4, 0x67, 0x29,
        0x66, 0x5b,
    ];
    assert_eq!(HMAC_SHA256, hmac_sha256(KEY, DATA)?);
    Ok(())
}

/// Test with a truncation of output to 128 bits.
#[test]
fn rfc4231_test_case_5() -> Result<()> {
    const KEY: &[u8; 20] = &[0x0c; 20];
    const DATA: &[u8] = b"Test With Truncation";
    const HMAC_SHA256: [u8; 16] = [
        0xa3, 0xb6, 0x16, 0x74, 0x73, 0x10, 0x0e, 0xe0, 0x6e, 0x0c, 0x79, 0x6c, 0x29, 0x55, 0x55,
        0x2b,
    ];
    let res = hmac_sha256(KEY, DATA)?;
    assert_eq!(HMAC_SHA256, res[..16]);
    Ok(())
}

#[test]
fn rfc4231_test_case_6() -> Result<()> {
    const KEY: &[u8; 131] = &[0xaa; 131];
    const DATA: &[u8] = b"Test Using Larger Than Block-Size Key - Hash Key First";
    const HMAC_SHA256: [u8; 32] = [
        0x60, 0xe4, 0x31, 0x59, 0x1e, 0xe0, 0xb6, 0x7f, 0x0d, 0x8a, 0x26, 0xaa, 0xcb, 0xf5, 0xb7,
        0x7f, 0x8e, 0x0b, 0xc6, 0x21, 0x37, 0x28, 0xc5, 0x14, 0x05, 0x46, 0x04, 0x0f, 0x0e, 0xe3,
        0x7f, 0x54,
    ];
    assert_eq!(HMAC_SHA256, hmac_sha256(KEY, DATA)?);
    Ok(())
}

#[test]
fn rfc4231_test_case_7() -> Result<()> {
    const KEY: &[u8; 131] = &[0xaa; 131];
    const DATA: &str = "This is a test using a larger than block-size key and a larger than \
           block-size data. The key needs to be hashed before being used by the HMAC algorithm.";
    const HMAC_SHA256: [u8; 32] = [
        0x9b, 0x09, 0xff, 0xa7, 0x1b, 0x94, 0x2f, 0xcb, 0x27, 0x63, 0x5f, 0xbc, 0xd5, 0xb0, 0xe9,
        0x44, 0xbf, 0xdc, 0x63, 0x64, 0x4f, 0x07, 0x13, 0x93, 0x8a, 0x7f, 0x51, 0x53, 0x5c, 0x3a,
        0x35, 0xe2,
    ];
    assert_eq!(HMAC_SHA256, hmac_sha256(KEY, DATA.as_bytes())?);
    Ok(())
}
