/*
 * Copyright (C) 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package android.net.thread.utils;

import static android.net.thread.utils.IntegrationTestUtils.getRaPios;
import static android.net.thread.utils.IntegrationTestUtils.pollForPacket;
import static android.net.thread.utils.IntegrationTestUtils.waitFor;

import static com.android.net.module.util.NetworkStackConstants.ICMPV6_ND_OPTION_SLLA;
import static com.android.net.module.util.NetworkStackConstants.IPV6_ADDR_ALL_ROUTERS_MULTICAST;

import android.net.InetAddresses;
import android.net.MacAddress;

import com.android.net.module.util.Ipv6Utils;
import com.android.net.module.util.structs.LlaOption;
import com.android.net.module.util.structs.PrefixInformationOption;
import com.android.testutils.TapPacketReader;

import java.io.IOException;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.List;
import java.util.Random;
import java.util.concurrent.TimeoutException;

/**
 * A class that simulates a device on the infrastructure network.
 *
 * <p>This class directly interacts with the TUN interface of the test network to pretend there's a
 * device on the infrastructure network.
 */
public final class InfraNetworkDevice {
    // The MAC address of this device.
    public final MacAddress macAddr;
    // The packet reader of the TUN interface of the test network.
    public final TapPacketReader packetReader;
    // The IPv6 address generated by SLAAC for the device.
    public Inet6Address ipv6Addr;

    /**
     * Constructs an InfraNetworkDevice with the given {@link MAC address} and {@link
     * TapPacketReader}.
     *
     * @param macAddr the MAC address of the device
     * @param packetReader the packet reader of the TUN interface of the test network.
     */
    public InfraNetworkDevice(MacAddress macAddr, TapPacketReader packetReader) {
        this.macAddr = macAddr;
        this.packetReader = packetReader;
    }

    /**
     * Sends an ICMPv6 echo request message to the given {@link Inet6Address}.
     *
     * @param dstAddr the destination address of the packet.
     * @throws IOException when it fails to send the packet.
     */
    public void sendEchoRequest(Inet6Address dstAddr) throws IOException {
        ByteBuffer icmp6Packet = Ipv6Utils.buildEchoRequestPacket(ipv6Addr, dstAddr);
        packetReader.sendResponse(icmp6Packet);
    }

    /**
     * Sends an ICMPv6 Router Solicitation (RS) message to all routers on the network.
     *
     * @throws IOException when it fails to send the packet.
     */
    public void sendRsPacket() throws IOException {
        ByteBuffer slla = LlaOption.build((byte) ICMPV6_ND_OPTION_SLLA, macAddr);
        ByteBuffer rs =
                Ipv6Utils.buildRsPacket(
                        (Inet6Address) InetAddresses.parseNumericAddress("fe80::1"),
                        IPV6_ADDR_ALL_ROUTERS_MULTICAST,
                        slla);
        packetReader.sendResponse(rs);
    }

    /**
     * Runs SLAAC to generate an IPv6 address for the device.
     *
     * <p>The devices sends an RS message, processes the received RA messages and generates an IPv6
     * address if there's any available Prefix Information Option (PIO). For now it only generates
     * one address in total and doesn't track the expiration.
     *
     * @param timeoutSeconds the number of seconds to wait for.
     * @throws TimeoutException when the device fails to generate a SLAAC address in given timeout.
     */
    public void runSlaac(Duration timeout) throws TimeoutException {
        waitFor(() -> (ipv6Addr = runSlaac()) != null, timeout);
    }

    private Inet6Address runSlaac() {
        try {
            sendRsPacket();

            final byte[] raPacket = pollForPacket(packetReader, p -> !getRaPios(p).isEmpty());

            final List<PrefixInformationOption> options = getRaPios(raPacket);

            for (PrefixInformationOption pio : options) {
                if (pio.validLifetime > 0 && pio.preferredLifetime > 0) {
                    final byte[] addressBytes = pio.prefix;
                    addressBytes[addressBytes.length - 1] = (byte) (new Random()).nextInt();
                    addressBytes[addressBytes.length - 2] = (byte) (new Random()).nextInt();
                    return (Inet6Address) InetAddress.getByAddress(addressBytes);
                }
            }
        } catch (IOException e) {
            throw new IllegalStateException("Failed to generate an address by SLAAC", e);
        }
        return null;
    }
}
