/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.android.inputmethod.research;

import com.android.inputmethod.latin.CollectionUtils;

import java.util.ArrayList;

/**
 * A group of log statements related to each other.
 *
 * A LogUnit is collection of LogStatements, each of which is generated by at a particular point
 * in the code.  (There is no LogStatement class; the data is stored across the instance variables
 * here.)  A single LogUnit's statements can correspond to all the calls made while in the same
 * composing region, or all the calls between committing the last composing region, and the first
 * character of the next composing region.
 *
 * Individual statements in a log may be marked as potentially private.  If so, then they are only
 * published to a ResearchLog if the ResearchLogger determines that publishing the entire LogUnit
 * will not violate the user's privacy.  Checks for this may include whether other LogUnits have
 * been published recently, or whether the LogUnit contains numbers, etc.
 */
/* package */ class LogUnit {
    private final ArrayList<String[]> mKeysList = CollectionUtils.newArrayList();
    private final ArrayList<Object[]> mValuesList = CollectionUtils.newArrayList();
    private final ArrayList<Boolean> mIsPotentiallyPrivate = CollectionUtils.newArrayList();
    private String mWord;
    private boolean mContainsDigit;

    public void addLogStatement(final String[] keys, final Object[] values,
            final Boolean isPotentiallyPrivate) {
        mKeysList.add(keys);
        mValuesList.add(values);
        mIsPotentiallyPrivate.add(isPotentiallyPrivate);
    }

    public void publishTo(final ResearchLog researchLog, final boolean isIncludingPrivateData) {
        final int size = mKeysList.size();
        for (int i = 0; i < size; i++) {
            if (!mIsPotentiallyPrivate.get(i) || isIncludingPrivateData) {
                researchLog.outputEvent(mKeysList.get(i), mValuesList.get(i));
            }
        }
    }

    public void setWord(String word) {
        mWord = word;
    }

    public String getWord() {
        return mWord;
    }

    public boolean hasWord() {
        return mWord != null;
    }

    public void setContainsDigit() {
        mContainsDigit = true;
    }

    public boolean hasDigit() {
        return mContainsDigit;
    }

    public boolean isEmpty() {
        return mKeysList.isEmpty();
    }
}
