/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.inputmethod.keyboard.internal;

import com.android.inputmethod.latin.utils.CollectionUtils;

import java.util.HashMap;

/**
 * !!!!! DO NOT EDIT THIS FILE !!!!!
 *
 * This file is generated by tools/make-keyboard-text. The base template file is
 *   tools/make-keyboard-text/res/src/com/android/inputmethod/keyboard/internal/
 *   KeyboardTextsTable.tmpl
 *
 * This file must be updated when any text resources in keyboard layout files have been changed.
 * These text resources are referred as "!text/<resource_name>" in keyboard XML definitions,
 * and should be defined in
 *   tools/make-keyboard-text/res/values-<locale>/donottranslate-more-keys.xml
 *
 * To update this file, please run the following commands.
 *   $ cd $ANDROID_BUILD_TOP
 *   $ mmm packages/inputmethods/LatinIME/tools/make-keyboard-text
 *   $ make-keyboard-text -java packages/inputmethods/LatinIME/java
 *
 * The updated source file will be generated to the following path (this file).
 *   packages/inputmethods/LatinIME/java/src/com/android/inputmethod/keyboard/internal/
 *   KeyboardTextsTable.java
 */
public final class KeyboardTextsTable {
    // Name to index map.
    private static final HashMap<String, Integer> sNameToIndexesMap = CollectionUtils.newHashMap();
    // Language to texts table map.
    private static final HashMap<String, String[]> sLanguageToTextsTableMap =
            CollectionUtils.newHashMap();
    // TODO: Remove this variable after debugging.
    // Texts table to language maps.
    private static final HashMap<String[], String> sTextsTableToLanguageMap =
            CollectionUtils.newHashMap();

    public static String getText(final String name, final String[] textsTable) {
        final Integer indexObj = sNameToIndexesMap.get(name);
        if (indexObj == null) {
            throw new RuntimeException("Unknown text name=" + name + " language="
                    + sTextsTableToLanguageMap.get(textsTable));
        }
        final int index = indexObj;
        final String text = (index < textsTable.length) ? textsTable[index] : null;
        if (text != null) {
            return text;
        }
        // Sanity check.
        if (index >= 0 && index < LANGUAGE_DEFAULT.length) {
            return LANGUAGE_DEFAULT[index];
        }
        // Throw exception for debugging purpose.
        throw new RuntimeException("Illegal index=" + index + " for name=" + name
                + " language=" + sTextsTableToLanguageMap.get(textsTable));
    }

    public static String[] getTextsTable(final String language) {
        final String[] textsTable = sLanguageToTextsTableMap.get(language);
        return textsTable != null ? textsTable : LANGUAGE_DEFAULT;
    }

    private static final String[] NAMES = {
    //  /* index:histogram */ "name",
        /*   0:30 */ "more_keys_for_a",
        /*   1:30 */ "more_keys_for_o",
        /*   2:28 */ "more_keys_for_u",
        /*   3:27 */ "more_keys_for_e",
        /*   4:26 */ "more_keys_for_i",
        /*   5:23 */ "double_quotes",
        /*   6:22 */ "single_quotes",
        /*   7:21 */ "more_keys_for_c",
        /*   8:20 */ "more_keys_for_s",
        /*   9:20 */ "more_keys_for_n",
        /*  10:20 */ "label_to_alpha_key",
        /*  11:14 */ "more_keys_for_y",
        /*  12:13 */ "more_keys_for_d",
        /*  13:12 */ "more_keys_for_z",
        /*  14:10 */ "more_keys_for_t",
        /*  15:10 */ "more_keys_for_l",
        /*  16: 9 */ "more_keys_for_g",
        /*  17: 9 */ "single_angle_quotes",
        /*  18: 9 */ "double_angle_quotes",
        /*  19: 9 */ "keylabel_for_currency",
        /*  20: 8 */ "more_keys_for_r",
        /*  21: 6 */ "more_keys_for_k",
        /*  22: 6 */ "keylabel_for_nordic_row1_11",
        /*  23: 6 */ "keylabel_for_nordic_row2_10",
        /*  24: 6 */ "keylabel_for_nordic_row2_11",
        /*  25: 6 */ "more_keys_for_cyrillic_ie",
        /*  26: 5 */ "more_keys_for_nordic_row2_10",
        /*  27: 5 */ "keylabel_for_east_slavic_row1_9",
        /*  28: 5 */ "keylabel_for_east_slavic_row2_2",
        /*  29: 5 */ "keylabel_for_east_slavic_row2_11",
        /*  30: 5 */ "keylabel_for_east_slavic_row3_5",
        /*  31: 5 */ "more_keys_for_cyrillic_soft_sign",
        /*  32: 5 */ "more_keys_for_punctuation",
        /*  33: 4 */ "more_keys_for_nordic_row2_11",
        /*  34: 4 */ "keylabel_for_symbols_1",
        /*  35: 4 */ "keylabel_for_symbols_2",
        /*  36: 4 */ "keylabel_for_symbols_3",
        /*  37: 4 */ "keylabel_for_symbols_4",
        /*  38: 4 */ "keylabel_for_symbols_5",
        /*  39: 4 */ "keylabel_for_symbols_6",
        /*  40: 4 */ "keylabel_for_symbols_7",
        /*  41: 4 */ "keylabel_for_symbols_8",
        /*  42: 4 */ "keylabel_for_symbols_9",
        /*  43: 4 */ "keylabel_for_symbols_0",
        /*  44: 4 */ "label_to_symbol_key",
        /*  45: 4 */ "label_to_symbol_with_microphone_key",
        /*  46: 4 */ "additional_more_keys_for_symbols_1",
        /*  47: 4 */ "additional_more_keys_for_symbols_2",
        /*  48: 4 */ "additional_more_keys_for_symbols_3",
        /*  49: 4 */ "additional_more_keys_for_symbols_4",
        /*  50: 4 */ "additional_more_keys_for_symbols_5",
        /*  51: 4 */ "additional_more_keys_for_symbols_6",
        /*  52: 4 */ "additional_more_keys_for_symbols_7",
        /*  53: 4 */ "additional_more_keys_for_symbols_8",
        /*  54: 4 */ "additional_more_keys_for_symbols_9",
        /*  55: 4 */ "additional_more_keys_for_symbols_0",
        /*  56: 3 */ "more_keys_for_star",
        /*  57: 3 */ "keyspec_left_parenthesis",
        /*  58: 3 */ "keyspec_right_parenthesis",
        /*  59: 3 */ "keyspec_left_square_bracket",
        /*  60: 3 */ "keyspec_right_square_bracket",
        /*  61: 3 */ "keyspec_left_curly_bracket",
        /*  62: 3 */ "keyspec_right_curly_bracket",
        /*  63: 3 */ "keyspec_less_than",
        /*  64: 3 */ "keyspec_greater_than",
        /*  65: 3 */ "keyspec_less_than_equal",
        /*  66: 3 */ "keyspec_greater_than_equal",
        /*  67: 3 */ "keyspec_left_double_angle_quote",
        /*  68: 3 */ "keyspec_right_double_angle_quote",
        /*  69: 3 */ "keyspec_left_single_angle_quote",
        /*  70: 3 */ "keyspec_right_single_angle_quote",
        /*  71: 3 */ "keylabel_for_tablet_comma",
        /*  72: 3 */ "more_keys_for_tablet_period",
        /*  73: 3 */ "more_keys_for_question",
        /*  74: 2 */ "more_keys_for_h",
        /*  75: 2 */ "more_keys_for_w",
        /*  76: 2 */ "more_keys_for_cyrillic_u",
        /*  77: 2 */ "more_keys_for_cyrillic_en",
        /*  78: 2 */ "more_keys_for_cyrillic_ghe",
        /*  79: 2 */ "more_keys_for_east_slavic_row2_2",
        /*  80: 2 */ "more_keys_for_cyrillic_o",
        /*  81: 2 */ "keylabel_for_south_slavic_row1_6",
        /*  82: 2 */ "keylabel_for_south_slavic_row2_11",
        /*  83: 2 */ "keylabel_for_south_slavic_row3_1",
        /*  84: 2 */ "keylabel_for_south_slavic_row3_8",
        /*  85: 2 */ "more_keys_for_cyrillic_i",
        /*  86: 2 */ "keylabel_for_swiss_row1_11",
        /*  87: 2 */ "keylabel_for_swiss_row2_10",
        /*  88: 2 */ "keylabel_for_swiss_row2_11",
        /*  89: 2 */ "more_keys_for_swiss_row1_11",
        /*  90: 2 */ "more_keys_for_swiss_row2_10",
        /*  91: 2 */ "more_keys_for_swiss_row2_11",
        /*  92: 2 */ "keylabel_for_spanish_row2_10",
        /*  93: 2 */ "more_keys_for_bullet",
        /*  94: 2 */ "more_keys_for_left_parenthesis",
        /*  95: 2 */ "more_keys_for_right_parenthesis",
        /*  96: 2 */ "more_keys_for_arabic_diacritics",
        /*  97: 2 */ "keylabel_for_comma",
        /*  98: 2 */ "more_keys_for_comma",
        /*  99: 2 */ "keyhintlabel_for_tablet_comma",
        /* 100: 2 */ "more_keys_for_tablet_comma",
        /* 101: 2 */ "keyhintlabel_for_period",
        /* 102: 2 */ "more_keys_for_period",
        /* 103: 2 */ "keyhintlabel_for_tablet_period",
        /* 104: 2 */ "keylabel_for_symbols_question",
        /* 105: 2 */ "keylabel_for_symbols_semicolon",
        /* 106: 2 */ "keylabel_for_symbols_percent",
        /* 107: 2 */ "more_keys_for_symbols_semicolon",
        /* 108: 2 */ "more_keys_for_symbols_percent",
        /* 109: 1 */ "more_keys_for_v",
        /* 110: 1 */ "more_keys_for_j",
        /* 111: 1 */ "more_keys_for_cyrillic_ka",
        /* 112: 1 */ "more_keys_for_cyrillic_a",
        /* 113: 1 */ "more_keys_for_east_slavic_row2_11",
        /* 114: 1 */ "more_keys_for_currency_dollar",
        /* 115: 1 */ "more_keys_for_tablet_punctuation",
        /* 116: 1 */ "more_keys_for_plus",
        /* 117: 1 */ "more_keys_for_less_than",
        /* 118: 1 */ "more_keys_for_greater_than",
        /* 119: 1 */ "keylabel_for_period",
        /* 120: 1 */ "keylabel_for_tablet_period",
        /* 121: 1 */ "more_keys_for_exclamation",
        /* 122: 1 */ "more_keys_for_q",
        /* 123: 1 */ "more_keys_for_x",
        /* 124: 1 */ "keylabel_for_q",
        /* 125: 1 */ "keylabel_for_w",
        /* 126: 1 */ "keylabel_for_y",
        /* 127: 1 */ "keylabel_for_x",
        /* 128: 0 */ "more_keys_for_currency",
        /* 129: 0 */ "more_keys_for_symbols_1",
        /* 130: 0 */ "more_keys_for_symbols_2",
        /* 131: 0 */ "more_keys_for_symbols_3",
        /* 132: 0 */ "more_keys_for_symbols_4",
        /* 133: 0 */ "more_keys_for_symbols_5",
        /* 134: 0 */ "more_keys_for_symbols_6",
        /* 135: 0 */ "more_keys_for_symbols_7",
        /* 136: 0 */ "more_keys_for_symbols_8",
        /* 137: 0 */ "more_keys_for_symbols_9",
        /* 138: 0 */ "more_keys_for_symbols_0",
        /* 139: 0 */ "more_keys_for_am_pm",
        /* 140: 0 */ "settings_as_more_key",
        /* 141: 0 */ "shortcut_as_more_key",
        /* 142: 0 */ "action_next_as_more_key",
        /* 143: 0 */ "action_previous_as_more_key",
        /* 144: 0 */ "label_to_more_symbol_key",
        /* 145: 0 */ "label_to_more_symbol_for_tablet_key",
        /* 146: 0 */ "label_to_phone_numeric_key",
        /* 147: 0 */ "label_to_phone_symbols_key",
        /* 148: 0 */ "label_time_am",
        /* 149: 0 */ "label_time_pm",
        /* 150: 0 */ "keylabel_for_popular_domain",
        /* 151: 0 */ "more_keys_for_popular_domain",
        /* 152: 0 */ "keyspecs_for_left_parenthesis_more_keys",
        /* 153: 0 */ "keyspecs_for_right_parenthesis_more_keys",
        /* 154: 0 */ "single_laqm_raqm",
        /* 155: 0 */ "single_raqm_laqm",
        /* 156: 0 */ "double_laqm_raqm",
        /* 157: 0 */ "double_raqm_laqm",
        /* 158: 0 */ "single_lqm_rqm",
        /* 159: 0 */ "single_9qm_lqm",
        /* 160: 0 */ "single_9qm_rqm",
        /* 161: 0 */ "single_rqm_9qm",
        /* 162: 0 */ "double_lqm_rqm",
        /* 163: 0 */ "double_9qm_lqm",
        /* 164: 0 */ "double_9qm_rqm",
        /* 165: 0 */ "double_rqm_9qm",
        /* 166: 0 */ "more_keys_for_single_quote",
        /* 167: 0 */ "more_keys_for_double_quote",
        /* 168: 0 */ "more_keys_for_tablet_double_quote",
        /* 169: 0 */ "emoji_key_as_more_key",
    };

    private static final String EMPTY = "";

    /* Default texts */
    private static final String[] LANGUAGE_DEFAULT = {
        /* more_keys_for_a ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_lqm_rqm",
        /* single_quotes */ "!text/single_lqm_rqm",
        /* more_keys_for_c ~ */
        EMPTY, EMPTY, EMPTY,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        /* label_to_alpha_key */ "ABC",
        /* more_keys_for_y ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ more_keys_for_g */
        /* single_angle_quotes */ "!text/single_laqm_raqm",
        /* double_angle_quotes */ "!text/double_laqm_raqm",
        /* keylabel_for_currency */ "$",
        /* more_keys_for_r ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ more_keys_for_cyrillic_soft_sign */
        /* more_keys_for_punctuation */ "!fixedColumnOrder!8,;,/,!text/keyspec_left_parenthesis,!text/keyspec_right_parenthesis,#,!,\\,,?,&,\\%,+,\",-,:,',@",
        /* more_keys_for_nordic_row2_11 */ EMPTY,
        /* keylabel_for_symbols_1 */ "1",
        /* keylabel_for_symbols_2 */ "2",
        /* keylabel_for_symbols_3 */ "3",
        /* keylabel_for_symbols_4 */ "4",
        /* keylabel_for_symbols_5 */ "5",
        /* keylabel_for_symbols_6 */ "6",
        /* keylabel_for_symbols_7 */ "7",
        /* keylabel_for_symbols_8 */ "8",
        /* keylabel_for_symbols_9 */ "9",
        /* keylabel_for_symbols_0 */ "0",
        // Label for "switch to symbols" key.
        /* label_to_symbol_key */ "?123",
        // Label for "switch to symbols with microphone" key. This string shouldn't include the "mic"
        // part because it'll be appended by the code.
        /* label_to_symbol_with_microphone_key */ "123",
        /* additional_more_keys_for_symbols_1 ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ additional_more_keys_for_symbols_0 */
        // U+2020: "†" DAGGER
        // U+2021: "‡" DOUBLE DAGGER
        // U+2605: "★" BLACK STAR
        /* more_keys_for_star */ "\u2020,\u2021,\u2605",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        /* keyspec_left_parenthesis */ "(",
        /* keyspec_right_parenthesis */ ")",
        /* keyspec_left_square_bracket */ "[",
        /* keyspec_right_square_bracket */ "]",
        /* keyspec_left_curly_bracket */ "{",
        /* keyspec_right_curly_bracket */ "}",
        /* keyspec_less_than */ "<",
        /* keyspec_greater_than */ ">",
        /* keyspec_less_than_equal */ "\u2264",
        /* keyspec_greater_than_equal */ "\u2265",
        /* keyspec_left_double_angle_quote */ "\u00AB",
        /* keyspec_right_double_angle_quote */ "\u00BB",
        /* keyspec_left_single_angle_quote */ "\u2039",
        /* keyspec_right_single_angle_quote */ "\u203A",
        /* keylabel_for_tablet_comma */ ",",
        /* more_keys_for_tablet_period */ "!text/more_keys_for_tablet_punctuation",
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* more_keys_for_question */ "\u00BF",
        /* more_keys_for_h ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ more_keys_for_swiss_row2_11 */
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* keylabel_for_spanish_row2_10 */ "\u00F1",
        // U+266A: "♪" EIGHTH NOTE
        // U+2665: "♥" BLACK HEART SUIT
        // U+2660: "♠" BLACK SPADE SUIT
        // U+2666: "♦" BLACK DIAMOND SUIT
        // U+2663: "♣" BLACK CLUB SUIT
        /* more_keys_for_bullet */ "\u266A,\u2665,\u2660,\u2666,\u2663",
        /* more_keys_for_left_parenthesis */ "!fixedColumnOrder!3,!text/keyspecs_for_left_parenthesis_more_keys",
        /* more_keys_for_right_parenthesis */ "!fixedColumnOrder!3,!text/keyspecs_for_right_parenthesis_more_keys",
        /* more_keys_for_arabic_diacritics */ EMPTY,
        // Comma key
        /* keylabel_for_comma */ ",",
        /* more_keys_for_comma ~ */
        EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ keyhintlabel_for_period */
        /* more_keys_for_period */ "!text/more_keys_for_punctuation",
        /* keyhintlabel_for_tablet_period */ EMPTY,
        /* keylabel_for_symbols_question */ "?",
        /* keylabel_for_symbols_semicolon */ ";",
        /* keylabel_for_symbols_percent */ "%",
        /* more_keys_for_symbols_semicolon */ EMPTY,
        // U+2030: "‰" PER MILLE SIGN
        /* more_keys_for_symbols_percent */ "\u2030",
        /* more_keys_for_v ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ more_keys_for_east_slavic_row2_11 */
        // U+00A2: "¢" CENT SIGN
        // U+00A3: "£" POUND SIGN
        // U+20AC: "€" EURO SIGN
        // U+00A5: "¥" YEN SIGN
        // U+20B1: "₱" PESO SIGN
        /* more_keys_for_currency_dollar */ "\u00A2,\u00A3,\u20AC,\u00A5,\u20B1",
        /* more_keys_for_tablet_punctuation */ "!fixedColumnOrder!7,;,/,!text/keyspec_left_parenthesis,!text/keyspec_right_parenthesis,#,',\\,,&,\\%,+,\",-,:,@",
        // U+00B1: "±" PLUS-MINUS SIGN
        /* more_keys_for_plus */ "\u00B1",
        /* more_keys_for_less_than */ "!fixedColumnOrder!3,!text/keyspec_left_single_angle_quote,!text/keyspec_less_than_equal,!text/keyspec_left_double_angle_quote",
        /* more_keys_for_greater_than */ "!fixedColumnOrder!3,!text/keyspec_right_single_angle_quote,!text/keyspec_greater_than_equal,!text/keyspec_right_double_angle_quote",
        // Period key
        /* keylabel_for_period */ ".",
        /* keylabel_for_tablet_period */ ".",
        // U+00A1: "¡" INVERTED EXCLAMATION MARK
        /* more_keys_for_exclamation */ "\u00A1",
        /* more_keys_for_q */ EMPTY,
        /* more_keys_for_x */ EMPTY,
        /* keylabel_for_q */ "q",
        /* keylabel_for_w */ "w",
        /* keylabel_for_y */ "y",
        /* keylabel_for_x */ "x",
        /* more_keys_for_currency */ "$,\u00A2,\u20AC,\u00A3,\u00A5,\u20B1",
        // U+00B9: "¹" SUPERSCRIPT ONE
        // U+00BD: "½" VULGAR FRACTION ONE HALF
        // U+2153: "⅓" VULGAR FRACTION ONE THIRD
        // U+00BC: "¼" VULGAR FRACTION ONE QUARTER
        // U+215B: "⅛" VULGAR FRACTION ONE EIGHTH
        /* more_keys_for_symbols_1 */ "\u00B9,\u00BD,\u2153,\u00BC,\u215B",
        // U+00B2: "²" SUPERSCRIPT TWO
        // U+2154: "⅔" VULGAR FRACTION TWO THIRDS
        /* more_keys_for_symbols_2 */ "\u00B2,\u2154",
        // U+00B3: "³" SUPERSCRIPT THREE
        // U+00BE: "¾" VULGAR FRACTION THREE QUARTERS
        // U+215C: "⅜" VULGAR FRACTION THREE EIGHTHS
        /* more_keys_for_symbols_3 */ "\u00B3,\u00BE,\u215C",
        // U+2074: "⁴" SUPERSCRIPT FOUR
        /* more_keys_for_symbols_4 */ "\u2074",
        // U+215D: "⅝" VULGAR FRACTION FIVE EIGHTHS
        /* more_keys_for_symbols_5 */ "\u215D",
        /* more_keys_for_symbols_6 */ EMPTY,
        // U+215E: "⅞" VULGAR FRACTION SEVEN EIGHTHS
        /* more_keys_for_symbols_7 */ "\u215E",
        /* more_keys_for_symbols_8 */ EMPTY,
        /* more_keys_for_symbols_9 */ EMPTY,
        // U+207F: "ⁿ" SUPERSCRIPT LATIN SMALL LETTER N
        // U+2205: "∅" EMPTY SET
        /* more_keys_for_symbols_0 */ "\u207F,\u2205",
        /* more_keys_for_am_pm */ "!fixedColumnOrder!2,!hasLabels!,!text/label_time_am,!text/label_time_pm",
        /* settings_as_more_key */ "!icon/settings_key|!code/key_settings",
        /* shortcut_as_more_key */ "!icon/shortcut_key|!code/key_shortcut",
        /* action_next_as_more_key */ "!hasLabels!,!text/label_next_key|!code/key_action_next",
        /* action_previous_as_more_key */ "!hasLabels!,!text/label_previous_key|!code/key_action_previous",
        // Label for "switch to more symbol" modifier key ("= \ <"). Must be short to fit on key!
        /* label_to_more_symbol_key */ "= \\\\ <",
        // Label for "switch to more symbol" modifier key on tablets.  Must be short to fit on key!
        /* label_to_more_symbol_for_tablet_key */ "~ [ <",
        // Label for "switch to phone numeric" key.  Must be short to fit on key!
        /* label_to_phone_numeric_key */ "123",
        // Label for "switch to phone symbols" key.  Must be short to fit on key!
        // U+FF0A: "＊" FULLWIDTH ASTERISK
        // U+FF03: "＃" FULLWIDTH NUMBER SIGN
        /* label_to_phone_symbols_key */ "\uFF0A\uFF03",
        // Key label for "ante meridiem"
        /* label_time_am */ "AM",
        // Key label for "post meridiem"
        /* label_time_pm */ "PM",
        /* keylabel_for_popular_domain */ ".com",
        // popular web domains for the locale - most popular, displayed on the keyboard
        /* more_keys_for_popular_domain */ "!hasLabels!,.net,.org,.gov,.edu",
        /* keyspecs_for_left_parenthesis_more_keys */ "!text/keyspec_less_than,!text/keyspec_left_curly_bracket,!text/keyspec_left_square_bracket",
        /* keyspecs_for_right_parenthesis_more_keys */ "!text/keyspec_greater_than,!text/keyspec_right_curly_bracket,!text/keyspec_right_square_bracket",
        // The following characters don't need BIDI mirroring.
        // U+2018: "‘" LEFT SINGLE QUOTATION MARK
        // U+2019: "’" RIGHT SINGLE QUOTATION MARK
        // U+201A: "‚" SINGLE LOW-9 QUOTATION MARK
        // U+201C: "“" LEFT DOUBLE QUOTATION MARK
        // U+201D: "”" RIGHT DOUBLE QUOTATION MARK
        // U+201E: "„" DOUBLE LOW-9 QUOTATION MARK
        // Abbreviations are:
        // laqm: LEFT-POINTING ANGLE QUOTATION MARK
        // raqm: RIGHT-POINTING ANGLE QUOTATION MARK
        // lqm: LEFT QUOTATION MARK
        // rqm: RIGHT QUOTATION MARK
        // 9qm: LOW-9 QUOTATION MARK
        // The following each quotation mark pair consist of
        // <opening quotation mark>, <closing quotation mark>
        // and is named after (single|double)_<opening quotation mark>_<closing quotation mark>.
        /* single_laqm_raqm */ "!text/keyspec_left_single_angle_quote,!text/keyspec_right_single_angle_quote",
        /* single_raqm_laqm */ "!text/keyspec_right_single_angle_quote,!text/keyspec_left_single_angle_quote",
        /* double_laqm_raqm */ "!text/keyspec_left_double_angle_quote,!text/keyspec_right_double_angle_quote",
        /* double_raqm_laqm */ "!text/keyspec_right_double_angle_quote,!text/keyspec_left_double_angle_quote",
        // The following each quotation mark triplet consists of
        // <another quotation mark>, <opening quotation mark>, <closing quotation mark>
        // and is named after (single|double)_<opening quotation mark>_<closing quotation mark>.
        /* single_lqm_rqm */ "\u201A,\u2018,\u2019",
        /* single_9qm_lqm */ "\u2019,\u201A,\u2018",
        /* single_9qm_rqm */ "\u2018,\u201A,\u2019",
        /* single_rqm_9qm */ "\u2018,\u2019,\u201A",
        /* double_lqm_rqm */ "\u201E,\u201C,\u201D",
        /* double_9qm_lqm */ "\u201D,\u201E,\u201C",
        /* double_9qm_rqm */ "\u201C,\u201E,\u201D",
        /* double_rqm_9qm */ "\u201C,\u201D,\u201E",
        /* more_keys_for_single_quote */ "!fixedColumnOrder!5,!text/single_quotes,!text/single_angle_quotes",
        /* more_keys_for_double_quote */ "!fixedColumnOrder!5,!text/double_quotes,!text/double_angle_quotes",
        /* more_keys_for_tablet_double_quote */ "!fixedColumnOrder!6,!text/double_quotes,!text/single_quotes,!text/double_angle_quotes,!text/single_angle_quotes",
        /* emoji_key_as_more_key */ "!icon/emoji_key|!code/key_emoji",
    };

    /* Language af: Afrikaans */
    private static final String[] LANGUAGE_af = {
        // This is the same as Dutch except more keys of y and demoting vowels with diaeresis.
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E1,\u00E2,\u00E4,\u00E0,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F3,\u00F4,\u00F6,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FB,\u00FC,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* more_keys_for_i */ "\u00ED,\u00EC,\u00EF,\u00EE,\u012F,\u012B,\u0133",
        /* double_quotes ~ */
        null, null, null, null,
        /* ~ more_keys_for_s */
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u00F1,\u0144",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* more_keys_for_y */ "\u00FD,\u0133",
    };

    /* Language ar: Arabic */
    private static final String[] LANGUAGE_ar = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0623: "ا" ARABIC LETTER ALEF
        // U+200C: ZERO WIDTH NON-JOINER
        // U+0628: "ب" ARABIC LETTER BEH
        // U+062C: "پ" ARABIC LETTER PEH
        /* label_to_alpha_key */ "\u0623\u200C\u0628\u200C\u062C",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ more_keys_for_cyrillic_soft_sign */
        /* more_keys_for_punctuation */ "!fixedColumnOrder!8,\",\',#,-,:,!,\u060C,\u061F,@,&,\\%,+,\u061B,/,(|),)|(",
        /* more_keys_for_nordic_row2_11 */ null,
        // U+0661: "١" ARABIC-INDIC DIGIT ONE
        /* keylabel_for_symbols_1 */ "\u0661",
        // U+0662: "٢" ARABIC-INDIC DIGIT TWO
        /* keylabel_for_symbols_2 */ "\u0662",
        // U+0663: "٣" ARABIC-INDIC DIGIT THREE
        /* keylabel_for_symbols_3 */ "\u0663",
        // U+0664: "٤" ARABIC-INDIC DIGIT FOUR
        /* keylabel_for_symbols_4 */ "\u0664",
        // U+0665: "٥" ARABIC-INDIC DIGIT FIVE
        /* keylabel_for_symbols_5 */ "\u0665",
        // U+0666: "٦" ARABIC-INDIC DIGIT SIX
        /* keylabel_for_symbols_6 */ "\u0666",
        // U+0667: "٧" ARABIC-INDIC DIGIT SEVEN
        /* keylabel_for_symbols_7 */ "\u0667",
        // U+0668: "٨" ARABIC-INDIC DIGIT EIGHT
        /* keylabel_for_symbols_8 */ "\u0668",
        // U+0669: "٩" ARABIC-INDIC DIGIT NINE
        /* keylabel_for_symbols_9 */ "\u0669",
        // U+0660: "٠" ARABIC-INDIC DIGIT ZERO
        /* keylabel_for_symbols_0 */ "\u0660",
        // Label for "switch to symbols" key.
        // U+061F: "؟" ARABIC QUESTION MARK
        /* label_to_symbol_key */ "\u0663\u0662\u0661\u061F",
        // Label for "switch to symbols with microphone" key. This string shouldn't include the "mic"
        // part because it'll be appended by the code.
        /* label_to_symbol_with_microphone_key */ "\u0663\u0662\u0661",
        /* additional_more_keys_for_symbols_1 */ "1",
        /* additional_more_keys_for_symbols_2 */ "2",
        /* additional_more_keys_for_symbols_3 */ "3",
        /* additional_more_keys_for_symbols_4 */ "4",
        /* additional_more_keys_for_symbols_5 */ "5",
        /* additional_more_keys_for_symbols_6 */ "6",
        /* additional_more_keys_for_symbols_7 */ "7",
        /* additional_more_keys_for_symbols_8 */ "8",
        /* additional_more_keys_for_symbols_9 */ "9",
        // U+066B: "٫" ARABIC DECIMAL SEPARATOR
        // U+066C: "٬" ARABIC THOUSANDS SEPARATOR
        /* additional_more_keys_for_symbols_0 */ "0,\u066B,\u066C",
        // U+2605: "★" BLACK STAR
        // U+066D: "٭" ARABIC FIVE POINTED STAR
        /* more_keys_for_star */ "\u2605,\u066D",
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        /* keyspec_left_parenthesis */ "(|)",
        /* keyspec_right_parenthesis */ ")|(",
        /* keyspec_left_square_bracket */ "[|]",
        /* keyspec_right_square_bracket */ "]|[",
        /* keyspec_left_curly_bracket */ "{|}",
        /* keyspec_right_curly_bracket */ "}|{",
        /* keyspec_less_than */ "<|>",
        /* keyspec_greater_than */ ">|<",
        /* keyspec_less_than_equal */ "\u2264|\u2265",
        /* keyspec_greater_than_equal */ "\u2265|\u2264",
        /* keyspec_left_double_angle_quote */ "\u00AB|\u00BB",
        /* keyspec_right_double_angle_quote */ "\u00BB|\u00AB",
        /* keyspec_left_single_angle_quote */ "\u2039|\u203A",
        /* keyspec_right_single_angle_quote */ "\u203A|\u2039",
        // U+061F: "؟" ARABIC QUESTION MARK
        // U+060C: "،" ARABIC COMMA
        // U+061B: "؛" ARABIC SEMICOLON
        /* keylabel_for_tablet_comma */ "\u060C",
        /* more_keys_for_tablet_period */ "!text/more_keys_for_arabic_diacritics",
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* more_keys_for_question */ "?,\u00BF",
        /* more_keys_for_h ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null,
        /* ~ keylabel_for_spanish_row2_10 */
        // U+266A: "♪" EIGHTH NOTE
        /* more_keys_for_bullet */ "\u266A",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+FD3E: "﴾" ORNATE LEFT PARENTHESIS
        // U+FD3F: "﴿" ORNATE RIGHT PARENTHESIS
        /* more_keys_for_left_parenthesis */ "!fixedColumnOrder!4,\uFD3E|\uFD3F,!text/keyspecs_for_left_parenthesis_more_keys",
        /* more_keys_for_right_parenthesis */ "!fixedColumnOrder!4,\uFD3F|\uFD3E,!text/keyspecs_for_right_parenthesis_more_keys",
        // U+0655: "ٕ" ARABIC HAMZA BELOW
        // U+0654: "ٔ" ARABIC HAMZA ABOVE
        // U+0652: "ْ" ARABIC SUKUN
        // U+064D: "ٍ" ARABIC KASRATAN
        // U+064C: "ٌ" ARABIC DAMMATAN
        // U+064B: "ً" ARABIC FATHATAN
        // U+0651: "ّ" ARABIC SHADDA
        // U+0656: "ٖ" ARABIC SUBSCRIPT ALEF
        // U+0670: "ٰ" ARABIC LETTER SUPERSCRIPT ALEF
        // U+0653: "ٓ" ARABIC MADDAH ABOVE
        // U+0650: "ِ" ARABIC KASRA
        // U+064F: "ُ" ARABIC DAMMA
        // U+064E: "َ" ARABIC FATHA
        // U+0640: "ـ" ARABIC TATWEEL
        // In order to make Tatweel easily distinguishable from other punctuations, we use consecutive Tatweels only for its displayed label.
        // Note: The space character is needed as a preceding letter to draw Arabic diacritics characters correctly.
        /* more_keys_for_arabic_diacritics */ "!fixedColumnOrder!7, \u0655|\u0655, \u0654|\u0654, \u0652|\u0652, \u064D|\u064D, \u064C|\u064C, \u064B|\u064B, \u0651|\u0651, \u0656|\u0656, \u0670|\u0670, \u0653|\u0653, \u0650|\u0650, \u064F|\u064F, \u064E|\u064E,\u0640\u0640\u0640|\u0640",
        // U+060C: "،" ARABIC COMMA
        /* keylabel_for_comma */ "\u060C",
        /* more_keys_for_comma */ "\\,",
        /* keyhintlabel_for_tablet_comma */ "\u061F",
        /* more_keys_for_tablet_comma */ "!fixedColumnOrder!4,:,!,\u061F,\u061B,-,/,\",\'",
        // U+0651: "ّ" ARABIC SHADDA
        /* keyhintlabel_for_period */ "\u0651",
        /* more_keys_for_period */ "!text/more_keys_for_arabic_diacritics",
        /* keyhintlabel_for_tablet_period */ "\u0651",
        /* keylabel_for_symbols_question */ "\u061F",
        /* keylabel_for_symbols_semicolon */ "\u061B",
        // U+066A: "٪" ARABIC PERCENT SIGN
        /* keylabel_for_symbols_percent */ "\u066A",
        /* more_keys_for_symbols_semicolon */ ";",
        // U+2030: "‰" PER MILLE SIGN
        /* more_keys_for_symbols_percent */ "\\%,\u2030",
    };

    /* Language az_AZ: Azerbaijani (Azerbaijan) */
    private static final String[] LANGUAGE_az_AZ = {
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        /* more_keys_for_a */ "\u00E2",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F6,\u00F4,\u0153,\u00F2,\u00F3,\u00F5,\u00F8,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FC,\u00FB,\u00F9,\u00FA,\u016B",
        // U+0259: "ə" LATIN SMALL LETTER SCHWA
        /* more_keys_for_e */ "\u0259",
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u0131,\u00EE,\u00EF,\u00EC,\u00ED,\u012F,\u012B",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u00E7,\u0107,\u010D",
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* more_keys_for_s */ "\u015F,\u00DF,\u015B,\u0161",
        /* more_keys_for_n ~ */
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_l */
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* more_keys_for_g */ "\u011F",
    };

    /* Language be_BY: Belarusian (Belarus) */
    private static final String[] LANGUAGE_be_BY = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c ~ */
        null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ keylabel_for_nordic_row2_11 */
        // U+0451: "ё" CYRILLIC SMALL LETTER IO
        /* more_keys_for_cyrillic_ie */ "\u0451",
        /* more_keys_for_nordic_row2_10 */ null,
        // U+045E: "ў" CYRILLIC SMALL LETTER SHORT U
        /* keylabel_for_east_slavic_row1_9 */ "\u045E",
        // U+044B: "ы" CYRILLIC SMALL LETTER YERU
        /* keylabel_for_east_slavic_row2_2 */ "\u044B",
        // U+044D: "э" CYRILLIC SMALL LETTER E
        /* keylabel_for_east_slavic_row2_11 */ "\u044D",
        // U+0456: "і" CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
        /* keylabel_for_east_slavic_row3_5 */ "\u0456",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* more_keys_for_cyrillic_soft_sign */ "\u044A",
    };

    /* Language bg: Bulgarian */
    private static final String[] LANGUAGE_bg = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        // single_quotes of Bulgarian is default single_quotes_right_left.
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes ~ */
        null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
    };

    /* Language ca: Catalan */
    private static final String[] LANGUAGE_ca = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* more_keys_for_a */ "\u00E0,\u00E1,\u00E4,\u00E2,\u00E3,\u00E5,\u0105,\u00E6,\u0101,\u00AA",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* more_keys_for_o */ "\u00F2,\u00F3,\u00F6,\u00F4,\u00F5,\u00F8,\u0153,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E8,\u00E9,\u00EB,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u00E7,\u0107,\u010D",
        /* more_keys_for_s */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u00F1,\u0144",
        /* label_to_alpha_key ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_t */
        // U+00B7: "·" MIDDLE DOT
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* more_keys_for_l */ "l\u00B7l,\u0142",
        /* more_keys_for_g ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null,
        /* ~ more_keys_for_cyrillic_soft_sign */
        // U+00B7: "·" MIDDLE DOT
        /* more_keys_for_punctuation */ "!fixedColumnOrder!9,;,/,(,),#,\u00B7,!,\\,,?,&,\\%,+,\",-,:,',@",
        /* more_keys_for_nordic_row2_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_swiss_row2_11 */
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* keylabel_for_spanish_row2_10 */ "\u00E7",
        /* more_keys_for_bullet ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_currency_dollar */
        /* more_keys_for_tablet_punctuation */ "!fixedColumnOrder!8,;,/,(,),#,\u00B7,',\\,,&,\\%,+,\",-,:,@",
    };

    /* Language cs: Czech */
    private static final String[] LANGUAGE_cs = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E1,\u00E0,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u016F,\u00FB,\u00FC,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u011B,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00ED,\u00EE,\u00EF,\u00EC,\u012F,\u012B",
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* more_keys_for_c */ "\u010D,\u00E7,\u0107",
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        /* more_keys_for_s */ "\u0161,\u00DF,\u015B",
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u0148,\u00F1,\u0144",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* more_keys_for_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* more_keys_for_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        /* more_keys_for_z */ "\u017E,\u017A,\u017C",
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        /* more_keys_for_t */ "\u0165",
        /* more_keys_for_l */ null,
        /* more_keys_for_g */ null,
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keylabel_for_currency */ null,
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        /* more_keys_for_r */ "\u0159",
    };

    /* Language da: Danish */
    private static final String[] LANGUAGE_da = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E1,\u00E4,\u00E0,\u00E2,\u00E3,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F3,\u00F4,\u00F2,\u00F5,\u0153,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FC,\u00FB,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        /* more_keys_for_e */ "\u00E9,\u00EB",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        /* more_keys_for_i */ "\u00ED,\u00EF",
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* more_keys_for_s */ "\u00DF,\u015B,\u0161",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u00F1,\u0144",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* more_keys_for_y */ "\u00FD,\u00FF",
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        /* more_keys_for_d */ "\u00F0",
        /* more_keys_for_z */ null,
        /* more_keys_for_t */ null,
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* more_keys_for_l */ "\u0142",
        /* more_keys_for_g */ null,
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keylabel_for_currency ~ */
        null, null, null,
        /* ~ more_keys_for_k */
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        /* keylabel_for_nordic_row1_11 */ "\u00E5",
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* keylabel_for_nordic_row2_10 */ "\u00E6",
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* keylabel_for_nordic_row2_11 */ "\u00F8",
        /* more_keys_for_cyrillic_ie */ null,
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* more_keys_for_nordic_row2_10 */ "\u00E4",
        /* keylabel_for_east_slavic_row1_9 ~ */
        null, null, null, null, null, null,
        /* ~ more_keys_for_punctuation */
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* more_keys_for_nordic_row2_11 */ "\u00F6",
    };

    /* Language de: German */
    private static final String[] LANGUAGE_de = {
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E4,\u00E2,\u00E0,\u00E1,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F6,\u00F4,\u00F2,\u00F3,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FC,\u00FB,\u00F9,\u00FA,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        /* more_keys_for_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0117",
        /* more_keys_for_i */ null,
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* more_keys_for_s */ "\u00DF,\u015B,\u0161",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u00F1,\u0144",
        /* label_to_alpha_key ~ */
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keylabel_for_currency ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_cyrillic_i */
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* keylabel_for_swiss_row1_11 */ "\u00FC",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* keylabel_for_swiss_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* keylabel_for_swiss_row2_11 */ "\u00E4",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        /* more_keys_for_swiss_row1_11 */ "\u00E8",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        /* more_keys_for_swiss_row2_10 */ "\u00E9",
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        /* more_keys_for_swiss_row2_11 */ "\u00E0",
    };

    /* Language el: Greek */
    private static final String[] LANGUAGE_el = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0391: "Α" GREEK CAPITAL LETTER ALPHA
        // U+0392: "Β" GREEK CAPITAL LETTER BETA
        // U+0393: "Γ" GREEK CAPITAL LETTER GAMMA
        /* label_to_alpha_key */ "\u0391\u0392\u0393",
    };

    /* Language en: English */
    private static final String[] LANGUAGE_en = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* more_keys_for_o */ "\u00F4,\u00F6,\u00F2,\u00F3,\u0153,\u00F8,\u014D,\u00F5",
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FB,\u00FC,\u00F9,\u00FA,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113",
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        /* more_keys_for_i */ "\u00EE,\u00EF,\u00ED,\u012B,\u00EC",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* more_keys_for_c */ "\u00E7",
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* more_keys_for_s */ "\u00DF",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* more_keys_for_n */ "\u00F1",
    };

    /* Language eo: Esperanto */
    private static final String[] LANGUAGE_eo = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* more_keys_for_a */ "\u00E1,\u00E0,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101,\u0103,\u0105,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* more_keys_for_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D,\u0151,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0169: "ũ" LATIN SMALL LETTER U WITH TILDE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00B5: "µ" MICRO SIGN
        /* more_keys_for_u */ "\u00FA,\u016F,\u00FB,\u00FC,\u00F9,\u016B,\u0169,\u0171,\u0173,\u00B5",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u011B,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0129: "ĩ" LATIN SMALL LETTER I WITH TILDE
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* more_keys_for_i */ "\u00ED,\u00EE,\u00EF,\u0129,\u00EC,\u012F,\u012B,\u0131,\u0133",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+010B: "ċ" LATIN SMALL LETTER C WITH DOT ABOVE
        /* more_keys_for_c */ "\u0107,\u010D,\u00E7,\u010B",
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0219: "ș" LATIN SMALL LETTER S WITH COMMA BELOW
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* more_keys_for_s */ "\u00DF,\u0161,\u015B,\u0219,\u015F",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+0149: "ŉ" LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
        // U+014B: "ŋ" LATIN SMALL LETTER ENG
        /* more_keys_for_n */ "\u00F1,\u0144,\u0146,\u0148,\u0149,\u014B",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+0177: "ŷ" LATIN SMALL LETTER Y WITH CIRCUMFLEX
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        /* more_keys_for_y */ "y,\u00FD,\u0177,\u00FF,\u00FE",
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        /* more_keys_for_d */ "\u00F0,\u010F,\u0111",
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* more_keys_for_z */ "\u017A,\u017C,\u017E",
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+021B: "ț" LATIN SMALL LETTER T WITH COMMA BELOW
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0167: "ŧ" LATIN SMALL LETTER T WITH STROKE
        /* more_keys_for_t */ "\u0165,\u021B,\u0163,\u0167",
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        // U+0140: "ŀ" LATIN SMALL LETTER L WITH MIDDLE DOT
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* more_keys_for_l */ "\u013A,\u013C,\u013E,\u0140,\u0142",
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        // U+0121: "ġ" LATIN SMALL LETTER G WITH DOT ABOVE
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        /* more_keys_for_g */ "\u011F,\u0121,\u0123",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keylabel_for_currency */
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        /* more_keys_for_r */ "\u0159,\u0155,\u0157",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        // U+0138: "ĸ" LATIN SMALL LETTER KRA
        /* more_keys_for_k */ "\u0137,\u0138",
        /* keylabel_for_nordic_row1_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_question */
        // U+0125: "ĥ" LATIN SMALL LETTER H WITH CIRCUMFLEX
        // U+0127: "ħ" LATIN SMALL LETTER H WITH STROKE
        /* more_keys_for_h */ "\u0125,\u0127",
        // U+0175: "ŵ" LATIN SMALL LETTER W WITH CIRCUMFLEX
        /* more_keys_for_w */ "w,\u0175",
        /* more_keys_for_cyrillic_u ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null,
        /* ~ more_keys_for_swiss_row2_11 */
        // U+0135: "ĵ" LATIN SMALL LETTER J WITH CIRCUMFLEX
        /* keylabel_for_spanish_row2_10 */ "\u0135",
        /* more_keys_for_bullet ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null,
        /* ~ more_keys_for_symbols_percent */
        // U+0175: "ŵ" LATIN SMALL LETTER W WITH CIRCUMFLEX
        /* more_keys_for_v */ "w,\u0175",
        /* more_keys_for_j ~ */
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_exclamation */
        /* more_keys_for_q */ "q",
        /* more_keys_for_x */ "x",
        // U+015D: "ŝ" LATIN SMALL LETTER S WITH CIRCUMFLEX
        /* keylabel_for_q */ "\u015D",
        // U+011D: "ĝ" LATIN SMALL LETTER G WITH CIRCUMFLEX
        /* keylabel_for_w */ "\u011D",
        // U+016D: "ŭ" LATIN SMALL LETTER U WITH BREVE
        /* keylabel_for_y */ "\u016D",
        // U+0109: "ĉ" LATIN SMALL LETTER C WITH CIRCUMFLEX
        /* keylabel_for_x */ "\u0109",
    };

    /* Language es: Spanish */
    private static final String[] LANGUAGE_es = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* more_keys_for_a */ "\u00E1,\u00E0,\u00E4,\u00E2,\u00E3,\u00E5,\u0105,\u00E6,\u0101,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* more_keys_for_o */ "\u00F3,\u00F2,\u00F6,\u00F4,\u00F5,\u00F8,\u0153,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u00E8,\u00EB,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u00E7,\u0107,\u010D",
        /* more_keys_for_s */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u00F1,\u0144",
        /* label_to_alpha_key ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_cyrillic_soft_sign */
        // U+00A1: "¡" INVERTED EXCLAMATION MARK
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* more_keys_for_punctuation */ "!fixedColumnOrder!9,\u00A1,;,/,(,),#,!,\\,,?,\u00BF,&,\\%,+,\",-,:,',@",
    };

    /* Language et_EE: Estonian (Estonia) */
    private static final String[] LANGUAGE_et_EE = {
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        /* more_keys_for_a */ "\u00E4,\u0101,\u00E0,\u00E1,\u00E2,\u00E3,\u00E5,\u00E6,\u0105",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* more_keys_for_o */ "\u00F6,\u00F5,\u00F2,\u00F3,\u00F4,\u0153,\u0151,\u00F8",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        /* more_keys_for_u */ "\u00FC,\u016B,\u0173,\u00F9,\u00FA,\u00FB,\u016F,\u0171",
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* more_keys_for_e */ "\u0113,\u00E8,\u0117,\u00E9,\u00EA,\u00EB,\u0119,\u011B",
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        /* more_keys_for_i */ "\u012B,\u00EC,\u012F,\u00ED,\u00EE,\u00EF,\u0131",
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* more_keys_for_c */ "\u010D,\u00E7,\u0107",
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* more_keys_for_s */ "\u0161,\u00DF,\u015B,\u015F",
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u0146,\u00F1,\u0144,\u0144",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* more_keys_for_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* more_keys_for_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        /* more_keys_for_z */ "\u017E,\u017C,\u017A",
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        /* more_keys_for_t */ "\u0163,\u0165",
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        /* more_keys_for_l */ "\u013C,\u0142,\u013A,\u013E",
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* more_keys_for_g */ "\u0123,\u011F",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keylabel_for_currency */
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        /* more_keys_for_r */ "\u0157,\u0159,\u0155",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        /* more_keys_for_k */ "\u0137",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* keylabel_for_nordic_row1_11 */ "\u00FC",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* keylabel_for_nordic_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* keylabel_for_nordic_row2_11 */ "\u00E4",
        /* more_keys_for_cyrillic_ie */ null,
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* more_keys_for_nordic_row2_10 */ "\u00F5",
    };

    /* Language fa: Persian */
    private static final String[] LANGUAGE_fa = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0627: "ا" ARABIC LETTER ALEF
        // U+200C: ZERO WIDTH NON-JOINER
        // U+0628: "ب" ARABIC LETTER BEH
        // U+067E: "پ" ARABIC LETTER PEH
        /* label_to_alpha_key */ "\u0627\u200C\u0628\u200C\u067E",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+FDFC: "﷼" RIAL SIGN
        /* keylabel_for_currency */ "\uFDFC",
        /* more_keys_for_r ~ */
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_cyrillic_soft_sign */
        // U+061F: "؟" ARABIC QUESTION MARK
        // U+060C: "،" ARABIC COMMA
        // U+061B: "؛" ARABIC SEMICOLON
        /* more_keys_for_punctuation */ "!fixedColumnOrder!8,\",\',#,-,:,!,\u060C,\u061F,@,&,\\%,+,\u061B,/,!text/keyspec_left_parenthesis,!text/keyspec_right_parenthesis",
        /* more_keys_for_nordic_row2_11 */ null,
        // U+06F1: "۱" EXTENDED ARABIC-INDIC DIGIT ONE
        /* keylabel_for_symbols_1 */ "\u06F1",
        // U+06F2: "۲" EXTENDED ARABIC-INDIC DIGIT TWO
        /* keylabel_for_symbols_2 */ "\u06F2",
        // U+06F3: "۳" EXTENDED ARABIC-INDIC DIGIT THREE
        /* keylabel_for_symbols_3 */ "\u06F3",
        // U+06F4: "۴" EXTENDED ARABIC-INDIC DIGIT FOUR
        /* keylabel_for_symbols_4 */ "\u06F4",
        // U+06F5: "۵" EXTENDED ARABIC-INDIC DIGIT FIVE
        /* keylabel_for_symbols_5 */ "\u06F5",
        // U+06F6: "۶" EXTENDED ARABIC-INDIC DIGIT SIX
        /* keylabel_for_symbols_6 */ "\u06F6",
        // U+06F7: "۷" EXTENDED ARABIC-INDIC DIGIT SEVEN
        /* keylabel_for_symbols_7 */ "\u06F7",
        // U+06F8: "۸" EXTENDED ARABIC-INDIC DIGIT EIGHT
        /* keylabel_for_symbols_8 */ "\u06F8",
        // U+06F9: "۹" EXTENDED ARABIC-INDIC DIGIT NINE
        /* keylabel_for_symbols_9 */ "\u06F9",
        // U+06F0: "۰" EXTENDED ARABIC-INDIC DIGIT ZERO
        /* keylabel_for_symbols_0 */ "\u06F0",
        // Label for "switch to symbols" key.
        // U+061F: "؟" ARABIC QUESTION MARK
        /* label_to_symbol_key */ "\u06F3\u06F2\u06F1\u061F",
        // Label for "switch to symbols with microphone" key. This string shouldn't include the "mic"
        // part because it'll be appended by the code.
        /* label_to_symbol_with_microphone_key */ "\u06F3\u06F2\u06F1",
        /* additional_more_keys_for_symbols_1 */ "1",
        /* additional_more_keys_for_symbols_2 */ "2",
        /* additional_more_keys_for_symbols_3 */ "3",
        /* additional_more_keys_for_symbols_4 */ "4",
        /* additional_more_keys_for_symbols_5 */ "5",
        /* additional_more_keys_for_symbols_6 */ "6",
        /* additional_more_keys_for_symbols_7 */ "7",
        /* additional_more_keys_for_symbols_8 */ "8",
        /* additional_more_keys_for_symbols_9 */ "9",
        // U+066B: "٫" ARABIC DECIMAL SEPARATOR
        // U+066C: "٬" ARABIC THOUSANDS SEPARATOR
        /* additional_more_keys_for_symbols_0 */ "0,\u066B,\u066C",
        // U+2605: "★" BLACK STAR
        // U+066D: "٭" ARABIC FIVE POINTED STAR
        /* more_keys_for_star */ "\u2605,\u066D",
        /* keyspec_left_parenthesis */ "(|)",
        /* keyspec_right_parenthesis */ ")|(",
        /* keyspec_left_square_bracket */ "[|]",
        /* keyspec_right_square_bracket */ "]|[",
        /* keyspec_left_curly_bracket */ "{|}",
        /* keyspec_right_curly_bracket */ "}|{",
        /* keyspec_less_than */ "<|>",
        /* keyspec_greater_than */ ">|<",
        /* keyspec_less_than_equal */ "\u2264|\u2265",
        /* keyspec_greater_than_equal */ "\u2265|\u2264",
        /* keyspec_left_double_angle_quote */ "\u00AB|\u00BB",
        /* keyspec_right_double_angle_quote */ "\u00BB|\u00AB",
        /* keyspec_left_single_angle_quote */ "\u2039|\u203A",
        /* keyspec_right_single_angle_quote */ "\u203A|\u2039",
        // U+060C: "،" ARABIC COMMA
        // U+061B: "؛" ARABIC SEMICOLON
        // U+061F: "؟" ARABIC QUESTION MARK
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        /* keylabel_for_tablet_comma */ "\u060C",
        /* more_keys_for_tablet_period */ "!text/more_keys_for_arabic_diacritics",
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* more_keys_for_question */ "?,\u00BF",
        /* more_keys_for_h ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null,
        /* ~ keylabel_for_spanish_row2_10 */
        // U+266A: "♪" EIGHTH NOTE
        /* more_keys_for_bullet */ "\u266A",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+FD3E: "﴾" ORNATE LEFT PARENTHESIS
        // U+FD3F: "﴿" ORNATE RIGHT PARENTHESIS
        /* more_keys_for_left_parenthesis */ "!fixedColumnOrder!4,\uFD3E|\uFD3F,!text/keyspecs_for_left_parenthesis_more_keys",
        /* more_keys_for_right_parenthesis */ "!fixedColumnOrder!4,\uFD3F|\uFD3E,!text/keyspecs_for_right_parenthesis_more_keys",
        // U+0655: "ٕ" ARABIC HAMZA BELOW
        // U+0652: "ْ" ARABIC SUKUN
        // U+0651: "ّ" ARABIC SHADDA
        // U+064C: "ٌ" ARABIC DAMMATAN
        // U+064D: "ٍ" ARABIC KASRATAN
        // U+064B: "ً" ARABIC FATHATAN
        // U+0654: "ٔ" ARABIC HAMZA ABOVE
        // U+0656: "ٖ" ARABIC SUBSCRIPT ALEF
        // U+0670: "ٰ" ARABIC LETTER SUPERSCRIPT ALEF
        // U+0653: "ٓ" ARABIC MADDAH ABOVE
        // U+064F: "ُ" ARABIC DAMMA
        // U+0650: "ِ" ARABIC KASRA
        // U+064E: "َ" ARABIC FATHA
        // U+0640: "ـ" ARABIC TATWEEL
        // In order to make Tatweel easily distinguishable from other punctuations, we use consecutive Tatweels only for its displayed label.
        // Note: The space character is needed as a preceding letter to draw Arabic diacritics characters correctly.
        /* more_keys_for_arabic_diacritics */ "!fixedColumnOrder!7, \u0655|\u0655, \u0652|\u0652, \u0651|\u0651, \u064C|\u064C, \u064D|\u064D, \u064B|\u064B, \u0654|\u0654, \u0656|\u0656, \u0670|\u0670, \u0653|\u0653, \u064F|\u064F, \u0650|\u0650, \u064E|\u064E,\u0640\u0640\u0640|\u0640",
        // U+060C: "،" ARABIC COMMA
        /* keylabel_for_comma */ "\u060C",
        /* more_keys_for_comma */ "\\,",
        /* keyhintlabel_for_tablet_comma */ "\u061F",
        /* more_keys_for_tablet_comma */ "!fixedColumnOrder!4,:,!,\u061F,\u061B,-,/,!text/keyspec_left_double_angle_quote,!text/keyspec_right_double_angle_quote",
        // U+064B: "ً" ARABIC FATHATAN
        /* keyhintlabel_for_period */ "\u064B",
        /* more_keys_for_period */ "!text/more_keys_for_arabic_diacritics",
        /* keyhintlabel_for_tablet_period */ "\u064B",
        /* keylabel_for_symbols_question */ "\u061F",
        /* keylabel_for_symbols_semicolon */ "\u061B",
        // U+066A: "٪" ARABIC PERCENT SIGN
        /* keylabel_for_symbols_percent */ "\u066A",
        /* more_keys_for_symbols_semicolon */ ";",
        // U+2030: "‰" PER MILLE SIGN
        /* more_keys_for_symbols_percent */ "\\%,\u2030",
        /* more_keys_for_v ~ */
        null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_plus */
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        /* more_keys_for_less_than */ "!fixedColumnOrder!3,!text/keyspec_left_single_angle_quote;,!text/keyspec_less_than_equal;,!text/keyspec_less_than",
        /* more_keys_for_greater_than */ "!fixedColumnOrder!3,!text/keyspec_right_single_angle_quote;,!text/keyspec_greater_than_equal;,!text/keyspec_greater_than",
    };

    /* Language fi: Finnish */
    private static final String[] LANGUAGE_fi = {
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E6,\u00E0,\u00E1,\u00E2,\u00E3,\u0101",
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F8,\u00F4,\u00F2,\u00F3,\u00F5,\u0153,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* more_keys_for_u */ "\u00FC",
        /* more_keys_for_e ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_c */
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        /* more_keys_for_s */ "\u0161,\u00DF,\u015B",
        /* more_keys_for_n ~ */
        null, null, null, null,
        /* ~ more_keys_for_d */
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        /* more_keys_for_z */ "\u017E,\u017A,\u017C",
        /* more_keys_for_t ~ */
        null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_k */
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        /* keylabel_for_nordic_row1_11 */ "\u00E5",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* keylabel_for_nordic_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* keylabel_for_nordic_row2_11 */ "\u00E4",
        /* more_keys_for_cyrillic_ie */ null,
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* more_keys_for_nordic_row2_10 */ "\u00F8",
        /* keylabel_for_east_slavic_row1_9 ~ */
        null, null, null, null, null, null,
        /* ~ more_keys_for_punctuation */
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* more_keys_for_nordic_row2_11 */ "\u00E6",
    };

    /* Language fr: French */
    private static final String[] LANGUAGE_fr = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* more_keys_for_a */ "\u00E0,\u00E2,%,\u00E6,\u00E1,\u00E4,\u00E3,\u00E5,\u0101,\u00AA",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* more_keys_for_o */ "\u00F4,\u0153,%,\u00F6,\u00F2,\u00F3,\u00F5,\u00F8,\u014D,\u00BA",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00F9,\u00FB,%,\u00FC,\u00FA,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u00E8,\u00EA,\u00EB,%,\u0119,\u0117,\u0113",
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00EE,%,\u00EF,\u00EC,\u00ED,\u012F,\u012B",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u00E7,\u0107,\u010D",
        /* more_keys_for_s ~ */
        null, null, null,
        /* ~ label_to_alpha_key */
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* more_keys_for_y */ "%,\u00FF",
        /* more_keys_for_d ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_cyrillic_i */
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        /* keylabel_for_swiss_row1_11 */ "\u00E8",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        /* keylabel_for_swiss_row2_10 */ "\u00E9",
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        /* keylabel_for_swiss_row2_11 */ "\u00E0",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* more_keys_for_swiss_row1_11 */ "\u00FC",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* more_keys_for_swiss_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* more_keys_for_swiss_row2_11 */ "\u00E4",
    };

    /* Language hi: Hindi */
    private static final String[] LANGUAGE_hi = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0915: "क" DEVANAGARI LETTER KA
        // U+0916: "ख" DEVANAGARI LETTER KHA
        // U+0917: "ग" DEVANAGARI LETTER GA
        /* label_to_alpha_key */ "\u0915\u0916\u0917",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20B9: "₹" INDIAN RUPEE SIGN
        /* keylabel_for_currency */ "\u20B9",
        /* more_keys_for_r ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_nordic_row2_11 */
        // U+0967: "१" DEVANAGARI DIGIT ONE
        /* keylabel_for_symbols_1 */ "\u0967",
        // U+0968: "२" DEVANAGARI DIGIT TWO
        /* keylabel_for_symbols_2 */ "\u0968",
        // U+0969: "३" DEVANAGARI DIGIT THREE
        /* keylabel_for_symbols_3 */ "\u0969",
        // U+096A: "४" DEVANAGARI DIGIT FOUR
        /* keylabel_for_symbols_4 */ "\u096A",
        // U+096B: "५" DEVANAGARI DIGIT FIVE
        /* keylabel_for_symbols_5 */ "\u096B",
        // U+096C: "६" DEVANAGARI DIGIT SIX
        /* keylabel_for_symbols_6 */ "\u096C",
        // U+096D: "७" DEVANAGARI DIGIT SEVEN
        /* keylabel_for_symbols_7 */ "\u096D",
        // U+096E: "८" DEVANAGARI DIGIT EIGHT
        /* keylabel_for_symbols_8 */ "\u096E",
        // U+096F: "९" DEVANAGARI DIGIT NINE
        /* keylabel_for_symbols_9 */ "\u096F",
        // U+0966: "०" DEVANAGARI DIGIT ZERO
        /* keylabel_for_symbols_0 */ "\u0966",
        // Label for "switch to symbols" key.
        /* label_to_symbol_key */ "?\u0967\u0968\u0969",
        // Label for "switch to symbols with microphone" key. This string shouldn't include the "mic"
        // part because it'll be appended by the code.
        /* label_to_symbol_with_microphone_key */ "\u0967\u0968\u0969",
        /* additional_more_keys_for_symbols_1 */ "1",
        /* additional_more_keys_for_symbols_2 */ "2",
        /* additional_more_keys_for_symbols_3 */ "3",
        /* additional_more_keys_for_symbols_4 */ "4",
        /* additional_more_keys_for_symbols_5 */ "5",
        /* additional_more_keys_for_symbols_6 */ "6",
        /* additional_more_keys_for_symbols_7 */ "7",
        /* additional_more_keys_for_symbols_8 */ "8",
        /* additional_more_keys_for_symbols_9 */ "9",
        /* additional_more_keys_for_symbols_0 */ "0",
    };

    /* Language hr: Croatian */
    private static final String[] LANGUAGE_hr = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_9qm_rqm",
        /* single_quotes */ "!text/single_9qm_rqm",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* more_keys_for_c */ "\u010D,\u0107,\u00E7",
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* more_keys_for_s */ "\u0161,\u015B,\u00DF",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u00F1,\u0144",
        /* label_to_alpha_key */ null,
        /* more_keys_for_y */ null,
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        /* more_keys_for_d */ "\u0111",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        /* more_keys_for_z */ "\u017E,\u017A,\u017C",
        /* more_keys_for_t ~ */
        null, null, null,
        /* ~ more_keys_for_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
    };

    /* Language hu: Hungarian */
    private static final String[] LANGUAGE_hu = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E1,\u00E0,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F3,\u00F6,\u0151,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FC,\u0171,\u00FB,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00ED,\u00EE,\u00EF,\u00EC,\u012F,\u012B",
        /* double_quotes */ "!text/double_9qm_rqm",
        /* single_quotes */ "!text/single_9qm_rqm",
        /* more_keys_for_c ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
    };

    /* Language hy_AM: Armenian (Armenia) */
    private static final String[] LANGUAGE_hy_AM = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0531: "Ա" ARMENIAN CAPITAL LETTER AYB
        // U+0532: "Բ" ARMENIAN CAPITAL LETTER BEN
        // U+0533: "Գ" ARMENIAN CAPITAL LETTER GIM
        /* label_to_alpha_key */ "\u0531\u0532\u0533",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ more_keys_for_cyrillic_soft_sign */
        // U+058A: "֊" ARMENIAN HYPHEN
        // U+055C: "՜" ARMENIAN EXCLAMATION MARK
        // U+055D: "՝" ARMENIAN COMMA
        // U+055E: "՞" ARMENIAN QUESTION MARK
        // U+0559: "ՙ" ARMENIAN MODIFIER LETTER LEFT HALF RING
        // U+055A: "՚" ARMENIAN APOSTROPHE
        // U+055B: "՛" ARMENIAN EMPHASIS MARK
        // U+055F: "՟" ARMENIAN ABBREVIATION MARK
        /* more_keys_for_punctuation */ "!fixedColumnOrder!8,!,?,\u0559,\u055A,.,\u055C,\\,,\u055E,:,;,\u055F,\u00AB,\u00BB,\u058A,\u055D,\u055B",
        /* more_keys_for_nordic_row2_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null,
        /* ~ keyspec_right_single_angle_quote */
        // U+058F: "֏" ARMENIAN DRAM SIGN
        // TODO: Enable this when we have glyph for the following letter
        // <string name="keylabel_for_currency">&#x058F;</string>
        // 
        // U+055D: "՝" ARMENIAN COMMA
        /* keylabel_for_tablet_comma */ "\u055D",
        /* more_keys_for_tablet_period */ "!text/more_keys_for_punctuation",
        // U+055E: "՞" ARMENIAN QUESTION MARK
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* more_keys_for_question */ "\u055E,\u00BF",
        /* more_keys_for_h ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_greater_than */
        // U+0589: "։" ARMENIAN FULL STOP
        /* keylabel_for_period */ "\u0589",
        /* keylabel_for_tablet_period */ "\u0589",
        // U+055C: "՜" ARMENIAN EXCLAMATION MARK
        // U+00A1: "¡" INVERTED EXCLAMATION MARK
        /* more_keys_for_exclamation */ "\u055C,\u00A1",
    };

    /* Language is: Icelandic */
    private static final String[] LANGUAGE_is = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E1,\u00E4,\u00E6,\u00E5,\u00E0,\u00E2,\u00E3,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FC,\u00FB,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u00EB,\u00E8,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00ED,\u00EF,\u00EE,\u00EC,\u012F,\u012B",
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c ~ */
        null, null, null, null,
        /* ~ label_to_alpha_key */
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* more_keys_for_y */ "\u00FD,\u00FF",
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        /* more_keys_for_d */ "\u00F0",
        /* more_keys_for_z */ null,
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        /* more_keys_for_t */ "\u00FE",
        /* more_keys_for_l ~ */
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_k */
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        /* keylabel_for_nordic_row1_11 */ "\u00F0",
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* keylabel_for_nordic_row2_10 */ "\u00E6",
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        /* keylabel_for_nordic_row2_11 */ "\u00FE",
    };

    /* Language it: Italian */
    private static final String[] LANGUAGE_it = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* more_keys_for_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101,\u00AA",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* more_keys_for_o */ "\u00F2,\u00F3,\u00F4,\u00F6,\u00F5,\u0153,\u00F8,\u014D,\u00BA",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00F9,\u00FA,\u00FB,\u00FC,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00EC,\u00ED,\u00EE,\u00EF,\u012F,\u012B",
    };

    /* Language iw: Hebrew */
    private static final String[] LANGUAGE_iw = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_rqm_9qm",
        /* single_quotes */ "!text/single_rqm_9qm",
        /* more_keys_for_c ~ */
        null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+05D0: "א" HEBREW LETTER ALEF
        // U+05D1: "ב" HEBREW LETTER BET
        // U+05D2: "ג" HEBREW LETTER GIMEL
        /* label_to_alpha_key */ "\u05D0\u05D1\u05D2",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20AA: "₪" NEW SHEQEL SIGN
        /* keylabel_for_currency */ "\u20AA",
        /* more_keys_for_r ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ additional_more_keys_for_symbols_0 */
        // U+2605: "★" BLACK STAR
        /* more_keys_for_star */ "\u2605",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        /* keyspec_left_parenthesis */ "(|)",
        /* keyspec_right_parenthesis */ ")|(",
        /* keyspec_left_square_bracket */ "[|]",
        /* keyspec_right_square_bracket */ "]|[",
        /* keyspec_left_curly_bracket */ "{|}",
        /* keyspec_right_curly_bracket */ "}|{",
        /* keyspec_less_than */ "<|>",
        /* keyspec_greater_than */ ">|<",
        /* keyspec_less_than_equal */ "\u2264|\u2265",
        /* keyspec_greater_than_equal */ "\u2265|\u2264",
        /* keyspec_left_double_angle_quote */ "\u00AB|\u00BB",
        /* keyspec_right_double_angle_quote */ "\u00BB|\u00AB",
        /* keyspec_left_single_angle_quote */ "\u2039|\u203A",
        /* keyspec_right_single_angle_quote */ "\u203A|\u2039",
        /* keylabel_for_tablet_comma ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_tablet_punctuation */
        // U+00B1: "±" PLUS-MINUS SIGN
        // U+FB29: "﬩" HEBREW LETTER ALTERNATIVE PLUS SIGN
        /* more_keys_for_plus */ "\u00B1,\uFB29",
    };

    /* Language ka_GE: Georgian (Georgia) */
    private static final String[] LANGUAGE_ka_GE = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c ~ */
        null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+10D0: "ა" GEORGIAN LETTER AN
        // U+10D1: "ბ" GEORGIAN LETTER BAN
        // U+10D2: "გ" GEORGIAN LETTER GAN
        /* label_to_alpha_key */ "\u10D0\u10D1\u10D2",
    };

    /* Language kk: Kazakh */
    private static final String[] LANGUAGE_kk = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ keylabel_for_nordic_row2_11 */
        // U+0451: "ё" CYRILLIC SMALL LETTER IO
        /* more_keys_for_cyrillic_ie */ "\u0451",
        /* more_keys_for_nordic_row2_10 */ null,
        // U+0449: "щ" CYRILLIC SMALL LETTER SHCHA
        /* keylabel_for_east_slavic_row1_9 */ "\u0449",
        // U+044B: "ы" CYRILLIC SMALL LETTER YERU
        /* keylabel_for_east_slavic_row2_2 */ "\u044B",
        // U+044D: "э" CYRILLIC SMALL LETTER E
        /* keylabel_for_east_slavic_row2_11 */ "\u044D",
        // U+0438: "и" CYRILLIC SMALL LETTER I
        /* keylabel_for_east_slavic_row3_5 */ "\u0438",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* more_keys_for_cyrillic_soft_sign */ "\u044A",
        /* more_keys_for_punctuation ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_w */
        // U+04AF: "ү" CYRILLIC SMALL LETTER STRAIGHT U
        // U+04B1: "ұ" CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
        /* more_keys_for_cyrillic_u */ "\u04AF,\u04B1",
        // U+04A3: "ң" CYRILLIC SMALL LETTER EN WITH DESCENDER
        /* more_keys_for_cyrillic_en */ "\u04A3",
        // U+0493: "ғ" CYRILLIC SMALL LETTER GHE WITH STROKE
        /* more_keys_for_cyrillic_ghe */ "\u0493",
        // U+0456: "і" CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
        /* more_keys_for_east_slavic_row2_2 */ "\u0456",
        // U+04E9: "ө" CYRILLIC SMALL LETTER BARRED O
        /* more_keys_for_cyrillic_o */ "\u04E9",
        /* keylabel_for_south_slavic_row1_6 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_j */
        // U+049B: "қ" CYRILLIC SMALL LETTER KA WITH DESCENDER
        /* more_keys_for_cyrillic_ka */ "\u049B",
        // U+04D9: "ә" CYRILLIC SMALL LETTER SCHWA
        /* more_keys_for_cyrillic_a */ "\u04D9",
        // U+04BB: "һ" CYRILLIC SMALL LETTER SHHA
        /* more_keys_for_east_slavic_row2_11 */ "\u04BB",
    };

    /* Language km_KH: Khmer (Cambodia) */
    private static final String[] LANGUAGE_km_KH = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+1780: "ក" KHMER LETTER KA
        // U+1781: "ខ" KHMER LETTER KHA
        // U+1782: "គ" KHMER LETTER KO
        /* label_to_alpha_key */ "\u1780\u1781\u1782",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_east_slavic_row2_11 */
        // U+17DB: "៛" KHMER CURRENCY SYMBOL RIEL
        /* more_keys_for_currency_dollar */ "\u17DB,\u00A2,\u00A3,\u20AC,\u00A5,\u20B1",
    };

    /* Language ky: Kirghiz */
    private static final String[] LANGUAGE_ky = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ keylabel_for_nordic_row2_11 */
        // U+0451: "ё" CYRILLIC SMALL LETTER IO
        /* more_keys_for_cyrillic_ie */ "\u0451",
        /* more_keys_for_nordic_row2_10 */ null,
        // U+0449: "щ" CYRILLIC SMALL LETTER SHCHA
        /* keylabel_for_east_slavic_row1_9 */ "\u0449",
        // U+044B: "ы" CYRILLIC SMALL LETTER YERU
        /* keylabel_for_east_slavic_row2_2 */ "\u044B",
        // U+044D: "э" CYRILLIC SMALL LETTER E
        /* keylabel_for_east_slavic_row2_11 */ "\u044D",
        // U+0438: "и" CYRILLIC SMALL LETTER I
        /* keylabel_for_east_slavic_row3_5 */ "\u0438",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* more_keys_for_cyrillic_soft_sign */ "\u044A",
        /* more_keys_for_punctuation ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_w */
        // U+04AF: "ү" CYRILLIC SMALL LETTER STRAIGHT U
        /* more_keys_for_cyrillic_u */ "\u04AF",
        // U+04A3: "ң" CYRILLIC SMALL LETTER EN WITH DESCENDER
        /* more_keys_for_cyrillic_en */ "\u04A3",
        /* more_keys_for_cyrillic_ghe */ null,
        /* more_keys_for_east_slavic_row2_2 */ null,
        // U+04E9: "ө" CYRILLIC SMALL LETTER BARRED O
        /* more_keys_for_cyrillic_o */ "\u04E9",
    };

    /* Language lo_LA: Lao (Laos) */
    private static final String[] LANGUAGE_lo_LA = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0E81: "ກ" LAO LETTER KO
        // U+0E82: "ຂ" LAO LETTER KHO SUNG
        // U+0E84: "ຄ" LAO LETTER KHO TAM
        /* label_to_alpha_key */ "\u0E81\u0E82\u0E84",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20AD: "₭" KIP SIGN
        /* keylabel_for_currency */ "\u20AD",
    };

    /* Language lt: Lithuanian */
    private static final String[] LANGUAGE_lt = {
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* more_keys_for_a */ "\u0105,\u00E4,\u0101,\u00E0,\u00E1,\u00E2,\u00E3,\u00E5,\u00E6",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* more_keys_for_o */ "\u00F6,\u00F5,\u00F2,\u00F3,\u00F4,\u0153,\u0151,\u00F8",
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        /* more_keys_for_u */ "\u016B,\u0173,\u00FC,\u016B,\u00F9,\u00FA,\u00FB,\u016F,\u0171",
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* more_keys_for_e */ "\u0117,\u0119,\u0113,\u00E8,\u00E9,\u00EA,\u00EB,\u011B",
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        /* more_keys_for_i */ "\u012F,\u012B,\u00EC,\u00ED,\u00EE,\u00EF,\u0131",
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* more_keys_for_c */ "\u010D,\u00E7,\u0107",
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* more_keys_for_s */ "\u0161,\u00DF,\u015B,\u015F",
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u0146,\u00F1,\u0144,\u0144",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* more_keys_for_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* more_keys_for_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        /* more_keys_for_z */ "\u017E,\u017C,\u017A",
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        /* more_keys_for_t */ "\u0163,\u0165",
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        /* more_keys_for_l */ "\u013C,\u0142,\u013A,\u013E",
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* more_keys_for_g */ "\u0123,\u011F",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keylabel_for_currency */
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        /* more_keys_for_r */ "\u0157,\u0159,\u0155",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        /* more_keys_for_k */ "\u0137",
    };

    /* Language lv: Latvian */
    private static final String[] LANGUAGE_lv = {
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        /* more_keys_for_a */ "\u0101,\u00E0,\u00E1,\u00E2,\u00E3,\u00E4,\u00E5,\u00E6,\u0105",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* more_keys_for_o */ "\u00F2,\u00F3,\u00F4,\u00F5,\u00F6,\u0153,\u0151,\u00F8",
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        /* more_keys_for_u */ "\u016B,\u0173,\u00F9,\u00FA,\u00FB,\u00FC,\u016F,\u0171",
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* more_keys_for_e */ "\u0113,\u0117,\u00E8,\u00E9,\u00EA,\u00EB,\u0119,\u011B",
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        /* more_keys_for_i */ "\u012B,\u012F,\u00EC,\u00ED,\u00EE,\u00EF,\u0131",
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* more_keys_for_c */ "\u010D,\u00E7,\u0107",
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* more_keys_for_s */ "\u0161,\u00DF,\u015B,\u015F",
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u0146,\u00F1,\u0144,\u0144",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* more_keys_for_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* more_keys_for_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        /* more_keys_for_z */ "\u017E,\u017C,\u017A",
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        /* more_keys_for_t */ "\u0163,\u0165",
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        /* more_keys_for_l */ "\u013C,\u0142,\u013A,\u013E",
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* more_keys_for_g */ "\u0123,\u011F",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keylabel_for_currency */
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        /* more_keys_for_r */ "\u0157,\u0159,\u0155",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        /* more_keys_for_k */ "\u0137",
    };

    /* Language mk: Macedonian */
    private static final String[] LANGUAGE_mk = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c ~ */
        null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ keylabel_for_nordic_row2_11 */
        // U+0450: "ѐ" CYRILLIC SMALL LETTER IE WITH GRAVE
        /* more_keys_for_cyrillic_ie */ "\u0450",
        /* more_keys_for_nordic_row2_10 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_cyrillic_o */
        // U+0455: "ѕ" CYRILLIC SMALL LETTER DZE
        /* keylabel_for_south_slavic_row1_6 */ "\u0455",
        // U+045C: "ќ" CYRILLIC SMALL LETTER KJE
        /* keylabel_for_south_slavic_row2_11 */ "\u045C",
        // U+0437: "з" CYRILLIC SMALL LETTER ZE
        /* keylabel_for_south_slavic_row3_1 */ "\u0437",
        // U+0453: "ѓ" CYRILLIC SMALL LETTER GJE
        /* keylabel_for_south_slavic_row3_8 */ "\u0453",
        // U+045D: "ѝ" CYRILLIC SMALL LETTER I WITH GRAVE
        /* more_keys_for_cyrillic_i */ "\u045D",
    };

    /* Language mn_MN: Mongolian (Mongolia) */
    private static final String[] LANGUAGE_mn_MN = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20AE: "₮" TUGRIK SIGN
        /* keylabel_for_currency */ "\u20AE",
    };

    /* Language nb: Norwegian Bokmål */
    private static final String[] LANGUAGE_nb = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E0,\u00E4,\u00E1,\u00E2,\u00E3,\u0101",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F4,\u00F2,\u00F3,\u00F6,\u00F5,\u0153,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FC,\u00FB,\u00F9,\u00FA,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        /* more_keys_for_i */ null,
        /* double_quotes */ "!text/double_9qm_rqm",
        /* single_quotes */ "!text/single_9qm_rqm",
        /* more_keys_for_c ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_k */
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        /* keylabel_for_nordic_row1_11 */ "\u00E5",
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* keylabel_for_nordic_row2_10 */ "\u00F8",
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* keylabel_for_nordic_row2_11 */ "\u00E6",
        /* more_keys_for_cyrillic_ie */ null,
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* more_keys_for_nordic_row2_10 */ "\u00F6",
        /* keylabel_for_east_slavic_row1_9 ~ */
        null, null, null, null, null, null,
        /* ~ more_keys_for_punctuation */
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* more_keys_for_nordic_row2_11 */ "\u00E4",
    };

    /* Language ne_NP: Nepali (Nepal) */
    private static final String[] LANGUAGE_ne_NP = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0915: "क" DEVANAGARI LETTER KA
        // U+0916: "ख" DEVANAGARI LETTER KHA
        // U+0917: "ग" DEVANAGARI LETTER GA
        /* label_to_alpha_key */ "\u0915\u0916\u0917",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+0930/U+0941/U+002E "रु." NEPALESE RUPEE SIGN
        /* keylabel_for_currency */ "\u0930\u0941.",
        /* more_keys_for_r ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_nordic_row2_11 */
        // U+0967: "१" DEVANAGARI DIGIT ONE
        /* keylabel_for_symbols_1 */ "\u0967",
        // U+0968: "२" DEVANAGARI DIGIT TWO
        /* keylabel_for_symbols_2 */ "\u0968",
        // U+0969: "३" DEVANAGARI DIGIT THREE
        /* keylabel_for_symbols_3 */ "\u0969",
        // U+096A: "४" DEVANAGARI DIGIT FOUR
        /* keylabel_for_symbols_4 */ "\u096A",
        // U+096B: "५" DEVANAGARI DIGIT FIVE
        /* keylabel_for_symbols_5 */ "\u096B",
        // U+096C: "६" DEVANAGARI DIGIT SIX
        /* keylabel_for_symbols_6 */ "\u096C",
        // U+096D: "७" DEVANAGARI DIGIT SEVEN
        /* keylabel_for_symbols_7 */ "\u096D",
        // U+096E: "८" DEVANAGARI DIGIT EIGHT
        /* keylabel_for_symbols_8 */ "\u096E",
        // U+096F: "९" DEVANAGARI DIGIT NINE
        /* keylabel_for_symbols_9 */ "\u096F",
        // U+0966: "०" DEVANAGARI DIGIT ZERO
        /* keylabel_for_symbols_0 */ "\u0966",
        // Label for "switch to symbols" key.
        /* label_to_symbol_key */ "?\u0967\u0968\u0969",
        // Label for "switch to symbols with microphone" key. This string shouldn't include the "mic"
        // part because it'll be appended by the code.
        /* label_to_symbol_with_microphone_key */ "\u0967\u0968\u0969",
        /* additional_more_keys_for_symbols_1 */ "1",
        /* additional_more_keys_for_symbols_2 */ "2",
        /* additional_more_keys_for_symbols_3 */ "3",
        /* additional_more_keys_for_symbols_4 */ "4",
        /* additional_more_keys_for_symbols_5 */ "5",
        /* additional_more_keys_for_symbols_6 */ "6",
        /* additional_more_keys_for_symbols_7 */ "7",
        /* additional_more_keys_for_symbols_8 */ "8",
        /* additional_more_keys_for_symbols_9 */ "9",
        /* additional_more_keys_for_symbols_0 */ "0",
    };

    /* Language nl: Dutch */
    private static final String[] LANGUAGE_nl = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E1,\u00E4,\u00E2,\u00E0,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FC,\u00FB,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u00EB,\u00EA,\u00E8,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* more_keys_for_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B,\u0133",
        /* double_quotes */ "!text/double_9qm_rqm",
        /* single_quotes */ "!text/single_9qm_rqm",
        /* more_keys_for_c */ null,
        /* more_keys_for_s */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u00F1,\u0144",
        /* label_to_alpha_key */ null,
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* more_keys_for_y */ "\u0133",
    };

    /* Language pl: Polish */
    private static final String[] LANGUAGE_pl = {
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u0105,\u00E1,\u00E0,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        /* more_keys_for_u */ null,
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u0119,\u00E8,\u00E9,\u00EA,\u00EB,\u0117,\u0113",
        /* more_keys_for_i */ null,
        /* double_quotes */ "!text/double_9qm_rqm",
        /* single_quotes */ "!text/single_9qm_rqm",
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u0107,\u00E7,\u010D",
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* more_keys_for_s */ "\u015B,\u00DF,\u0161",
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* more_keys_for_n */ "\u0144,\u00F1",
        /* label_to_alpha_key ~ */
        null, null, null,
        /* ~ more_keys_for_d */
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* more_keys_for_z */ "\u017C,\u017A,\u017E",
        /* more_keys_for_t */ null,
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* more_keys_for_l */ "\u0142",
    };

    /* Language pt: Portuguese */
    private static final String[] LANGUAGE_pt = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* more_keys_for_a */ "\u00E1,\u00E3,\u00E0,\u00E2,\u00E4,\u00E5,\u00E6,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* more_keys_for_o */ "\u00F3,\u00F5,\u00F4,\u00F2,\u00F6,\u0153,\u00F8,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        /* more_keys_for_e */ "\u00E9,\u00EA,\u00E8,\u0119,\u0117,\u0113,\u00EB",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00ED,\u00EE,\u00EC,\u00EF,\u012F,\u012B",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* more_keys_for_c */ "\u00E7,\u010D,\u0107",
    };

    /* Language rm: Raeto-Romance */
    private static final String[] LANGUAGE_rm = {
        /* more_keys_for_a */ null,
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* more_keys_for_o */ "\u00F2,\u00F3,\u00F6,\u00F4,\u00F5,\u0153,\u00F8",
    };

    /* Language ro: Romanian */
    private static final String[] LANGUAGE_ro = {
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E2,\u00E3,\u0103,\u00E0,\u00E1,\u00E4,\u00E6,\u00E5,\u0101",
        /* more_keys_for_o ~ */
        null, null, null,
        /* ~ more_keys_for_e */
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00EE,\u00EF,\u00EC,\u00ED,\u012F,\u012B",
        /* double_quotes */ "!text/double_9qm_rqm",
        /* single_quotes */ "!text/single_9qm_rqm",
        /* more_keys_for_c */ null,
        // U+0219: "ș" LATIN SMALL LETTER S WITH COMMA BELOW
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* more_keys_for_s */ "\u0219,\u00DF,\u015B,\u0161",
        /* more_keys_for_n ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_z */
        // U+021B: "ț" LATIN SMALL LETTER T WITH COMMA BELOW
        /* more_keys_for_t */ "\u021B",
    };

    /* Language ru: Russian */
    private static final String[] LANGUAGE_ru = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c ~ */
        null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ keylabel_for_nordic_row2_11 */
        // U+0451: "ё" CYRILLIC SMALL LETTER IO
        /* more_keys_for_cyrillic_ie */ "\u0451",
        /* more_keys_for_nordic_row2_10 */ null,
        // U+0449: "щ" CYRILLIC SMALL LETTER SHCHA
        /* keylabel_for_east_slavic_row1_9 */ "\u0449",
        // U+044B: "ы" CYRILLIC SMALL LETTER YERU
        /* keylabel_for_east_slavic_row2_2 */ "\u044B",
        // U+044D: "э" CYRILLIC SMALL LETTER E
        /* keylabel_for_east_slavic_row2_11 */ "\u044D",
        // U+0438: "и" CYRILLIC SMALL LETTER I
        /* keylabel_for_east_slavic_row3_5 */ "\u0438",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* more_keys_for_cyrillic_soft_sign */ "\u044A",
    };

    /* Language sk: Slovak */
    private static final String[] LANGUAGE_sk = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        /* more_keys_for_a */ "\u00E1,\u00E4,\u0101,\u00E0,\u00E2,\u00E3,\u00E5,\u00E6,\u0105",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* more_keys_for_o */ "\u00F4,\u00F3,\u00F6,\u00F2,\u00F5,\u0153,\u0151,\u00F8",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        /* more_keys_for_u */ "\u00FA,\u016F,\u00FC,\u016B,\u0173,\u00F9,\u00FB,\u0171",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        /* more_keys_for_e */ "\u00E9,\u011B,\u0113,\u0117,\u00E8,\u00EA,\u00EB,\u0119",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        /* more_keys_for_i */ "\u00ED,\u012B,\u012F,\u00EC,\u00EE,\u00EF,\u0131",
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* more_keys_for_c */ "\u010D,\u00E7,\u0107",
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* more_keys_for_s */ "\u0161,\u00DF,\u015B,\u015F",
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u0148,\u0146,\u00F1,\u0144,\u0144",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* more_keys_for_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* more_keys_for_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        /* more_keys_for_z */ "\u017E,\u017C,\u017A",
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        /* more_keys_for_t */ "\u0165,\u0163",
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* more_keys_for_l */ "\u013E,\u013A,\u013C,\u0142",
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* more_keys_for_g */ "\u0123,\u011F",
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keylabel_for_currency */ null,
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        /* more_keys_for_r */ "\u0155,\u0159,\u0157",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        /* more_keys_for_k */ "\u0137",
    };

    /* Language sl: Slovenian */
    private static final String[] LANGUAGE_sl = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* more_keys_for_c */ "\u010D,\u0107",
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* more_keys_for_s */ "\u0161",
        /* more_keys_for_n ~ */
        null, null, null,
        /* ~ more_keys_for_y */
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        /* more_keys_for_d */ "\u0111",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* more_keys_for_z */ "\u017E",
        /* more_keys_for_t ~ */
        null, null, null,
        /* ~ more_keys_for_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
    };

    /* Language sr: Serbian */
    private static final String[] LANGUAGE_sr = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c ~ */
        null, null, null,
        /* ~ more_keys_for_n */
        // END: More keys definitions for Serbian (Cyrillic)
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null,
        /* ~ more_keys_for_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keylabel_for_currency ~ */
        null, null, null, null, null, null,
        /* ~ keylabel_for_nordic_row2_11 */
        // U+0450: "ѐ" CYRILLIC SMALL LETTER IE WITH GRAVE
        /* more_keys_for_cyrillic_ie */ "\u0450",
        /* more_keys_for_nordic_row2_10 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_cyrillic_o */
        // TODO: Move these to sr-Latn once we can handle IETF language tag with script name specified.
        // BEGIN: More keys definitions for Serbian (Latin)
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // <string name="more_keys_for_s">&#x0161;,&#x00DF;,&#x015B;</string>
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // <string name="more_keys_for_c">&#x010D;,&#x00E7;,&#x0107;</string>
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        // <string name="more_keys_for_d">&#x010F;</string>
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // <string name="more_keys_for_z">&#x017E;,&#x017A;,&#x017C;</string>
        // END: More keys definitions for Serbian (Latin)
        // BEGIN: More keys definitions for Serbian (Cyrillic)
        // U+0437: "з" CYRILLIC SMALL LETTER ZE
        /* keylabel_for_south_slavic_row1_6 */ "\u0437",
        // U+045B: "ћ" CYRILLIC SMALL LETTER TSHE
        /* keylabel_for_south_slavic_row2_11 */ "\u045B",
        // U+0455: "ѕ" CYRILLIC SMALL LETTER DZE
        /* keylabel_for_south_slavic_row3_1 */ "\u0455",
        // U+0452: "ђ" CYRILLIC SMALL LETTER DJE
        /* keylabel_for_south_slavic_row3_8 */ "\u0452",
        // U+045D: "ѝ" CYRILLIC SMALL LETTER I WITH GRAVE
        /* more_keys_for_cyrillic_i */ "\u045D",
    };

    /* Language sv: Swedish */
    private static final String[] LANGUAGE_sv = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        /* more_keys_for_a */ "\u00E1,\u00E0,\u00E2,\u0105,\u00E3",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F3,\u00F2,\u00F4,\u00F5,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FC,\u00FA,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        /* more_keys_for_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0119",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        /* more_keys_for_i */ "\u00ED,\u00EC,\u00EE,\u00EF",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u00E7,\u0107,\u010D",
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* more_keys_for_s */ "\u015B,\u0161,\u015F,\u00DF",
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        /* more_keys_for_n */ "\u0144,\u00F1,\u0148",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* more_keys_for_y */ "\u00FD,\u00FF,\u00FC",
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* more_keys_for_d */ "\u00F0,\u010F",
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        /* more_keys_for_z */ "\u017A,\u017E,\u017C",
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        /* more_keys_for_t */ "\u0165,\u00FE",
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* more_keys_for_l */ "\u0142",
        /* more_keys_for_g */ null,
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keylabel_for_currency */ null,
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        /* more_keys_for_r */ "\u0159",
        /* more_keys_for_k */ null,
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        /* keylabel_for_nordic_row1_11 */ "\u00E5",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* keylabel_for_nordic_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* keylabel_for_nordic_row2_11 */ "\u00E4",
        /* more_keys_for_cyrillic_ie */ null,
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        /* more_keys_for_nordic_row2_10 */ "\u00F8,\u0153",
        /* keylabel_for_east_slavic_row1_9 ~ */
        null, null, null, null, null, null,
        /* ~ more_keys_for_punctuation */
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* more_keys_for_nordic_row2_11 */ "\u00E6",
    };

    /* Language sw: Swahili */
    private static final String[] LANGUAGE_sw = {
        // This is the same as English except more_keys_for_g.
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* more_keys_for_o */ "\u00F4,\u00F6,\u00F2,\u00F3,\u0153,\u00F8,\u014D,\u00F5",
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FB,\u00FC,\u00F9,\u00FA,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113",
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        /* more_keys_for_i */ "\u00EE,\u00EF,\u00ED,\u012B,\u00EC",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* more_keys_for_c */ "\u00E7",
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* more_keys_for_s */ "\u00DF",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* more_keys_for_n */ "\u00F1",
        /* label_to_alpha_key ~ */
        null, null, null, null, null, null,
        /* ~ more_keys_for_l */
        /* more_keys_for_g */ "g\'",
    };

    /* Language th: Thai */
    private static final String[] LANGUAGE_th = {
        /* more_keys_for_a ~ */
        null, null, null, null, null, null, null, null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0E01: "ก" THAI CHARACTER KO KAI
        // U+0E02: "ข" THAI CHARACTER KHO KHAI
        // U+0E04: "ค" THAI CHARACTER KHO KHWAI
        /* label_to_alpha_key */ "\u0E01\u0E02\u0E04",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+0E3F: "฿" THAI CURRENCY SYMBOL BAHT
        /* keylabel_for_currency */ "\u0E3F",
    };

    /* Language tl: Tagalog */
    private static final String[] LANGUAGE_tl = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* more_keys_for_a */ "\u00E1,\u00E0,\u00E4,\u00E2,\u00E3,\u00E5,\u0105,\u00E6,\u0101,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* more_keys_for_o */ "\u00F3,\u00F2,\u00F6,\u00F4,\u00F5,\u00F8,\u0153,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E9,\u00E8,\u00EB,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u00E7,\u0107,\u010D",
        /* more_keys_for_s */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* more_keys_for_n */ "\u00F1,\u0144",
    };

    /* Language tr: Turkish */
    private static final String[] LANGUAGE_tr = {
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        /* more_keys_for_a */ "\u00E2",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* more_keys_for_o */ "\u00F6,\u00F4,\u0153,\u00F2,\u00F3,\u00F5,\u00F8,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FC,\u00FB,\u00F9,\u00FA,\u016B",
        /* more_keys_for_e */ null,
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* more_keys_for_i */ "\u0131,\u00EE,\u00EF,\u00EC,\u00ED,\u012F,\u012B",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u00E7,\u0107,\u010D",
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* more_keys_for_s */ "\u015F,\u00DF,\u015B,\u0161",
        /* more_keys_for_n ~ */
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_l */
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* more_keys_for_g */ "\u011F",
    };

    /* Language uk: Ukrainian */
    private static final String[] LANGUAGE_uk = {
        /* more_keys_for_a ~ */
        null, null, null, null, null,
        /* ~ more_keys_for_i */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* more_keys_for_c ~ */
        null, null, null,
        /* ~ more_keys_for_n */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* label_to_alpha_key */ "\u0410\u0411\u0412",
        /* more_keys_for_y ~ */
        null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20B4: "₴" HRYVNIA SIGN
        /* keylabel_for_currency */ "\u20B4",
        /* more_keys_for_r ~ */
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_nordic_row2_10 */
        // U+0449: "щ" CYRILLIC SMALL LETTER SHCHA
        /* keylabel_for_east_slavic_row1_9 */ "\u0449",
        // U+0456: "і" CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
        /* keylabel_for_east_slavic_row2_2 */ "\u0456",
        // U+0454: "є" CYRILLIC SMALL LETTER UKRAINIAN IE
        /* keylabel_for_east_slavic_row2_11 */ "\u0454",
        // U+0438: "и" CYRILLIC SMALL LETTER I
        /* keylabel_for_east_slavic_row3_5 */ "\u0438",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* more_keys_for_cyrillic_soft_sign */ "\u044A",
        /* more_keys_for_punctuation ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null,
        /* ~ more_keys_for_cyrillic_en */
        // U+0491: "ґ" CYRILLIC SMALL LETTER GHE WITH UPTURN
        /* more_keys_for_cyrillic_ghe */ "\u0491",
        // U+0457: "ї" CYRILLIC SMALL LETTER YI
        /* more_keys_for_east_slavic_row2_2 */ "\u0457",
    };

    /* Language vi: Vietnamese */
    private static final String[] LANGUAGE_vi = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+1EA3: "ả" LATIN SMALL LETTER A WITH HOOK ABOVE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+1EA1: "ạ" LATIN SMALL LETTER A WITH DOT BELOW
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+1EB1: "ằ" LATIN SMALL LETTER A WITH BREVE AND GRAVE
        // U+1EAF: "ắ" LATIN SMALL LETTER A WITH BREVE AND ACUTE
        // U+1EB3: "ẳ" LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
        // U+1EB5: "ẵ" LATIN SMALL LETTER A WITH BREVE AND TILDE
        // U+1EB7: "ặ" LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+1EA7: "ầ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
        // U+1EA5: "ấ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
        // U+1EA9: "ẩ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
        // U+1EAB: "ẫ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
        // U+1EAD: "ậ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
        /* more_keys_for_a */ "\u00E0,\u00E1,\u1EA3,\u00E3,\u1EA1,\u0103,\u1EB1,\u1EAF,\u1EB3,\u1EB5,\u1EB7,\u00E2,\u1EA7,\u1EA5,\u1EA9,\u1EAB,\u1EAD",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+1ECF: "ỏ" LATIN SMALL LETTER O WITH HOOK ABOVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+1ECD: "ọ" LATIN SMALL LETTER O WITH DOT BELOW
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+1ED3: "ồ" LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
        // U+1ED1: "ố" LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
        // U+1ED5: "ổ" LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
        // U+1ED7: "ỗ" LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
        // U+1ED9: "ộ" LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
        // U+01A1: "ơ" LATIN SMALL LETTER O WITH HORN
        // U+1EDD: "ờ" LATIN SMALL LETTER O WITH HORN AND GRAVE
        // U+1EDB: "ớ" LATIN SMALL LETTER O WITH HORN AND ACUTE
        // U+1EDF: "ở" LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
        // U+1EE1: "ỡ" LATIN SMALL LETTER O WITH HORN AND TILDE
        // U+1EE3: "ợ" LATIN SMALL LETTER O WITH HORN AND DOT BELOW
        /* more_keys_for_o */ "\u00F2,\u00F3,\u1ECF,\u00F5,\u1ECD,\u00F4,\u1ED3,\u1ED1,\u1ED5,\u1ED7,\u1ED9,\u01A1,\u1EDD,\u1EDB,\u1EDF,\u1EE1,\u1EE3",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+1EE7: "ủ" LATIN SMALL LETTER U WITH HOOK ABOVE
        // U+0169: "ũ" LATIN SMALL LETTER U WITH TILDE
        // U+1EE5: "ụ" LATIN SMALL LETTER U WITH DOT BELOW
        // U+01B0: "ư" LATIN SMALL LETTER U WITH HORN
        // U+1EEB: "ừ" LATIN SMALL LETTER U WITH HORN AND GRAVE
        // U+1EE9: "ứ" LATIN SMALL LETTER U WITH HORN AND ACUTE
        // U+1EED: "ử" LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
        // U+1EEF: "ữ" LATIN SMALL LETTER U WITH HORN AND TILDE
        // U+1EF1: "ự" LATIN SMALL LETTER U WITH HORN AND DOT BELOW
        /* more_keys_for_u */ "\u00F9,\u00FA,\u1EE7,\u0169,\u1EE5,\u01B0,\u1EEB,\u1EE9,\u1EED,\u1EEF,\u1EF1",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+1EBB: "ẻ" LATIN SMALL LETTER E WITH HOOK ABOVE
        // U+1EBD: "ẽ" LATIN SMALL LETTER E WITH TILDE
        // U+1EB9: "ẹ" LATIN SMALL LETTER E WITH DOT BELOW
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+1EC1: "ề" LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
        // U+1EBF: "ế" LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
        // U+1EC3: "ể" LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
        // U+1EC5: "ễ" LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
        // U+1EC7: "ệ" LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
        /* more_keys_for_e */ "\u00E8,\u00E9,\u1EBB,\u1EBD,\u1EB9,\u00EA,\u1EC1,\u1EBF,\u1EC3,\u1EC5,\u1EC7",
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+1EC9: "ỉ" LATIN SMALL LETTER I WITH HOOK ABOVE
        // U+0129: "ĩ" LATIN SMALL LETTER I WITH TILDE
        // U+1ECB: "ị" LATIN SMALL LETTER I WITH DOT BELOW
        /* more_keys_for_i */ "\u00EC,\u00ED,\u1EC9,\u0129,\u1ECB",
        /* double_quotes ~ */
        null, null, null, null, null, null,
        /* ~ label_to_alpha_key */
        // U+1EF3: "ỳ" LATIN SMALL LETTER Y WITH GRAVE
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+1EF7: "ỷ" LATIN SMALL LETTER Y WITH HOOK ABOVE
        // U+1EF9: "ỹ" LATIN SMALL LETTER Y WITH TILDE
        // U+1EF5: "ỵ" LATIN SMALL LETTER Y WITH DOT BELOW
        /* more_keys_for_y */ "\u1EF3,\u00FD,\u1EF7,\u1EF9,\u1EF5",
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        /* more_keys_for_d */ "\u0111",
        /* more_keys_for_z ~ */
        null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20AB: "₫" DONG SIGN
        /* keylabel_for_currency */ "\u20AB",
    };

    /* Language zu: Zulu */
    private static final String[] LANGUAGE_zu = {
        // This is the same as English
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* more_keys_for_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* more_keys_for_o */ "\u00F4,\u00F6,\u00F2,\u00F3,\u0153,\u00F8,\u014D,\u00F5",
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* more_keys_for_u */ "\u00FB,\u00FC,\u00F9,\u00FA,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* more_keys_for_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113",
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        /* more_keys_for_i */ "\u00EE,\u00EF,\u00ED,\u012B,\u00EC",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* more_keys_for_c */ "\u00E7",
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* more_keys_for_s */ "\u00DF",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* more_keys_for_n */ "\u00F1",
    };

    /* Language zz: Alphabet */
    private static final String[] LANGUAGE_zz = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* more_keys_for_a */ "\u00E0,\u00E1,\u00E2,\u00E3,\u00E4,\u00E5,\u00E6,\u00E3,\u00E5,\u0101,\u0103,\u0105,\u00AA",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+014F: "ŏ" LATIN SMALL LETTER O WITH BREVE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* more_keys_for_o */ "\u00F2,\u00F3,\u00F4,\u00F5,\u00F6,\u00F8,\u014D,\u014F,\u0151,\u0153,\u00BA",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+0169: "ũ" LATIN SMALL LETTER U WITH TILDE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+016D: "ŭ" LATIN SMALL LETTER U WITH BREVE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        /* more_keys_for_u */ "\u00F9,\u00FA,\u00FB,\u00FC,\u0169,\u016B,\u016D,\u016F,\u0171,\u0173",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+0115: "ĕ" LATIN SMALL LETTER E WITH BREVE
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* more_keys_for_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113,\u0115,\u0117,\u0119,\u011B",
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0129: "ĩ" LATIN SMALL LETTER I WITH TILDE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+012D: "ĭ" LATIN SMALL LETTER I WITH BREVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* more_keys_for_i */ "\u00EC,\u00ED,\u00EE,\u00EF,\u0129,\u012B,\u012D,\u012F,\u0131,\u0133",
        /* double_quotes */ null,
        /* single_quotes */ null,
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+0109: "ĉ" LATIN SMALL LETTER C WITH CIRCUMFLEX
        // U+010B: "ċ" LATIN SMALL LETTER C WITH DOT ABOVE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* more_keys_for_c */ "\u00E7,\u0107,\u0109,\u010B,\u010D",
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015D: "ŝ" LATIN SMALL LETTER S WITH CIRCUMFLEX
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+017F: "ſ" LATIN SMALL LETTER LONG S
        /* more_keys_for_s */ "\u00DF,\u015B,\u015D,\u015F,\u0161,\u017F",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+0149: "ŉ" LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
        // U+014B: "ŋ" LATIN SMALL LETTER ENG
        /* more_keys_for_n */ "\u00F1,\u0144,\u0146,\u0148,\u0149,\u014B",
        /* label_to_alpha_key */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+0177: "ŷ" LATIN SMALL LETTER Y WITH CIRCUMFLEX
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* more_keys_for_y */ "\u00FD,\u0177,\u00FF,\u0133",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        /* more_keys_for_d */ "\u010F,\u0111,\u00F0",
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* more_keys_for_z */ "\u017A,\u017C,\u017E",
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+0167: "ŧ" LATIN SMALL LETTER T WITH STROKE
        /* more_keys_for_t */ "\u00FE,\u0163,\u0165,\u0167",
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        // U+0140: "ŀ" LATIN SMALL LETTER L WITH MIDDLE DOT
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* more_keys_for_l */ "\u013A,\u013C,\u013E,\u0140,\u0142",
        // U+011D: "ĝ" LATIN SMALL LETTER G WITH CIRCUMFLEX
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        // U+0121: "ġ" LATIN SMALL LETTER G WITH DOT ABOVE
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        /* more_keys_for_g */ "\u011D,\u011F,\u0121,\u0123",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keylabel_for_currency */
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        /* more_keys_for_r */ "\u0155,\u0157,\u0159",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        // U+0138: "ĸ" LATIN SMALL LETTER KRA
        /* more_keys_for_k */ "\u0137,\u0138",
        /* keylabel_for_nordic_row1_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null,
        /* ~ more_keys_for_question */
        // U+0125: "ĥ" LATIN SMALL LETTER H WITH CIRCUMFLEX
        /* more_keys_for_h */ "\u0125",
        // U+0175: "ŵ" LATIN SMALL LETTER W WITH CIRCUMFLEX
        /* more_keys_for_w */ "\u0175",
        /* more_keys_for_cyrillic_u ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null,
        /* ~ more_keys_for_v */
        // U+0135: "ĵ" LATIN SMALL LETTER J WITH CIRCUMFLEX
        /* more_keys_for_j */ "\u0135",
    };

    // TODO: Use the language + "_" + region representation for the locale string key.
    // Currently we are dropping the region from the key.
    private static final Object[] LANGUAGES_AND_TEXTS = {
    // "locale", TEXT_ARRAY,  /* numberOfNonNullText/lengthOf_TEXT_ARRAY localeName */
        "DEFAULT", LANGUAGE_DEFAULT, /* 170/170 default */
        "af", LANGUAGE_af,    /*   7/ 12 Afrikaans */
        "ar", LANGUAGE_ar,    /*  58/109 Arabic */
        "az", LANGUAGE_az_AZ, /*   8/ 17 Azerbaijani (Azerbaijan) */
        "be", LANGUAGE_be_BY, /*   9/ 32 Belarusian (Belarus) */
        "bg", LANGUAGE_bg,    /*   2/ 11 Bulgarian */
        "ca", LANGUAGE_ca,    /*  11/116 Catalan */
        "cs", LANGUAGE_cs,    /*  17/ 21 Czech */
        "da", LANGUAGE_da,    /*  19/ 34 Danish */
        "de", LANGUAGE_de,    /*  16/ 92 German */
        "el", LANGUAGE_el,    /*   1/ 11 Greek */
        "en", LANGUAGE_en,    /*   8/ 10 English */
        "eo", LANGUAGE_eo,    /*  26/128 Esperanto */
        "es", LANGUAGE_es,    /*   8/ 33 Spanish */
        "et", LANGUAGE_et_EE, /*  22/ 27 Estonian (Estonia) */
        "fa", LANGUAGE_fa,    /*  61/119 Persian */
        "fi", LANGUAGE_fi,    /*  10/ 34 Finnish */
        "fr", LANGUAGE_fr,    /*  13/ 92 French */
        "hi", LANGUAGE_hi,    /*  24/ 56 Hindi */
        "hr", LANGUAGE_hr,    /*   9/ 19 Croatian */
        "hu", LANGUAGE_hu,    /*   9/ 19 Hungarian */
        "hy", LANGUAGE_hy_AM, /*   8/122 Armenian (Armenia) */
        "is", LANGUAGE_is,    /*  13/ 25 Icelandic */
        "it", LANGUAGE_it,    /*   5/  5 Italian */
        "iw", LANGUAGE_iw,    /*  20/117 Hebrew */
        "ka", LANGUAGE_ka_GE, /*   3/ 11 Georgian (Georgia) */
        "kk", LANGUAGE_kk,    /*  15/114 Kazakh */
        "km", LANGUAGE_km_KH, /*   2/115 Khmer (Cambodia) */
        "ky", LANGUAGE_ky,    /*  10/ 81 Kirghiz */
        "lo", LANGUAGE_lo_LA, /*   2/ 20 Lao (Laos) */
        "lt", LANGUAGE_lt,    /*  18/ 22 Lithuanian */
        "lv", LANGUAGE_lv,    /*  18/ 22 Latvian */
        "mk", LANGUAGE_mk,    /*   9/ 86 Macedonian */
        "mn", LANGUAGE_mn_MN, /*   2/ 20 Mongolian (Mongolia) */
        "nb", LANGUAGE_nb,    /*  11/ 34 Norwegian Bokmål */
        "ne", LANGUAGE_ne_NP, /*  24/ 56 Nepali (Nepal) */
        "nl", LANGUAGE_nl,    /*   9/ 12 Dutch */
        "pl", LANGUAGE_pl,    /*  10/ 16 Polish */
        "pt", LANGUAGE_pt,    /*   6/  8 Portuguese */
        "rm", LANGUAGE_rm,    /*   1/  2 Raeto-Romance */
        "ro", LANGUAGE_ro,    /*   6/ 15 Romanian */
        "ru", LANGUAGE_ru,    /*   9/ 32 Russian */
        "sk", LANGUAGE_sk,    /*  20/ 22 Slovak */
        "sl", LANGUAGE_sl,    /*   8/ 19 Slovenian */
        "sr", LANGUAGE_sr,    /*  11/ 86 Serbian */
        "sv", LANGUAGE_sv,    /*  21/ 34 Swedish */
        "sw", LANGUAGE_sw,    /*   9/ 17 Swahili */
        "th", LANGUAGE_th,    /*   2/ 20 Thai */
        "tl", LANGUAGE_tl,    /*   7/ 10 Tagalog */
        "tr", LANGUAGE_tr,    /*   7/ 17 Turkish */
        "uk", LANGUAGE_uk,    /*  11/ 80 Ukrainian */
        "vi", LANGUAGE_vi,    /*   8/ 20 Vietnamese */
        "zu", LANGUAGE_zu,    /*   8/ 10 Zulu */
        "zz", LANGUAGE_zz,    /*  19/111 Alphabet */
    };

    static {
        for (int index = 0; index < NAMES.length; index++) {
            sNameToIndexesMap.put(NAMES[index], index);
        }

        for (int i = 0; i < LANGUAGES_AND_TEXTS.length; i += 2) {
            final String language = (String)LANGUAGES_AND_TEXTS[i];
            final String[] textsTable = (String[])LANGUAGES_AND_TEXTS[i + 1];
            sLanguageToTextsTableMap.put(language, textsTable);
            sTextsTableToLanguageMap.put(textsTable, language);
        }
    }
}
