/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.inputmethod.keyboard.internal;

import com.android.inputmethod.latin.utils.CollectionUtils;

import java.util.HashMap;

/**
 * !!!!! DO NOT EDIT THIS FILE !!!!!
 *
 * This file is generated by tools/make-keyboard-text. The base template file is
 *   tools/make-keyboard-text/res/src/com/android/inputmethod/keyboard/internal/
 *   KeyboardTextsTable.tmpl
 *
 * This file must be updated when any text resources in keyboard layout files have been changed.
 * These text resources are referred as "!text/<resource_name>" in keyboard XML definitions,
 * and should be defined in
 *   tools/make-keyboard-text/res/values-<locale>/donottranslate-more-keys.xml
 *
 * To update this file, please run the following commands.
 *   $ cd $ANDROID_BUILD_TOP
 *   $ mmm packages/inputmethods/LatinIME/tools/make-keyboard-text
 *   $ make-keyboard-text -java packages/inputmethods/LatinIME/java
 *
 * The updated source file will be generated to the following path (this file).
 *   packages/inputmethods/LatinIME/java/src/com/android/inputmethod/keyboard/internal/
 *   KeyboardTextsTable.java
 */
public final class KeyboardTextsTable {
    // Name to index map.
    private static final HashMap<String, Integer> sNameToIndexesMap = CollectionUtils.newHashMap();
    // Language to texts table map.
    private static final HashMap<String, String[]> sLanguageToTextsTableMap =
            CollectionUtils.newHashMap();
    // TODO: Remove this variable after debugging.
    // Texts table to language maps.
    private static final HashMap<String[], String> sTextsTableToLanguageMap =
            CollectionUtils.newHashMap();

    public static String getText(final String name, final String[] textsTable) {
        final Integer indexObj = sNameToIndexesMap.get(name);
        if (indexObj == null) {
            throw new RuntimeException("Unknown text name=" + name + " language="
                    + sTextsTableToLanguageMap.get(textsTable));
        }
        final int index = indexObj;
        final String text = (index < textsTable.length) ? textsTable[index] : null;
        if (text != null) {
            return text;
        }
        // Sanity check.
        if (index >= 0 && index < LANGUAGE_DEFAULT.length) {
            return LANGUAGE_DEFAULT[index];
        }
        // Throw exception for debugging purpose.
        throw new RuntimeException("Illegal index=" + index + " for name=" + name
                + " language=" + sTextsTableToLanguageMap.get(textsTable));
    }

    public static String[] getTextsTable(final String language) {
        final String[] textsTable = sLanguageToTextsTableMap.get(language);
        return textsTable != null ? textsTable : LANGUAGE_DEFAULT;
    }

    private static final String[] NAMES = {
    //  /* index:histogram */ "name",
        /*   0:32 */ "morekeys_a",
        /*   1:32 */ "morekeys_o",
        /*   2:30 */ "morekeys_u",
        /*   3:29 */ "morekeys_e",
        /*   4:28 */ "morekeys_i",
        /*   5:23 */ "morekeys_c",
        /*   6:23 */ "double_quotes",
        /*   7:22 */ "morekeys_n",
        /*   8:22 */ "single_quotes",
        /*   9:21 */ "keylabel_to_alpha",
        /*  10:20 */ "morekeys_s",
        /*  11:14 */ "morekeys_y",
        /*  12:13 */ "morekeys_d",
        /*  13:12 */ "morekeys_z",
        /*  14:10 */ "morekeys_t",
        /*  15:10 */ "morekeys_l",
        /*  16: 9 */ "morekeys_g",
        /*  17: 9 */ "single_angle_quotes",
        /*  18: 9 */ "double_angle_quotes",
        /*  19: 9 */ "keyspec_currency",
        /*  20: 8 */ "morekeys_r",
        /*  21: 6 */ "morekeys_k",
        /*  22: 6 */ "morekeys_cyrillic_ie",
        /*  23: 5 */ "keyspec_nordic_row1_11",
        /*  24: 5 */ "keyspec_nordic_row2_10",
        /*  25: 5 */ "keyspec_nordic_row2_11",
        /*  26: 5 */ "morekeys_nordic_row2_10",
        /*  27: 5 */ "keyspec_east_slavic_row1_9",
        /*  28: 5 */ "keyspec_east_slavic_row2_2",
        /*  29: 5 */ "keyspec_east_slavic_row2_11",
        /*  30: 5 */ "keyspec_east_slavic_row3_5",
        /*  31: 5 */ "morekeys_cyrillic_soft_sign",
        /*  32: 4 */ "morekeys_nordic_row2_11",
        /*  33: 4 */ "keyspec_symbols_1",
        /*  34: 4 */ "keyspec_symbols_2",
        /*  35: 4 */ "keyspec_symbols_3",
        /*  36: 4 */ "keyspec_symbols_4",
        /*  37: 4 */ "keyspec_symbols_5",
        /*  38: 4 */ "keyspec_symbols_6",
        /*  39: 4 */ "keyspec_symbols_7",
        /*  40: 4 */ "keyspec_symbols_8",
        /*  41: 4 */ "keyspec_symbols_9",
        /*  42: 4 */ "keyspec_symbols_0",
        /*  43: 4 */ "keylabel_to_symbol",
        /*  44: 4 */ "additional_morekeys_symbols_1",
        /*  45: 4 */ "additional_morekeys_symbols_2",
        /*  46: 4 */ "additional_morekeys_symbols_3",
        /*  47: 4 */ "additional_morekeys_symbols_4",
        /*  48: 4 */ "additional_morekeys_symbols_5",
        /*  49: 4 */ "additional_morekeys_symbols_6",
        /*  50: 4 */ "additional_morekeys_symbols_7",
        /*  51: 4 */ "additional_morekeys_symbols_8",
        /*  52: 4 */ "additional_morekeys_symbols_9",
        /*  53: 4 */ "additional_morekeys_symbols_0",
        /*  54: 3 */ "morekeys_punctuation",
        /*  55: 3 */ "morekeys_star",
        /*  56: 3 */ "keyspec_left_parenthesis",
        /*  57: 3 */ "keyspec_right_parenthesis",
        /*  58: 3 */ "keyspec_left_square_bracket",
        /*  59: 3 */ "keyspec_right_square_bracket",
        /*  60: 3 */ "keyspec_left_curly_bracket",
        /*  61: 3 */ "keyspec_right_curly_bracket",
        /*  62: 3 */ "keyspec_less_than",
        /*  63: 3 */ "keyspec_greater_than",
        /*  64: 3 */ "keyspec_less_than_equal",
        /*  65: 3 */ "keyspec_greater_than_equal",
        /*  66: 3 */ "keyspec_left_double_angle_quote",
        /*  67: 3 */ "keyspec_right_double_angle_quote",
        /*  68: 3 */ "keyspec_left_single_angle_quote",
        /*  69: 3 */ "keyspec_right_single_angle_quote",
        /*  70: 3 */ "keyspec_tablet_comma",
        /*  71: 3 */ "morekeys_tablet_period",
        /*  72: 3 */ "morekeys_question",
        /*  73: 2 */ "morekeys_h",
        /*  74: 2 */ "morekeys_w",
        /*  75: 2 */ "morekeys_east_slavic_row2_2",
        /*  76: 2 */ "morekeys_cyrillic_u",
        /*  77: 2 */ "morekeys_cyrillic_en",
        /*  78: 2 */ "morekeys_cyrillic_ghe",
        /*  79: 2 */ "morekeys_cyrillic_o",
        /*  80: 2 */ "morekeys_cyrillic_i",
        /*  81: 2 */ "keyspec_south_slavic_row1_6",
        /*  82: 2 */ "keyspec_south_slavic_row2_11",
        /*  83: 2 */ "keyspec_south_slavic_row3_1",
        /*  84: 2 */ "keyspec_south_slavic_row3_8",
        /*  85: 2 */ "keyspec_swiss_row1_11",
        /*  86: 2 */ "keyspec_swiss_row2_10",
        /*  87: 2 */ "keyspec_swiss_row2_11",
        /*  88: 2 */ "morekeys_swiss_row1_11",
        /*  89: 2 */ "morekeys_swiss_row2_10",
        /*  90: 2 */ "morekeys_swiss_row2_11",
        /*  91: 2 */ "keyspec_spanish_row2_10",
        /*  92: 2 */ "morekeys_bullet",
        /*  93: 2 */ "morekeys_left_parenthesis",
        /*  94: 2 */ "morekeys_right_parenthesis",
        /*  95: 2 */ "morekeys_arabic_diacritics",
        /*  96: 2 */ "keyspec_comma",
        /*  97: 2 */ "keyhintlabel_tablet_comma",
        /*  98: 2 */ "morekeys_tablet_comma",
        /*  99: 2 */ "keyhintlabel_period",
        /* 100: 2 */ "morekeys_period",
        /* 101: 2 */ "keyhintlabel_tablet_period",
        /* 102: 2 */ "keyspec_symbols_question",
        /* 103: 2 */ "keyspec_symbols_semicolon",
        /* 104: 2 */ "keyspec_symbols_percent",
        /* 105: 2 */ "morekeys_symbols_semicolon",
        /* 106: 2 */ "morekeys_symbols_percent",
        /* 107: 1 */ "morekeys_v",
        /* 108: 1 */ "morekeys_j",
        /* 109: 1 */ "morekeys_east_slavic_row2_11",
        /* 110: 1 */ "morekeys_cyrillic_ka",
        /* 111: 1 */ "morekeys_cyrillic_a",
        /* 112: 1 */ "morekeys_currency_dollar",
        /* 113: 1 */ "morekeys_tablet_punctuation",
        /* 114: 1 */ "morekeys_plus",
        /* 115: 1 */ "morekeys_less_than",
        /* 116: 1 */ "morekeys_greater_than",
        /* 117: 1 */ "keyspec_period",
        /* 118: 1 */ "keyspec_tablet_period",
        /* 119: 1 */ "morekeys_exclamation",
        /* 120: 1 */ "morekeys_q",
        /* 121: 1 */ "morekeys_x",
        /* 122: 1 */ "keyspec_q",
        /* 123: 1 */ "keyspec_w",
        /* 124: 1 */ "keyspec_y",
        /* 125: 1 */ "keyspec_x",
        /* 126: 0 */ "morekeys_currency",
        /* 127: 0 */ "morekeys_symbols_1",
        /* 128: 0 */ "morekeys_symbols_2",
        /* 129: 0 */ "morekeys_symbols_3",
        /* 130: 0 */ "morekeys_symbols_4",
        /* 131: 0 */ "morekeys_symbols_5",
        /* 132: 0 */ "morekeys_symbols_6",
        /* 133: 0 */ "morekeys_symbols_7",
        /* 134: 0 */ "morekeys_symbols_8",
        /* 135: 0 */ "morekeys_symbols_9",
        /* 136: 0 */ "morekeys_symbols_0",
        /* 137: 0 */ "morekeys_am_pm",
        /* 138: 0 */ "keyspec_settings",
        /* 139: 0 */ "keyspec_shortcut",
        /* 140: 0 */ "keyspec_action_next",
        /* 141: 0 */ "keyspec_action_previous",
        /* 142: 0 */ "keylabel_to_more_symbol",
        /* 143: 0 */ "keylabel_tablet_to_more_symbol",
        /* 144: 0 */ "keylabel_to_phone_numeric",
        /* 145: 0 */ "keylabel_to_phone_symbols",
        /* 146: 0 */ "keylabel_time_am",
        /* 147: 0 */ "keylabel_time_pm",
        /* 148: 0 */ "keyspec_popular_domain",
        /* 149: 0 */ "morekeys_popular_domain",
        /* 150: 0 */ "keyspecs_left_parenthesis_more_keys",
        /* 151: 0 */ "keyspecs_right_parenthesis_more_keys",
        /* 152: 0 */ "single_laqm_raqm",
        /* 153: 0 */ "single_raqm_laqm",
        /* 154: 0 */ "double_laqm_raqm",
        /* 155: 0 */ "double_raqm_laqm",
        /* 156: 0 */ "single_lqm_rqm",
        /* 157: 0 */ "single_9qm_lqm",
        /* 158: 0 */ "single_9qm_rqm",
        /* 159: 0 */ "single_rqm_9qm",
        /* 160: 0 */ "double_lqm_rqm",
        /* 161: 0 */ "double_9qm_lqm",
        /* 162: 0 */ "double_9qm_rqm",
        /* 163: 0 */ "double_rqm_9qm",
        /* 164: 0 */ "morekeys_single_quote",
        /* 165: 0 */ "morekeys_double_quote",
        /* 166: 0 */ "morekeys_tablet_double_quote",
        /* 167: 0 */ "keyspec_emoji_key",
    };

    private static final String EMPTY = "";

    /* Default texts */
    private static final String[] LANGUAGE_DEFAULT = {
        /* morekeys_a ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ morekeys_c */
        /* double_quotes */ "!text/double_lqm_rqm",
        /* morekeys_n */ EMPTY,
        /* single_quotes */ "!text/single_lqm_rqm",
        // Label for "switch to alphabetic" key.
        /* keylabel_to_alpha */ "ABC",
        /* morekeys_s ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ morekeys_g */
        /* single_angle_quotes */ "!text/single_laqm_raqm",
        /* double_angle_quotes */ "!text/double_laqm_raqm",
        /* keyspec_currency */ "$",
        /* morekeys_r ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ morekeys_nordic_row2_11 */
        /* keyspec_symbols_1 */ "1",
        /* keyspec_symbols_2 */ "2",
        /* keyspec_symbols_3 */ "3",
        /* keyspec_symbols_4 */ "4",
        /* keyspec_symbols_5 */ "5",
        /* keyspec_symbols_6 */ "6",
        /* keyspec_symbols_7 */ "7",
        /* keyspec_symbols_8 */ "8",
        /* keyspec_symbols_9 */ "9",
        /* keyspec_symbols_0 */ "0",
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "?123",
        /* additional_morekeys_symbols_1 ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ additional_morekeys_symbols_0 */
        /* morekeys_punctuation */ "!autoColumnOrder!8,\\,,?,!,#,!text/keyspec_right_parenthesis,!text/keyspec_left_parenthesis,/,;,',@,:,-,\",+,\\%,&",
        // U+2020: "†" DAGGER
        // U+2021: "‡" DOUBLE DAGGER
        // U+2605: "★" BLACK STAR
        /* morekeys_star */ "\u2020,\u2021,\u2605",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        /* keyspec_left_parenthesis */ "(",
        /* keyspec_right_parenthesis */ ")",
        /* keyspec_left_square_bracket */ "[",
        /* keyspec_right_square_bracket */ "]",
        /* keyspec_left_curly_bracket */ "{",
        /* keyspec_right_curly_bracket */ "}",
        /* keyspec_less_than */ "<",
        /* keyspec_greater_than */ ">",
        /* keyspec_less_than_equal */ "\u2264",
        /* keyspec_greater_than_equal */ "\u2265",
        /* keyspec_left_double_angle_quote */ "\u00AB",
        /* keyspec_right_double_angle_quote */ "\u00BB",
        /* keyspec_left_single_angle_quote */ "\u2039",
        /* keyspec_right_single_angle_quote */ "\u203A",
        /* keyspec_tablet_comma */ ",",
        /* morekeys_tablet_period */ "!text/morekeys_tablet_punctuation",
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* morekeys_question */ "\u00BF",
        /* morekeys_h ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ morekeys_swiss_row2_11 */
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* keyspec_spanish_row2_10 */ "\u00F1",
        // U+266A: "♪" EIGHTH NOTE
        // U+2665: "♥" BLACK HEART SUIT
        // U+2660: "♠" BLACK SPADE SUIT
        // U+2666: "♦" BLACK DIAMOND SUIT
        // U+2663: "♣" BLACK CLUB SUIT
        /* morekeys_bullet */ "\u266A,\u2665,\u2660,\u2666,\u2663",
        /* morekeys_left_parenthesis */ "!fixedColumnOrder!3,!text/keyspecs_left_parenthesis_more_keys",
        /* morekeys_right_parenthesis */ "!fixedColumnOrder!3,!text/keyspecs_right_parenthesis_more_keys",
        /* morekeys_arabic_diacritics */ EMPTY,
        // Comma key
        /* keyspec_comma */ ",",
        /* keyhintlabel_tablet_comma ~ */
        EMPTY, EMPTY, EMPTY,
        /* ~ keyhintlabel_period */
        /* morekeys_period */ "!text/morekeys_punctuation",
        /* keyhintlabel_tablet_period */ EMPTY,
        /* keyspec_symbols_question */ "?",
        /* keyspec_symbols_semicolon */ ";",
        /* keyspec_symbols_percent */ "%",
        /* morekeys_symbols_semicolon */ EMPTY,
        // U+2030: "‰" PER MILLE SIGN
        /* morekeys_symbols_percent */ "\u2030",
        /* morekeys_v ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ morekeys_cyrillic_a */
        // U+00A2: "¢" CENT SIGN
        // U+00A3: "£" POUND SIGN
        // U+20AC: "€" EURO SIGN
        // U+00A5: "¥" YEN SIGN
        // U+20B1: "₱" PESO SIGN
        /* morekeys_currency_dollar */ "\u00A2,\u00A3,\u20AC,\u00A5,\u20B1",
        /* morekeys_tablet_punctuation */ "!autoColumnOrder!7,\\,,',#,!text/keyspec_right_parenthesis,!text/keyspec_left_parenthesis,/,;,@,:,-,\",+,\\%,&",
        // U+00B1: "±" PLUS-MINUS SIGN
        /* morekeys_plus */ "\u00B1",
        /* morekeys_less_than */ "!fixedColumnOrder!3,!text/keyspec_left_single_angle_quote,!text/keyspec_less_than_equal,!text/keyspec_left_double_angle_quote",
        /* morekeys_greater_than */ "!fixedColumnOrder!3,!text/keyspec_right_single_angle_quote,!text/keyspec_greater_than_equal,!text/keyspec_right_double_angle_quote",
        // Period key
        /* keyspec_period */ ".",
        /* keyspec_tablet_period */ ".",
        // U+00A1: "¡" INVERTED EXCLAMATION MARK
        /* morekeys_exclamation */ "\u00A1",
        /* morekeys_q */ EMPTY,
        /* morekeys_x */ EMPTY,
        /* keyspec_q */ "q",
        /* keyspec_w */ "w",
        /* keyspec_y */ "y",
        /* keyspec_x */ "x",
        /* morekeys_currency */ "$,\u00A2,\u20AC,\u00A3,\u00A5,\u20B1",
        // U+00B9: "¹" SUPERSCRIPT ONE
        // U+00BD: "½" VULGAR FRACTION ONE HALF
        // U+2153: "⅓" VULGAR FRACTION ONE THIRD
        // U+00BC: "¼" VULGAR FRACTION ONE QUARTER
        // U+215B: "⅛" VULGAR FRACTION ONE EIGHTH
        /* morekeys_symbols_1 */ "\u00B9,\u00BD,\u2153,\u00BC,\u215B",
        // U+00B2: "²" SUPERSCRIPT TWO
        // U+2154: "⅔" VULGAR FRACTION TWO THIRDS
        /* morekeys_symbols_2 */ "\u00B2,\u2154",
        // U+00B3: "³" SUPERSCRIPT THREE
        // U+00BE: "¾" VULGAR FRACTION THREE QUARTERS
        // U+215C: "⅜" VULGAR FRACTION THREE EIGHTHS
        /* morekeys_symbols_3 */ "\u00B3,\u00BE,\u215C",
        // U+2074: "⁴" SUPERSCRIPT FOUR
        /* morekeys_symbols_4 */ "\u2074",
        // U+215D: "⅝" VULGAR FRACTION FIVE EIGHTHS
        /* morekeys_symbols_5 */ "\u215D",
        /* morekeys_symbols_6 */ EMPTY,
        // U+215E: "⅞" VULGAR FRACTION SEVEN EIGHTHS
        /* morekeys_symbols_7 */ "\u215E",
        /* morekeys_symbols_8 */ EMPTY,
        /* morekeys_symbols_9 */ EMPTY,
        // U+207F: "ⁿ" SUPERSCRIPT LATIN SMALL LETTER N
        // U+2205: "∅" EMPTY SET
        /* morekeys_symbols_0 */ "\u207F,\u2205",
        /* morekeys_am_pm */ "!fixedColumnOrder!2,!hasLabels!,!text/keylabel_time_am,!text/keylabel_time_pm",
        /* keyspec_settings */ "!icon/settings_key|!code/key_settings",
        /* keyspec_shortcut */ "!icon/shortcut_key|!code/key_shortcut",
        /* keyspec_action_next */ "!hasLabels!,!text/label_next_key|!code/key_action_next",
        /* keyspec_action_previous */ "!hasLabels!,!text/label_previous_key|!code/key_action_previous",
        // Label for "switch to more symbol" modifier key ("= \ <"). Must be short to fit on key!
        /* keylabel_to_more_symbol */ "= \\\\ <",
        // Label for "switch to more symbol" modifier key on tablets.  Must be short to fit on key!
        /* keylabel_tablet_to_more_symbol */ "~ [ <",
        // Label for "switch to phone numeric" key.  Must be short to fit on key!
        /* keylabel_to_phone_numeric */ "123",
        // Label for "switch to phone symbols" key.  Must be short to fit on key!
        // U+FF0A: "＊" FULLWIDTH ASTERISK
        // U+FF03: "＃" FULLWIDTH NUMBER SIGN
        /* keylabel_to_phone_symbols */ "\uFF0A\uFF03",
        // Key label for "ante meridiem"
        /* keylabel_time_am */ "AM",
        // Key label for "post meridiem"
        /* keylabel_time_pm */ "PM",
        /* keyspec_popular_domain */ ".com",
        // popular web domains for the locale - most popular, displayed on the keyboard
        /* morekeys_popular_domain */ "!hasLabels!,.net,.org,.gov,.edu",
        /* keyspecs_left_parenthesis_more_keys */ "!text/keyspec_less_than,!text/keyspec_left_curly_bracket,!text/keyspec_left_square_bracket",
        /* keyspecs_right_parenthesis_more_keys */ "!text/keyspec_greater_than,!text/keyspec_right_curly_bracket,!text/keyspec_right_square_bracket",
        // The following characters don't need BIDI mirroring.
        // U+2018: "‘" LEFT SINGLE QUOTATION MARK
        // U+2019: "’" RIGHT SINGLE QUOTATION MARK
        // U+201A: "‚" SINGLE LOW-9 QUOTATION MARK
        // U+201C: "“" LEFT DOUBLE QUOTATION MARK
        // U+201D: "”" RIGHT DOUBLE QUOTATION MARK
        // U+201E: "„" DOUBLE LOW-9 QUOTATION MARK
        // Abbreviations are:
        // laqm: LEFT-POINTING ANGLE QUOTATION MARK
        // raqm: RIGHT-POINTING ANGLE QUOTATION MARK
        // lqm: LEFT QUOTATION MARK
        // rqm: RIGHT QUOTATION MARK
        // 9qm: LOW-9 QUOTATION MARK
        // The following each quotation mark pair consist of
        // <opening quotation mark>, <closing quotation mark>
        // and is named after (single|double)_<opening quotation mark>_<closing quotation mark>.
        /* single_laqm_raqm */ "!text/keyspec_left_single_angle_quote,!text/keyspec_right_single_angle_quote",
        /* single_raqm_laqm */ "!text/keyspec_right_single_angle_quote,!text/keyspec_left_single_angle_quote",
        /* double_laqm_raqm */ "!text/keyspec_left_double_angle_quote,!text/keyspec_right_double_angle_quote",
        /* double_raqm_laqm */ "!text/keyspec_right_double_angle_quote,!text/keyspec_left_double_angle_quote",
        // The following each quotation mark triplet consists of
        // <another quotation mark>, <opening quotation mark>, <closing quotation mark>
        // and is named after (single|double)_<opening quotation mark>_<closing quotation mark>.
        /* single_lqm_rqm */ "\u201A,\u2018,\u2019",
        /* single_9qm_lqm */ "\u2019,\u201A,\u2018",
        /* single_9qm_rqm */ "\u2018,\u201A,\u2019",
        /* single_rqm_9qm */ "\u2018,\u2019,\u201A",
        /* double_lqm_rqm */ "\u201E,\u201C,\u201D",
        /* double_9qm_lqm */ "\u201D,\u201E,\u201C",
        /* double_9qm_rqm */ "\u201C,\u201E,\u201D",
        /* double_rqm_9qm */ "\u201C,\u201D,\u201E",
        /* morekeys_single_quote */ "!fixedColumnOrder!5,!text/single_quotes,!text/single_angle_quotes",
        /* morekeys_double_quote */ "!fixedColumnOrder!5,!text/double_quotes,!text/double_angle_quotes",
        /* morekeys_tablet_double_quote */ "!fixedColumnOrder!6,!text/double_quotes,!text/single_quotes,!text/double_angle_quotes,!text/single_angle_quotes",
        /* keyspec_emoji_key */ "!icon/emoji_key|!code/key_emoji",
    };

    /* Language af: Afrikaans */
    private static final String[] LANGUAGE_af = {
        // This is the same as Dutch except more keys of y and demoting vowels with diaeresis.
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E1,\u00E2,\u00E4,\u00E0,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F3,\u00F4,\u00F6,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FB,\u00FC,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_i */ "\u00ED,\u00EC,\u00EF,\u00EE,\u012F,\u012B,\u0133",
        /* morekeys_c */ null,
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
        /* single_quotes ~ */
        null, null, null,
        /* ~ morekeys_s */
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_y */ "\u00FD,\u0133",
    };

    /* Language ar: Arabic */
    private static final String[] LANGUAGE_ar = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0623: "أ" ARABIC LETTER ALEF WITH HAMZA ABOVE
        // U+200C: ZERO WIDTH NON-JOINER
        // U+0628: "ب" ARABIC LETTER BEH
        // U+062C: "ج" ARABIC LETTER JEEM
        /* keylabel_to_alpha */ "\u0623\u200C\u0628\u200C\u062C",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null,
        /* ~ morekeys_nordic_row2_11 */
        // U+0661: "١" ARABIC-INDIC DIGIT ONE
        /* keyspec_symbols_1 */ "\u0661",
        // U+0662: "٢" ARABIC-INDIC DIGIT TWO
        /* keyspec_symbols_2 */ "\u0662",
        // U+0663: "٣" ARABIC-INDIC DIGIT THREE
        /* keyspec_symbols_3 */ "\u0663",
        // U+0664: "٤" ARABIC-INDIC DIGIT FOUR
        /* keyspec_symbols_4 */ "\u0664",
        // U+0665: "٥" ARABIC-INDIC DIGIT FIVE
        /* keyspec_symbols_5 */ "\u0665",
        // U+0666: "٦" ARABIC-INDIC DIGIT SIX
        /* keyspec_symbols_6 */ "\u0666",
        // U+0667: "٧" ARABIC-INDIC DIGIT SEVEN
        /* keyspec_symbols_7 */ "\u0667",
        // U+0668: "٨" ARABIC-INDIC DIGIT EIGHT
        /* keyspec_symbols_8 */ "\u0668",
        // U+0669: "٩" ARABIC-INDIC DIGIT NINE
        /* keyspec_symbols_9 */ "\u0669",
        // U+0660: "٠" ARABIC-INDIC DIGIT ZERO
        /* keyspec_symbols_0 */ "\u0660",
        // Label for "switch to symbols" key.
        // U+061F: "؟" ARABIC QUESTION MARK
        /* keylabel_to_symbol */ "\u0663\u0662\u0661\u061F",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        // U+066B: "٫" ARABIC DECIMAL SEPARATOR
        // U+066C: "٬" ARABIC THOUSANDS SEPARATOR
        /* additional_morekeys_symbols_0 */ "0,\u066B,\u066C",
        /* morekeys_punctuation */ null,
        // U+2605: "★" BLACK STAR
        // U+066D: "٭" ARABIC FIVE POINTED STAR
        /* morekeys_star */ "\u2605,\u066D",
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        /* keyspec_left_parenthesis */ "(|)",
        /* keyspec_right_parenthesis */ ")|(",
        /* keyspec_left_square_bracket */ "[|]",
        /* keyspec_right_square_bracket */ "]|[",
        /* keyspec_left_curly_bracket */ "{|}",
        /* keyspec_right_curly_bracket */ "}|{",
        /* keyspec_less_than */ "<|>",
        /* keyspec_greater_than */ ">|<",
        /* keyspec_less_than_equal */ "\u2264|\u2265",
        /* keyspec_greater_than_equal */ "\u2265|\u2264",
        /* keyspec_left_double_angle_quote */ "\u00AB|\u00BB",
        /* keyspec_right_double_angle_quote */ "\u00BB|\u00AB",
        /* keyspec_left_single_angle_quote */ "\u2039|\u203A",
        /* keyspec_right_single_angle_quote */ "\u203A|\u2039",
        // U+061F: "؟" ARABIC QUESTION MARK
        // U+060C: "،" ARABIC COMMA
        // U+061B: "؛" ARABIC SEMICOLON
        /* keyspec_tablet_comma */ "\u060C",
        /* morekeys_tablet_period */ "!text/morekeys_arabic_diacritics",
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* morekeys_question */ "?,\u00BF",
        /* morekeys_h ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null,
        /* ~ keyspec_spanish_row2_10 */
        // U+266A: "♪" EIGHTH NOTE
        /* morekeys_bullet */ "\u266A",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+FD3E: "﴾" ORNATE LEFT PARENTHESIS
        // U+FD3F: "﴿" ORNATE RIGHT PARENTHESIS
        /* morekeys_left_parenthesis */ "!fixedColumnOrder!4,\uFD3E|\uFD3F,!text/keyspecs_left_parenthesis_more_keys",
        /* morekeys_right_parenthesis */ "!fixedColumnOrder!4,\uFD3F|\uFD3E,!text/keyspecs_right_parenthesis_more_keys",
        // U+0655: "ٕ" ARABIC HAMZA BELOW
        // U+0654: "ٔ" ARABIC HAMZA ABOVE
        // U+0652: "ْ" ARABIC SUKUN
        // U+064D: "ٍ" ARABIC KASRATAN
        // U+064C: "ٌ" ARABIC DAMMATAN
        // U+064B: "ً" ARABIC FATHATAN
        // U+0651: "ّ" ARABIC SHADDA
        // U+0656: "ٖ" ARABIC SUBSCRIPT ALEF
        // U+0670: "ٰ" ARABIC LETTER SUPERSCRIPT ALEF
        // U+0653: "ٓ" ARABIC MADDAH ABOVE
        // U+0650: "ِ" ARABIC KASRA
        // U+064F: "ُ" ARABIC DAMMA
        // U+064E: "َ" ARABIC FATHA
        // U+0640: "ـ" ARABIC TATWEEL
        // In order to make Tatweel easily distinguishable from other punctuations, we use consecutive Tatweels only for its displayed label.
        // Note: The space character is needed as a preceding letter to draw Arabic diacritics characters correctly.
        /* morekeys_arabic_diacritics */ "!fixedColumnOrder!7, \u0655|\u0655, \u0654|\u0654, \u0652|\u0652, \u064D|\u064D, \u064C|\u064C, \u064B|\u064B, \u0651|\u0651, \u0656|\u0656, \u0670|\u0670, \u0653|\u0653, \u0650|\u0650, \u064F|\u064F, \u064E|\u064E,\u0640\u0640\u0640|\u0640",
        // U+060C: "،" ARABIC COMMA
        /* keyspec_comma */ "\u060C",
        /* keyhintlabel_tablet_comma */ "\u061F",
        /* morekeys_tablet_comma */ "!fixedColumnOrder!4,:,!,\u061F,\u061B,-,/,\",\'",
        // U+0651: "ّ" ARABIC SHADDA
        /* keyhintlabel_period */ "\u0651",
        /* morekeys_period */ "!text/morekeys_arabic_diacritics",
        /* keyhintlabel_tablet_period */ "\u0651",
        /* keyspec_symbols_question */ "\u061F",
        /* keyspec_symbols_semicolon */ "\u061B",
        // U+066A: "٪" ARABIC PERCENT SIGN
        /* keyspec_symbols_percent */ "\u066A",
        /* morekeys_symbols_semicolon */ ";",
        // U+2030: "‰" PER MILLE SIGN
        /* morekeys_symbols_percent */ "\\%,\u2030",
    };

    /* Language az_AZ: Azerbaijani (Azerbaijan) */
    private static final String[] LANGUAGE_az_AZ = {
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        /* morekeys_a */ "\u00E2",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F6,\u00F4,\u0153,\u00F2,\u00F3,\u00F5,\u00F8,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FC,\u00FB,\u00F9,\u00FA,\u016B",
        // U+0259: "ə" LATIN SMALL LETTER SCHWA
        /* morekeys_e */ "\u0259",
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u0131,\u00EE,\u00EF,\u00EC,\u00ED,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u010D",
        /* double_quotes ~ */
        null, null, null, null,
        /* ~ keylabel_to_alpha */
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* morekeys_s */ "\u015F,\u00DF,\u015B,\u0161",
        /* morekeys_y ~ */
        null, null, null, null, null,
        /* ~ morekeys_l */
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* morekeys_g */ "\u011F",
    };

    /* Language be_BY: Belarusian (Belarus) */
    private static final String[] LANGUAGE_be_BY = {
        /* morekeys_a ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_c */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_lqm",
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_k */
        // U+0451: "ё" CYRILLIC SMALL LETTER IO
        /* morekeys_cyrillic_ie */ "\u0451",
        /* keyspec_nordic_row1_11 ~ */
        null, null, null, null,
        /* ~ morekeys_nordic_row2_10 */
        // U+045E: "ў" CYRILLIC SMALL LETTER SHORT U
        /* keyspec_east_slavic_row1_9 */ "\u045E",
        // U+044B: "ы" CYRILLIC SMALL LETTER YERU
        /* keyspec_east_slavic_row2_2 */ "\u044B",
        // U+044D: "э" CYRILLIC SMALL LETTER E
        /* keyspec_east_slavic_row2_11 */ "\u044D",
        // U+0456: "і" CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
        /* keyspec_east_slavic_row3_5 */ "\u0456",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* morekeys_cyrillic_soft_sign */ "\u044A",
    };

    /* Language bg: Bulgarian */
    private static final String[] LANGUAGE_bg = {
        /* morekeys_a ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_c */
        // single_quotes of Bulgarian is default single_quotes_right_left.
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ null,
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
    };

    /* Language ca: Catalan */
    private static final String[] LANGUAGE_ca = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E0,\u00E1,\u00E4,\u00E2,\u00E3,\u00E5,\u0105,\u00E6,\u0101,\u00AA",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F2,\u00F3,\u00F6,\u00F4,\u00F5,\u00F8,\u0153,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E8,\u00E9,\u00EB,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u010D",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
        /* single_quotes ~ */
        null, null, null, null, null, null, null,
        /* ~ morekeys_t */
        // U+00B7: "·" MIDDLE DOT
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* morekeys_l */ "l\u00B7l,\u0142",
        /* morekeys_g ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null,
        /* ~ additional_morekeys_symbols_0 */
        // U+00B7: "·" MIDDLE DOT
        /* morekeys_punctuation */ "!autoColumnOrder!9,\\,,?,!,\u00B7,#,),(,/,;,',@,:,-,\",+,\\%,&",
        /* morekeys_star ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ morekeys_swiss_row2_11 */
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* keyspec_spanish_row2_10 */ "\u00E7",
        /* morekeys_bullet ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ morekeys_currency_dollar */
        /* morekeys_tablet_punctuation */ "!autoColumnOrder!8,\\,,',\u00B7,#,),(,/,;,@,:,-,\",+,\\%,&",
    };

    /* Language cs: Czech */
    private static final String[] LANGUAGE_cs = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E1,\u00E0,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u016F,\u00FB,\u00FC,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u011B,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EE,\u00EF,\u00EC,\u012F,\u012B",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* morekeys_c */ "\u010D,\u00E7,\u0107",
        /* double_quotes */ "!text/double_9qm_lqm",
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u0148,\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        /* morekeys_s */ "\u0161,\u00DF,\u015B",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* morekeys_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        /* morekeys_z */ "\u017E,\u017A,\u017C",
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        /* morekeys_t */ "\u0165",
        /* morekeys_l */ null,
        /* morekeys_g */ null,
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keyspec_currency */ null,
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        /* morekeys_r */ "\u0159",
    };

    /* Language da: Danish */
    private static final String[] LANGUAGE_da = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E1,\u00E4,\u00E0,\u00E2,\u00E3,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F3,\u00F4,\u00F2,\u00F5,\u0153,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00FB,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        /* morekeys_e */ "\u00E9,\u00EB",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        /* morekeys_i */ "\u00ED,\u00EF",
        /* morekeys_c */ null,
        /* double_quotes */ "!text/double_9qm_lqm",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* morekeys_s */ "\u00DF,\u015B,\u0161",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "\u00FD,\u00FF",
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        /* morekeys_d */ "\u00F0",
        /* morekeys_z */ null,
        /* morekeys_t */ null,
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* morekeys_l */ "\u0142",
        /* morekeys_g */ null,
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keyspec_currency ~ */
        null, null, null, null,
        /* ~ morekeys_cyrillic_ie */
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        /* keyspec_nordic_row1_11 */ "\u00E5",
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* keyspec_nordic_row2_10 */ "\u00E6",
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* keyspec_nordic_row2_11 */ "\u00F8",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* morekeys_nordic_row2_10 */ "\u00E4",
        /* keyspec_east_slavic_row1_9 ~ */
        null, null, null, null, null,
        /* ~ morekeys_cyrillic_soft_sign */
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* morekeys_nordic_row2_11 */ "\u00F6",
    };

    /* Language de: German */
    private static final String[] LANGUAGE_de = {
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E4,\u00E2,\u00E0,\u00E1,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F6,\u00F4,\u00F2,\u00F3,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FC,\u00FB,\u00F9,\u00FA,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        /* morekeys_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0117",
        /* morekeys_i */ null,
        /* morekeys_c */ null,
        /* double_quotes */ "!text/double_9qm_lqm",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* morekeys_s */ "\u00DF,\u015B,\u0161",
        /* morekeys_y ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keyspec_currency ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ keyspec_south_slavic_row3_8 */
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* keyspec_swiss_row1_11 */ "\u00FC",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* keyspec_swiss_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* keyspec_swiss_row2_11 */ "\u00E4",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        /* morekeys_swiss_row1_11 */ "\u00E8",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        /* morekeys_swiss_row2_10 */ "\u00E9",
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        /* morekeys_swiss_row2_11 */ "\u00E0",
    };

    /* Language el: Greek */
    private static final String[] LANGUAGE_el = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0391: "Α" GREEK CAPITAL LETTER ALPHA
        // U+0392: "Β" GREEK CAPITAL LETTER BETA
        // U+0393: "Γ" GREEK CAPITAL LETTER GAMMA
        /* keylabel_to_alpha */ "\u0391\u0392\u0393",
    };

    /* Language en: English */
    private static final String[] LANGUAGE_en = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* morekeys_o */ "\u00F4,\u00F6,\u00F2,\u00F3,\u0153,\u00F8,\u014D,\u00F5",
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FB,\u00FC,\u00F9,\u00FA,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113",
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        /* morekeys_i */ "\u00EE,\u00EF,\u00ED,\u012B,\u00EC",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* morekeys_c */ "\u00E7",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* morekeys_n */ "\u00F1",
        /* single_quotes */ null,
        /* keylabel_to_alpha */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* morekeys_s */ "\u00DF",
    };

    /* Language eo: Esperanto */
    private static final String[] LANGUAGE_eo = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E1,\u00E0,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101,\u0103,\u0105,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D,\u0151,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0169: "ũ" LATIN SMALL LETTER U WITH TILDE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00B5: "µ" MICRO SIGN
        /* morekeys_u */ "\u00FA,\u016F,\u00FB,\u00FC,\u00F9,\u016B,\u0169,\u0171,\u0173,\u00B5",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u011B,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0129: "ĩ" LATIN SMALL LETTER I WITH TILDE
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_i */ "\u00ED,\u00EE,\u00EF,\u0129,\u00EC,\u012F,\u012B,\u0131,\u0133",
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+010B: "ċ" LATIN SMALL LETTER C WITH DOT ABOVE
        /* morekeys_c */ "\u0107,\u010D,\u00E7,\u010B",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+0149: "ŉ" LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
        // U+014B: "ŋ" LATIN SMALL LETTER ENG
        /* morekeys_n */ "\u00F1,\u0144,\u0146,\u0148,\u0149,\u014B",
        /* single_quotes */ null,
        /* keylabel_to_alpha */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0219: "ș" LATIN SMALL LETTER S WITH COMMA BELOW
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* morekeys_s */ "\u00DF,\u0161,\u015B,\u0219,\u015F",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+0177: "ŷ" LATIN SMALL LETTER Y WITH CIRCUMFLEX
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        /* morekeys_y */ "y,\u00FD,\u0177,\u00FF,\u00FE",
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        /* morekeys_d */ "\u00F0,\u010F,\u0111",
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* morekeys_z */ "\u017A,\u017C,\u017E",
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+021B: "ț" LATIN SMALL LETTER T WITH COMMA BELOW
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0167: "ŧ" LATIN SMALL LETTER T WITH STROKE
        /* morekeys_t */ "\u0165,\u021B,\u0163,\u0167",
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        // U+0140: "ŀ" LATIN SMALL LETTER L WITH MIDDLE DOT
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* morekeys_l */ "\u013A,\u013C,\u013E,\u0140,\u0142",
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        // U+0121: "ġ" LATIN SMALL LETTER G WITH DOT ABOVE
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        /* morekeys_g */ "\u011F,\u0121,\u0123",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keyspec_currency */
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        /* morekeys_r */ "\u0159,\u0155,\u0157",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        // U+0138: "ĸ" LATIN SMALL LETTER KRA
        /* morekeys_k */ "\u0137,\u0138",
        /* morekeys_cyrillic_ie ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ morekeys_question */
        // U+0125: "ĥ" LATIN SMALL LETTER H WITH CIRCUMFLEX
        // U+0127: "ħ" LATIN SMALL LETTER H WITH STROKE
        /* morekeys_h */ "\u0125,\u0127",
        // U+0175: "ŵ" LATIN SMALL LETTER W WITH CIRCUMFLEX
        /* morekeys_w */ "w,\u0175",
        /* morekeys_east_slavic_row2_2 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null,
        /* ~ morekeys_swiss_row2_11 */
        // U+0135: "ĵ" LATIN SMALL LETTER J WITH CIRCUMFLEX
        /* keyspec_spanish_row2_10 */ "\u0135",
        /* morekeys_bullet ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_symbols_percent */
        // U+0175: "ŵ" LATIN SMALL LETTER W WITH CIRCUMFLEX
        /* morekeys_v */ "w,\u0175",
        /* morekeys_j ~ */
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_exclamation */
        /* morekeys_q */ "q",
        /* morekeys_x */ "x",
        // U+015D: "ŝ" LATIN SMALL LETTER S WITH CIRCUMFLEX
        /* keyspec_q */ "\u015D",
        // U+011D: "ĝ" LATIN SMALL LETTER G WITH CIRCUMFLEX
        /* keyspec_w */ "\u011D",
        // U+016D: "ŭ" LATIN SMALL LETTER U WITH BREVE
        /* keyspec_y */ "\u016D",
        // U+0109: "ĉ" LATIN SMALL LETTER C WITH CIRCUMFLEX
        /* keyspec_x */ "\u0109",
    };

    /* Language es: Spanish */
    private static final String[] LANGUAGE_es = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E1,\u00E0,\u00E4,\u00E2,\u00E3,\u00E5,\u0105,\u00E6,\u0101,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F3,\u00F2,\u00F6,\u00F4,\u00F5,\u00F8,\u0153,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EB,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u010D",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
        /* single_quotes ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null,
        /* ~ additional_morekeys_symbols_0 */
        // U+00A1: "¡" INVERTED EXCLAMATION MARK
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* morekeys_punctuation */ "!autoColumnOrder!9,\\,,?,!,#,),(,/,;,\u00A1,',@,:,-,\",+,\\%,&,\u00BF",
    };

    /* Language et_EE: Estonian (Estonia) */
    private static final String[] LANGUAGE_et_EE = {
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        /* morekeys_a */ "\u00E4,\u0101,\u00E0,\u00E1,\u00E2,\u00E3,\u00E5,\u00E6,\u0105",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* morekeys_o */ "\u00F6,\u00F5,\u00F2,\u00F3,\u00F4,\u0153,\u0151,\u00F8",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        /* morekeys_u */ "\u00FC,\u016B,\u0173,\u00F9,\u00FA,\u00FB,\u016F,\u0171",
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* morekeys_e */ "\u0113,\u00E8,\u0117,\u00E9,\u00EA,\u00EB,\u0119,\u011B",
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        /* morekeys_i */ "\u012B,\u00EC,\u012F,\u00ED,\u00EE,\u00EF,\u0131",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* morekeys_c */ "\u010D,\u00E7,\u0107",
        /* double_quotes */ "!text/double_9qm_lqm",
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u0146,\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* morekeys_s */ "\u0161,\u00DF,\u015B,\u015F",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* morekeys_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        /* morekeys_z */ "\u017E,\u017C,\u017A",
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        /* morekeys_t */ "\u0163,\u0165",
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        /* morekeys_l */ "\u013C,\u0142,\u013A,\u013E",
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* morekeys_g */ "\u0123,\u011F",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keyspec_currency */
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        /* morekeys_r */ "\u0157,\u0159,\u0155",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        /* morekeys_k */ "\u0137",
        /* morekeys_cyrillic_ie */ null,
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* keyspec_nordic_row1_11 */ "\u00FC",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* keyspec_nordic_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* keyspec_nordic_row2_11 */ "\u00E4",
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* morekeys_nordic_row2_10 */ "\u00F5",
    };

    /* Language eu_ES: Basque (Spain) */
    private static final String[] LANGUAGE_eu_ES = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E1,\u00E0,\u00E4,\u00E2,\u00E3,\u00E5,\u0105,\u00E6,\u0101,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F3,\u00F2,\u00F6,\u00F4,\u00F5,\u00F8,\u0153,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EB,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u010D",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
    };

    /* Language fa: Persian */
    private static final String[] LANGUAGE_fa = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0627: "ا" ARABIC LETTER ALEF
        // U+200C: ZERO WIDTH NON-JOINER
        // U+0628: "ب" ARABIC LETTER BEH
        // U+067E: "پ" ARABIC LETTER PEH
        /* keylabel_to_alpha */ "\u0627\u200C\u0628\u200C\u067E",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+FDFC: "﷼" RIAL SIGN
        /* keyspec_currency */ "\uFDFC",
        /* morekeys_r ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_nordic_row2_11 */
        // U+06F1: "۱" EXTENDED ARABIC-INDIC DIGIT ONE
        /* keyspec_symbols_1 */ "\u06F1",
        // U+06F2: "۲" EXTENDED ARABIC-INDIC DIGIT TWO
        /* keyspec_symbols_2 */ "\u06F2",
        // U+06F3: "۳" EXTENDED ARABIC-INDIC DIGIT THREE
        /* keyspec_symbols_3 */ "\u06F3",
        // U+06F4: "۴" EXTENDED ARABIC-INDIC DIGIT FOUR
        /* keyspec_symbols_4 */ "\u06F4",
        // U+06F5: "۵" EXTENDED ARABIC-INDIC DIGIT FIVE
        /* keyspec_symbols_5 */ "\u06F5",
        // U+06F6: "۶" EXTENDED ARABIC-INDIC DIGIT SIX
        /* keyspec_symbols_6 */ "\u06F6",
        // U+06F7: "۷" EXTENDED ARABIC-INDIC DIGIT SEVEN
        /* keyspec_symbols_7 */ "\u06F7",
        // U+06F8: "۸" EXTENDED ARABIC-INDIC DIGIT EIGHT
        /* keyspec_symbols_8 */ "\u06F8",
        // U+06F9: "۹" EXTENDED ARABIC-INDIC DIGIT NINE
        /* keyspec_symbols_9 */ "\u06F9",
        // U+06F0: "۰" EXTENDED ARABIC-INDIC DIGIT ZERO
        /* keyspec_symbols_0 */ "\u06F0",
        // Label for "switch to symbols" key.
        // U+061F: "؟" ARABIC QUESTION MARK
        /* keylabel_to_symbol */ "\u06F3\u06F2\u06F1\u061F",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        // U+066B: "٫" ARABIC DECIMAL SEPARATOR
        // U+066C: "٬" ARABIC THOUSANDS SEPARATOR
        /* additional_morekeys_symbols_0 */ "0,\u066B,\u066C",
        /* morekeys_punctuation */ null,
        // U+2605: "★" BLACK STAR
        // U+066D: "٭" ARABIC FIVE POINTED STAR
        /* morekeys_star */ "\u2605,\u066D",
        /* keyspec_left_parenthesis */ "(|)",
        /* keyspec_right_parenthesis */ ")|(",
        /* keyspec_left_square_bracket */ "[|]",
        /* keyspec_right_square_bracket */ "]|[",
        /* keyspec_left_curly_bracket */ "{|}",
        /* keyspec_right_curly_bracket */ "}|{",
        /* keyspec_less_than */ "<|>",
        /* keyspec_greater_than */ ">|<",
        /* keyspec_less_than_equal */ "\u2264|\u2265",
        /* keyspec_greater_than_equal */ "\u2265|\u2264",
        /* keyspec_left_double_angle_quote */ "\u00AB|\u00BB",
        /* keyspec_right_double_angle_quote */ "\u00BB|\u00AB",
        /* keyspec_left_single_angle_quote */ "\u2039|\u203A",
        /* keyspec_right_single_angle_quote */ "\u203A|\u2039",
        // U+060C: "،" ARABIC COMMA
        // U+061B: "؛" ARABIC SEMICOLON
        // U+061F: "؟" ARABIC QUESTION MARK
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        /* keyspec_tablet_comma */ "\u060C",
        /* morekeys_tablet_period */ "!text/morekeys_arabic_diacritics",
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* morekeys_question */ "?,\u00BF",
        /* morekeys_h ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null,
        /* ~ keyspec_spanish_row2_10 */
        // U+266A: "♪" EIGHTH NOTE
        /* morekeys_bullet */ "\u266A",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+FD3E: "﴾" ORNATE LEFT PARENTHESIS
        // U+FD3F: "﴿" ORNATE RIGHT PARENTHESIS
        /* morekeys_left_parenthesis */ "!fixedColumnOrder!4,\uFD3E|\uFD3F,!text/keyspecs_left_parenthesis_more_keys",
        /* morekeys_right_parenthesis */ "!fixedColumnOrder!4,\uFD3F|\uFD3E,!text/keyspecs_right_parenthesis_more_keys",
        // U+0655: "ٕ" ARABIC HAMZA BELOW
        // U+0652: "ْ" ARABIC SUKUN
        // U+0651: "ّ" ARABIC SHADDA
        // U+064C: "ٌ" ARABIC DAMMATAN
        // U+064D: "ٍ" ARABIC KASRATAN
        // U+064B: "ً" ARABIC FATHATAN
        // U+0654: "ٔ" ARABIC HAMZA ABOVE
        // U+0656: "ٖ" ARABIC SUBSCRIPT ALEF
        // U+0670: "ٰ" ARABIC LETTER SUPERSCRIPT ALEF
        // U+0653: "ٓ" ARABIC MADDAH ABOVE
        // U+064F: "ُ" ARABIC DAMMA
        // U+0650: "ِ" ARABIC KASRA
        // U+064E: "َ" ARABIC FATHA
        // U+0640: "ـ" ARABIC TATWEEL
        // In order to make Tatweel easily distinguishable from other punctuations, we use consecutive Tatweels only for its displayed label.
        // Note: The space character is needed as a preceding letter to draw Arabic diacritics characters correctly.
        /* morekeys_arabic_diacritics */ "!fixedColumnOrder!7, \u0655|\u0655, \u0652|\u0652, \u0651|\u0651, \u064C|\u064C, \u064D|\u064D, \u064B|\u064B, \u0654|\u0654, \u0656|\u0656, \u0670|\u0670, \u0653|\u0653, \u064F|\u064F, \u0650|\u0650, \u064E|\u064E,\u0640\u0640\u0640|\u0640",
        // U+060C: "،" ARABIC COMMA
        /* keyspec_comma */ "\u060C",
        /* keyhintlabel_tablet_comma */ "\u061F",
        /* morekeys_tablet_comma */ "!fixedColumnOrder!4,:,!,\u061F,\u061B,-,/,!text/keyspec_left_double_angle_quote,!text/keyspec_right_double_angle_quote",
        // U+064B: "ً" ARABIC FATHATAN
        /* keyhintlabel_period */ "\u064B",
        /* morekeys_period */ "!text/morekeys_arabic_diacritics",
        /* keyhintlabel_tablet_period */ "\u064B",
        /* keyspec_symbols_question */ "\u061F",
        /* keyspec_symbols_semicolon */ "\u061B",
        // U+066A: "٪" ARABIC PERCENT SIGN
        /* keyspec_symbols_percent */ "\u066A",
        /* morekeys_symbols_semicolon */ ";",
        // U+2030: "‰" PER MILLE SIGN
        /* morekeys_symbols_percent */ "\\%,\u2030",
        /* morekeys_v ~ */
        null, null, null, null, null, null, null, null,
        /* ~ morekeys_plus */
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        /* morekeys_less_than */ "!fixedColumnOrder!3,!text/keyspec_left_single_angle_quote,!text/keyspec_less_than_equal,!text/keyspec_less_than",
        /* morekeys_greater_than */ "!fixedColumnOrder!3,!text/keyspec_right_single_angle_quote,!text/keyspec_greater_than_equal,!text/keyspec_greater_than",
    };

    /* Language fi: Finnish */
    private static final String[] LANGUAGE_fi = {
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E6,\u00E0,\u00E1,\u00E2,\u00E3,\u0101",
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F8,\u00F4,\u00F2,\u00F3,\u00F5,\u0153,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* morekeys_u */ "\u00FC",
        /* morekeys_e ~ */
        null, null, null, null, null, null, null,
        /* ~ keylabel_to_alpha */
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        /* morekeys_s */ "\u0161,\u00DF,\u015B",
        /* morekeys_y */ null,
        /* morekeys_d */ null,
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        /* morekeys_z */ "\u017E,\u017A,\u017C",
        /* morekeys_t ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_cyrillic_ie */
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        /* keyspec_nordic_row1_11 */ "\u00E5",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* keyspec_nordic_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* keyspec_nordic_row2_11 */ "\u00E4",
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* morekeys_nordic_row2_10 */ "\u00F8",
        /* keyspec_east_slavic_row1_9 ~ */
        null, null, null, null, null,
        /* ~ morekeys_cyrillic_soft_sign */
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* morekeys_nordic_row2_11 */ "\u00E6",
    };

    /* Language fr: French */
    private static final String[] LANGUAGE_fr = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E0,\u00E2,%,\u00E6,\u00E1,\u00E4,\u00E3,\u00E5,\u0101,\u00AA",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F4,\u0153,%,\u00F6,\u00F2,\u00F3,\u00F5,\u00F8,\u014D,\u00BA",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00F9,\u00FB,%,\u00FC,\u00FA,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EA,\u00EB,%,\u0119,\u0117,\u0113",
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00EE,%,\u00EF,\u00EC,\u00ED,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,%,\u0107,\u010D",
        /* double_quotes ~ */
        null, null, null, null, null,
        /* ~ morekeys_s */
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "%,\u00FF",
        /* morekeys_d ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ keyspec_south_slavic_row3_8 */
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        /* keyspec_swiss_row1_11 */ "\u00E8",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        /* keyspec_swiss_row2_10 */ "\u00E9",
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        /* keyspec_swiss_row2_11 */ "\u00E0",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        /* morekeys_swiss_row1_11 */ "\u00FC",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* morekeys_swiss_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* morekeys_swiss_row2_11 */ "\u00E4",
    };

    /* Language gl_ES: Gallegan (Spain) */
    private static final String[] LANGUAGE_gl_ES = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E1,\u00E0,\u00E4,\u00E2,\u00E3,\u00E5,\u0105,\u00E6,\u0101,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F3,\u00F2,\u00F6,\u00F4,\u00F5,\u00F8,\u0153,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EB,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u010D",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
    };

    /* Language hi: Hindi */
    private static final String[] LANGUAGE_hi = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0915: "क" DEVANAGARI LETTER KA
        // U+0916: "ख" DEVANAGARI LETTER KHA
        // U+0917: "ग" DEVANAGARI LETTER GA
        /* keylabel_to_alpha */ "\u0915\u0916\u0917",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20B9: "₹" INDIAN RUPEE SIGN
        /* keyspec_currency */ "\u20B9",
        /* morekeys_r ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_nordic_row2_11 */
        // U+0967: "१" DEVANAGARI DIGIT ONE
        /* keyspec_symbols_1 */ "\u0967",
        // U+0968: "२" DEVANAGARI DIGIT TWO
        /* keyspec_symbols_2 */ "\u0968",
        // U+0969: "३" DEVANAGARI DIGIT THREE
        /* keyspec_symbols_3 */ "\u0969",
        // U+096A: "४" DEVANAGARI DIGIT FOUR
        /* keyspec_symbols_4 */ "\u096A",
        // U+096B: "५" DEVANAGARI DIGIT FIVE
        /* keyspec_symbols_5 */ "\u096B",
        // U+096C: "६" DEVANAGARI DIGIT SIX
        /* keyspec_symbols_6 */ "\u096C",
        // U+096D: "७" DEVANAGARI DIGIT SEVEN
        /* keyspec_symbols_7 */ "\u096D",
        // U+096E: "८" DEVANAGARI DIGIT EIGHT
        /* keyspec_symbols_8 */ "\u096E",
        // U+096F: "९" DEVANAGARI DIGIT NINE
        /* keyspec_symbols_9 */ "\u096F",
        // U+0966: "०" DEVANAGARI DIGIT ZERO
        /* keyspec_symbols_0 */ "\u0966",
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "?\u0967\u0968\u0969",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
    };

    /* Language hr: Croatian */
    private static final String[] LANGUAGE_hr = {
        /* morekeys_a ~ */
        null, null, null, null, null,
        /* ~ morekeys_i */
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* morekeys_c */ "\u010D,\u0107,\u00E7",
        /* double_quotes */ "!text/double_9qm_rqm",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_rqm",
        /* keylabel_to_alpha */ null,
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* morekeys_s */ "\u0161,\u015B,\u00DF",
        /* morekeys_y */ null,
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        /* morekeys_d */ "\u0111",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        /* morekeys_z */ "\u017E,\u017A,\u017C",
        /* morekeys_t ~ */
        null, null, null,
        /* ~ morekeys_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
    };

    /* Language hu: Hungarian */
    private static final String[] LANGUAGE_hu = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E1,\u00E0,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F3,\u00F6,\u0151,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u0171,\u00FB,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EE,\u00EF,\u00EC,\u012F,\u012B",
        /* morekeys_c */ null,
        /* double_quotes */ "!text/double_9qm_rqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_rqm",
        /* keylabel_to_alpha ~ */
        null, null, null, null, null, null, null, null,
        /* ~ morekeys_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
    };

    /* Language hy_AM: Armenian (Armenia) */
    private static final String[] LANGUAGE_hy_AM = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0531: "Ա" ARMENIAN CAPITAL LETTER AYB
        // U+0532: "Բ" ARMENIAN CAPITAL LETTER BEN
        // U+0533: "Գ" ARMENIAN CAPITAL LETTER GIM
        /* keylabel_to_alpha */ "\u0531\u0532\u0533",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ additional_morekeys_symbols_0 */
        // U+055E: "՞" ARMENIAN QUESTION MARK
        // U+055C: "՜" ARMENIAN EXCLAMATION MARK
        // U+055A: "՚" ARMENIAN APOSTROPHE
        // U+0559: "ՙ" ARMENIAN MODIFIER LETTER LEFT HALF RING
        // U+055D: "՝" ARMENIAN COMMA
        // U+055B: "՛" ARMENIAN EMPHASIS MARK
        // U+058A: "֊" ARMENIAN HYPHEN
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+055F: "՟" ARMENIAN ABBREVIATION MARK
        /* morekeys_punctuation */ "!autoColumnOrder!8,\\,,\u055E,\u055C,.,\u055A,\u0559,?,!,\u055D,\u055B,\u058A,\u00BB,\u00AB,\u055F,;,:",
        /* morekeys_star ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ keyspec_right_single_angle_quote */
        // U+058F: "֏" ARMENIAN DRAM SIGN
        // TODO: Enable this when we have glyph for the following letter
        // <string name="keyspec_currency">&#x058F;</string>
        // 
        // U+055D: "՝" ARMENIAN COMMA
        /* keyspec_tablet_comma */ "\u055D",
        /* morekeys_tablet_period */ "!text/morekeys_punctuation",
        // U+055E: "՞" ARMENIAN QUESTION MARK
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* morekeys_question */ "\u055E,\u00BF",
        /* morekeys_h ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_greater_than */
        // U+0589: "։" ARMENIAN FULL STOP
        /* keyspec_period */ "\u0589",
        /* keyspec_tablet_period */ "\u0589",
        // U+055C: "՜" ARMENIAN EXCLAMATION MARK
        // U+00A1: "¡" INVERTED EXCLAMATION MARK
        /* morekeys_exclamation */ "\u055C,\u00A1",
    };

    /* Language is: Icelandic */
    private static final String[] LANGUAGE_is = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E1,\u00E4,\u00E6,\u00E5,\u00E0,\u00E2,\u00E3,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00FB,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00EB,\u00E8,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EF,\u00EE,\u00EC,\u012F,\u012B",
        /* morekeys_c */ null,
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        /* morekeys_s */ null,
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "\u00FD,\u00FF",
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        /* morekeys_d */ "\u00F0",
        /* morekeys_z */ null,
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        /* morekeys_t */ "\u00FE",
    };

    /* Language it: Italian */
    private static final String[] LANGUAGE_it = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101,\u00AA",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F2,\u00F3,\u00F4,\u00F6,\u00F5,\u0153,\u00F8,\u014D,\u00BA",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00F9,\u00FA,\u00FB,\u00FC,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00EC,\u00ED,\u00EE,\u00EF,\u012F,\u012B",
    };

    /* Language iw: Hebrew */
    private static final String[] LANGUAGE_iw = {
        /* morekeys_a ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_c */
        /* double_quotes */ "!text/double_rqm_9qm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_rqm_9qm",
        // Label for "switch to alphabetic" key.
        // U+05D0: "א" HEBREW LETTER ALEF
        // U+05D1: "ב" HEBREW LETTER BET
        // U+05D2: "ג" HEBREW LETTER GIMEL
        /* keylabel_to_alpha */ "\u05D0\u05D1\u05D2",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20AA: "₪" NEW SHEQEL SIGN
        /* keyspec_currency */ "\u20AA",
        /* morekeys_r ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null,
        /* ~ morekeys_punctuation */
        // U+2605: "★" BLACK STAR
        /* morekeys_star */ "\u2605",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        /* keyspec_left_parenthesis */ "(|)",
        /* keyspec_right_parenthesis */ ")|(",
        /* keyspec_left_square_bracket */ "[|]",
        /* keyspec_right_square_bracket */ "]|[",
        /* keyspec_left_curly_bracket */ "{|}",
        /* keyspec_right_curly_bracket */ "}|{",
        /* keyspec_less_than */ "<|>",
        /* keyspec_greater_than */ ">|<",
        /* keyspec_less_than_equal */ "\u2264|\u2265",
        /* keyspec_greater_than_equal */ "\u2265|\u2264",
        /* keyspec_left_double_angle_quote */ "\u00AB|\u00BB",
        /* keyspec_right_double_angle_quote */ "\u00BB|\u00AB",
        /* keyspec_left_single_angle_quote */ "\u2039|\u203A",
        /* keyspec_right_single_angle_quote */ "\u203A|\u2039",
        /* keyspec_tablet_comma ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_tablet_punctuation */
        // U+00B1: "±" PLUS-MINUS SIGN
        // U+FB29: "﬩" HEBREW LETTER ALTERNATIVE PLUS SIGN
        /* morekeys_plus */ "\u00B1,\uFB29",
    };

    /* Language ka_GE: Georgian (Georgia) */
    private static final String[] LANGUAGE_ka_GE = {
        /* morekeys_a ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_c */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_lqm",
        // Label for "switch to alphabetic" key.
        // U+10D0: "ა" GEORGIAN LETTER AN
        // U+10D1: "ბ" GEORGIAN LETTER BAN
        // U+10D2: "გ" GEORGIAN LETTER GAN
        /* keylabel_to_alpha */ "\u10D0\u10D1\u10D2",
    };

    /* Language kk: Kazakh */
    private static final String[] LANGUAGE_kk = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_k */
        // U+0451: "ё" CYRILLIC SMALL LETTER IO
        /* morekeys_cyrillic_ie */ "\u0451",
        /* keyspec_nordic_row1_11 ~ */
        null, null, null, null,
        /* ~ morekeys_nordic_row2_10 */
        // U+0449: "щ" CYRILLIC SMALL LETTER SHCHA
        /* keyspec_east_slavic_row1_9 */ "\u0449",
        // U+044B: "ы" CYRILLIC SMALL LETTER YERU
        /* keyspec_east_slavic_row2_2 */ "\u044B",
        // U+044D: "э" CYRILLIC SMALL LETTER E
        /* keyspec_east_slavic_row2_11 */ "\u044D",
        // U+0438: "и" CYRILLIC SMALL LETTER I
        /* keyspec_east_slavic_row3_5 */ "\u0438",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* morekeys_cyrillic_soft_sign */ "\u044A",
        /* morekeys_nordic_row2_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_w */
        // U+0456: "і" CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
        /* morekeys_east_slavic_row2_2 */ "\u0456",
        // U+04AF: "ү" CYRILLIC SMALL LETTER STRAIGHT U
        // U+04B1: "ұ" CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
        /* morekeys_cyrillic_u */ "\u04AF,\u04B1",
        // U+04A3: "ң" CYRILLIC SMALL LETTER EN WITH DESCENDER
        /* morekeys_cyrillic_en */ "\u04A3",
        // U+0493: "ғ" CYRILLIC SMALL LETTER GHE WITH STROKE
        /* morekeys_cyrillic_ghe */ "\u0493",
        // U+04E9: "ө" CYRILLIC SMALL LETTER BARRED O
        /* morekeys_cyrillic_o */ "\u04E9",
        /* morekeys_cyrillic_i ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_j */
        // U+04BB: "һ" CYRILLIC SMALL LETTER SHHA
        /* morekeys_east_slavic_row2_11 */ "\u04BB",
        // U+049B: "қ" CYRILLIC SMALL LETTER KA WITH DESCENDER
        /* morekeys_cyrillic_ka */ "\u049B",
        // U+04D9: "ә" CYRILLIC SMALL LETTER SCHWA
        /* morekeys_cyrillic_a */ "\u04D9",
    };

    /* Language km_KH: Khmer (Cambodia) */
    private static final String[] LANGUAGE_km_KH = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+1780: "ក" KHMER LETTER KA
        // U+1781: "ខ" KHMER LETTER KHA
        // U+1782: "គ" KHMER LETTER KO
        /* keylabel_to_alpha */ "\u1780\u1781\u1782",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_cyrillic_a */
        // U+17DB: "៛" KHMER CURRENCY SYMBOL RIEL
        /* morekeys_currency_dollar */ "\u17DB,\u00A2,\u00A3,\u20AC,\u00A5,\u20B1",
    };

    /* Language ky: Kirghiz */
    private static final String[] LANGUAGE_ky = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_k */
        // U+0451: "ё" CYRILLIC SMALL LETTER IO
        /* morekeys_cyrillic_ie */ "\u0451",
        /* keyspec_nordic_row1_11 ~ */
        null, null, null, null,
        /* ~ morekeys_nordic_row2_10 */
        // U+0449: "щ" CYRILLIC SMALL LETTER SHCHA
        /* keyspec_east_slavic_row1_9 */ "\u0449",
        // U+044B: "ы" CYRILLIC SMALL LETTER YERU
        /* keyspec_east_slavic_row2_2 */ "\u044B",
        // U+044D: "э" CYRILLIC SMALL LETTER E
        /* keyspec_east_slavic_row2_11 */ "\u044D",
        // U+0438: "и" CYRILLIC SMALL LETTER I
        /* keyspec_east_slavic_row3_5 */ "\u0438",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* morekeys_cyrillic_soft_sign */ "\u044A",
        /* morekeys_nordic_row2_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_east_slavic_row2_2 */
        // U+04AF: "ү" CYRILLIC SMALL LETTER STRAIGHT U
        /* morekeys_cyrillic_u */ "\u04AF",
        // U+04A3: "ң" CYRILLIC SMALL LETTER EN WITH DESCENDER
        /* morekeys_cyrillic_en */ "\u04A3",
        /* morekeys_cyrillic_ghe */ null,
        // U+04E9: "ө" CYRILLIC SMALL LETTER BARRED O
        /* morekeys_cyrillic_o */ "\u04E9",
    };

    /* Language lo_LA: Lao (Laos) */
    private static final String[] LANGUAGE_lo_LA = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0E81: "ກ" LAO LETTER KO
        // U+0E82: "ຂ" LAO LETTER KHO SUNG
        // U+0E84: "ຄ" LAO LETTER KHO TAM
        /* keylabel_to_alpha */ "\u0E81\u0E82\u0E84",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20AD: "₭" KIP SIGN
        /* keyspec_currency */ "\u20AD",
    };

    /* Language lt: Lithuanian */
    private static final String[] LANGUAGE_lt = {
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* morekeys_a */ "\u0105,\u00E4,\u0101,\u00E0,\u00E1,\u00E2,\u00E3,\u00E5,\u00E6",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* morekeys_o */ "\u00F6,\u00F5,\u00F2,\u00F3,\u00F4,\u0153,\u0151,\u00F8",
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        /* morekeys_u */ "\u016B,\u0173,\u00FC,\u016B,\u00F9,\u00FA,\u00FB,\u016F,\u0171",
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* morekeys_e */ "\u0117,\u0119,\u0113,\u00E8,\u00E9,\u00EA,\u00EB,\u011B",
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        /* morekeys_i */ "\u012F,\u012B,\u00EC,\u00ED,\u00EE,\u00EF,\u0131",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* morekeys_c */ "\u010D,\u00E7,\u0107",
        /* double_quotes */ "!text/double_9qm_lqm",
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u0146,\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* morekeys_s */ "\u0161,\u00DF,\u015B,\u015F",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* morekeys_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        /* morekeys_z */ "\u017E,\u017C,\u017A",
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        /* morekeys_t */ "\u0163,\u0165",
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        /* morekeys_l */ "\u013C,\u0142,\u013A,\u013E",
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* morekeys_g */ "\u0123,\u011F",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keyspec_currency */
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        /* morekeys_r */ "\u0157,\u0159,\u0155",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        /* morekeys_k */ "\u0137",
    };

    /* Language lv: Latvian */
    private static final String[] LANGUAGE_lv = {
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        /* morekeys_a */ "\u0101,\u00E0,\u00E1,\u00E2,\u00E3,\u00E4,\u00E5,\u00E6,\u0105",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* morekeys_o */ "\u00F2,\u00F3,\u00F4,\u00F5,\u00F6,\u0153,\u0151,\u00F8",
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        /* morekeys_u */ "\u016B,\u0173,\u00F9,\u00FA,\u00FB,\u00FC,\u016F,\u0171",
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* morekeys_e */ "\u0113,\u0117,\u00E8,\u00E9,\u00EA,\u00EB,\u0119,\u011B",
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        /* morekeys_i */ "\u012B,\u012F,\u00EC,\u00ED,\u00EE,\u00EF,\u0131",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* morekeys_c */ "\u010D,\u00E7,\u0107",
        /* double_quotes */ "!text/double_9qm_lqm",
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u0146,\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* morekeys_s */ "\u0161,\u00DF,\u015B,\u015F",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* morekeys_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        /* morekeys_z */ "\u017E,\u017C,\u017A",
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        /* morekeys_t */ "\u0163,\u0165",
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        /* morekeys_l */ "\u013C,\u0142,\u013A,\u013E",
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* morekeys_g */ "\u0123,\u011F",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keyspec_currency */
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        /* morekeys_r */ "\u0157,\u0159,\u0155",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        /* morekeys_k */ "\u0137",
    };

    /* Language mk: Macedonian */
    private static final String[] LANGUAGE_mk = {
        /* morekeys_a ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_c */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_lqm",
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_k */
        // U+0450: "ѐ" CYRILLIC SMALL LETTER IE WITH GRAVE
        /* morekeys_cyrillic_ie */ "\u0450",
        /* keyspec_nordic_row1_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_cyrillic_o */
        // U+045D: "ѝ" CYRILLIC SMALL LETTER I WITH GRAVE
        /* morekeys_cyrillic_i */ "\u045D",
        // U+0455: "ѕ" CYRILLIC SMALL LETTER DZE
        /* keyspec_south_slavic_row1_6 */ "\u0455",
        // U+045C: "ќ" CYRILLIC SMALL LETTER KJE
        /* keyspec_south_slavic_row2_11 */ "\u045C",
        // U+0437: "з" CYRILLIC SMALL LETTER ZE
        /* keyspec_south_slavic_row3_1 */ "\u0437",
        // U+0453: "ѓ" CYRILLIC SMALL LETTER GJE
        /* keyspec_south_slavic_row3_8 */ "\u0453",
    };

    /* Language mn_MN: Mongolian (Mongolia) */
    private static final String[] LANGUAGE_mn_MN = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20AE: "₮" TUGRIK SIGN
        /* keyspec_currency */ "\u20AE",
    };

    /* Language my_MM: Burmese (Myanmar) */
    private static final String[] LANGUAGE_my_MM = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+1000: "က" MYANMAR LETTER KA
        // U+1001: "ခ" MYANMAR LETTER KHA
        // U+1002: "ဂ" MYANMAR LETTER GA
        /* keylabel_to_alpha */ "\u1000\u1001\u1002",
    };

    /* Language nb: Norwegian Bokmål */
    private static final String[] LANGUAGE_nb = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E0,\u00E4,\u00E1,\u00E2,\u00E3,\u0101",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F4,\u00F2,\u00F3,\u00F6,\u00F5,\u0153,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FC,\u00FB,\u00F9,\u00FA,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0119,\u0117,\u0113",
        /* morekeys_i */ null,
        /* morekeys_c */ null,
        /* double_quotes */ "!text/double_9qm_rqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_rqm",
        /* keylabel_to_alpha ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_cyrillic_ie */
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        /* keyspec_nordic_row1_11 */ "\u00E5",
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* keyspec_nordic_row2_10 */ "\u00F8",
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* keyspec_nordic_row2_11 */ "\u00E6",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* morekeys_nordic_row2_10 */ "\u00F6",
        /* keyspec_east_slavic_row1_9 ~ */
        null, null, null, null, null,
        /* ~ morekeys_cyrillic_soft_sign */
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* morekeys_nordic_row2_11 */ "\u00E4",
    };

    /* Language ne_NP: Nepali (Nepal) */
    private static final String[] LANGUAGE_ne_NP = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0915: "क" DEVANAGARI LETTER KA
        // U+0916: "ख" DEVANAGARI LETTER KHA
        // U+0917: "ग" DEVANAGARI LETTER GA
        /* keylabel_to_alpha */ "\u0915\u0916\u0917",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+0930/U+0941/U+002E "रु." NEPALESE RUPEE SIGN
        /* keyspec_currency */ "\u0930\u0941.",
        /* morekeys_r ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_nordic_row2_11 */
        // U+0967: "१" DEVANAGARI DIGIT ONE
        /* keyspec_symbols_1 */ "\u0967",
        // U+0968: "२" DEVANAGARI DIGIT TWO
        /* keyspec_symbols_2 */ "\u0968",
        // U+0969: "३" DEVANAGARI DIGIT THREE
        /* keyspec_symbols_3 */ "\u0969",
        // U+096A: "४" DEVANAGARI DIGIT FOUR
        /* keyspec_symbols_4 */ "\u096A",
        // U+096B: "५" DEVANAGARI DIGIT FIVE
        /* keyspec_symbols_5 */ "\u096B",
        // U+096C: "६" DEVANAGARI DIGIT SIX
        /* keyspec_symbols_6 */ "\u096C",
        // U+096D: "७" DEVANAGARI DIGIT SEVEN
        /* keyspec_symbols_7 */ "\u096D",
        // U+096E: "८" DEVANAGARI DIGIT EIGHT
        /* keyspec_symbols_8 */ "\u096E",
        // U+096F: "९" DEVANAGARI DIGIT NINE
        /* keyspec_symbols_9 */ "\u096F",
        // U+0966: "०" DEVANAGARI DIGIT ZERO
        /* keyspec_symbols_0 */ "\u0966",
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "?\u0967\u0968\u0969",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
    };

    /* Language nl: Dutch */
    private static final String[] LANGUAGE_nl = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E1,\u00E4,\u00E2,\u00E0,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00FB,\u00F9,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00EB,\u00EA,\u00E8,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B,\u0133",
        /* morekeys_c */ null,
        /* double_quotes */ "!text/double_9qm_rqm",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_rqm",
        /* keylabel_to_alpha */ null,
        /* morekeys_s */ null,
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_y */ "\u0133",
    };

    /* Language pl: Polish */
    private static final String[] LANGUAGE_pl = {
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u0105,\u00E1,\u00E0,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F3,\u00F6,\u00F4,\u00F2,\u00F5,\u0153,\u00F8,\u014D",
        /* morekeys_u */ null,
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u0119,\u00E8,\u00E9,\u00EA,\u00EB,\u0117,\u0113",
        /* morekeys_i */ null,
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u0107,\u00E7,\u010D",
        /* double_quotes */ "!text/double_9qm_rqm",
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* morekeys_n */ "\u0144,\u00F1",
        /* single_quotes */ "!text/single_9qm_rqm",
        /* keylabel_to_alpha */ null,
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* morekeys_s */ "\u015B,\u00DF,\u0161",
        /* morekeys_y */ null,
        /* morekeys_d */ null,
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* morekeys_z */ "\u017C,\u017A,\u017E",
        /* morekeys_t */ null,
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* morekeys_l */ "\u0142",
    };

    /* Language pt: Portuguese */
    private static final String[] LANGUAGE_pt = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E1,\u00E3,\u00E0,\u00E2,\u00E4,\u00E5,\u00E6,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F3,\u00F5,\u00F4,\u00F2,\u00F6,\u0153,\u00F8,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        /* morekeys_e */ "\u00E9,\u00EA,\u00E8,\u0119,\u0117,\u0113,\u00EB",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EE,\u00EC,\u00EF,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* morekeys_c */ "\u00E7,\u010D,\u0107",
    };

    /* Language rm: Raeto-Romance */
    private static final String[] LANGUAGE_rm = {
        /* morekeys_a */ null,
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* morekeys_o */ "\u00F2,\u00F3,\u00F6,\u00F4,\u00F5,\u0153,\u00F8",
    };

    /* Language ro: Romanian */
    private static final String[] LANGUAGE_ro = {
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E2,\u00E3,\u0103,\u00E0,\u00E1,\u00E4,\u00E6,\u00E5,\u0101",
        /* morekeys_o ~ */
        null, null, null,
        /* ~ morekeys_e */
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00EE,\u00EF,\u00EC,\u00ED,\u012F,\u012B",
        /* morekeys_c */ null,
        /* double_quotes */ "!text/double_9qm_rqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_rqm",
        /* keylabel_to_alpha */ null,
        // U+0219: "ș" LATIN SMALL LETTER S WITH COMMA BELOW
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* morekeys_s */ "\u0219,\u00DF,\u015B,\u0161",
        /* morekeys_y ~ */
        null, null, null,
        /* ~ morekeys_z */
        // U+021B: "ț" LATIN SMALL LETTER T WITH COMMA BELOW
        /* morekeys_t */ "\u021B",
    };

    /* Language ru: Russian */
    private static final String[] LANGUAGE_ru = {
        /* morekeys_a ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_c */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_lqm",
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_k */
        // U+0451: "ё" CYRILLIC SMALL LETTER IO
        /* morekeys_cyrillic_ie */ "\u0451",
        /* keyspec_nordic_row1_11 ~ */
        null, null, null, null,
        /* ~ morekeys_nordic_row2_10 */
        // U+0449: "щ" CYRILLIC SMALL LETTER SHCHA
        /* keyspec_east_slavic_row1_9 */ "\u0449",
        // U+044B: "ы" CYRILLIC SMALL LETTER YERU
        /* keyspec_east_slavic_row2_2 */ "\u044B",
        // U+044D: "э" CYRILLIC SMALL LETTER E
        /* keyspec_east_slavic_row2_11 */ "\u044D",
        // U+0438: "и" CYRILLIC SMALL LETTER I
        /* keyspec_east_slavic_row3_5 */ "\u0438",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* morekeys_cyrillic_soft_sign */ "\u044A",
    };

    /* Language sk: Slovak */
    private static final String[] LANGUAGE_sk = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        /* morekeys_a */ "\u00E1,\u00E4,\u0101,\u00E0,\u00E2,\u00E3,\u00E5,\u00E6,\u0105",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        /* morekeys_o */ "\u00F4,\u00F3,\u00F6,\u00F2,\u00F5,\u0153,\u0151,\u00F8",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        /* morekeys_u */ "\u00FA,\u016F,\u00FC,\u016B,\u0173,\u00F9,\u00FB,\u0171",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        /* morekeys_e */ "\u00E9,\u011B,\u0113,\u0117,\u00E8,\u00EA,\u00EB,\u0119",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        /* morekeys_i */ "\u00ED,\u012B,\u012F,\u00EC,\u00EE,\u00EF,\u0131",
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* morekeys_c */ "\u010D,\u00E7,\u0107",
        /* double_quotes */ "!text/double_9qm_lqm",
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u0148,\u0146,\u00F1,\u0144",
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        /* morekeys_s */ "\u0161,\u00DF,\u015B,\u015F",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "\u00FD,\u00FF",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* morekeys_d */ "\u010F",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        /* morekeys_z */ "\u017E,\u017C,\u017A",
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        /* morekeys_t */ "\u0165,\u0163",
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* morekeys_l */ "\u013E,\u013A,\u013C,\u0142",
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* morekeys_g */ "\u0123,\u011F",
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keyspec_currency */ null,
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        /* morekeys_r */ "\u0155,\u0159,\u0157",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        /* morekeys_k */ "\u0137",
    };

    /* Language sl: Slovenian */
    private static final String[] LANGUAGE_sl = {
        /* morekeys_a ~ */
        null, null, null, null, null,
        /* ~ morekeys_i */
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        /* morekeys_c */ "\u010D,\u0107",
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_lqm",
        /* keylabel_to_alpha */ null,
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* morekeys_s */ "\u0161",
        /* morekeys_y */ null,
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        /* morekeys_d */ "\u0111",
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* morekeys_z */ "\u017E",
        /* morekeys_t ~ */
        null, null, null,
        /* ~ morekeys_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
    };

    /* Language sr: Serbian */
    private static final String[] LANGUAGE_sr = {
        /* morekeys_a ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_c */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_lqm",
        // END: More keys definitions for Serbian (Cyrillic)
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null,
        /* ~ morekeys_g */
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keyspec_currency ~ */
        null, null, null,
        /* ~ morekeys_k */
        // U+0450: "ѐ" CYRILLIC SMALL LETTER IE WITH GRAVE
        /* morekeys_cyrillic_ie */ "\u0450",
        /* keyspec_nordic_row1_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_cyrillic_o */
        // U+045D: "ѝ" CYRILLIC SMALL LETTER I WITH GRAVE
        /* morekeys_cyrillic_i */ "\u045D",
        // TODO: Move these to sr-Latn once we can handle IETF language tag with script name specified.
        // BEGIN: More keys definitions for Serbian (Latin)
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // <string name="morekeys_s">&#x0161;,&#x00DF;,&#x015B;</string>
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // <string name="morekeys_c">&#x010D;,&#x00E7;,&#x0107;</string>
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        // <string name="morekeys_d">&#x010F;</string>
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // <string name="morekeys_z">&#x017E;,&#x017A;,&#x017C;</string>
        // END: More keys definitions for Serbian (Latin)
        // BEGIN: More keys definitions for Serbian (Cyrillic)
        // U+0437: "з" CYRILLIC SMALL LETTER ZE
        /* keyspec_south_slavic_row1_6 */ "\u0437",
        // U+045B: "ћ" CYRILLIC SMALL LETTER TSHE
        /* keyspec_south_slavic_row2_11 */ "\u045B",
        // U+0455: "ѕ" CYRILLIC SMALL LETTER DZE
        /* keyspec_south_slavic_row3_1 */ "\u0455",
        // U+0452: "ђ" CYRILLIC SMALL LETTER DJE
        /* keyspec_south_slavic_row3_8 */ "\u0452",
    };

    /* Language sv: Swedish */
    private static final String[] LANGUAGE_sv = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        /* morekeys_a */ "\u00E1,\u00E0,\u00E2,\u0105,\u00E3",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F3,\u00F2,\u00F4,\u00F5,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FC,\u00FA,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        /* morekeys_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0119",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        /* morekeys_i */ "\u00ED,\u00EC,\u00EE,\u00EF",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u010D",
        /* double_quotes */ null,
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        /* morekeys_n */ "\u0144,\u00F1,\u0148",
        /* single_quotes */ null,
        /* keylabel_to_alpha */ null,
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* morekeys_s */ "\u015B,\u0161,\u015F,\u00DF",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        /* morekeys_y */ "\u00FD,\u00FF",
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        /* morekeys_d */ "\u00F0,\u010F",
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        /* morekeys_z */ "\u017A,\u017E,\u017C",
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        /* morekeys_t */ "\u0165,\u00FE",
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* morekeys_l */ "\u0142",
        /* morekeys_g */ null,
        /* single_angle_quotes */ "!text/single_raqm_laqm",
        /* double_angle_quotes */ "!text/double_raqm_laqm",
        /* keyspec_currency */ null,
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        /* morekeys_r */ "\u0159",
        /* morekeys_k */ null,
        /* morekeys_cyrillic_ie */ null,
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        /* keyspec_nordic_row1_11 */ "\u00E5",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        /* keyspec_nordic_row2_10 */ "\u00F6",
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        /* keyspec_nordic_row2_11 */ "\u00E4",
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        /* morekeys_nordic_row2_10 */ "\u00F8,\u0153",
        /* keyspec_east_slavic_row1_9 ~ */
        null, null, null, null, null,
        /* ~ morekeys_cyrillic_soft_sign */
        // U+00E6: "æ" LATIN SMALL LETTER AE
        /* morekeys_nordic_row2_11 */ "\u00E6",
    };

    /* Language sw: Swahili */
    private static final String[] LANGUAGE_sw = {
        // This is the same as English except morekeys_g.
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* morekeys_o */ "\u00F4,\u00F6,\u00F2,\u00F3,\u0153,\u00F8,\u014D,\u00F5",
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FB,\u00FC,\u00F9,\u00FA,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113",
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        /* morekeys_i */ "\u00EE,\u00EF,\u00ED,\u012B,\u00EC",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* morekeys_c */ "\u00E7",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* morekeys_n */ "\u00F1",
        /* single_quotes */ null,
        /* keylabel_to_alpha */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* morekeys_s */ "\u00DF",
        /* morekeys_y ~ */
        null, null, null, null, null,
        /* ~ morekeys_l */
        /* morekeys_g */ "g\'",
    };

    /* Language th: Thai */
    private static final String[] LANGUAGE_th = {
        /* morekeys_a ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ single_quotes */
        // Label for "switch to alphabetic" key.
        // U+0E01: "ก" THAI CHARACTER KO KAI
        // U+0E02: "ข" THAI CHARACTER KHO KHAI
        // U+0E04: "ค" THAI CHARACTER KHO KHWAI
        /* keylabel_to_alpha */ "\u0E01\u0E02\u0E04",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+0E3F: "฿" THAI CURRENCY SYMBOL BAHT
        /* keyspec_currency */ "\u0E3F",
    };

    /* Language tl: Tagalog */
    private static final String[] LANGUAGE_tl = {
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E1,\u00E0,\u00E4,\u00E2,\u00E3,\u00E5,\u0105,\u00E6,\u0101,\u00AA",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F3,\u00F2,\u00F6,\u00F4,\u00F5,\u00F8,\u0153,\u014D,\u00BA",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FC,\u00F9,\u00FB,\u016B",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EB,\u00EA,\u0119,\u0117,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u00ED,\u00EF,\u00EC,\u00EE,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u010D",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        /* morekeys_n */ "\u00F1,\u0144",
    };

    /* Language tr: Turkish */
    private static final String[] LANGUAGE_tr = {
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        /* morekeys_a */ "\u00E2",
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        /* morekeys_o */ "\u00F6,\u00F4,\u0153,\u00F2,\u00F3,\u00F5,\u00F8,\u014D",
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FC,\u00FB,\u00F9,\u00FA,\u016B",
        /* morekeys_e */ null,
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        /* morekeys_i */ "\u0131,\u00EE,\u00EF,\u00EC,\u00ED,\u012F,\u012B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u010D",
        /* double_quotes ~ */
        null, null, null, null,
        /* ~ keylabel_to_alpha */
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        /* morekeys_s */ "\u015F,\u00DF,\u015B,\u0161",
        /* morekeys_y ~ */
        null, null, null, null, null,
        /* ~ morekeys_l */
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        /* morekeys_g */ "\u011F",
    };

    /* Language uk: Ukrainian */
    private static final String[] LANGUAGE_uk = {
        /* morekeys_a ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_c */
        /* double_quotes */ "!text/double_9qm_lqm",
        /* morekeys_n */ null,
        /* single_quotes */ "!text/single_9qm_lqm",
        // Label for "switch to alphabetic" key.
        // U+0410: "А" CYRILLIC CAPITAL LETTER A
        // U+0411: "Б" CYRILLIC CAPITAL LETTER BE
        // U+0412: "В" CYRILLIC CAPITAL LETTER VE
        /* keylabel_to_alpha */ "\u0410\u0411\u0412",
        /* morekeys_s ~ */
        null, null, null, null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20B4: "₴" HRYVNIA SIGN
        /* keyspec_currency */ "\u20B4",
        /* morekeys_r ~ */
        null, null, null, null, null, null, null,
        /* ~ morekeys_nordic_row2_10 */
        // U+0449: "щ" CYRILLIC SMALL LETTER SHCHA
        /* keyspec_east_slavic_row1_9 */ "\u0449",
        // U+0456: "і" CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
        /* keyspec_east_slavic_row2_2 */ "\u0456",
        // U+0454: "є" CYRILLIC SMALL LETTER UKRAINIAN IE
        /* keyspec_east_slavic_row2_11 */ "\u0454",
        // U+0438: "и" CYRILLIC SMALL LETTER I
        /* keyspec_east_slavic_row3_5 */ "\u0438",
        // U+044A: "ъ" CYRILLIC SMALL LETTER HARD SIGN
        /* morekeys_cyrillic_soft_sign */ "\u044A",
        /* morekeys_nordic_row2_11 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_w */
        // U+0457: "ї" CYRILLIC SMALL LETTER YI
        /* morekeys_east_slavic_row2_2 */ "\u0457",
        /* morekeys_cyrillic_u */ null,
        /* morekeys_cyrillic_en */ null,
        // U+0491: "ґ" CYRILLIC SMALL LETTER GHE WITH UPTURN
        /* morekeys_cyrillic_ghe */ "\u0491",
    };

    /* Language vi: Vietnamese */
    private static final String[] LANGUAGE_vi = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+1EA3: "ả" LATIN SMALL LETTER A WITH HOOK ABOVE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+1EA1: "ạ" LATIN SMALL LETTER A WITH DOT BELOW
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+1EB1: "ằ" LATIN SMALL LETTER A WITH BREVE AND GRAVE
        // U+1EAF: "ắ" LATIN SMALL LETTER A WITH BREVE AND ACUTE
        // U+1EB3: "ẳ" LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
        // U+1EB5: "ẵ" LATIN SMALL LETTER A WITH BREVE AND TILDE
        // U+1EB7: "ặ" LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+1EA7: "ầ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
        // U+1EA5: "ấ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
        // U+1EA9: "ẩ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
        // U+1EAB: "ẫ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
        // U+1EAD: "ậ" LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
        /* morekeys_a */ "\u00E0,\u00E1,\u1EA3,\u00E3,\u1EA1,\u0103,\u1EB1,\u1EAF,\u1EB3,\u1EB5,\u1EB7,\u00E2,\u1EA7,\u1EA5,\u1EA9,\u1EAB,\u1EAD",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+1ECF: "ỏ" LATIN SMALL LETTER O WITH HOOK ABOVE
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+1ECD: "ọ" LATIN SMALL LETTER O WITH DOT BELOW
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+1ED3: "ồ" LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
        // U+1ED1: "ố" LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
        // U+1ED5: "ổ" LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
        // U+1ED7: "ỗ" LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
        // U+1ED9: "ộ" LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
        // U+01A1: "ơ" LATIN SMALL LETTER O WITH HORN
        // U+1EDD: "ờ" LATIN SMALL LETTER O WITH HORN AND GRAVE
        // U+1EDB: "ớ" LATIN SMALL LETTER O WITH HORN AND ACUTE
        // U+1EDF: "ở" LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
        // U+1EE1: "ỡ" LATIN SMALL LETTER O WITH HORN AND TILDE
        // U+1EE3: "ợ" LATIN SMALL LETTER O WITH HORN AND DOT BELOW
        /* morekeys_o */ "\u00F2,\u00F3,\u1ECF,\u00F5,\u1ECD,\u00F4,\u1ED3,\u1ED1,\u1ED5,\u1ED7,\u1ED9,\u01A1,\u1EDD,\u1EDB,\u1EDF,\u1EE1,\u1EE3",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+1EE7: "ủ" LATIN SMALL LETTER U WITH HOOK ABOVE
        // U+0169: "ũ" LATIN SMALL LETTER U WITH TILDE
        // U+1EE5: "ụ" LATIN SMALL LETTER U WITH DOT BELOW
        // U+01B0: "ư" LATIN SMALL LETTER U WITH HORN
        // U+1EEB: "ừ" LATIN SMALL LETTER U WITH HORN AND GRAVE
        // U+1EE9: "ứ" LATIN SMALL LETTER U WITH HORN AND ACUTE
        // U+1EED: "ử" LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
        // U+1EEF: "ữ" LATIN SMALL LETTER U WITH HORN AND TILDE
        // U+1EF1: "ự" LATIN SMALL LETTER U WITH HORN AND DOT BELOW
        /* morekeys_u */ "\u00F9,\u00FA,\u1EE7,\u0169,\u1EE5,\u01B0,\u1EEB,\u1EE9,\u1EED,\u1EEF,\u1EF1",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+1EBB: "ẻ" LATIN SMALL LETTER E WITH HOOK ABOVE
        // U+1EBD: "ẽ" LATIN SMALL LETTER E WITH TILDE
        // U+1EB9: "ẹ" LATIN SMALL LETTER E WITH DOT BELOW
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+1EC1: "ề" LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
        // U+1EBF: "ế" LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
        // U+1EC3: "ể" LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
        // U+1EC5: "ễ" LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
        // U+1EC7: "ệ" LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
        /* morekeys_e */ "\u00E8,\u00E9,\u1EBB,\u1EBD,\u1EB9,\u00EA,\u1EC1,\u1EBF,\u1EC3,\u1EC5,\u1EC7",
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+1EC9: "ỉ" LATIN SMALL LETTER I WITH HOOK ABOVE
        // U+0129: "ĩ" LATIN SMALL LETTER I WITH TILDE
        // U+1ECB: "ị" LATIN SMALL LETTER I WITH DOT BELOW
        /* morekeys_i */ "\u00EC,\u00ED,\u1EC9,\u0129,\u1ECB",
        /* morekeys_c ~ */
        null, null, null, null, null, null,
        /* ~ morekeys_s */
        // U+1EF3: "ỳ" LATIN SMALL LETTER Y WITH GRAVE
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+1EF7: "ỷ" LATIN SMALL LETTER Y WITH HOOK ABOVE
        // U+1EF9: "ỹ" LATIN SMALL LETTER Y WITH TILDE
        // U+1EF5: "ỵ" LATIN SMALL LETTER Y WITH DOT BELOW
        /* morekeys_y */ "\u1EF3,\u00FD,\u1EF7,\u1EF9,\u1EF5",
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        /* morekeys_d */ "\u0111",
        /* morekeys_z ~ */
        null, null, null, null, null, null,
        /* ~ double_angle_quotes */
        // U+20AB: "₫" DONG SIGN
        /* keyspec_currency */ "\u20AB",
    };

    /* Language zu: Zulu */
    private static final String[] LANGUAGE_zu = {
        // This is the same as English
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* morekeys_o */ "\u00F4,\u00F6,\u00F2,\u00F3,\u0153,\u00F8,\u014D,\u00F5",
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FB,\u00FC,\u00F9,\u00FA,\u016B",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113",
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        /* morekeys_i */ "\u00EE,\u00EF,\u00ED,\u012B,\u00EC",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* morekeys_c */ "\u00E7",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* morekeys_n */ "\u00F1",
        /* single_quotes */ null,
        /* keylabel_to_alpha */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* morekeys_s */ "\u00DF",
    };

    /* Language zz: Alphabet */
    private static final String[] LANGUAGE_zz = {
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E0,\u00E1,\u00E2,\u00E3,\u00E4,\u00E5,\u00E6,\u0101,\u0103,\u0105,\u00AA",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+014F: "ŏ" LATIN SMALL LETTER O WITH BREVE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F2,\u00F3,\u00F4,\u00F5,\u00F6,\u00F8,\u014D,\u014F,\u0151,\u0153,\u00BA",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+0169: "ũ" LATIN SMALL LETTER U WITH TILDE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+016D: "ŭ" LATIN SMALL LETTER U WITH BREVE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        /* morekeys_u */ "\u00F9,\u00FA,\u00FB,\u00FC,\u0169,\u016B,\u016D,\u016F,\u0171,\u0173",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+0115: "ĕ" LATIN SMALL LETTER E WITH BREVE
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* morekeys_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113,\u0115,\u0117,\u0119,\u011B",
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0129: "ĩ" LATIN SMALL LETTER I WITH TILDE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+012D: "ĭ" LATIN SMALL LETTER I WITH BREVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_i */ "\u00EC,\u00ED,\u00EE,\u00EF,\u0129,\u012B,\u012D,\u012F,\u0131,\u0133",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+0109: "ĉ" LATIN SMALL LETTER C WITH CIRCUMFLEX
        // U+010B: "ċ" LATIN SMALL LETTER C WITH DOT ABOVE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u0109,\u010B,\u010D",
        /* double_quotes */ null,
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+0149: "ŉ" LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
        // U+014B: "ŋ" LATIN SMALL LETTER ENG
        /* morekeys_n */ "\u00F1,\u0144,\u0146,\u0148,\u0149,\u014B",
        /* single_quotes */ null,
        /* keylabel_to_alpha */ null,
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015D: "ŝ" LATIN SMALL LETTER S WITH CIRCUMFLEX
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+017F: "ſ" LATIN SMALL LETTER LONG S
        /* morekeys_s */ "\u00DF,\u015B,\u015D,\u015F,\u0161,\u017F",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+0177: "ŷ" LATIN SMALL LETTER Y WITH CIRCUMFLEX
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_y */ "\u00FD,\u0177,\u00FF,\u0133",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        /* morekeys_d */ "\u010F,\u0111,\u00F0",
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* morekeys_z */ "\u017A,\u017C,\u017E",
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+0167: "ŧ" LATIN SMALL LETTER T WITH STROKE
        /* morekeys_t */ "\u00FE,\u0163,\u0165,\u0167",
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        // U+0140: "ŀ" LATIN SMALL LETTER L WITH MIDDLE DOT
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* morekeys_l */ "\u013A,\u013C,\u013E,\u0140,\u0142",
        // U+011D: "ĝ" LATIN SMALL LETTER G WITH CIRCUMFLEX
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        // U+0121: "ġ" LATIN SMALL LETTER G WITH DOT ABOVE
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        /* morekeys_g */ "\u011D,\u011F,\u0121,\u0123",
        /* single_angle_quotes ~ */
        null, null, null,
        /* ~ keyspec_currency */
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        /* morekeys_r */ "\u0155,\u0157,\u0159",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        // U+0138: "ĸ" LATIN SMALL LETTER KRA
        /* morekeys_k */ "\u0137,\u0138",
        /* morekeys_cyrillic_ie ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ morekeys_question */
        // U+0125: "ĥ" LATIN SMALL LETTER H WITH CIRCUMFLEX
        /* morekeys_h */ "\u0125",
        // U+0175: "ŵ" LATIN SMALL LETTER W WITH CIRCUMFLEX
        /* morekeys_w */ "\u0175",
        /* morekeys_east_slavic_row2_2 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null,
        /* ~ morekeys_v */
        // U+0135: "ĵ" LATIN SMALL LETTER J WITH CIRCUMFLEX
        /* morekeys_j */ "\u0135",
    };

    // TODO: Use the language + "_" + region representation for the locale string key.
    // Currently we are dropping the region from the key.
    private static final Object[] LANGUAGES_AND_TEXTS = {
    // "locale", TEXT_ARRAY,  /* numberOfNonNullText/lengthOf_TEXT_ARRAY localeName */
        "DEFAULT", LANGUAGE_DEFAULT, /* 168/168 default */
        "af", LANGUAGE_af,    /*   7/ 12 Afrikaans */
        "ar", LANGUAGE_ar,    /*  55/107 Arabic */
        "az", LANGUAGE_az_AZ, /*   8/ 17 Azerbaijani (Azerbaijan) */
        "be", LANGUAGE_be_BY, /*   9/ 32 Belarusian (Belarus) */
        "bg", LANGUAGE_bg,    /*   2/ 10 Bulgarian */
        "ca", LANGUAGE_ca,    /*  11/114 Catalan */
        "cs", LANGUAGE_cs,    /*  17/ 21 Czech */
        "da", LANGUAGE_da,    /*  19/ 33 Danish */
        "de", LANGUAGE_de,    /*  16/ 91 German */
        "el", LANGUAGE_el,    /*   1/ 10 Greek */
        "en", LANGUAGE_en,    /*   8/ 11 English */
        "eo", LANGUAGE_eo,    /*  26/126 Esperanto */
        "es", LANGUAGE_es,    /*   8/ 55 Spanish */
        "et", LANGUAGE_et_EE, /*  22/ 27 Estonian (Estonia) */
        "eu", LANGUAGE_eu_ES, /*   7/  8 Basque (Spain) */
        "fa", LANGUAGE_fa,    /*  58/117 Persian */
        "fi", LANGUAGE_fi,    /*  10/ 33 Finnish */
        "fr", LANGUAGE_fr,    /*  13/ 91 French */
        "gl", LANGUAGE_gl_ES, /*   7/  8 Gallegan (Spain) */
        "hi", LANGUAGE_hi,    /*  23/ 54 Hindi */
        "hr", LANGUAGE_hr,    /*   9/ 19 Croatian */
        "hu", LANGUAGE_hu,    /*   9/ 19 Hungarian */
        "hy", LANGUAGE_hy_AM, /*   8/120 Armenian (Armenia) */
        "is", LANGUAGE_is,    /*  10/ 15 Icelandic */
        "it", LANGUAGE_it,    /*   5/  5 Italian */
        "iw", LANGUAGE_iw,    /*  20/115 Hebrew */
        "ka", LANGUAGE_ka_GE, /*   3/ 10 Georgian (Georgia) */
        "kk", LANGUAGE_kk,    /*  15/112 Kazakh */
        "km", LANGUAGE_km_KH, /*   2/113 Khmer (Cambodia) */
        "ky", LANGUAGE_ky,    /*  10/ 80 Kirghiz */
        "lo", LANGUAGE_lo_LA, /*   2/ 20 Lao (Laos) */
        "lt", LANGUAGE_lt,    /*  18/ 22 Lithuanian */
        "lv", LANGUAGE_lv,    /*  18/ 22 Latvian */
        "mk", LANGUAGE_mk,    /*   9/ 85 Macedonian */
        "mn", LANGUAGE_mn_MN, /*   2/ 20 Mongolian (Mongolia) */
        "my", LANGUAGE_my_MM, /*   1/ 10 Burmese (Myanmar) */
        "nb", LANGUAGE_nb,    /*  11/ 33 Norwegian Bokmål */
        "ne", LANGUAGE_ne_NP, /*  23/ 54 Nepali (Nepal) */
        "nl", LANGUAGE_nl,    /*   9/ 12 Dutch */
        "pl", LANGUAGE_pl,    /*  10/ 16 Polish */
        "pt", LANGUAGE_pt,    /*   6/  6 Portuguese */
        "rm", LANGUAGE_rm,    /*   1/  2 Raeto-Romance */
        "ro", LANGUAGE_ro,    /*   6/ 15 Romanian */
        "ru", LANGUAGE_ru,    /*   9/ 32 Russian */
        "sk", LANGUAGE_sk,    /*  20/ 22 Slovak */
        "sl", LANGUAGE_sl,    /*   8/ 19 Slovenian */
        "sr", LANGUAGE_sr,    /*  11/ 85 Serbian */
        "sv", LANGUAGE_sv,    /*  21/ 33 Swedish */
        "sw", LANGUAGE_sw,    /*   9/ 17 Swahili */
        "th", LANGUAGE_th,    /*   2/ 20 Thai */
        "tl", LANGUAGE_tl,    /*   7/  8 Tagalog */
        "tr", LANGUAGE_tr,    /*   7/ 17 Turkish */
        "uk", LANGUAGE_uk,    /*  11/ 79 Ukrainian */
        "vi", LANGUAGE_vi,    /*   8/ 20 Vietnamese */
        "zu", LANGUAGE_zu,    /*   8/ 11 Zulu */
        "zz", LANGUAGE_zz,    /*  19/109 Alphabet */
    };

    static {
        for (int index = 0; index < NAMES.length; index++) {
            sNameToIndexesMap.put(NAMES[index], index);
        }

        for (int i = 0; i < LANGUAGES_AND_TEXTS.length; i += 2) {
            final String language = (String)LANGUAGES_AND_TEXTS[i];
            final String[] textsTable = (String[])LANGUAGES_AND_TEXTS[i + 1];
            sLanguageToTextsTableMap.put(language, textsTable);
            sTextsTableToLanguageMap.put(textsTable, language);
        }
    }
}
