/*
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.inputmethod.keyboard.internal;

import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Path;
import android.view.View;

import com.android.inputmethod.keyboard.PointerTracker;
import com.android.inputmethod.latin.CoordinateUtils;
import com.android.inputmethod.latin.R;

/**
 * Draw rubber band preview graphics during sliding key input.
 */
public final class SlidingKeyInputPreview extends AbstractDrawingPreview {
    private final int mPreviewWidth;

    private boolean mShowSlidingKeyInputPreview;
    private final int[] mPreviewFrom = CoordinateUtils.newInstance();
    private final int[] mPreviewTo = CoordinateUtils.newInstance();

    // TODO: Finalize the rubber band preview implementation.
    private final RoundedLine mRoundedLine = new RoundedLine();
    private final Paint mPaint = new Paint();

    public SlidingKeyInputPreview(final View drawingView, final TypedArray mainKeyboardViewAttr) {
        super(drawingView);
        final int previewColor = mainKeyboardViewAttr.getColor(
                R.styleable.MainKeyboardView_slidingKeyInputPreviewColor, 0);
        mPreviewWidth = mainKeyboardViewAttr.getDimensionPixelSize(
                R.styleable.MainKeyboardView_slidingKeyInputPreviewWidth, 0);
        mPaint.setColor(previewColor);
    }

    public void dismissSlidingKeyInputPreview() {
        mShowSlidingKeyInputPreview = false;
        getDrawingView().invalidate();
    }

    /**
     * Draws the preview
     * @param canvas The canvas where the preview is drawn.
     */
    @Override
    public void drawPreview(final Canvas canvas) {
        if (!isPreviewEnabled() || !mShowSlidingKeyInputPreview) {
            return;
        }

        // TODO: Finalize the rubber band preview implementation.
        final int radius = mPreviewWidth / 2;
        final Path path = mRoundedLine.makePath(
                CoordinateUtils.x(mPreviewFrom), CoordinateUtils.y(mPreviewFrom), radius,
                CoordinateUtils.x(mPreviewTo), CoordinateUtils.y(mPreviewTo), radius);
        canvas.drawPath(path, mPaint);
    }

    /**
     * Set the position of the preview.
     * @param tracker The new location of the preview is based on the points in PointerTracker.
     */
    @Override
    public void setPreviewPosition(final PointerTracker tracker) {
        if (!tracker.isInSlidingKeyInputFromModifier()) {
            mShowSlidingKeyInputPreview = false;
            return;
        }
        tracker.getDownCoordinates(mPreviewFrom);
        tracker.getLastCoordinates(mPreviewTo);
        mShowSlidingKeyInputPreview = true;
        getDrawingView().invalidate();
    }
}
