/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef _LIBINPUT_TOUCHVIDEOFRAME_H
#define _LIBINPUT_TOUCHVIDEOFRAME_H

#include <stdint.h>
#include <sys/time.h>
#include <vector>

namespace android {

/**
 * Represents data from a single scan of the touchscreen device.
 * Similar in concept to a video frame, but the touch strength is used as
 * the values instead.
 */
class TouchVideoFrame {
public:
    TouchVideoFrame(uint32_t width, uint32_t height, std::vector<int16_t> data,
            const struct timeval& timestamp) :
            mWidth(width), mHeight(height), mData(std::move(data)), mTimestamp(timestamp) {
    }

    /**
     * Width of the frame
     */
    uint32_t getWidth() const { return mWidth; }
    /**
     * Height of the frame
     */
    uint32_t getHeight() const { return mHeight; }
    /**
     * The touch strength data.
     * The array is a 2-D row-major matrix, with dimensions (height, width).
     * Total size of the array should equal getHeight() * getWidth().
     * Data is allowed to be negative.
     */
    const std::vector<int16_t>& getData() const { return mData; }
    /**
     * Time at which the heatmap was taken.
     */
    const struct timeval& getTimestamp() const { return mTimestamp; }

private:
    uint32_t mWidth;
    uint32_t mHeight;
    std::vector<int16_t> mData;
    struct timeval mTimestamp;
};

} // namespace android

#endif // _LIBINPUT_TOUCHVIDEOFRAME_H
