/**
 * Copyright 2024, The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.os;

import android.os.IInputConstants;

/**
 * Flag definitions for MotionEvents.
 * @hide
 */
@Backing(type="int")
enum MotionEventFlag {

    /**
     * This flag indicates that the window that received this motion event is partly
     * or wholly obscured by another visible window above it and the event directly passed through
     * the obscured area.
     *
     * A security sensitive application can check this flag to identify situations in which
     * a malicious application may have covered up part of its content for the purpose
     * of misleading the user or hijacking touches.  An appropriate response might be
     * to drop the suspect touches or to take additional precautions to confirm the user's
     * actual intent.
     */
    WINDOW_IS_OBSCURED = 0x1,

    /**
     * This flag indicates that the window that received this motion event is partly
     * or wholly obscured by another visible window above it and the event did not directly pass
     * through the obscured area.
     *
     * A security sensitive application can check this flag to identify situations in which
     * a malicious application may have covered up part of its content for the purpose
     * of misleading the user or hijacking touches.  An appropriate response might be
     * to drop the suspect touches or to take additional precautions to confirm the user's
     * actual intent.
     *
     * Unlike FLAG_WINDOW_IS_OBSCURED, this is only true if the window that received this event is
     * obstructed in areas other than the touched location.
     */
    WINDOW_IS_PARTIALLY_OBSCURED = 0x2,

    /**
     * This private flag is only set on {@link #ACTION_HOVER_MOVE} events and indicates that
     * this event will be immediately followed by a {@link #ACTION_HOVER_EXIT}. It is used to
     * prevent generating redundant {@link #ACTION_HOVER_ENTER} events.
     * @hide
     */
    HOVER_EXIT_PENDING = 0x4,

    /**
     * This flag indicates that the event has been generated by a gesture generator. It
     * provides a hint to the GestureDetector to not apply any touch slop.
     *
     * @hide
     */
    IS_GENERATED_GESTURE = 0x8,

    /**
     * This flag is only set for events with {@link #ACTION_POINTER_UP} and {@link #ACTION_CANCEL}.
     * It indicates that the pointer going up was an unintentional user touch. When FLAG_CANCELED
     * is set, the typical actions that occur in response for a pointer going up (such as click
     * handlers, end of drawing) should be aborted. This flag is typically set when the user was
     * accidentally touching the screen, such as by gripping the device, or placing the palm on the
     * screen.
     *
     * @see #ACTION_POINTER_UP
     * @see #ACTION_CANCEL
     */
    CANCELED = IInputConstants.INPUT_EVENT_FLAG_CANCELED,

    /**
     * This flag indicates that the event will not cause a focus change if it is directed to an
     * unfocused window, even if it an {@link #ACTION_DOWN}. This is typically used with pointer
     * gestures to allow the user to direct gestures to an unfocused window without bringing the
     * window into focus.
     * @hide
     */
    NO_FOCUS_CHANGE = 0x40,

    /**
     * This flag indicates that the event has a valid value for AXIS_ORIENTATION.
     *
     * This is a private flag that is not used in Java.
     * @hide
     */
    PRIVATE_FLAG_SUPPORTS_ORIENTATION = 0x80,

    /**
     * This flag indicates that the pointers' AXIS_ORIENTATION can be used to precisely determine
     * the direction in which the tool is pointing. The value of the orientation axis will be in
     * the range [-pi, pi], which represents a full circle. This is usually supported by devices
     * like styluses.
     *
     * Conversely, AXIS_ORIENTATION cannot be used to tell which direction the tool is pointing
     * when this flag is not set. In this case, the axis value will have a range of [-pi/2, pi/2],
     * which represents half a circle. This is usually the case for devices like touchscreens and
     * touchpads, for which it is difficult to tell which direction along the major axis of the
     * touch ellipse the finger is pointing.
     *
     * This is a private flag that is not used in Java.
     * @hide
     */
    PRIVATE_FLAG_SUPPORTS_DIRECTIONAL_ORIENTATION = 0x100,

    /**
     * The input event was generated or modified by accessibility service.
     * Shared by both KeyEvent and MotionEvent flags, so this value should not overlap with either
     * set of flags, including in input/Input.h and in android/input.h.
     */
    IS_ACCESSIBILITY_EVENT = IInputConstants.INPUT_EVENT_FLAG_IS_ACCESSIBILITY_EVENT,

    /**
     * Private flag that indicates when the system has detected that this motion event
     * may be inconsistent with respect to the sequence of previously delivered motion events,
     * such as when a pointer move event is sent but the pointer is not down.
     *
     * @hide
     * @see #isTainted
     * @see #setTainted
     */
    TAINTED = IInputConstants.INPUT_EVENT_FLAG_TAINTED,

    /**
     * Private flag indicating that this event was synthesized by the system and should be delivered
     * to the accessibility focused view first. When being dispatched such an event is not handled
     * by predecessors of the accessibility focused view and after the event reaches that view the
     * flag is cleared and normal event dispatch is performed. This ensures that the platform can
     * click on any view that has accessibility focus which is semantically equivalent to asking the
     * view to perform a click accessibility action but more generic as views not implementing click
     * action correctly can still be activated.
     *
     * @hide
     * @see #isTargetAccessibilityFocus()
     * @see #setTargetAccessibilityFocus(boolean)
     */
    TARGET_ACCESSIBILITY_FOCUS = 0x40000000,
}
