/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <gtest/gtest.h>
#include <gui/ISurfaceComposer.h>
#include <gui/SurfaceComposerClient.h>
#include <private/gui/ComposerService.h>
#include <ui/DisplayConfig.h>
#include <utils/Errors.h>
#include <utils/Vector.h>

#include "utils/TransactionUtils.h"

namespace android {

::testing::Environment* const binderEnv =
        ::testing::AddGlobalTestEnvironment(new BinderEnvironment());

/**
 * Test class for setting display configs and passing around refresh rate ranges.
 */
class RefreshRateRangeTest : public ::testing::Test {
private:
    int32_t initialDefaultConfig;
    bool initialAllowGroupSwitching;
    float initialPrimaryMin;
    float initialPrimaryMax;
    float initialAppRequestMin;
    float initialAppRequestMax;

protected:
    void SetUp() override {
        mDisplayToken = SurfaceComposerClient::getInternalDisplayToken();
        status_t res =
                SurfaceComposerClient::getDesiredDisplayConfigSpecs(mDisplayToken,
                                                                    &initialDefaultConfig,
                                                                    &initialAllowGroupSwitching,
                                                                    &initialPrimaryMin,
                                                                    &initialPrimaryMax,
                                                                    &initialAppRequestMin,
                                                                    &initialAppRequestMax);
        ASSERT_EQ(res, NO_ERROR);
    }

    void TearDown() override {
        status_t res =
                SurfaceComposerClient::setDesiredDisplayConfigSpecs(mDisplayToken,
                                                                    initialDefaultConfig,
                                                                    initialAllowGroupSwitching,
                                                                    initialPrimaryMin,
                                                                    initialPrimaryMax,
                                                                    initialAppRequestMin,
                                                                    initialAppRequestMax);
        ASSERT_EQ(res, NO_ERROR);
    }

    void testSetAllowGroupSwitching(bool allowGroupSwitching);

    sp<IBinder> mDisplayToken;
};

TEST_F(RefreshRateRangeTest, setAllConfigs) {
    Vector<DisplayConfig> configs;
    status_t res = SurfaceComposerClient::getDisplayConfigs(mDisplayToken, &configs);
    ASSERT_EQ(res, NO_ERROR);
    ASSERT_GT(configs.size(), 0);

    for (size_t i = 0; i < configs.size(); i++) {
        res = SurfaceComposerClient::setDesiredDisplayConfigSpecs(mDisplayToken,
                                                                  static_cast<int32_t>(i), false,
                                                                  configs[i].refreshRate,
                                                                  configs[i].refreshRate,
                                                                  configs[i].refreshRate,
                                                                  configs[i].refreshRate);
        ASSERT_EQ(res, NO_ERROR);

        int defaultConfig;
        bool allowGroupSwitching;
        float primaryRefreshRateMin;
        float primaryRefreshRateMax;
        float appRequestRefreshRateMin;
        float appRequestRefreshRateMax;
        res = SurfaceComposerClient::getDesiredDisplayConfigSpecs(mDisplayToken, &defaultConfig,
                                                                  &allowGroupSwitching,
                                                                  &primaryRefreshRateMin,
                                                                  &primaryRefreshRateMax,
                                                                  &appRequestRefreshRateMin,
                                                                  &appRequestRefreshRateMax);
        ASSERT_EQ(res, NO_ERROR);
        ASSERT_EQ(defaultConfig, i);
        ASSERT_EQ(allowGroupSwitching, false);
        ASSERT_EQ(primaryRefreshRateMin, configs[i].refreshRate);
        ASSERT_EQ(primaryRefreshRateMax, configs[i].refreshRate);
        ASSERT_EQ(appRequestRefreshRateMin, configs[i].refreshRate);
        ASSERT_EQ(appRequestRefreshRateMax, configs[i].refreshRate);
    }
}

void RefreshRateRangeTest::testSetAllowGroupSwitching(bool allowGroupSwitching) {
    status_t res = SurfaceComposerClient::setDesiredDisplayConfigSpecs(mDisplayToken, 0,
                                                                       allowGroupSwitching, 0.f,
                                                                       90.f, 0.f, 90.f);
    ASSERT_EQ(res, NO_ERROR);
    int defaultConfig;
    bool newAllowGroupSwitching;
    float primaryRefreshRateMin;
    float primaryRefreshRateMax;
    float appRequestRefreshRateMin;
    float appRequestRefreshRateMax;

    res = SurfaceComposerClient::getDesiredDisplayConfigSpecs(mDisplayToken, &defaultConfig,
                                                              &newAllowGroupSwitching,
                                                              &primaryRefreshRateMin,
                                                              &primaryRefreshRateMax,
                                                              &appRequestRefreshRateMin,
                                                              &appRequestRefreshRateMax);
    ASSERT_EQ(res, NO_ERROR);
    ASSERT_EQ(defaultConfig, 0);
    ASSERT_EQ(newAllowGroupSwitching, allowGroupSwitching);
    ASSERT_EQ(primaryRefreshRateMin, 0.f);
    ASSERT_EQ(primaryRefreshRateMax, 90.f);
    ASSERT_EQ(appRequestRefreshRateMin, 0.f);
    ASSERT_EQ(appRequestRefreshRateMax, 90.f);
}

TEST_F(RefreshRateRangeTest, setAllowGroupSwitching) {
    testSetAllowGroupSwitching(true);
    testSetAllowGroupSwitching(false);
    testSetAllowGroupSwitching(true);
}

} // namespace android
