/*
 * Copyright (C) 2007 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef ANDROID_UI_BUFFER_MAPPER_H
#define ANDROID_UI_BUFFER_MAPPER_H

#include <stdint.h>
#include <sys/types.h>

#include <ui/Gralloc1.h>

#include <utils/Singleton.h>

namespace android {

// ---------------------------------------------------------------------------

namespace Gralloc2 {
class Mapper;
}

class Rect;

class GraphicBufferMapper : public Singleton<GraphicBufferMapper>
{
public:
    static inline GraphicBufferMapper& get() { return getInstance(); }

    // This may NOT work on devices without a valid Gralloc2::Mapper.
    status_t registerBuffer(buffer_handle_t handle);

    status_t registerBuffer(const GraphicBuffer* buffer);

    status_t unregisterBuffer(buffer_handle_t handle);

    status_t getDimensions(buffer_handle_t handle,
            uint32_t* outWidth, uint32_t* outHeight) const;

    status_t getFormat(buffer_handle_t handle, int32_t* outFormat) const;

    status_t getLayerCount(buffer_handle_t handle,
            uint32_t* outLayerCount) const;

    status_t getProducerUsage(buffer_handle_t handle,
            uint64_t* outProducerUsage) const;

    status_t getConsumerUsage(buffer_handle_t handle,
            uint64_t* outConsumerUsage) const;

    status_t getBackingStore(buffer_handle_t handle,
            uint64_t* outBackingStore) const;

    status_t getStride(buffer_handle_t handle, uint32_t* outStride) const;

    status_t lock(buffer_handle_t handle,
            uint32_t usage, const Rect& bounds, void** vaddr);

    status_t lockYCbCr(buffer_handle_t handle,
            uint32_t usage, const Rect& bounds, android_ycbcr *ycbcr);

    status_t unlock(buffer_handle_t handle);

    status_t lockAsync(buffer_handle_t handle,
            uint32_t usage, const Rect& bounds, void** vaddr, int fenceFd);

    status_t lockAsync(buffer_handle_t handle,
            uint64_t producerUsage, uint64_t consumerUsage, const Rect& bounds,
            void** vaddr, int fenceFd);

    status_t lockAsyncYCbCr(buffer_handle_t handle,
            uint32_t usage, const Rect& bounds, android_ycbcr *ycbcr,
            int fenceFd);

    status_t unlockAsync(buffer_handle_t handle, int *fenceFd);

    const Gralloc2::Mapper& getGrallocMapper() const
    {
        return *mMapper;
    }

private:
    friend class Singleton<GraphicBufferMapper>;

    GraphicBufferMapper();

    const std::unique_ptr<const Gralloc2::Mapper> mMapper;

    std::unique_ptr<Gralloc1::Loader> mLoader;
    std::unique_ptr<Gralloc1::Device> mDevice;
};

// ---------------------------------------------------------------------------

}; // namespace android

#endif // ANDROID_UI_BUFFER_MAPPER_H

