/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.systemui.biometrics

import android.annotation.RawRes
import android.content.Context
import com.airbnb.lottie.LottieAnimationView
import com.android.systemui.res.R
import com.android.systemui.biometrics.ui.binder.Spaghetti.BiometricState
import com.android.systemui.biometrics.ui.binder.Spaghetti.BiometricState.STATE_AUTHENTICATED
import com.android.systemui.biometrics.ui.binder.Spaghetti.BiometricState.STATE_ERROR
import com.android.systemui.biometrics.ui.binder.Spaghetti.BiometricState.STATE_HELP
import com.android.systemui.biometrics.ui.binder.Spaghetti.BiometricState.STATE_PENDING_CONFIRMATION

/** Face/Fingerprint combined icon animator for BiometricPrompt. */
open class AuthBiometricFingerprintAndFaceIconController(
    context: Context,
    iconView: LottieAnimationView,
    iconViewOverlay: LottieAnimationView,
) : AuthBiometricFingerprintIconController(context, iconView, iconViewOverlay) {

    override val actsAsConfirmButton: Boolean = true

    override fun shouldAnimateIconViewForTransition(
            oldState: BiometricState,
            newState: BiometricState
    ): Boolean = when (newState) {
        STATE_PENDING_CONFIRMATION -> true
        else -> super.shouldAnimateIconViewForTransition(oldState, newState)
    }

    @RawRes
    override fun getAnimationForTransition(
        oldState: BiometricState,
        newState: BiometricState
    ): Int? = when (newState) {
        STATE_AUTHENTICATED -> {
           if (oldState == STATE_PENDING_CONFIRMATION) {
               R.raw.fingerprint_dialogue_unlocked_to_checkmark_success_lottie
           } else {
               super.getAnimationForTransition(oldState, newState)
           }
        }
        STATE_PENDING_CONFIRMATION -> {
            if (oldState == STATE_ERROR || oldState == STATE_HELP) {
                R.raw.fingerprint_dialogue_error_to_unlock_lottie
            } else {
                R.raw.fingerprint_dialogue_fingerprint_to_unlock_lottie
            }
        }
        else -> super.getAnimationForTransition(oldState, newState)
    }
}
