/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.view.translation;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.annotation.SuppressLint;
import android.os.Parcel;
import android.os.Parcelable;
import android.util.ArrayMap;
import android.view.autofill.AutofillId;

import com.android.internal.util.DataClass;

import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * Wrapper class representing a translation response associated with a {@link android.view.View} to
 * be used by {@link android.service.translation.TranslationService}.
 */
@DataClass(genBuilder = true, genToString = true, genEqualsHashCode = true, genGetters = false)
public final class ViewTranslationResponse implements Parcelable {

    /**
     * The {@link AutofillId} of the view associated with this response.
     */
    @NonNull
    private final AutofillId mAutofillId;

    @NonNull
    @DataClass.PluralOf("translationResponseValue")
    private final Map<String, TranslationResponseValue> mTranslationResponseValues;

    /**
     * Gets the {@link TranslationResponseValue} of the corresponding key.
     * @param key String id of the translated translation response value.
     * @return the {@link TranslationResponseValue}.
     * @throws IllegalArgumentException if the key does not exist.
     */
    @NonNull
    public TranslationResponseValue getValue(@NonNull String key) {
        Objects.requireNonNull(key);
        if (!mTranslationResponseValues.containsKey(key)) {
            throw new IllegalArgumentException("Request does not contain value for key=" + key);
        }
        return mTranslationResponseValues.get(key);
    }

    /**
     * Returns all keys in this response as a {@link Set} of Strings. The keys are used by
     * {@link #getValue(String)} to get the {@link TranslationResponseValue}s.
     */
    @NonNull
    public Set<String> getKeys() {
        return mTranslationResponseValues.keySet();
    }


    /**
     * Returns the associated {@link AutofillId} of this response.
     */
    @NonNull
    public AutofillId getAutofillId() {
        return mAutofillId;
    }

    private static Map<String, TranslationResponseValue> defaultTranslationResponseValues() {
        return Collections.emptyMap();
    }

    @DataClass.Suppress({"addTranslationResponseValue", "setAutofillId"})
    abstract static class BaseBuilder {

        abstract Builder setTranslationResponseValues(Map<String, TranslationResponseValue> value);

        /**
         * Sets the corresponding {@link TranslationResponseValue} for the provided key.
         *
         * @param key The key for this translation response value.
         * @param value the translation response value holding the translated content.
         * @return this builder.
         */
        @SuppressLint("MissingGetterMatchingBuilder")
        public Builder setValue(String key,
                TranslationResponseValue value) {
            final Builder builder = (Builder) this;
            if (builder.mTranslationResponseValues == null) {
                setTranslationResponseValues(new ArrayMap<>());
            }
            builder.mTranslationResponseValues.put(key, value);
            return builder;
        }
    }



    // Code below generated by codegen v1.0.22.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/view/translation/ViewTranslationResponse.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    @DataClass.Generated.Member
    /* package-private */ ViewTranslationResponse(
            @NonNull AutofillId autofillId,
            @NonNull Map<String,TranslationResponseValue> translationResponseValues) {
        this.mAutofillId = autofillId;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mAutofillId);
        this.mTranslationResponseValues = translationResponseValues;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mTranslationResponseValues);

        // onConstructed(); // You can define this method to get a callback
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "ViewTranslationResponse { " +
                "autofillId = " + mAutofillId + ", " +
                "translationResponseValues = " + mTranslationResponseValues +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public boolean equals(@Nullable Object o) {
        // You can override field equality logic by defining either of the methods like:
        // boolean fieldNameEquals(ViewTranslationResponse other) { ... }
        // boolean fieldNameEquals(FieldType otherValue) { ... }

        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        @SuppressWarnings("unchecked")
        ViewTranslationResponse that = (ViewTranslationResponse) o;
        //noinspection PointlessBooleanExpression
        return true
                && Objects.equals(mAutofillId, that.mAutofillId)
                && Objects.equals(mTranslationResponseValues, that.mTranslationResponseValues);
    }

    @Override
    @DataClass.Generated.Member
    public int hashCode() {
        // You can override field hashCode logic by defining methods like:
        // int fieldNameHashCode() { ... }

        int _hash = 1;
        _hash = 31 * _hash + Objects.hashCode(mAutofillId);
        _hash = 31 * _hash + Objects.hashCode(mTranslationResponseValues);
        return _hash;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeTypedObject(mAutofillId, flags);
        dest.writeMap(mTranslationResponseValues);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ ViewTranslationResponse(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        AutofillId autofillId = (AutofillId) in.readTypedObject(AutofillId.CREATOR);
        Map<String,TranslationResponseValue> translationResponseValues = new java.util.LinkedHashMap<>();
        in.readMap(translationResponseValues, TranslationResponseValue.class.getClassLoader());

        this.mAutofillId = autofillId;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mAutofillId);
        this.mTranslationResponseValues = translationResponseValues;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mTranslationResponseValues);

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<ViewTranslationResponse> CREATOR
            = new Parcelable.Creator<ViewTranslationResponse>() {
        @Override
        public ViewTranslationResponse[] newArray(int size) {
            return new ViewTranslationResponse[size];
        }

        @Override
        public ViewTranslationResponse createFromParcel(@NonNull Parcel in) {
            return new ViewTranslationResponse(in);
        }
    };

    /**
     * A builder for {@link ViewTranslationResponse}
     */
    @SuppressWarnings("WeakerAccess")
    @DataClass.Generated.Member
    public static final class Builder extends BaseBuilder {

        private @NonNull AutofillId mAutofillId;
        private @NonNull Map<String,TranslationResponseValue> mTranslationResponseValues;

        private long mBuilderFieldsSet = 0L;

        /**
         * Creates a new Builder.
         *
         * @param autofillId
         *   The {@link AutofillId} of the view associated with this response.
         */
        public Builder(
                @NonNull AutofillId autofillId) {
            mAutofillId = autofillId;
            com.android.internal.util.AnnotationValidations.validate(
                    NonNull.class, null, mAutofillId);
        }

        @DataClass.Generated.Member
        @Override
        @NonNull Builder setTranslationResponseValues(@NonNull Map<String,TranslationResponseValue> value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x2;
            mTranslationResponseValues = value;
            return this;
        }

        /** Builds the instance. This builder should not be touched after calling this! */
        public @NonNull ViewTranslationResponse build() {
            checkNotUsed();
            mBuilderFieldsSet |= 0x4; // Mark builder used

            if ((mBuilderFieldsSet & 0x2) == 0) {
                mTranslationResponseValues = defaultTranslationResponseValues();
            }
            ViewTranslationResponse o = new ViewTranslationResponse(
                    mAutofillId,
                    mTranslationResponseValues);
            return o;
        }

        private void checkNotUsed() {
            if ((mBuilderFieldsSet & 0x4) != 0) {
                throw new IllegalStateException(
                        "This Builder should not be reused. Use a new Builder instance instead");
            }
        }
    }

    @DataClass.Generated(
            time = 1614992272865L,
            codegenVersion = "1.0.22",
            sourceFile = "frameworks/base/core/java/android/view/translation/ViewTranslationResponse.java",
            inputSignatures = "private final @android.annotation.NonNull android.view.autofill.AutofillId mAutofillId\nprivate final @android.annotation.NonNull @com.android.internal.util.DataClass.PluralOf(\"translationResponseValue\") java.util.Map<java.lang.String,android.view.translation.TranslationResponseValue> mTranslationResponseValues\npublic @android.annotation.NonNull android.view.translation.TranslationResponseValue getValue(java.lang.String)\npublic @android.annotation.NonNull java.util.Set<java.lang.String> getKeys()\npublic @android.annotation.NonNull android.view.autofill.AutofillId getAutofillId()\nprivate static  java.util.Map<java.lang.String,android.view.translation.TranslationResponseValue> defaultTranslationResponseValues()\nclass ViewTranslationResponse extends java.lang.Object implements [android.os.Parcelable]\nabstract  android.view.translation.ViewTranslationResponse.Builder setTranslationResponseValues(java.util.Map<java.lang.String,android.view.translation.TranslationResponseValue>)\npublic @android.annotation.SuppressLint android.view.translation.ViewTranslationResponse.Builder setValue(java.lang.String,android.view.translation.TranslationResponseValue)\nclass BaseBuilder extends java.lang.Object implements []\n@com.android.internal.util.DataClass(genBuilder=true, genToString=true, genEqualsHashCode=true, genGetters=false)\nabstract  android.view.translation.ViewTranslationResponse.Builder setTranslationResponseValues(java.util.Map<java.lang.String,android.view.translation.TranslationResponseValue>)\npublic @android.annotation.SuppressLint android.view.translation.ViewTranslationResponse.Builder setValue(java.lang.String,android.view.translation.TranslationResponseValue)\nclass BaseBuilder extends java.lang.Object implements []")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
