/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.service.voice;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.annotation.SuppressLint;
import android.annotation.SystemApi;
import android.os.Parcel;
import android.os.Parcelable;

import com.android.internal.util.DataClass;
import com.android.internal.util.Preconditions;

import java.util.Objects;

/**
 * Represents a result supporting the visual query detection.
 *
 * @hide
 */
@DataClass(
        genConstructor = false,
        genBuilder = true,
        genEqualsHashCode = true,
        genHiddenConstDefs = true,
        genParcelable = true,
        genToString = true
)
@SystemApi
@SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
public final class VisualQueryDetectedResult implements Parcelable {

    /**
     * Text query being associated with the detection result.
     **/
    @NonNull
    private final String mPartialQuery;
    private static String defaultPartialQuery() {
        return "";
    }

    /** Id of the current speaker
     *
     * <p>Only values between 0 and {@link #getMaxSpeakerId} (inclusive) are accepted.
     */
    private final int mSpeakerId;
    private static int defaultSpeakerId() {
        return 0;
    }
    /** Maximum number of active speaker ids. **/
    @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
    public static int getMaxSpeakerId() {
        return 15;
    }

    /**
     * Detected signal representing the arbitrary data that will make accessibility detections work.
     *
     * This field should only be set if the device setting for allowing accessibility data is on
     * based on the result of {@link VisualQueryDetector#isAccessibilityDetectionEnabled()}. If the
     * enable bit return by the method is {@code false}, it would suggest a failure to egress the
     * {@link VisualQueryDetectedResult} object with this field set. The system server will prevent
     * egress and invoke
     * {@link VisualQueryDetector.Callback#onFailure(VisualQueryDetectionServiceFailure)}.
     */
    @Nullable
    private final byte[] mAccessibilityDetectionData;
    private static byte[] defaultAccessibilityDetectionData() {
        return null;
    }

    private void onConstructed() {
        Preconditions.checkArgumentInRange(mSpeakerId, 0, getMaxSpeakerId(), "speakerId");
    }

    /**
     * Provides an instance of {@link Builder} with state corresponding to this instance.
     *
     * @hide
     */
    public Builder buildUpon() {
        return new Builder()
                .setPartialQuery(mPartialQuery)
                .setSpeakerId(mSpeakerId)
                .setAccessibilityDetectionData(mAccessibilityDetectionData);
    }

    /**
     * TODO(b/301491148): Remove suppressLint on generated API when fixed or sdk finalized.
     * Codegen does not support flaggedAPI, so needs to review manually on the generated code
     * and makes sure the following:
     *     1. SuppressLint is added back to the API after each run of codegen
     *     2. No unwanted method is modified due to suppressLint annotation
     *
     *  Run $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/service/voice/VisualQueryAttentionResult.java
     *  for codegen on new APIs.
     */

    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/service/voice/VisualQueryDetectedResult.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    @DataClass.Generated.Member
    /* package-private */ VisualQueryDetectedResult(
            @NonNull String partialQuery,
            int speakerId,
            @Nullable byte[] accessibilityDetectionData) {
        this.mPartialQuery = partialQuery;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mPartialQuery);
        this.mSpeakerId = speakerId;
        this.mAccessibilityDetectionData = accessibilityDetectionData;

        onConstructed();
    }

    /**
     * Text query being associated with the detection result.
     */
    @DataClass.Generated.Member
    @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
    public @NonNull String getPartialQuery() {
        return mPartialQuery;
    }

    /**
     * Id of the current speaker
     *
     * <p>Only values between 0 and {@link #getMaxSpeakerId} (inclusive) are accepted.
     */
    @DataClass.Generated.Member
    @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
    public int getSpeakerId() {
        return mSpeakerId;
    }

    /**
     * Detected signal representing the data for allowing accessibility feature. This field can
     * only be set when the secure device settings is set to true by either settings page UI or
     * {@link VisualQueryDetector@setAccessibilityDetectionEnabled(boolean)}
     */
    @DataClass.Generated.Member
    @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
    public @Nullable byte[] getAccessibilityDetectionData() {
        return mAccessibilityDetectionData;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "VisualQueryDetectedResult { " +
                "partialQuery = " + mPartialQuery + ", " +
                "speakerId = " + mSpeakerId + ", " +
                "accessibilityDetectionData = " + java.util.Arrays.toString(mAccessibilityDetectionData) +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public boolean equals(@Nullable Object o) {
        // You can override field equality logic by defining either of the methods like:
        // boolean fieldNameEquals(VisualQueryDetectedResult other) { ... }
        // boolean fieldNameEquals(FieldType otherValue) { ... }

        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        @SuppressWarnings("unchecked")
        VisualQueryDetectedResult that = (VisualQueryDetectedResult) o;
        //noinspection PointlessBooleanExpression
        return true
                && Objects.equals(mPartialQuery, that.mPartialQuery)
                && mSpeakerId == that.mSpeakerId
                && java.util.Arrays.equals(mAccessibilityDetectionData, that.mAccessibilityDetectionData);
    }

    @Override
    @DataClass.Generated.Member
    public int hashCode() {
        // You can override field hashCode logic by defining methods like:
        // int fieldNameHashCode() { ... }

        int _hash = 1;
        _hash = 31 * _hash + Objects.hashCode(mPartialQuery);
        _hash = 31 * _hash + mSpeakerId;
        _hash = 31 * _hash + java.util.Arrays.hashCode(mAccessibilityDetectionData);
        return _hash;
    }

    @Override
    @DataClass.Generated.Member
    @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeString(mPartialQuery);
        dest.writeInt(mSpeakerId);
        dest.writeByteArray(mAccessibilityDetectionData);
    }

    @Override
    @DataClass.Generated.Member
    @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ VisualQueryDetectedResult(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        String partialQuery = in.readString();
        int speakerId = in.readInt();
        byte[] accessibilityDetectionData = in.createByteArray();

        this.mPartialQuery = partialQuery;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mPartialQuery);
        this.mSpeakerId = speakerId;
        this.mAccessibilityDetectionData = accessibilityDetectionData;

        onConstructed();
    }

    @DataClass.Generated.Member
    @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
    public static final @NonNull Parcelable.Creator<VisualQueryDetectedResult> CREATOR
            = new Parcelable.Creator<VisualQueryDetectedResult>() {
        @Override
        public VisualQueryDetectedResult[] newArray(int size) {
            return new VisualQueryDetectedResult[size];
        }

        @Override
        public VisualQueryDetectedResult createFromParcel(@NonNull Parcel in) {
            return new VisualQueryDetectedResult(in);
        }
    };

    /**
     * A builder for {@link VisualQueryDetectedResult}
     */
    @SuppressWarnings("WeakerAccess")
    @DataClass.Generated.Member
    @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
    public static final class Builder {

        private @NonNull String mPartialQuery;
        private int mSpeakerId;
        private @Nullable byte[] mAccessibilityDetectionData;

        private long mBuilderFieldsSet = 0L;

        @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
        public Builder() {
        }

        /**
         * Text query being associated with the detection result.
         */
        @DataClass.Generated.Member
        @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
        public @NonNull Builder setPartialQuery(@NonNull String value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x1;
            mPartialQuery = value;
            return this;
        }

        /**
         * Id of the current speaker
         *
         * <p>Only values between 0 and {@link #getMaxSpeakerId} (inclusive) are accepted.
         */
        @DataClass.Generated.Member
        @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
        public @NonNull Builder setSpeakerId(int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x2;
            mSpeakerId = value;
            return this;
        }

        /**
         * Detected signal representing the data for allowing accessibility feature. This field can
         * only be set when the secure device settings is set to true by either settings page UI or
         * {@link VisualQueryDetector@setAccessibilityDetectionEnabled(boolean)}
         */
        @DataClass.Generated.Member
        @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
        public @NonNull Builder setAccessibilityDetectionData(@NonNull byte... value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x4;
            mAccessibilityDetectionData = value;
            return this;
        }

        /** Builds the instance. This builder should not be touched after calling this! */
        @SuppressLint("UnflaggedApi") // b/325678077 flags not supported in isolated process
        public @NonNull VisualQueryDetectedResult build() {
            checkNotUsed();
            mBuilderFieldsSet |= 0x8; // Mark builder used

            if ((mBuilderFieldsSet & 0x1) == 0) {
                mPartialQuery = defaultPartialQuery();
            }
            if ((mBuilderFieldsSet & 0x2) == 0) {
                mSpeakerId = defaultSpeakerId();
            }
            if ((mBuilderFieldsSet & 0x4) == 0) {
                mAccessibilityDetectionData = defaultAccessibilityDetectionData();
            }
            VisualQueryDetectedResult o = new VisualQueryDetectedResult(
                    mPartialQuery,
                    mSpeakerId,
                    mAccessibilityDetectionData);
            return o;
        }

        private void checkNotUsed() {
            if ((mBuilderFieldsSet & 0x8) != 0) {
                throw new IllegalStateException(
                        "This Builder should not be reused. Use a new Builder instance instead");
            }
        }
    }

    @DataClass.Generated(
            time = 1710958903381L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/service/voice/VisualQueryDetectedResult.java",
            inputSignatures = "private final @android.annotation.NonNull java.lang.String mPartialQuery\nprivate final  int mSpeakerId\nprivate final @android.annotation.Nullable byte[] mAccessibilityDetectionData\nprivate static  java.lang.String defaultPartialQuery()\nprivate static  int defaultSpeakerId()\npublic static @android.annotation.SuppressLint int getMaxSpeakerId()\nprivate static  byte[] defaultAccessibilityDetectionData()\nprivate  void onConstructed()\npublic  android.service.voice.VisualQueryDetectedResult.Builder buildUpon()\nclass VisualQueryDetectedResult extends java.lang.Object implements [android.os.Parcelable]\n@com.android.internal.util.DataClass(genConstructor=false, genBuilder=true, genEqualsHashCode=true, genHiddenConstDefs=true, genParcelable=true, genToString=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
