/*
 * Copyright (C) 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.service.voice;

import android.annotation.FlaggedApi;
import android.annotation.NonNull;
import android.annotation.SystemApi;
import android.os.Parcel;
import android.os.Parcelable;
import android.service.voice.flags.Flags;
import android.text.TextUtils;

import com.android.internal.util.DataClass;
import com.android.internal.util.Preconditions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Contains training data related to hotword detection service.
 *
 * <p>The constructed object's size must be within
 * {@link HotwordTrainingData#getMaxTrainingDataBytes()} or an
 * {@link IllegalArgumentException} will be thrown on construction. Size of the object is calculated
 * by converting object to a {@link Parcel} and using the {@link Parcel#dataSize()}.
 *
 * @hide
 */
@DataClass(
        genConstructor = false,
        genBuilder = true,
        genEqualsHashCode = true,
        genHiddenConstDefs = true,
        genParcelable = true,
        genToString = true)
@SystemApi
@FlaggedApi(Flags.FLAG_ALLOW_TRAINING_DATA_EGRESS_FROM_HDS)
public final class HotwordTrainingData implements Parcelable {
    /** Max size for hotword training data in bytes. */
    public static int getMaxTrainingDataBytes() {
        return 1024 * 1024; // 1 MB;
    }

    /** The list containing hotword audio that is useful for training. */
    @NonNull
    @DataClass.PluralOf("trainingAudio")
    private final List<HotwordTrainingAudio> mTrainingAudioList;

    private static List<HotwordTrainingAudio> defaultTrainingAudioList() {
        return Collections.emptyList();
    }

    /** App-defined stage when hotword model timed-out while running.
     * <p> Returns -1 if unset. */
    private final int mTimeoutStage;

    private static int defaultTimeoutStage() {
        return -1;
    }

    private void onConstructed() {
        // Verify size of object is within limit.
        Parcel parcel = Parcel.obtain();
        parcel.writeValue(this);
        int dataSizeBytes = parcel.dataSize();
        parcel.recycle();
        Preconditions.checkArgument(
                dataSizeBytes < getMaxTrainingDataBytes(),
                TextUtils.formatSimple(
                        "Hotword training data of size %s exceeds size limit of %s bytes!",
                        dataSizeBytes, getMaxTrainingDataBytes()));
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/service/voice/HotwordTrainingData.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    @DataClass.Generated.Member
    /* package-private */ HotwordTrainingData(
            @NonNull List<HotwordTrainingAudio> trainingAudioList,
            int timeoutStage) {
        this.mTrainingAudioList = trainingAudioList;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mTrainingAudioList);
        this.mTimeoutStage = timeoutStage;

        onConstructed();
    }

    /**
     * The list containing hotword audio that is useful for training.
     */
    @DataClass.Generated.Member
    public @NonNull List<HotwordTrainingAudio> getTrainingAudioList() {
        return mTrainingAudioList;
    }

    /**
     * App-defined stage when hotword model timed-out while running.
     * <p> Returns -1 if unset.
     */
    @DataClass.Generated.Member
    public int getTimeoutStage() {
        return mTimeoutStage;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "HotwordTrainingData { " +
                "trainingAudioList = " + mTrainingAudioList + ", " +
                "timeoutStage = " + mTimeoutStage +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public boolean equals(@android.annotation.Nullable Object o) {
        // You can override field equality logic by defining either of the methods like:
        // boolean fieldNameEquals(HotwordTrainingData other) { ... }
        // boolean fieldNameEquals(FieldType otherValue) { ... }

        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        @SuppressWarnings("unchecked")
        HotwordTrainingData that = (HotwordTrainingData) o;
        //noinspection PointlessBooleanExpression
        return true
                && java.util.Objects.equals(mTrainingAudioList, that.mTrainingAudioList)
                && mTimeoutStage == that.mTimeoutStage;
    }

    @Override
    @DataClass.Generated.Member
    public int hashCode() {
        // You can override field hashCode logic by defining methods like:
        // int fieldNameHashCode() { ... }

        int _hash = 1;
        _hash = 31 * _hash + java.util.Objects.hashCode(mTrainingAudioList);
        _hash = 31 * _hash + mTimeoutStage;
        return _hash;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeParcelableList(mTrainingAudioList, flags);
        dest.writeInt(mTimeoutStage);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ HotwordTrainingData(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        List<HotwordTrainingAudio> trainingAudioList = new ArrayList<>();
        in.readParcelableList(trainingAudioList, HotwordTrainingAudio.class.getClassLoader());
        int timeoutStage = in.readInt();

        this.mTrainingAudioList = trainingAudioList;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mTrainingAudioList);
        this.mTimeoutStage = timeoutStage;

        onConstructed();
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<HotwordTrainingData> CREATOR
            = new Parcelable.Creator<HotwordTrainingData>() {
        @Override
        public HotwordTrainingData[] newArray(int size) {
            return new HotwordTrainingData[size];
        }

        @Override
        public HotwordTrainingData createFromParcel(@NonNull Parcel in) {
            return new HotwordTrainingData(in);
        }
    };

    /**
     * A builder for {@link HotwordTrainingData}
     */
    @FlaggedApi(Flags.FLAG_ALLOW_TRAINING_DATA_EGRESS_FROM_HDS)
    @SuppressWarnings("WeakerAccess")
    @DataClass.Generated.Member
    public static final class Builder {

        private @NonNull List<HotwordTrainingAudio> mTrainingAudioList;
        private int mTimeoutStage;

        private long mBuilderFieldsSet = 0L;

        public Builder() {
        }

        /**
         * The list containing hotword audio that is useful for training.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setTrainingAudioList(@NonNull List<HotwordTrainingAudio> value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x1;
            mTrainingAudioList = value;
            return this;
        }

        /** @see #setTrainingAudioList */
        @DataClass.Generated.Member
        public @NonNull Builder addTrainingAudio(@NonNull HotwordTrainingAudio value) {
            if (mTrainingAudioList == null) setTrainingAudioList(new ArrayList<>());
            mTrainingAudioList.add(value);
            return this;
        }

        /**
         * App-defined stage when hotword model timed-out while running.
         * <p> Returns -1 if unset.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setTimeoutStage(int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x2;
            mTimeoutStage = value;
            return this;
        }

        /** Builds the instance. This builder should not be touched after calling this! */
        public @NonNull HotwordTrainingData build() {
            checkNotUsed();
            mBuilderFieldsSet |= 0x4; // Mark builder used

            if ((mBuilderFieldsSet & 0x1) == 0) {
                mTrainingAudioList = defaultTrainingAudioList();
            }
            if ((mBuilderFieldsSet & 0x2) == 0) {
                mTimeoutStage = defaultTimeoutStage();
            }
            HotwordTrainingData o = new HotwordTrainingData(
                    mTrainingAudioList,
                    mTimeoutStage);
            return o;
        }

        private void checkNotUsed() {
            if ((mBuilderFieldsSet & 0x4) != 0) {
                throw new IllegalStateException(
                        "This Builder should not be reused. Use a new Builder instance instead");
            }
        }
    }

    @DataClass.Generated(
            time = 1697826948280L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/service/voice/HotwordTrainingData.java",
            inputSignatures = "private final @android.annotation.NonNull @com.android.internal.util.DataClass.PluralOf(\"trainingAudio\") java.util.List<android.service.voice.HotwordTrainingAudio> mTrainingAudioList\nprivate final  int mTimeoutStage\npublic static  int getMaxTrainingDataBytes()\nprivate static  java.util.List<android.service.voice.HotwordTrainingAudio> defaultTrainingAudioList()\nprivate static  int defaultTimeoutStage()\nprivate  void onConstructed()\nclass HotwordTrainingData extends java.lang.Object implements [android.os.Parcelable]\n@com.android.internal.util.DataClass(genConstructor=false, genBuilder=true, genEqualsHashCode=true, genHiddenConstDefs=true, genParcelable=true, genToString=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
