/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.android.lint.aidl

import com.android.tools.lint.detector.api.getUMethod
import com.google.android.lint.ANNOTATION_PERMISSION_METHOD
import com.google.android.lint.ANNOTATION_PERMISSION_NAME
import com.google.android.lint.CLASS_STUB
import com.intellij.psi.PsiAnonymousClass
import org.jetbrains.uast.UCallExpression
import org.jetbrains.uast.UMethod
import org.jetbrains.uast.UParameter

/**
 * Given a UMethod, determine if this method is
 * an entrypoint to an interface generated by AIDL,
 * returning the interface name if so
 */
fun getContainingAidlInterface(node: UMethod): String? {
    if (!isInClassCalledStub(node)) return null
    for (superMethod in node.findSuperMethods()) {
        for (extendsInterface in superMethod.containingClass?.extendsList?.referenceElements
            ?: continue) {
            if (extendsInterface.qualifiedName == IINTERFACE_INTERFACE) {
                return superMethod.containingClass?.name
            }
        }
    }
    return null
}

private fun isInClassCalledStub(node: UMethod): Boolean {
    (node.containingClass as? PsiAnonymousClass)?.let {
        return it.baseClassReference.referenceName == CLASS_STUB
    }
    return node.containingClass?.extendsList?.referenceElements?.any {
        it.referenceName == CLASS_STUB
    } ?: false
}

fun isPermissionMethodCall(callExpression: UCallExpression): Boolean {
    val method = callExpression.resolve()?.getUMethod() ?: return false
    return hasPermissionMethodAnnotation(method)
}

fun hasPermissionMethodAnnotation(method: UMethod): Boolean = method.annotations
    .any {
        it.hasQualifiedName(ANNOTATION_PERMISSION_METHOD)
    }

fun hasPermissionNameAnnotation(parameter: UParameter) = parameter.annotations.any {
    it.hasQualifiedName(ANNOTATION_PERMISSION_NAME)
}
