/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.internal.protolog;

import com.android.internal.protolog.IProtoLogClient;

/**
 * The ProtoLog Service interface.
 *
 * This service runs in system server.
 *
 * All ProtoLog clients (in theory one per process) will register themselves to this service.
 * This service will then serve as the entry point for any shell command based interactions with
 * protolog and get/forward any required information/actions from/to all the registered ProtoLog
 * clients.
 *
 * Clients will be responsible for directly sending all their trace data to Perfetto without passing
 * through this service, except viewer config data, the mappings from messages hashes (generated by
 * the ProtoLog Tool if and when the tool processed the source code). So, this service is
 * responsible for dumping the viewer config data from all clients. The reason for this is because
 * we do this on Perfetto's onFlush callback when a tracing instance is stopped. However, if a
 * client process is frozen then this will not dump; system server, where this service runs cannot
 * be frozen. Secondly many processes (i.e. apps) will run the same code with the same viewer config
 * data, so this service allows us to orchestrate which config gets dumped so we don't duplicate
 * this information in the trace and waste valuable trace space.
 *
 * {@hide}
 */
interface IProtoLogService {
    interface IRegisterClientArgs {
        String[] getGroups();
        boolean[] getGroupsDefaultLogcatStatus();
        String getViewerConfigFile();
    }

    void registerClient(IProtoLogClient client, IRegisterClientArgs args);
}