package com.android.systemui.biometrics.ui.binder

import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.repeatOnLifecycle
import com.android.systemui.R
import com.android.systemui.animation.Interpolators
import com.android.systemui.biometrics.AuthDialog
import com.android.systemui.biometrics.AuthPanelController
import com.android.systemui.biometrics.ui.CredentialPasswordView
import com.android.systemui.biometrics.ui.CredentialPatternView
import com.android.systemui.biometrics.ui.CredentialView
import com.android.systemui.biometrics.ui.viewmodel.CredentialViewModel
import com.android.systemui.lifecycle.repeatWhenAttached
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch

/**
 * View binder for all credential variants of BiometricPrompt, including [CredentialPatternView] and
 * [CredentialPasswordView].
 *
 * This binder delegates to sub-binders for each variant, such as the [CredentialPasswordViewBinder]
 * and [CredentialPatternViewBinder].
 */
object CredentialViewBinder {

    /** Binds a [CredentialPasswordView] or [CredentialPatternView] to a [CredentialViewModel]. */
    @JvmStatic
    fun bind(
        view: ViewGroup,
        host: CredentialView.Host,
        viewModel: CredentialViewModel,
        panelViewController: AuthPanelController,
        animatePanel: Boolean,
        maxErrorDuration: Long = 3_000L,
    ) {
        val titleView: TextView = view.requireViewById(R.id.title)
        val subtitleView: TextView = view.requireViewById(R.id.subtitle)
        val descriptionView: TextView = view.requireViewById(R.id.description)
        val iconView: ImageView? = view.findViewById(R.id.icon)
        val errorView: TextView = view.requireViewById(R.id.error)

        var errorTimer: Job? = null

        // bind common elements
        view.repeatWhenAttached {
            if (animatePanel) {
                with(panelViewController) {
                    // Credential view is always full screen.
                    setUseFullScreen(true)
                    updateForContentDimensions(
                        containerWidth,
                        containerHeight,
                        0 /* animateDurationMs */
                    )
                }
            }

            repeatOnLifecycle(Lifecycle.State.STARTED) {
                // show prompt metadata
                launch {
                    viewModel.header.collect { header ->
                        titleView.text = header.title
                        view.announceForAccessibility(header.title)

                        subtitleView.textOrHide = header.subtitle
                        descriptionView.textOrHide = header.description

                        iconView?.setImageDrawable(header.icon)

                        // Only animate this if we're transitioning from a biometric view.
                        if (viewModel.animateContents.value) {
                            view.animateCredentialViewIn()
                        }
                    }
                }

                // show transient error messages
                launch {
                    viewModel.errorMessage
                        .onEach { msg ->
                            errorTimer?.cancel()
                            if (msg.isNotBlank()) {
                                errorTimer = launch {
                                    delay(maxErrorDuration)
                                    viewModel.resetErrorMessage()
                                }
                            }
                        }
                        .collect { errorView.textOrHide = it }
                }

                // show an extra dialog if the remaining attempts becomes low
                launch {
                    viewModel.remainingAttempts
                        .filter { it.remaining != null }
                        .collect { info ->
                            host.onCredentialAttemptsRemaining(info.remaining!!, info.message)
                        }
                }
            }
        }

        // bind the auth widget
        when (view) {
            is CredentialPasswordView -> CredentialPasswordViewBinder.bind(view, host, viewModel)
            is CredentialPatternView -> CredentialPatternViewBinder.bind(view, host, viewModel)
            else -> throw IllegalStateException("unexpected view type: ${view.javaClass.name}")
        }
    }
}

private fun View.animateCredentialViewIn() {
    translationY = resources.getDimension(R.dimen.biometric_dialog_credential_translation_offset)
    alpha = 0f
    postOnAnimation {
        animate()
            .translationY(0f)
            .setDuration(AuthDialog.ANIMATE_CREDENTIAL_INITIAL_DURATION_MS.toLong())
            .alpha(1f)
            .setInterpolator(Interpolators.LINEAR_OUT_SLOW_IN)
            .withLayer()
            .start()
    }
}

private var TextView.textOrHide: String?
    set(value) {
        val gone = value.isNullOrBlank()
        visibility = if (gone) View.GONE else View.VISIBLE
        text = if (gone) "" else value
    }
    get() = text?.toString()
