/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

package com.android.server.wm;

import static com.android.dx.mockito.inline.extended.ExtendedMockito.doReturn;
import static com.android.dx.mockito.inline.extended.ExtendedMockito.eq;
import static com.android.dx.mockito.inline.extended.ExtendedMockito.mock;
import static com.android.dx.mockito.inline.extended.ExtendedMockito.spy;
import static com.android.dx.mockito.inline.extended.ExtendedMockito.times;
import static com.android.dx.mockito.inline.extended.ExtendedMockito.verify;

import static com.google.common.truth.Truth.assertThat;

import android.content.Intent;
import android.os.Binder;
import android.platform.test.annotations.Presubmit;

import androidx.test.filters.SmallTest;

import com.android.server.wm.ActivityStarter.Factory;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * Tests for the {@link ActivityStartController} class.
 *
 * Build/Install/Run:
 *  atest WmTests:ActivityStartControllerTests
 */
@SmallTest
@Presubmit
@RunWith(WindowTestRunner.class)
public class ActivityStartControllerTests extends WindowTestsBase {
    private ActivityStartController mController;
    private Factory mFactory;
    private ActivityStarter mStarter;

    private static final String TEST_TYPE = "testType";

    @Before
    public void setUp() throws Exception {
        mFactory = mock(Factory.class);
        mController = new ActivityStartController(mAtm, mAtm.mTaskSupervisor, mFactory);
        mStarter = spy(new ActivityStarter(mController, mAtm,
                mAtm.mTaskSupervisor, mock(ActivityStartInterceptor.class)));
        doReturn(mStarter).when(mFactory).obtain();
    }

    /**
     * Ensures that when an [Activity] is started in a [TaskFragment] the associated
     * [ActivityStarter.Request] has the intent's resolved type correctly set.
     */
    @Test
    public void testStartActivityInTaskFragment_setsActivityStarterRequestResolvedType() {
        final Intent intent = new Intent();
        intent.setType(TEST_TYPE);

        mController.startActivityInTaskFragment(
                mock(TaskFragment.class),
                intent,
                null /* activityOptions */,
                null /* resultTo */ ,
                Binder.getCallingPid(),
                Binder.getCallingUid(),
                null /* errorCallbackToken */
        );

        assertThat(mStarter.mRequest.resolvedType).isEqualTo(TEST_TYPE);
    }

    /**
     * Ensures instances are recycled after execution.
     */
    @Test
    public void testRecycling() {
        final Intent intent = new Intent();
        final ActivityStarter optionStarter = new ActivityStarter(mController, mAtm,
                mAtm.mTaskSupervisor, mock(ActivityStartInterceptor.class));
        optionStarter
                .setIntent(intent)
                .setReason("Test")
                .execute();
        verify(mFactory, times(1)).recycle(eq(optionStarter));
    }
}
