/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.server.pm.pkg.component;

import static java.util.Collections.emptySet;

import android.annotation.NonNull;
import android.content.pm.ApplicationInfo;
import android.os.Parcel;
import android.os.Parcelable;
import android.util.ArrayMap;
import android.util.ArraySet;

import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.util.CollectionUtils;
import com.android.internal.util.DataClass;
import com.android.internal.util.Parcelling;

import java.util.Set;

/** @hide */
@DataClass(genGetters = true, genSetters = true, genParcelable = true, genAidl = false,
        genBuilder = false)
@VisibleForTesting(visibility = VisibleForTesting.Visibility.PACKAGE)
public class ParsedProcessImpl implements ParsedProcess, Parcelable {

    @NonNull
    private String name;

    /** @see ParsedProcess#getAppClassNamesByPackage() */
    @NonNull
    private ArrayMap<String, String> appClassNamesByPackage = ArrayMap.EMPTY;

    @NonNull
    @DataClass.ParcelWith(Parcelling.BuiltIn.ForInternedStringSet.class)
    private Set<String> deniedPermissions = emptySet();

    @ApplicationInfo.GwpAsanMode
    private int gwpAsanMode = ApplicationInfo.GWP_ASAN_DEFAULT;
    @ApplicationInfo.MemtagMode
    private int memtagMode = ApplicationInfo.MEMTAG_DEFAULT;
    @ApplicationInfo.NativeHeapZeroInitialized
    private int nativeHeapZeroInitialized = ApplicationInfo.ZEROINIT_DEFAULT;

    public ParsedProcessImpl() {
    }

    public ParsedProcessImpl(@NonNull ParsedProcess other) {
        name = other.getName();
        appClassNamesByPackage = (other.getAppClassNamesByPackage().size() == 0)
                ? ArrayMap.EMPTY : new ArrayMap<>(other.getAppClassNamesByPackage());
        deniedPermissions = new ArraySet<>(other.getDeniedPermissions());
        gwpAsanMode = other.getGwpAsanMode();
        memtagMode = other.getMemtagMode();
        nativeHeapZeroInitialized = other.getNativeHeapZeroInitialized();
    }

    public void addStateFrom(@NonNull ParsedProcess other) {
        deniedPermissions = CollectionUtils.addAll(deniedPermissions, other.getDeniedPermissions());
        gwpAsanMode = other.getGwpAsanMode();
        memtagMode = other.getMemtagMode();
        nativeHeapZeroInitialized = other.getNativeHeapZeroInitialized();

        final ArrayMap<String, String> oacn = other.getAppClassNamesByPackage();
        for (int i = 0; i < oacn.size(); i++) {
            appClassNamesByPackage.put(oacn.keyAt(i), oacn.valueAt(i));
        }
    }

    /**
     * Sets a custom application name used in this process for a given package.
     */
    public void putAppClassNameForPackage(String packageName, String className) {
        if (appClassNamesByPackage.size() == 0) {
            appClassNamesByPackage = new ArrayMap<>(4);
        }
        appClassNamesByPackage.put(packageName, className);
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/content/pm/parsing/component/ParsedProcessImpl.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /**
     * Creates a new ParsedProcessImpl.
     *
     */
    @DataClass.Generated.Member
    public ParsedProcessImpl(
            @NonNull String name,
            @NonNull ArrayMap<String,String> appClassNamesByPackage,
            @NonNull Set<String> deniedPermissions,
            @ApplicationInfo.GwpAsanMode int gwpAsanMode,
            @ApplicationInfo.MemtagMode int memtagMode,
            @ApplicationInfo.NativeHeapZeroInitialized int nativeHeapZeroInitialized) {
        this.name = name;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, name);
        this.appClassNamesByPackage = appClassNamesByPackage;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, appClassNamesByPackage);
        this.deniedPermissions = deniedPermissions;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, deniedPermissions);
        this.gwpAsanMode = gwpAsanMode;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.GwpAsanMode.class, null, gwpAsanMode);
        this.memtagMode = memtagMode;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.MemtagMode.class, null, memtagMode);
        this.nativeHeapZeroInitialized = nativeHeapZeroInitialized;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.NativeHeapZeroInitialized.class, null, nativeHeapZeroInitialized);

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public @NonNull String getName() {
        return name;
    }

    /**
     * @see ParsedProcess#getAppClassNamesByPackage()
     */
    @DataClass.Generated.Member
    public @NonNull ArrayMap<String,String> getAppClassNamesByPackage() {
        return appClassNamesByPackage;
    }

    @DataClass.Generated.Member
    public @NonNull Set<String> getDeniedPermissions() {
        return deniedPermissions;
    }

    @DataClass.Generated.Member
    public @ApplicationInfo.GwpAsanMode int getGwpAsanMode() {
        return gwpAsanMode;
    }

    @DataClass.Generated.Member
    public @ApplicationInfo.MemtagMode int getMemtagMode() {
        return memtagMode;
    }

    @DataClass.Generated.Member
    public @ApplicationInfo.NativeHeapZeroInitialized int getNativeHeapZeroInitialized() {
        return nativeHeapZeroInitialized;
    }

    @DataClass.Generated.Member
    public @NonNull ParsedProcessImpl setName(@NonNull String value) {
        name = value;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, name);
        return this;
    }

    /**
     * @see ParsedProcess#getAppClassNamesByPackage()
     */
    @DataClass.Generated.Member
    public @NonNull ParsedProcessImpl setAppClassNamesByPackage(@NonNull ArrayMap<String,String> value) {
        appClassNamesByPackage = value;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, appClassNamesByPackage);
        return this;
    }

    @DataClass.Generated.Member
    public @NonNull ParsedProcessImpl setDeniedPermissions(@NonNull Set<String> value) {
        deniedPermissions = value;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, deniedPermissions);
        return this;
    }

    @DataClass.Generated.Member
    public @NonNull ParsedProcessImpl setGwpAsanMode(@ApplicationInfo.GwpAsanMode int value) {
        gwpAsanMode = value;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.GwpAsanMode.class, null, gwpAsanMode);
        return this;
    }

    @DataClass.Generated.Member
    public @NonNull ParsedProcessImpl setMemtagMode(@ApplicationInfo.MemtagMode int value) {
        memtagMode = value;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.MemtagMode.class, null, memtagMode);
        return this;
    }

    @DataClass.Generated.Member
    public @NonNull ParsedProcessImpl setNativeHeapZeroInitialized(@ApplicationInfo.NativeHeapZeroInitialized int value) {
        nativeHeapZeroInitialized = value;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.NativeHeapZeroInitialized.class, null, nativeHeapZeroInitialized);
        return this;
    }

    @DataClass.Generated.Member
    static Parcelling<Set<String>> sParcellingForDeniedPermissions =
            Parcelling.Cache.get(
                    Parcelling.BuiltIn.ForInternedStringSet.class);
    static {
        if (sParcellingForDeniedPermissions == null) {
            sParcellingForDeniedPermissions = Parcelling.Cache.put(
                    new Parcelling.BuiltIn.ForInternedStringSet());
        }
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeString(name);
        dest.writeMap(appClassNamesByPackage);
        sParcellingForDeniedPermissions.parcel(deniedPermissions, dest, flags);
        dest.writeInt(gwpAsanMode);
        dest.writeInt(memtagMode);
        dest.writeInt(nativeHeapZeroInitialized);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    protected ParsedProcessImpl(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        String _name = in.readString();
        ArrayMap<String,String> _appClassNamesByPackage = new ArrayMap();
        in.readMap(_appClassNamesByPackage, String.class.getClassLoader());
        Set<String> _deniedPermissions = sParcellingForDeniedPermissions.unparcel(in);
        int _gwpAsanMode = in.readInt();
        int _memtagMode = in.readInt();
        int _nativeHeapZeroInitialized = in.readInt();

        this.name = _name;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, name);
        this.appClassNamesByPackage = _appClassNamesByPackage;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, appClassNamesByPackage);
        this.deniedPermissions = _deniedPermissions;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, deniedPermissions);
        this.gwpAsanMode = _gwpAsanMode;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.GwpAsanMode.class, null, gwpAsanMode);
        this.memtagMode = _memtagMode;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.MemtagMode.class, null, memtagMode);
        this.nativeHeapZeroInitialized = _nativeHeapZeroInitialized;
        com.android.internal.util.AnnotationValidations.validate(
                ApplicationInfo.NativeHeapZeroInitialized.class, null, nativeHeapZeroInitialized);

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<ParsedProcessImpl> CREATOR
            = new Parcelable.Creator<ParsedProcessImpl>() {
        @Override
        public ParsedProcessImpl[] newArray(int size) {
            return new ParsedProcessImpl[size];
        }

        @Override
        public ParsedProcessImpl createFromParcel(@NonNull Parcel in) {
            return new ParsedProcessImpl(in);
        }
    };

    @DataClass.Generated(
            time = 1641431953775L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/content/pm/parsing/component/ParsedProcessImpl.java",
            inputSignatures = "private @android.annotation.NonNull java.lang.String name\nprivate @android.annotation.NonNull android.util.ArrayMap<java.lang.String,java.lang.String> appClassNamesByPackage\nprivate @android.annotation.NonNull @com.android.internal.util.DataClass.ParcelWith(com.android.internal.util.Parcelling.BuiltIn.ForInternedStringSet.class) java.util.Set<java.lang.String> deniedPermissions\nprivate @android.content.pm.ApplicationInfo.GwpAsanMode int gwpAsanMode\nprivate @android.content.pm.ApplicationInfo.MemtagMode int memtagMode\nprivate @android.content.pm.ApplicationInfo.NativeHeapZeroInitialized int nativeHeapZeroInitialized\npublic  void addStateFrom(android.content.pm.parsing.component.ParsedProcess)\npublic  void putAppClassNameForPackage(java.lang.String,java.lang.String)\nclass ParsedProcessImpl extends java.lang.Object implements [android.content.pm.parsing.component.ParsedProcess, android.os.Parcelable]\n@com.android.internal.util.DataClass(genGetters=true, genSetters=true, genParcelable=true, genAidl=false, genBuilder=false)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
