/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

package com.android.systemui.keyguard.domain.interactor

import android.animation.ValueAnimator
import android.annotation.SuppressLint
import android.app.DreamManager
import com.android.app.animation.Interpolators
import com.android.systemui.Flags.communalSceneKtfRefactor
import com.android.systemui.communal.domain.interactor.CommunalInteractor
import com.android.systemui.communal.domain.interactor.CommunalSceneInteractor
import com.android.systemui.communal.shared.model.CommunalScenes
import com.android.systemui.communal.shared.model.CommunalTransitionKeys
import com.android.systemui.dagger.SysUISingleton
import com.android.systemui.dagger.qualifiers.Background
import com.android.systemui.dagger.qualifiers.Main
import com.android.systemui.deviceentry.domain.interactor.DeviceEntryInteractor
import com.android.systemui.keyguard.KeyguardWmStateRefactor
import com.android.systemui.keyguard.data.repository.KeyguardTransitionRepository
import com.android.systemui.keyguard.shared.model.BiometricUnlockMode.Companion.isWakeAndUnlock
import com.android.systemui.keyguard.shared.model.KeyguardState
import com.android.systemui.power.domain.interactor.PowerInteractor
import com.android.systemui.power.shared.model.WakeSleepReason
import com.android.systemui.scene.shared.flag.SceneContainerFlag
import com.android.systemui.util.kotlin.Utils.Companion.sample
import com.android.systemui.util.kotlin.sample
import javax.inject.Inject
import kotlin.time.Duration.Companion.milliseconds
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.launch

@SysUISingleton
class FromDozingTransitionInteractor
@Inject
constructor(
    override val transitionRepository: KeyguardTransitionRepository,
    override val internalTransitionInteractor: InternalKeyguardTransitionInteractor,
    transitionInteractor: KeyguardTransitionInteractor,
    @Background private val scope: CoroutineScope,
    @Background bgDispatcher: CoroutineDispatcher,
    @Main mainDispatcher: CoroutineDispatcher,
    keyguardInteractor: KeyguardInteractor,
    powerInteractor: PowerInteractor,
    private val communalInteractor: CommunalInteractor,
    private val communalSceneInteractor: CommunalSceneInteractor,
    keyguardOcclusionInteractor: KeyguardOcclusionInteractor,
    val deviceEntryInteractor: DeviceEntryInteractor,
    private val wakeToGoneInteractor: KeyguardWakeDirectlyToGoneInteractor,
    private val dreamManager: DreamManager,
) :
    TransitionInteractor(
        fromState = KeyguardState.DOZING,
        transitionInteractor = transitionInteractor,
        mainDispatcher = mainDispatcher,
        bgDispatcher = bgDispatcher,
        powerInteractor = powerInteractor,
        keyguardOcclusionInteractor = keyguardOcclusionInteractor,
        keyguardInteractor = keyguardInteractor,
    ) {

    override fun start() {
        listenForDozingToAny()
        listenForDozingToGoneViaBiometrics()
        listenForWakeFromDozing()
        listenForTransitionToCamera(scope, keyguardInteractor)
    }

    private val canTransitionToGoneOnWake: Flow<Boolean> =
        combine(
            keyguardInteractor.isKeyguardShowing,
            keyguardInteractor.isKeyguardDismissible,
        ) { isKeyguardShowing, isKeyguardDismissible ->
            isKeyguardDismissible && !isKeyguardShowing
        }

    private fun listenForDozingToGoneViaBiometrics() {
        if (KeyguardWmStateRefactor.isEnabled) {
            return
        }

        // This is separate from `listenForDozingToAny` because any delay on wake and unlock will
        // cause a noticeable issue with animations
        scope.launch {
            powerInteractor.isAwake
                .filterRelevantKeyguardStateAnd { isAwake -> isAwake }
                .sample(
                    keyguardInteractor.biometricUnlockState,
                    ::Pair,
                )
                .collect {
                    (
                        _,
                        biometricUnlockState,
                    ) ->
                    if (isWakeAndUnlock(biometricUnlockState.mode)) {
                        if (SceneContainerFlag.isEnabled) {
                            // TODO(b/336576536): Check if adaptation for scene framework is needed
                        } else {
                            startTransitionTo(
                                KeyguardState.GONE,
                                ownerReason = "biometric wake and unlock",
                            )
                        }
                    }
                }
        }
    }

    @SuppressLint("MissingPermission")
    private fun listenForDozingToAny() {
        if (KeyguardWmStateRefactor.isEnabled) {
            return
        }

        scope.launch {
            powerInteractor.isAwake
                .debounce(50L)
                .filterRelevantKeyguardStateAnd { isAwake -> isAwake }
                .sample(
                    keyguardInteractor.isKeyguardOccluded,
                    communalInteractor.isCommunalAvailable,
                    communalSceneInteractor.isIdleOnCommunal,
                    canTransitionToGoneOnWake,
                    keyguardInteractor.primaryBouncerShowing,
                )
                .collect {
                    (
                        _,
                        occluded,
                        isCommunalAvailable,
                        isIdleOnCommunal,
                        canTransitionToGoneOnWake,
                        primaryBouncerShowing) ->
                    if (!deviceEntryInteractor.isLockscreenEnabled()) {
                        if (SceneContainerFlag.isEnabled) {
                            // TODO(b/336576536): Check if adaptation for scene framework is needed
                        } else {
                            startTransitionTo(KeyguardState.GONE)
                        }
                    } else if (canTransitionToGoneOnWake) {
                        if (SceneContainerFlag.isEnabled) {
                            // TODO(b/336576536): Check if adaptation for scene framework is needed
                        } else {
                            startTransitionTo(KeyguardState.GONE)
                        }
                    } else if (primaryBouncerShowing) {
                        if (SceneContainerFlag.isEnabled) {
                            // TODO(b/336576536): Check if adaptation for scene framework is needed
                        } else {
                            startTransitionTo(KeyguardState.PRIMARY_BOUNCER)
                        }
                    } else if (occluded) {
                        startTransitionTo(KeyguardState.OCCLUDED)
                    } else if (isIdleOnCommunal && !communalSceneKtfRefactor()) {
                        if (SceneContainerFlag.isEnabled) {
                            // TODO(b/336576536): Check if adaptation for scene framework is needed
                        } else {
                            startTransitionTo(KeyguardState.GLANCEABLE_HUB)
                        }
                    } else if (
                        powerInteractor.detailedWakefulness.value.lastWakeReason ==
                            WakeSleepReason.POWER_BUTTON &&
                            isCommunalAvailable &&
                            dreamManager.canStartDreaming(true)
                    ) {
                        // This case handles tapping the power button to transition through
                        // dream -> off -> hub.
                        if (SceneContainerFlag.isEnabled) {
                            // TODO(b/336576536): Check if adaptation for scene framework is needed
                        } else {
                            transitionToGlanceableHub()
                        }
                    } else {
                        startTransitionTo(KeyguardState.LOCKSCREEN)
                    }
                }
        }
    }

    /** Figure out what state to transition to when we awake from DOZING. */
    @SuppressLint("MissingPermission")
    private fun listenForWakeFromDozing() {
        if (!KeyguardWmStateRefactor.isEnabled) {
            return
        }

        scope.launch {
            powerInteractor.detailedWakefulness
                .filterRelevantKeyguardStateAnd { it.isAwake() }
                .sample(
                    communalInteractor.isCommunalAvailable,
                    communalSceneInteractor.isIdleOnCommunal,
                    keyguardInteractor.biometricUnlockState,
                    wakeToGoneInteractor.canWakeDirectlyToGone,
                    keyguardInteractor.primaryBouncerShowing,
                )
                .collect {
                    (
                        _,
                        isCommunalAvailable,
                        isIdleOnCommunal,
                        biometricUnlockState,
                        canWakeDirectlyToGone,
                        primaryBouncerShowing) ->
                    if (
                        !maybeStartTransitionToOccludedOrInsecureCamera { state, reason ->
                            startTransitionTo(state, ownerReason = reason)
                        } &&
                            // Handled by dismissFromDozing().
                            !isWakeAndUnlock(biometricUnlockState.mode)
                    ) {
                        if (canWakeDirectlyToGone) {
                            if (SceneContainerFlag.isEnabled) {
                                // TODO(b/336576536): Check if adaptation for scene framework is
                                // needed
                            } else {
                                startTransitionTo(
                                    KeyguardState.GONE,
                                    ownerReason = "waking from dozing"
                                )
                            }
                        } else if (primaryBouncerShowing) {
                            if (SceneContainerFlag.isEnabled) {
                                // TODO(b/336576536): Check if adaptation for scene framework is
                                // needed
                            } else {
                                startTransitionTo(
                                    KeyguardState.PRIMARY_BOUNCER,
                                    ownerReason = "waking from dozing"
                                )
                            }
                        } else if (isIdleOnCommunal && !communalSceneKtfRefactor()) {
                            if (SceneContainerFlag.isEnabled) {
                                // TODO(b/336576536): Check if adaptation for scene framework is
                                // needed
                            } else {
                                startTransitionTo(
                                    KeyguardState.GLANCEABLE_HUB,
                                    ownerReason = "waking from dozing"
                                )
                            }
                        } else if (
                            powerInteractor.detailedWakefulness.value.lastWakeReason ==
                                WakeSleepReason.POWER_BUTTON &&
                                isCommunalAvailable &&
                                dreamManager.canStartDreaming(true)
                        ) {
                            // This case handles tapping the power button to transition through
                            // dream -> off -> hub.
                            if (SceneContainerFlag.isEnabled) {
                                // TODO(b/336576536): Check if adaptation for scene framework is
                                // needed
                            } else {
                                transitionToGlanceableHub()
                            }
                        } else {
                            startTransitionTo(
                                KeyguardState.LOCKSCREEN,
                                ownerReason = "waking from dozing"
                            )
                        }
                    }
                }
        }
    }

    private suspend fun transitionToGlanceableHub() {
        if (communalSceneKtfRefactor()) {
            communalSceneInteractor.changeScene(
                CommunalScenes.Communal,
                // Immediately show the hub when transitioning from dozing to hub.
                CommunalTransitionKeys.Immediately,
            )
        } else {
            startTransitionTo(KeyguardState.GLANCEABLE_HUB)
        }
    }

    /** Dismisses keyguard from the DOZING state. */
    fun dismissFromDozing() {
        scope.launch { startTransitionTo(KeyguardState.GONE) }
    }

    override fun getDefaultAnimatorForTransitionsToState(toState: KeyguardState): ValueAnimator {
        return ValueAnimator().apply {
            interpolator = Interpolators.LINEAR
            duration = DEFAULT_DURATION.inWholeMilliseconds
        }
    }

    companion object {
        const val TAG = "FromDozingTransitionInteractor"
        private val DEFAULT_DURATION = 500.milliseconds
        val TO_LOCKSCREEN_DURATION = DEFAULT_DURATION
        val TO_GONE_DURATION = DEFAULT_DURATION
        val TO_PRIMARY_BOUNCER_DURATION = DEFAULT_DURATION
        val TO_GLANCEABLE_HUB_DURATION = DEFAULT_DURATION
    }
}
