/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.android.systemui.statusbar;

import android.content.pm.UserInfo;
import android.util.SparseArray;

import com.android.systemui.statusbar.notification.collection.NotificationEntry;

public interface NotificationLockscreenUserManager {
    String PERMISSION_SELF = "com.android.systemui.permission.SELF";
    String NOTIFICATION_UNLOCKED_BY_WORK_CHALLENGE_ACTION
            = "com.android.systemui.statusbar.work_challenge_unlocked_notification_action";

    boolean shouldAllowLockscreenRemoteInput();

    /**
     * @param userId user Id
     * @return true if we re on a secure lock screen
     */
    boolean isLockscreenPublicMode(int userId);

    /**
     * Does a user require a separate work challenge? If so, the unlock mechanism is decoupled from
     * the current user and has to be solved separately.
     */
    default boolean needsSeparateWorkChallenge(int userId) {
        return false;
    }

    void setUpWithPresenter(NotificationPresenter presenter);

    int getCurrentUserId();

    boolean isCurrentProfile(int userId);

    /** Adds a listener to be notified when the current user changes. */
    void addUserChangedListener(UserChangedListener listener);

    /**
     * Registers a [KeyguardNotificationSuppressor] that will be consulted during
     * {@link #shouldShowOnKeyguard(NotificationEntry)}
     */
    void addKeyguardNotificationSuppressor(KeyguardNotificationSuppressor suppressor);

    /**
     * Removes a listener previously registered with
     * {@link #addUserChangedListener(UserChangedListener)}
     */
    void removeUserChangedListener(UserChangedListener listener);

    SparseArray<UserInfo> getCurrentProfiles();

    void setLockscreenPublicMode(boolean isProfilePublic, int userId);

    boolean shouldShowLockscreenNotifications();

    boolean shouldHideNotifications(int userId);
    boolean shouldHideNotifications(String key);
    boolean shouldShowOnKeyguard(NotificationEntry entry);

    void addOnNeedsRedactionInPublicChangedListener(Runnable listener);

    void removeOnNeedsRedactionInPublicChangedListener(Runnable listener);

    boolean isAnyProfilePublicMode();

    void updatePublicMode();

    /** Does this notification require redaction if it is displayed when the device is public? */
    boolean notifNeedsRedactionInPublic(NotificationEntry entry);

    /**
     * Do all sensitive notifications belonging to the given user require redaction when they are
     * displayed in public?
     */
    boolean sensitiveNotifsNeedRedactionInPublic(int userId);

    /**
     * Has the given user chosen to allow notifications to be shown even when the lockscreen is in
     * "public" (secure & locked) mode?
     */
    boolean userAllowsNotificationsInPublic(int userId);

    /** Notified when the current user changes. */
    interface UserChangedListener {
        default void onUserChanged(int userId) {}
        default void onCurrentProfilesChanged(SparseArray<UserInfo> currentProfiles) {}
    }

    /** Used to hide notifications on the lockscreen */
    interface KeyguardNotificationSuppressor {
        boolean shouldSuppressOnKeyguard(NotificationEntry entry);
    }
}
