/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.service.displayhash;

import android.graphics.Rect;
import android.hardware.HardwareBuffer;
import android.os.RemoteCallback;
import android.view.displayhash.DisplayHash;

/**
 * Service used to handle DisplayHash requests.
 *
 * @hide
 */
oneway interface IDisplayHashingService {
    /**
     * Generates the DisplayHash that can be used to validate that the system generated the token.
     *
     * @param salt The salt to use when generating the hmac. This should be unique to the caller so
     *        the token cannot be verified by any other process.
     * @param buffer The buffer to generate the hash for.
     * @param bounds The size and position of the content being hashed in window space.
     * @param hashAlgorithm The String for the hash algorithm to use based on values in
     *        {@link #SERVICE_META_DATA_KEY_AVAILABLE_ALGORITHMS}.
     * @param Callback The callback invoked to send back the DisplayHash.
     */
    void generateDisplayHash(in byte[] salt, in HardwareBuffer buffer, in Rect bounds,
                                 in String hashAlgorithm, in RemoteCallback callback);

    /**
     * Call to verify that the DisplayHash passed in was generated by the system. The result
     * will be sent in the callback as a boolean with the key {@link #EXTRA_VERIFIED_DISPLAY_HASH}.
     *
     * @param salt The salt value to use when verifying the hmac. This should be the same value that
     *        was passed to {@link generateDisplayHash()} to generate the DisplayHash.
     * @param displayHash The hash to verify that it was generated by the system.
     * @param callback The callback invoked to send back the VerifiedDisplayHash.
     */
    void verifyDisplayHash(in byte[] salt, in DisplayHash displayHash, in RemoteCallback callback);

    /**
     * Call to get a map of supported algorithms and their {@link DisplayHashParams}
     *
     * @param callback The callback invoked to send back the map of algorithms to DisplayHashParams.
     */
    void getDisplayHashAlgorithms(in RemoteCallback callback);

    /**
     * Call to get the interval required between display hash requests. Requests made faster than
     * this will be throttled. The result will be sent in the callback as an int with the key
     * {@link #EXTRA_INTERVAL_BETWEEN_REQUESTS}.
     *
     * @param callback The callback invoked to send back the interval duration.
     */
    void getIntervalBetweenRequestsMillis(in RemoteCallback callback);
}
