/*
 * Copyright (C) 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.hardware.face;


import static android.os.PowerManager.WAKE_REASON_UNKNOWN;

import android.annotation.IntDef;
import android.annotation.NonNull;
import android.annotation.Nullable;
import android.hardware.biometrics.AuthenticateOptions;
import android.os.Parcelable;
import android.os.PowerManager;

import com.android.internal.util.DataClass;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * Additional options when requesting Face authentication or detection.
 *
 * @hide
 */
@DataClass(
        genParcelable = true,
        genAidl = true,
        genBuilder = true,
        genSetters = true,
        genEqualsHashCode = true
)
public class FaceAuthenticateOptions implements AuthenticateOptions, Parcelable {

    /** The user id for this operation. */
    private final int mUserId;
    private static int defaultUserId() {
        return 0;
    }

    /** The sensor id for this operation. */
    private int mSensorId;
    private static int defaultSensorId() {
        return -1;
    }

    /** The current doze state of the device. */
    @AuthenticateOptions.DisplayState
    private final int mDisplayState;
    private static int defaultDisplayState() {
        return DISPLAY_STATE_UNKNOWN;
    }

    public static final int AUTHENTICATE_REASON_UNKNOWN = 0;
    public static final int AUTHENTICATE_REASON_STARTED_WAKING_UP = 1;
    public static final int AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN = 2;
    public static final int AUTHENTICATE_REASON_ASSISTANT_VISIBLE = 3;
    public static final int AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN = 4;
    public static final int AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED = 5;
    public static final int AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED = 6;
    public static final int AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED = 7;
    public static final int AUTHENTICATE_REASON_QS_EXPANDED = 8;
    public static final int AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER = 9;
    public static final int AUTHENTICATE_REASON_UDFPS_POINTER_DOWN = 10;

    /**
     * The reason for this operation when requested by the system (sysui),
     * otherwise AUTHENTICATE_REASON_UNKNOWN.
     *
     * See frameworks/base/packages/SystemUI/src/com/android/keyguard/FaceAuthReason.kt
     * for more details about each reason.
     */
    @AuthenticateReason
    private final int mAuthenticateReason;
    private static int defaultAuthenticateReason() {
        return AUTHENTICATE_REASON_UNKNOWN;
    }

    /** A reason if this request was triggered due to a power event or WAKE_REASON_UNKNOWN. */
    @PowerManager.WakeReason
    private final int mWakeReason;
    private static int defaultWakeReason() {
        return WAKE_REASON_UNKNOWN;
    }

    /**
     * The package name for that operation that should be used for
     * {@link android.app.AppOpsManager} verification.
     *
     * This option may be overridden by the FingerprintManager using the caller's context.
     */
    @NonNull
    private String mOpPackageName;
    private static String defaultOpPackageName() {
        return "";
    }

    /**
     * The attribution tag, if any.
     *
     * This option may be overridden by the FingerprintManager using the caller's context.
     */
    @Nullable
    private String mAttributionTag;
    private static String defaultAttributionTag() {
        return null;
    }


    /** If the authentication is requested due to mandatory biometrics being active. */
    private boolean mIsMandatoryBiometrics;

    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/hardware/face/FaceAuthenticateOptions.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    @IntDef(prefix = "AUTHENTICATE_REASON_", value = {
        AUTHENTICATE_REASON_UNKNOWN,
        AUTHENTICATE_REASON_STARTED_WAKING_UP,
        AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN,
        AUTHENTICATE_REASON_ASSISTANT_VISIBLE,
        AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN,
        AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED,
        AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED,
        AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED,
        AUTHENTICATE_REASON_QS_EXPANDED,
        AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER,
        AUTHENTICATE_REASON_UDFPS_POINTER_DOWN
    })
    @Retention(RetentionPolicy.SOURCE)
    @DataClass.Generated.Member
    public @interface AuthenticateReason {}

    @DataClass.Generated.Member
    public static String authenticateReasonToString(@AuthenticateReason int value) {
        switch (value) {
            case AUTHENTICATE_REASON_UNKNOWN:
                    return "AUTHENTICATE_REASON_UNKNOWN";
            case AUTHENTICATE_REASON_STARTED_WAKING_UP:
                    return "AUTHENTICATE_REASON_STARTED_WAKING_UP";
            case AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN:
                    return "AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN";
            case AUTHENTICATE_REASON_ASSISTANT_VISIBLE:
                    return "AUTHENTICATE_REASON_ASSISTANT_VISIBLE";
            case AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN:
                    return "AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN";
            case AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED:
                    return "AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED";
            case AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED:
                    return "AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED";
            case AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED:
                    return "AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED";
            case AUTHENTICATE_REASON_QS_EXPANDED:
                    return "AUTHENTICATE_REASON_QS_EXPANDED";
            case AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER:
                    return "AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER";
            case AUTHENTICATE_REASON_UDFPS_POINTER_DOWN:
                    return "AUTHENTICATE_REASON_UDFPS_POINTER_DOWN";
            default: return Integer.toHexString(value);
        }
    }

    @DataClass.Generated.Member
    /* package-private */ FaceAuthenticateOptions(
            int userId,
            int sensorId,
            @AuthenticateOptions.DisplayState int displayState,
            @AuthenticateReason int authenticateReason,
            @PowerManager.WakeReason int wakeReason,
            @NonNull String opPackageName,
            @Nullable String attributionTag,
            boolean isMandatoryBiometrics) {
        this.mUserId = userId;
        this.mSensorId = sensorId;
        this.mDisplayState = displayState;
        com.android.internal.util.AnnotationValidations.validate(
                AuthenticateOptions.DisplayState.class, null, mDisplayState);
        this.mAuthenticateReason = authenticateReason;

        if (!(mAuthenticateReason == AUTHENTICATE_REASON_UNKNOWN)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_STARTED_WAKING_UP)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_ASSISTANT_VISIBLE)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_QS_EXPANDED)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_UDFPS_POINTER_DOWN)) {
            throw new java.lang.IllegalArgumentException(
                    "authenticateReason was " + mAuthenticateReason + " but must be one of: "
                            + "AUTHENTICATE_REASON_UNKNOWN(" + AUTHENTICATE_REASON_UNKNOWN + "), "
                            + "AUTHENTICATE_REASON_STARTED_WAKING_UP(" + AUTHENTICATE_REASON_STARTED_WAKING_UP + "), "
                            + "AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN(" + AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN + "), "
                            + "AUTHENTICATE_REASON_ASSISTANT_VISIBLE(" + AUTHENTICATE_REASON_ASSISTANT_VISIBLE + "), "
                            + "AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN(" + AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN + "), "
                            + "AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED(" + AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED + "), "
                            + "AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED(" + AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED + "), "
                            + "AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED(" + AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED + "), "
                            + "AUTHENTICATE_REASON_QS_EXPANDED(" + AUTHENTICATE_REASON_QS_EXPANDED + "), "
                            + "AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER(" + AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER + "), "
                            + "AUTHENTICATE_REASON_UDFPS_POINTER_DOWN(" + AUTHENTICATE_REASON_UDFPS_POINTER_DOWN + ")");
        }

        this.mWakeReason = wakeReason;
        com.android.internal.util.AnnotationValidations.validate(
                PowerManager.WakeReason.class, null, mWakeReason);
        this.mOpPackageName = opPackageName;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mOpPackageName);
        this.mAttributionTag = attributionTag;
        this.mIsMandatoryBiometrics = isMandatoryBiometrics;

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * The user id for this operation.
     */
    @DataClass.Generated.Member
    public int getUserId() {
        return mUserId;
    }

    /**
     * The sensor id for this operation.
     */
    @DataClass.Generated.Member
    public int getSensorId() {
        return mSensorId;
    }

    /**
     * The current doze state of the device.
     */
    @DataClass.Generated.Member
    public @AuthenticateOptions.DisplayState int getDisplayState() {
        return mDisplayState;
    }

    /**
     * The reason for this operation when requested by the system (sysui),
     * otherwise AUTHENTICATE_REASON_UNKNOWN.
     *
     * See frameworks/base/packages/SystemUI/src/com/android/keyguard/FaceAuthReason.kt
     * for more details about each reason.
     */
    @DataClass.Generated.Member
    public @AuthenticateReason int getAuthenticateReason() {
        return mAuthenticateReason;
    }

    /**
     * A reason if this request was triggered due to a power event or WAKE_REASON_UNKNOWN.
     */
    @DataClass.Generated.Member
    public @PowerManager.WakeReason int getWakeReason() {
        return mWakeReason;
    }

    /**
     * The package name for that operation that should be used for
     * {@link android.app.AppOpsManager} verification.
     *
     * This option may be overridden by the FingerprintManager using the caller's context.
     */
    @DataClass.Generated.Member
    public @NonNull String getOpPackageName() {
        return mOpPackageName;
    }

    /**
     * The attribution tag, if any.
     *
     * This option may be overridden by the FingerprintManager using the caller's context.
     */
    @DataClass.Generated.Member
    public @Nullable String getAttributionTag() {
        return mAttributionTag;
    }

    /**
     * If the authentication is requested due to mandatory biometrics being active.
     */
    @DataClass.Generated.Member
    public boolean isMandatoryBiometrics() {
        return mIsMandatoryBiometrics;
    }

    /**
     * The sensor id for this operation.
     */
    @DataClass.Generated.Member
    public @NonNull FaceAuthenticateOptions setSensorId( int value) {
        mSensorId = value;
        return this;
    }

    /**
     * The package name for that operation that should be used for
     * {@link android.app.AppOpsManager} verification.
     *
     * This option may be overridden by the FingerprintManager using the caller's context.
     */
    @DataClass.Generated.Member
    public @NonNull FaceAuthenticateOptions setOpPackageName(@NonNull String value) {
        mOpPackageName = value;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mOpPackageName);
        return this;
    }

    /**
     * The attribution tag, if any.
     *
     * This option may be overridden by the FingerprintManager using the caller's context.
     */
    @DataClass.Generated.Member
    public @NonNull FaceAuthenticateOptions setAttributionTag(@NonNull String value) {
        mAttributionTag = value;
        return this;
    }

    /**
     * If the authentication is requested due to mandatory biometrics being active.
     */
    @DataClass.Generated.Member
    public @NonNull FaceAuthenticateOptions setIsMandatoryBiometrics( boolean value) {
        mIsMandatoryBiometrics = value;
        return this;
    }

    @Override
    @DataClass.Generated.Member
    public boolean equals(@Nullable Object o) {
        // You can override field equality logic by defining either of the methods like:
        // boolean fieldNameEquals(FaceAuthenticateOptions other) { ... }
        // boolean fieldNameEquals(FieldType otherValue) { ... }

        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        @SuppressWarnings("unchecked")
        FaceAuthenticateOptions that = (FaceAuthenticateOptions) o;
        //noinspection PointlessBooleanExpression
        return true
                && mUserId == that.mUserId
                && mSensorId == that.mSensorId
                && mDisplayState == that.mDisplayState
                && mAuthenticateReason == that.mAuthenticateReason
                && mWakeReason == that.mWakeReason
                && java.util.Objects.equals(mOpPackageName, that.mOpPackageName)
                && java.util.Objects.equals(mAttributionTag, that.mAttributionTag)
                && mIsMandatoryBiometrics == that.mIsMandatoryBiometrics;
    }

    @Override
    @DataClass.Generated.Member
    public int hashCode() {
        // You can override field hashCode logic by defining methods like:
        // int fieldNameHashCode() { ... }

        int _hash = 1;
        _hash = 31 * _hash + mUserId;
        _hash = 31 * _hash + mSensorId;
        _hash = 31 * _hash + mDisplayState;
        _hash = 31 * _hash + mAuthenticateReason;
        _hash = 31 * _hash + mWakeReason;
        _hash = 31 * _hash + java.util.Objects.hashCode(mOpPackageName);
        _hash = 31 * _hash + java.util.Objects.hashCode(mAttributionTag);
        _hash = 31 * _hash + Boolean.hashCode(mIsMandatoryBiometrics);
        return _hash;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull android.os.Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        int flg = 0;
        if (mIsMandatoryBiometrics) flg |= 0x80;
        if (mAttributionTag != null) flg |= 0x40;
        dest.writeInt(flg);
        dest.writeInt(mUserId);
        dest.writeInt(mSensorId);
        dest.writeInt(mDisplayState);
        dest.writeInt(mAuthenticateReason);
        dest.writeInt(mWakeReason);
        dest.writeString(mOpPackageName);
        if (mAttributionTag != null) dest.writeString(mAttributionTag);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    protected FaceAuthenticateOptions(@NonNull android.os.Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        int flg = in.readInt();
        boolean isMandatoryBiometrics = (flg & 0x80) != 0;
        int userId = in.readInt();
        int sensorId = in.readInt();
        int displayState = in.readInt();
        int authenticateReason = in.readInt();
        int wakeReason = in.readInt();
        String opPackageName = in.readString();
        String attributionTag = (flg & 0x40) == 0 ? null : in.readString();

        this.mUserId = userId;
        this.mSensorId = sensorId;
        this.mDisplayState = displayState;
        com.android.internal.util.AnnotationValidations.validate(
                AuthenticateOptions.DisplayState.class, null, mDisplayState);
        this.mAuthenticateReason = authenticateReason;

        if (!(mAuthenticateReason == AUTHENTICATE_REASON_UNKNOWN)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_STARTED_WAKING_UP)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_ASSISTANT_VISIBLE)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_QS_EXPANDED)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER)
                && !(mAuthenticateReason == AUTHENTICATE_REASON_UDFPS_POINTER_DOWN)) {
            throw new java.lang.IllegalArgumentException(
                    "authenticateReason was " + mAuthenticateReason + " but must be one of: "
                            + "AUTHENTICATE_REASON_UNKNOWN(" + AUTHENTICATE_REASON_UNKNOWN + "), "
                            + "AUTHENTICATE_REASON_STARTED_WAKING_UP(" + AUTHENTICATE_REASON_STARTED_WAKING_UP + "), "
                            + "AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN(" + AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN + "), "
                            + "AUTHENTICATE_REASON_ASSISTANT_VISIBLE(" + AUTHENTICATE_REASON_ASSISTANT_VISIBLE + "), "
                            + "AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN(" + AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN + "), "
                            + "AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED(" + AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED + "), "
                            + "AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED(" + AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED + "), "
                            + "AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED(" + AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED + "), "
                            + "AUTHENTICATE_REASON_QS_EXPANDED(" + AUTHENTICATE_REASON_QS_EXPANDED + "), "
                            + "AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER(" + AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER + "), "
                            + "AUTHENTICATE_REASON_UDFPS_POINTER_DOWN(" + AUTHENTICATE_REASON_UDFPS_POINTER_DOWN + ")");
        }

        this.mWakeReason = wakeReason;
        com.android.internal.util.AnnotationValidations.validate(
                PowerManager.WakeReason.class, null, mWakeReason);
        this.mOpPackageName = opPackageName;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mOpPackageName);
        this.mAttributionTag = attributionTag;
        this.mIsMandatoryBiometrics = isMandatoryBiometrics;

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<FaceAuthenticateOptions> CREATOR
            = new Parcelable.Creator<FaceAuthenticateOptions>() {
        @Override
        public FaceAuthenticateOptions[] newArray(int size) {
            return new FaceAuthenticateOptions[size];
        }

        @Override
        public FaceAuthenticateOptions createFromParcel(@NonNull android.os.Parcel in) {
            return new FaceAuthenticateOptions(in);
        }
    };

    /**
     * A builder for {@link FaceAuthenticateOptions}
     */
    @SuppressWarnings("WeakerAccess")
    @DataClass.Generated.Member
    public static class Builder {

        private int mUserId;
        private int mSensorId;
        private @AuthenticateOptions.DisplayState int mDisplayState;
        private @AuthenticateReason int mAuthenticateReason;
        private @PowerManager.WakeReason int mWakeReason;
        private @NonNull String mOpPackageName;
        private @Nullable String mAttributionTag;
        private boolean mIsMandatoryBiometrics;

        private long mBuilderFieldsSet = 0L;

        public Builder() {
        }

        /**
         * The user id for this operation.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setUserId(int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x1;
            mUserId = value;
            return this;
        }

        /**
         * The sensor id for this operation.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setSensorId(int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x2;
            mSensorId = value;
            return this;
        }

        /**
         * The current doze state of the device.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setDisplayState(@AuthenticateOptions.DisplayState int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x4;
            mDisplayState = value;
            return this;
        }

        /**
         * The reason for this operation when requested by the system (sysui),
         * otherwise AUTHENTICATE_REASON_UNKNOWN.
         *
         * See frameworks/base/packages/SystemUI/src/com/android/keyguard/FaceAuthReason.kt
         * for more details about each reason.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setAuthenticateReason(@AuthenticateReason int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x8;
            mAuthenticateReason = value;
            return this;
        }

        /**
         * A reason if this request was triggered due to a power event or WAKE_REASON_UNKNOWN.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setWakeReason(@PowerManager.WakeReason int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x10;
            mWakeReason = value;
            return this;
        }

        /**
         * The package name for that operation that should be used for
         * {@link android.app.AppOpsManager} verification.
         *
         * This option may be overridden by the FingerprintManager using the caller's context.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setOpPackageName(@NonNull String value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x20;
            mOpPackageName = value;
            return this;
        }

        /**
         * The attribution tag, if any.
         *
         * This option may be overridden by the FingerprintManager using the caller's context.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setAttributionTag(@NonNull String value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x40;
            mAttributionTag = value;
            return this;
        }

        /**
         * If the authentication is requested due to mandatory biometrics being active.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setIsMandatoryBiometrics(boolean value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x80;
            mIsMandatoryBiometrics = value;
            return this;
        }

        /** Builds the instance. This builder should not be touched after calling this! */
        public @NonNull FaceAuthenticateOptions build() {
            checkNotUsed();
            mBuilderFieldsSet |= 0x100; // Mark builder used

            if ((mBuilderFieldsSet & 0x1) == 0) {
                mUserId = defaultUserId();
            }
            if ((mBuilderFieldsSet & 0x2) == 0) {
                mSensorId = defaultSensorId();
            }
            if ((mBuilderFieldsSet & 0x4) == 0) {
                mDisplayState = defaultDisplayState();
            }
            if ((mBuilderFieldsSet & 0x8) == 0) {
                mAuthenticateReason = defaultAuthenticateReason();
            }
            if ((mBuilderFieldsSet & 0x10) == 0) {
                mWakeReason = defaultWakeReason();
            }
            if ((mBuilderFieldsSet & 0x20) == 0) {
                mOpPackageName = defaultOpPackageName();
            }
            if ((mBuilderFieldsSet & 0x40) == 0) {
                mAttributionTag = defaultAttributionTag();
            }
            FaceAuthenticateOptions o = new FaceAuthenticateOptions(
                    mUserId,
                    mSensorId,
                    mDisplayState,
                    mAuthenticateReason,
                    mWakeReason,
                    mOpPackageName,
                    mAttributionTag,
                    mIsMandatoryBiometrics);
            return o;
        }

        private void checkNotUsed() {
            if ((mBuilderFieldsSet & 0x100) != 0) {
                throw new IllegalStateException(
                        "This Builder should not be reused. Use a new Builder instance instead");
            }
        }
    }

    @DataClass.Generated(
            time = 1723436679828L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/hardware/face/FaceAuthenticateOptions.java",
            inputSignatures = "private final  int mUserId\nprivate  int mSensorId\nprivate final @android.hardware.biometrics.AuthenticateOptions.DisplayState int mDisplayState\npublic static final  int AUTHENTICATE_REASON_UNKNOWN\npublic static final  int AUTHENTICATE_REASON_STARTED_WAKING_UP\npublic static final  int AUTHENTICATE_REASON_PRIMARY_BOUNCER_SHOWN\npublic static final  int AUTHENTICATE_REASON_ASSISTANT_VISIBLE\npublic static final  int AUTHENTICATE_REASON_ALTERNATE_BIOMETRIC_BOUNCER_SHOWN\npublic static final  int AUTHENTICATE_REASON_NOTIFICATION_PANEL_CLICKED\npublic static final  int AUTHENTICATE_REASON_OCCLUDING_APP_REQUESTED\npublic static final  int AUTHENTICATE_REASON_PICK_UP_GESTURE_TRIGGERED\npublic static final  int AUTHENTICATE_REASON_QS_EXPANDED\npublic static final  int AUTHENTICATE_REASON_SWIPE_UP_ON_BOUNCER\npublic static final  int AUTHENTICATE_REASON_UDFPS_POINTER_DOWN\nprivate final @android.hardware.face.FaceAuthenticateOptions.AuthenticateReason int mAuthenticateReason\nprivate final @android.os.PowerManager.WakeReason int mWakeReason\nprivate @android.annotation.NonNull java.lang.String mOpPackageName\nprivate @android.annotation.Nullable java.lang.String mAttributionTag\nprivate  boolean mIsMandatoryBiometrics\nprivate static  int defaultUserId()\nprivate static  int defaultSensorId()\nprivate static  int defaultDisplayState()\nprivate static  int defaultAuthenticateReason()\nprivate static  int defaultWakeReason()\nprivate static  java.lang.String defaultOpPackageName()\nprivate static  java.lang.String defaultAttributionTag()\nclass FaceAuthenticateOptions extends java.lang.Object implements [android.hardware.biometrics.AuthenticateOptions, android.os.Parcelable]\n@com.android.internal.util.DataClass(genParcelable=true, genAidl=true, genBuilder=true, genSetters=true, genEqualsHashCode=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
