/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.app.appfunctions;

import static android.app.appfunctions.AppFunctionRuntimeMetadata.PROPERTY_APP_FUNCTION_STATIC_METADATA_QUALIFIED_ID;
import static android.app.appfunctions.AppFunctionRuntimeMetadata.PROPERTY_ENABLED;
import static android.app.appfunctions.AppFunctionStaticMetadataHelper.APP_FUNCTION_INDEXER_PACKAGE;
import static android.app.appfunctions.AppFunctionStaticMetadataHelper.STATIC_PROPERTY_ENABLED_BY_DEFAULT;
import static android.app.appfunctions.flags.Flags.FLAG_ENABLE_APP_FUNCTION_MANAGER;

import android.annotation.CallbackExecutor;
import android.annotation.FlaggedApi;
import android.annotation.NonNull;
import android.app.appsearch.AppSearchManager;
import android.app.appsearch.AppSearchResult;
import android.app.appsearch.GlobalSearchSession;
import android.app.appsearch.JoinSpec;
import android.app.appsearch.SearchResult;
import android.app.appsearch.SearchResults;
import android.app.appsearch.SearchSpec;
import android.os.OutcomeReceiver;

import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;

/**
 * Helper class containing utilities for {@link AppFunctionManager}.
 *
 * @hide
 */
@FlaggedApi(FLAG_ENABLE_APP_FUNCTION_MANAGER)
public class AppFunctionManagerHelper {

    /**
     * Returns (through a callback) a boolean indicating whether the app function is enabled.
     *
     * <p>This method can only check app functions that are owned by the caller owned by packages
     * visible to the caller.
     *
     * <p>If operation fails, the callback's {@link OutcomeReceiver#onError} is called with errors:
     *
     * <ul>
     *   <li>{@link IllegalArgumentException}, if the function is not found
     *   <li>{@link SecurityException}, if the caller does not have permission to query the target
     *       package
     * </ul>
     *
     * @param functionIdentifier the identifier of the app function to check (unique within the
     *     target package) and in most cases, these are automatically generated by the AppFunctions
     *     SDK
     * @param targetPackage the package name of the app function's owner
     * @param appSearchExecutor the executor to run the metadata search mechanism through AppSearch
     * @param callbackExecutor the executor to run the callback
     * @param callback the callback to receive the function enabled check result
     * @hide
     */
    public static void isAppFunctionEnabled(
            @NonNull String functionIdentifier,
            @NonNull String targetPackage,
            @NonNull AppSearchManager appSearchManager,
            @NonNull Executor appSearchExecutor,
            @NonNull @CallbackExecutor Executor callbackExecutor,
            @NonNull OutcomeReceiver<Boolean, Exception> callback) {
        Objects.requireNonNull(functionIdentifier);
        Objects.requireNonNull(targetPackage);
        Objects.requireNonNull(appSearchManager);
        Objects.requireNonNull(appSearchExecutor);
        Objects.requireNonNull(callbackExecutor);
        Objects.requireNonNull(callback);

        appSearchManager.createGlobalSearchSession(
                appSearchExecutor,
                (searchSessionResult) -> {
                    if (!searchSessionResult.isSuccess()) {
                        callbackExecutor.execute(
                                () ->
                                        callback.onError(
                                                failedResultToException(searchSessionResult)));
                        return;
                    }
                    try (GlobalSearchSession searchSession = searchSessionResult.getResultValue()) {
                        SearchResults results =
                                searchJoinedStaticWithRuntimeAppFunctions(
                                        searchSession, targetPackage, functionIdentifier);
                        results.getNextPage(
                                appSearchExecutor,
                                listAppSearchResult ->
                                        callbackExecutor.execute(
                                                () -> {
                                                    if (listAppSearchResult.isSuccess()) {
                                                        callback.onResult(
                                                                getEnabledStateFromSearchResults(
                                                                        Objects.requireNonNull(
                                                                                listAppSearchResult
                                                                        .getResultValue())));
                                                    } else {
                                                        callback.onError(
                                                                failedResultToException(
                                                                        listAppSearchResult));
                                                    }
                                                }));
                    } catch (Exception e) {
                        callbackExecutor.execute(() -> callback.onError(e));
                    }
                });
    }

    /**
     * Searches joined app function static and runtime metadata using the function Id and the
     * package.
     *
     * @hide
     */
    private static @NonNull SearchResults searchJoinedStaticWithRuntimeAppFunctions(
            @NonNull GlobalSearchSession session,
            @NonNull String targetPackage,
            @NonNull String functionIdentifier) {
        SearchSpec runtimeSearchSpec =
                getAppFunctionRuntimeMetadataSearchSpecByFunctionId(targetPackage);
        JoinSpec joinSpec =
                new JoinSpec.Builder(PROPERTY_APP_FUNCTION_STATIC_METADATA_QUALIFIED_ID)
                        .setNestedSearch(functionIdentifier, runtimeSearchSpec)
                        .build();
        SearchSpec joinedStaticWithRuntimeSearchSpec =
                new SearchSpec.Builder()
                        .setJoinSpec(joinSpec)
                        .addFilterPackageNames(APP_FUNCTION_INDEXER_PACKAGE)
                        .addFilterSchemas(
                                AppFunctionStaticMetadataHelper.getStaticSchemaNameForPackage(
                                        targetPackage))
                        .setTermMatch(SearchSpec.TERM_MATCH_EXACT_ONLY)
                        .build();
        return session.search(functionIdentifier, joinedStaticWithRuntimeSearchSpec);
    }

    /**
     * Finds whether the function is enabled or not from the search results returned by {@link
     * #searchJoinedStaticWithRuntimeAppFunctions}.
     *
     * @throws IllegalArgumentException if the function is not found in the results
     * @hide
     */
    private static boolean getEnabledStateFromSearchResults(
            @NonNull List<SearchResult> joinedStaticRuntimeResults) {
        if (joinedStaticRuntimeResults.isEmpty()) {
            // Function not found.
            throw new IllegalArgumentException("App function not found.");
        } else {
            List<SearchResult> runtimeMetadataResults =
                    joinedStaticRuntimeResults.getFirst().getJoinedResults();
            if (!runtimeMetadataResults.isEmpty()) {
                Boolean result =
                        (Boolean)
                                runtimeMetadataResults
                                        .getFirst()
                                        .getGenericDocument()
                                        .getProperty(PROPERTY_ENABLED);
                if (result != null) {
                    return result;
                }
            }
            // Runtime metadata not found. Using the default value in the static metadata.
            return joinedStaticRuntimeResults
                    .getFirst()
                    .getGenericDocument()
                    .getPropertyBoolean(STATIC_PROPERTY_ENABLED_BY_DEFAULT);
        }
    }

    /**
     * Returns search spec that queries app function metadata for a specific package name by its
     * function identifier.
     *
     * @hide
     */
    public static @NonNull SearchSpec getAppFunctionRuntimeMetadataSearchSpecByFunctionId(
            @NonNull String targetPackage) {
        return new SearchSpec.Builder()
                .addFilterPackageNames(APP_FUNCTION_INDEXER_PACKAGE)
                .addFilterSchemas(
                        AppFunctionRuntimeMetadata.getRuntimeSchemaNameForPackage(targetPackage))
                .addFilterProperties(
                        AppFunctionRuntimeMetadata.getRuntimeSchemaNameForPackage(targetPackage),
                        List.of(AppFunctionRuntimeMetadata.PROPERTY_FUNCTION_ID))
                .setTermMatch(SearchSpec.TERM_MATCH_EXACT_ONLY)
                .build();
    }

    /**
     * Converts a failed app search result codes into an exception.
     *
     * @hide
     */
    public static @NonNull Exception failedResultToException(
            @NonNull AppSearchResult appSearchResult) {
        return switch (appSearchResult.getResultCode()) {
            case AppSearchResult.RESULT_INVALID_ARGUMENT ->
                    new IllegalArgumentException(appSearchResult.getErrorMessage());
            case AppSearchResult.RESULT_IO_ERROR ->
                    new IOException(appSearchResult.getErrorMessage());
            case AppSearchResult.RESULT_SECURITY_ERROR ->
                    new SecurityException(appSearchResult.getErrorMessage());
            default -> new IllegalStateException(appSearchResult.getErrorMessage());
        };
    }
}
