/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.server.media.quality;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.media.quality.AmbientBacklightSettings;
import android.media.quality.IAmbientBacklightCallback;
import android.media.quality.IMediaQualityManager;
import android.media.quality.IPictureProfileCallback;
import android.media.quality.ISoundProfileCallback;
import android.media.quality.MediaQualityContract;
import android.media.quality.ParamCapability;
import android.media.quality.PictureProfile;
import android.media.quality.SoundProfile;
import android.os.PersistableBundle;
import android.util.Log;

import com.android.server.SystemService;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

/**
 * This service manage picture profile and sound profile for TV setting. Also communicates with the
 * database to save, update the profiles.
 */
public class MediaQualityService extends SystemService {

    private static final boolean DEBUG = false;
    private static final String TAG = "MediaQualityService";
    private final Context mContext;
    private final MediaQualityDbHelper mMediaQualityDbHelper;

    public MediaQualityService(Context context) {
        super(context);
        mContext = context;
        mMediaQualityDbHelper = new MediaQualityDbHelper(mContext);
        mMediaQualityDbHelper.setWriteAheadLoggingEnabled(true);
        mMediaQualityDbHelper.setIdleConnectionTimeout(30);
    }

    @Override
    public void onStart() {
        publishBinderService(Context.MEDIA_QUALITY_SERVICE, new BinderService());
    }

    // TODO: Add additional APIs. b/373951081
    private final class BinderService extends IMediaQualityManager.Stub {

        @Override
        public PictureProfile createPictureProfile(PictureProfile pp) {
            SQLiteDatabase db = mMediaQualityDbHelper.getWritableDatabase();

            ContentValues values = new ContentValues();
            values.put(MediaQualityContract.BaseParameters.PARAMETER_TYPE, pp.getProfileType());
            values.put(MediaQualityContract.BaseParameters.PARAMETER_NAME, pp.getName());
            values.put(MediaQualityContract.BaseParameters.PARAMETER_PACKAGE, pp.getPackageName());
            values.put(MediaQualityContract.BaseParameters.PARAMETER_INPUT_ID, pp.getInputId());
            values.put(mMediaQualityDbHelper.SETTINGS, bundleToJson(pp.getParameters()));

            // id is auto-generated by SQLite upon successful insertion of row
            long id = db.insert(mMediaQualityDbHelper.PICTURE_QUALITY_TABLE_NAME, null, values);
            return new PictureProfile.Builder(pp).setProfileId(Long.toString(id)).build();
        }

        @Override
        public void updatePictureProfile(String id, PictureProfile pp) {
            // TODO: implement
        }
        @Override
        public void removePictureProfile(String id) {
            // TODO: implement
        }

        @Override
        public PictureProfile getPictureProfile(int type, String name) {
            SQLiteDatabase db = mMediaQualityDbHelper.getReadableDatabase();

            String selection = MediaQualityContract.BaseParameters.PARAMETER_TYPE + " = ? AND "
                    + MediaQualityContract.BaseParameters.PARAMETER_NAME + " = ?";
            String[] selectionArguments = {Integer.toString(type), name};

            try (
                    Cursor cursor = db.query(
                            mMediaQualityDbHelper.PICTURE_QUALITY_TABLE_NAME,
                            getAllPictureProfileColumns(),
                            selection,
                            selectionArguments,
                            /*groupBy=*/ null,
                            /*having=*/ null,
                            /*orderBy=*/ null)
            ) {
                int count = cursor.getCount();
                if (count == 0) {
                    return null;
                }
                if (count > 1) {
                    Log.wtf(TAG, String.format(Locale.US, "%d entries found for type=%d and name=%s"
                                    + " in %s. Should only ever be 0 or 1.", count, type, name,
                                    mMediaQualityDbHelper.PICTURE_QUALITY_TABLE_NAME));
                    return null;
                }
                cursor.moveToFirst();
                return getPictureProfileFromCursor(cursor);
            }
        }

        private String bundleToJson(PersistableBundle bundle) {
            JSONObject jsonObject = new JSONObject();
            if (bundle == null) {
                return jsonObject.toString();
            }
            for (String key : bundle.keySet()) {
                try {
                    jsonObject.put(key, bundle.getString(key));
                } catch (JSONException e) {
                    Log.e(TAG, "Unable to serialize ", e);
                }
            }
            return jsonObject.toString();
        }

        private PersistableBundle jsonToBundle(String jsonString) {
            JSONObject jsonObject = null;
            PersistableBundle bundle = new PersistableBundle();

            try {
                jsonObject = new JSONObject(jsonString);

                Iterator<String> keys = jsonObject.keys();
                while (keys.hasNext()) {
                    String key = keys.next();
                    Object value = jsonObject.get(key);

                    if (value instanceof String) {
                        bundle.putString(key, (String) value);
                    } else if (value instanceof Integer) {
                        bundle.putInt(key, (Integer) value);
                    } else if (value instanceof Boolean) {
                        bundle.putBoolean(key, (Boolean) value);
                    } else if (value instanceof Double) {
                        bundle.putDouble(key, (Double) value);
                    } else if (value instanceof Long) {
                        bundle.putLong(key, (Long) value);
                    }
                }
            } catch (JSONException e) {
                throw new RuntimeException(e);
            }

            return bundle;
        }

        private String[] getAllPictureProfileColumns() {
            return new String[]{
                    MediaQualityContract.BaseParameters.PARAMETER_ID,
                    MediaQualityContract.BaseParameters.PARAMETER_TYPE,
                    MediaQualityContract.BaseParameters.PARAMETER_NAME,
                    MediaQualityContract.BaseParameters.PARAMETER_INPUT_ID,
                    MediaQualityContract.BaseParameters.PARAMETER_PACKAGE,
                    mMediaQualityDbHelper.SETTINGS
            };
        }

        private PictureProfile getPictureProfileFromCursor(Cursor cursor) {
            String returnId = cursor.getString(cursor.getColumnIndexOrThrow(
                    MediaQualityContract.BaseParameters.PARAMETER_ID));
            int type = cursor.getInt(cursor.getColumnIndexOrThrow(
                    MediaQualityContract.BaseParameters.PARAMETER_TYPE));
            String name = cursor.getString(cursor.getColumnIndexOrThrow(
                    MediaQualityContract.BaseParameters.PARAMETER_NAME));
            String inputId = cursor.getString(cursor.getColumnIndexOrThrow(
                    MediaQualityContract.BaseParameters.PARAMETER_INPUT_ID));
            String packageName = cursor.getString(cursor.getColumnIndexOrThrow(
                    MediaQualityContract.BaseParameters.PARAMETER_PACKAGE));
            String settings = cursor.getString(
                    cursor.getColumnIndexOrThrow(mMediaQualityDbHelper.SETTINGS));
            return new PictureProfile(returnId, type, name, inputId,
                    packageName, jsonToBundle(settings));
        }

        @Override
        public List<PictureProfile> getPictureProfilesByPackage(String packageName) {
            String selection = MediaQualityContract.BaseParameters.PARAMETER_PACKAGE + " = ?";
            String[] selectionArguments = {packageName};
            return getPictureProfilesBasedOnConditions(getAllPictureProfileColumns(), selection,
                    selectionArguments);
        }

        @Override
        public List<PictureProfile> getAvailablePictureProfiles() {
            return new ArrayList<>();
        }

        @Override
        public List<String> getPictureProfilePackageNames() {
            String [] column = {MediaQualityContract.BaseParameters.PARAMETER_NAME};
            List<PictureProfile> pictureProfiles = getPictureProfilesBasedOnConditions(column,
                    null, null);
            List<String> packageNames = new ArrayList<>();
            for (PictureProfile pictureProfile: pictureProfiles) {
                packageNames.add(pictureProfile.getName());
            }
            return packageNames;
        }

        private List<PictureProfile> getPictureProfilesBasedOnConditions(String[] columns,
                String selection, String[] selectionArguments) {
            SQLiteDatabase db = mMediaQualityDbHelper.getReadableDatabase();

            try (
                    Cursor cursor = db.query(
                            mMediaQualityDbHelper.PICTURE_QUALITY_TABLE_NAME,
                            columns,
                            selection,
                            selectionArguments,
                            /*groupBy=*/ null,
                            /*having=*/ null,
                            /*orderBy=*/ null)
            ) {
                List<PictureProfile> pictureProfiles = new ArrayList<>();
                while (cursor.moveToNext()) {
                    pictureProfiles.add(getPictureProfileFromCursor(cursor));
                }
                return pictureProfiles;
            }
        }

        @Override
        public SoundProfile createSoundProfile(SoundProfile pp) {
            // TODO: implement
            return pp;
        }
        @Override
        public void updateSoundProfile(String id, SoundProfile pp) {
            // TODO: implement
        }
        @Override
        public void removeSoundProfile(String id) {
            // TODO: implement
        }
        @Override
        public SoundProfile getSoundProfile(int type, String id) {
            return null;
        }
        @Override
        public List<SoundProfile> getSoundProfilesByPackage(String packageName) {
            return new ArrayList<>();
        }
        @Override
        public List<SoundProfile> getAvailableSoundProfiles() {
            return new ArrayList<>();
        }
        @Override
        public List<String> getSoundProfilePackageNames() {
            return new ArrayList<>();
        }


        @Override
        public void registerPictureProfileCallback(final IPictureProfileCallback callback) {
        }
        @Override
        public void registerSoundProfileCallback(final ISoundProfileCallback callback) {
        }

        @Override
        public void registerAmbientBacklightCallback(IAmbientBacklightCallback callback) {
        }

        @Override
        public void setAmbientBacklightSettings(AmbientBacklightSettings settings) {
        }

        @Override
        public void setAmbientBacklightEnabled(boolean enabled) {
        }

        @Override
        public List<ParamCapability> getParamCapabilities(List<String> names) {
            return new ArrayList<>();
        }

        @Override
        public List<String> getPictureProfileAllowList() {
            return new ArrayList<>();
        }

        @Override
        public void setPictureProfileAllowList(List<String> packages) {
        }

        @Override
        public List<String> getSoundProfileAllowList() {
            return new ArrayList<>();
        }

        @Override
        public void setSoundProfileAllowList(List<String> packages) {
        }

        @Override
        public boolean isSupported() {
            return false;
        }

        @Override
        public void setAutoPictureQualityEnabled(boolean enabled) {
        }

        @Override
        public boolean isAutoPictureQualityEnabled() {
            return false;
        }

        @Override
        public void setSuperResolutionEnabled(boolean enabled) {
        }

        @Override
        public boolean isSuperResolutionEnabled() {
            return false;
        }

        @Override
        public void setAutoSoundQualityEnabled(boolean enabled) {
        }

        @Override
        public boolean isAutoSoundQualityEnabled() {
            return false;
        }
    }
}
