/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.server.input;

import android.annotation.NonNull;

import com.android.internal.util.AnnotationValidations;
import com.android.internal.util.DataClass;
import com.android.tools.r8.keepanno.annotations.KeepItemKind;
import com.android.tools.r8.keepanno.annotations.UsedByNative;

/**
 * This class represents a touchpad hardware state at a single moment in time.
 * It is only used by the touchpad visualization which is implemented in TouchpadDebugActivity.
 */
@DataClass(genToString = true)
@UsedByNative(
        description = "Called from JNI in jni/com_android_server_input_InputManagerService.cpp",
        kind = KeepItemKind.CLASS_AND_MEMBERS)
public final class TouchpadHardwareState{
    /**
     * The time at which the event was received by the system.
     * The time is in milliseconds and start counting when the program starts.
      */
    private final float mTimestamp;

    /**
     * Number of buttons pressed. Note that in our case while using
     * a touchpad only one button is available and can be pressed.
     */
    private final int mButtonsDown;

    /**
     * The number of FingerState structs pointed to by the fingers field.
     */
    private final int mFingerCount;

    /**
     * The number of fingers touching the pad, which may be more than fingerCount.
     */
    private final int mTouchCount;

    /**
     * Array of fingerStates that indicates the properties of each finger touching the touchpad.
     */
    private final @NonNull TouchpadFingerState[] mFingerStates;

    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/services/core/java/com/android/server/input/
    // TouchpadHardwareState.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /**
     * Creates a new TouchpadHardwareState.
     *
     * @param timestamp
     *   The time at which the event was received by the system.
     *   The time is in milliseconds and start counting when the program starts.
     * @param buttonsDown
     *   Number of buttons pressed. Note that in our case while using
     *   a touchpad only one button is available and can be pressed.
     * @param fingerCount
     *   The number of FingerState structs pointed to by the fingers field.
     * @param touchCount
     *   The number of fingers touching the pad, which may be more than fingerCount.
     * @param fingerStates
     *   Array of fingerStates that indicates the properties of each finger touching the touchpad.
     */
    @DataClass.Generated.Member
    public TouchpadHardwareState(
            float timestamp,
            int buttonsDown,
            int fingerCount,
            int touchCount,
            @NonNull TouchpadFingerState[] fingerStates) {
        this.mTimestamp = timestamp;
        this.mButtonsDown = buttonsDown;
        this.mFingerCount = fingerCount;
        this.mTouchCount = touchCount;
        this.mFingerStates = fingerStates;
        AnnotationValidations.validate(
                NonNull.class, null, mFingerStates);

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * The time at which the event was received by the system.
     * The time is in milliseconds and start counting when the program starts.
     */
    @DataClass.Generated.Member
    public float getTimestamp() {
        return mTimestamp;
    }

    /**
     * Number of buttons pressed. Note that in our case while using
     * a touchpad only one button is available and can be pressed.
     */
    @DataClass.Generated.Member
    public int getButtonsDown() {
        return mButtonsDown;
    }

    /**
     * The number of FingerState structs pointed to by the fingers field.
     */
    @DataClass.Generated.Member
    public int getFingerCount() {
        return mFingerCount;
    }

    /**
     * The number of fingers touching the pad, which may be more than fingerCount.
     */
    @DataClass.Generated.Member
    public int getTouchCount() {
        return mTouchCount;
    }

    /**
     * Array of fingerStates that indicates the properties of each finger touching the touchpad.
     */
    @DataClass.Generated.Member
    public @NonNull TouchpadFingerState[] getFingerStates() {
        return mFingerStates;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "TouchpadHardwareState { " +
                "timestamp = " + mTimestamp + ", " +
                "buttonsDown = " + mButtonsDown + ", " +
                "fingerCount = " + mFingerCount + ", " +
                "touchCount = " + mTouchCount + ", " +
                "fingerStates = " + java.util.Arrays.toString(mFingerStates) +
        " }";
    }

    @DataClass.Generated(
            time = 1724079048292L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/services/core/java/com/android/server/input/"
                    + "TouchpadHardwareState.java",
            inputSignatures = "private final  float mTimestamp\nprivate final  int mButtonsDown\n"
                    + "private final  int mFingerCount\nprivate final  int mTouchCount\nprivate "
                    + "final @android.annotation.NonNull com.android.server.input."
                    + "TouchpadFingerState[] mFingerStates\nclass TouchpadHardwareState extends "
                    + "java.lang.Object implements []\n@com.android.internal.util.DataClass"
                    + "(genToString=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
