/*
 * Copyright (C) 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.systemui.bluetooth.qsdialog

import android.bluetooth.BluetoothAdapter
import android.bluetooth.BluetoothDevice
import android.content.Context
import android.media.AudioManager
import android.testing.TestableLooper
import androidx.test.ext.junit.runners.AndroidJUnit4
import androidx.test.filters.SmallTest
import com.android.settingslib.bluetooth.CachedBluetoothDevice
import com.android.settingslib.bluetooth.LocalBluetoothManager
import com.android.systemui.SysuiTestCase
import com.android.systemui.coroutines.collectLastValue
import com.android.systemui.testKosmos
import com.android.systemui.util.time.FakeSystemClock
import com.google.common.truth.Truth.assertThat
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.test.TestScope
import kotlinx.coroutines.test.UnconfinedTestDispatcher
import kotlinx.coroutines.test.runTest
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.mockito.Mock
import org.mockito.Mockito.`when`
import org.mockito.junit.MockitoJUnit
import org.mockito.junit.MockitoRule

@SmallTest
@RunWith(AndroidJUnit4::class)
@TestableLooper.RunWithLooper(setAsMainLooper = true)
class DeviceItemInteractorTest : SysuiTestCase() {

    @get:Rule val mockitoRule: MockitoRule = MockitoJUnit.rule()
    private val kosmos = testKosmos()

    @Mock private lateinit var bluetoothTileDialogRepository: BluetoothTileDialogRepository

    @Mock private lateinit var cachedDevice1: CachedBluetoothDevice

    @Mock private lateinit var cachedDevice2: CachedBluetoothDevice

    @Mock private lateinit var device1: BluetoothDevice

    @Mock private lateinit var device2: BluetoothDevice

    @Mock private lateinit var deviceItem1: DeviceItem

    @Mock private lateinit var deviceItem2: DeviceItem

    @Mock private lateinit var audioManager: AudioManager

    @Mock private lateinit var adapter: BluetoothAdapter

    @Mock private lateinit var localBluetoothManager: LocalBluetoothManager

    @Mock private lateinit var logger: BluetoothTileDialogLogger

    private val fakeSystemClock = FakeSystemClock()

    private lateinit var interactor: DeviceItemInteractor

    private lateinit var dispatcher: CoroutineDispatcher

    private lateinit var testScope: TestScope

    @Before
    fun setUp() {
        dispatcher = UnconfinedTestDispatcher()
        testScope = TestScope(dispatcher)
        `when`(deviceItem1.cachedBluetoothDevice).thenReturn(cachedDevice1)
        `when`(deviceItem2.cachedBluetoothDevice).thenReturn(cachedDevice2)
        `when`(cachedDevice1.address).thenReturn("ADDRESS")
        `when`(cachedDevice1.device).thenReturn(device1)
        `when`(cachedDevice2.device).thenReturn(device2)
        `when`(bluetoothTileDialogRepository.cachedDevices)
            .thenReturn(listOf(cachedDevice1, cachedDevice2))
    }

    @Test
    fun testUpdateDeviceItems_noCachedDevice_returnEmpty() {
        testScope.runTest {
            `when`(bluetoothTileDialogRepository.cachedDevices).thenReturn(emptyList())
            interactor =
                DeviceItemInteractor(
                    bluetoothTileDialogRepository,
                    kosmos.audioSharingInteractor,
                    audioManager,
                    adapter,
                    localBluetoothManager,
                    fakeSystemClock,
                    logger,
                    listOf(createFactory({ true }, deviceItem1)),
                    emptyList(),
                    testScope.backgroundScope,
                    dispatcher,
                )

            val latest by collectLastValue(interactor.deviceItemUpdate)
            val latestShowSeeAll by collectLastValue(interactor.showSeeAllUpdate)
            interactor.updateDeviceItems(mContext, DeviceFetchTrigger.FIRST_LOAD)

            assertThat(latest).isEqualTo(emptyList<DeviceItem>())
            assertThat(latestShowSeeAll).isFalse()
        }
    }

    @Test
    fun testUpdateDeviceItems_hasCachedDevice_filterNotMatch_returnEmpty() {
        testScope.runTest {
            `when`(bluetoothTileDialogRepository.cachedDevices).thenReturn(listOf(cachedDevice1))
            interactor =
                DeviceItemInteractor(
                    bluetoothTileDialogRepository,
                    kosmos.audioSharingInteractor,
                    audioManager,
                    adapter,
                    localBluetoothManager,
                    fakeSystemClock,
                    logger,
                    listOf(createFactory({ false }, deviceItem1)),
                    emptyList(),
                    testScope.backgroundScope,
                    dispatcher,
                )

            val latest by collectLastValue(interactor.deviceItemUpdate)
            val latestShowSeeAll by collectLastValue(interactor.showSeeAllUpdate)
            interactor.updateDeviceItems(mContext, DeviceFetchTrigger.FIRST_LOAD)

            assertThat(latest).isEqualTo(emptyList<DeviceItem>())
            assertThat(latestShowSeeAll).isFalse()
        }
    }

    @Test
    fun testUpdateDeviceItems_hasCachedDevice_filterMatch_returnDeviceItem() {
        testScope.runTest {
            `when`(bluetoothTileDialogRepository.cachedDevices).thenReturn(listOf(cachedDevice1))
            interactor =
                DeviceItemInteractor(
                    bluetoothTileDialogRepository,
                    kosmos.audioSharingInteractor,
                    audioManager,
                    adapter,
                    localBluetoothManager,
                    fakeSystemClock,
                    logger,
                    listOf(createFactory({ true }, deviceItem1)),
                    emptyList(),
                    testScope.backgroundScope,
                    dispatcher,
                )

            val latest by collectLastValue(interactor.deviceItemUpdate)
            val latestShowSeeAll by collectLastValue(interactor.showSeeAllUpdate)
            interactor.updateDeviceItems(mContext, DeviceFetchTrigger.FIRST_LOAD)

            assertThat(latest).isEqualTo(listOf(deviceItem1))
            assertThat(latestShowSeeAll).isFalse()
        }
    }

    @Test
    fun testUpdateDeviceItems_hasCachedDevice_filterMatch_returnMultipleDeviceItem() {
        testScope.runTest {
            `when`(adapter.mostRecentlyConnectedDevices).thenReturn(null)
            interactor =
                DeviceItemInteractor(
                    bluetoothTileDialogRepository,
                    kosmos.audioSharingInteractor,
                    audioManager,
                    adapter,
                    localBluetoothManager,
                    fakeSystemClock,
                    logger,
                    listOf(
                        createFactory({ false }, deviceItem1),
                        createFactory({ true }, deviceItem2),
                    ),
                    emptyList(),
                    testScope.backgroundScope,
                    dispatcher,
                )

            val latest by collectLastValue(interactor.deviceItemUpdate)
            val latestShowSeeAll by collectLastValue(interactor.showSeeAllUpdate)
            interactor.updateDeviceItems(mContext, DeviceFetchTrigger.FIRST_LOAD)

            assertThat(latest).isEqualTo(listOf(deviceItem2, deviceItem2))
            assertThat(latestShowSeeAll).isFalse()
        }
    }

    @Test
    fun testUpdateDeviceItems_sortByDisplayPriority() {
        testScope.runTest {
            `when`(adapter.mostRecentlyConnectedDevices).thenReturn(null)
            interactor =
                DeviceItemInteractor(
                    bluetoothTileDialogRepository,
                    kosmos.audioSharingInteractor,
                    audioManager,
                    adapter,
                    localBluetoothManager,
                    fakeSystemClock,
                    logger,
                    listOf(
                        createFactory(
                            { cachedDevice -> cachedDevice.device == device1 },
                            deviceItem1,
                        ),
                        createFactory(
                            { cachedDevice -> cachedDevice.device == device2 },
                            deviceItem2,
                        ),
                    ),
                    listOf(
                        DeviceItemType.SAVED_BLUETOOTH_DEVICE,
                        DeviceItemType.CONNECTED_BLUETOOTH_DEVICE,
                    ),
                    testScope.backgroundScope,
                    dispatcher,
                )
            `when`(deviceItem1.type).thenReturn(DeviceItemType.CONNECTED_BLUETOOTH_DEVICE)
            `when`(deviceItem2.type).thenReturn(DeviceItemType.SAVED_BLUETOOTH_DEVICE)

            val latest by collectLastValue(interactor.deviceItemUpdate)
            val latestShowSeeAll by collectLastValue(interactor.showSeeAllUpdate)
            interactor.updateDeviceItems(mContext, DeviceFetchTrigger.FIRST_LOAD)

            assertThat(latest).isEqualTo(listOf(deviceItem2, deviceItem1))
            assertThat(latestShowSeeAll).isFalse()
        }
    }

    @Test
    fun testUpdateDeviceItems_sameType_sortByRecentlyConnected() {
        testScope.runTest {
            `when`(adapter.mostRecentlyConnectedDevices).thenReturn(listOf(device2, device1))
            interactor =
                DeviceItemInteractor(
                    bluetoothTileDialogRepository,
                    kosmos.audioSharingInteractor,
                    audioManager,
                    adapter,
                    localBluetoothManager,
                    fakeSystemClock,
                    logger,
                    listOf(
                        createFactory(
                            { cachedDevice -> cachedDevice.device == device1 },
                            deviceItem1,
                        ),
                        createFactory(
                            { cachedDevice -> cachedDevice.device == device2 },
                            deviceItem2,
                        ),
                    ),
                    listOf(DeviceItemType.CONNECTED_BLUETOOTH_DEVICE),
                    testScope.backgroundScope,
                    dispatcher,
                )
            `when`(deviceItem1.type).thenReturn(DeviceItemType.CONNECTED_BLUETOOTH_DEVICE)
            `when`(deviceItem2.type).thenReturn(DeviceItemType.CONNECTED_BLUETOOTH_DEVICE)

            val latest by collectLastValue(interactor.deviceItemUpdate)
            val latestShowSeeAll by collectLastValue(interactor.showSeeAllUpdate)
            interactor.updateDeviceItems(mContext, DeviceFetchTrigger.FIRST_LOAD)

            assertThat(latest).isEqualTo(listOf(deviceItem2, deviceItem1))
            assertThat(latestShowSeeAll).isFalse()
        }
    }

    @Test
    fun testUpdateDeviceItems_showMaxDeviceItems_showSeeAll() {
        testScope.runTest {
            `when`(bluetoothTileDialogRepository.cachedDevices)
                .thenReturn(listOf(cachedDevice2, cachedDevice2, cachedDevice2, cachedDevice2))
            `when`(adapter.mostRecentlyConnectedDevices).thenReturn(null)
            interactor =
                DeviceItemInteractor(
                    bluetoothTileDialogRepository,
                    kosmos.audioSharingInteractor,
                    audioManager,
                    adapter,
                    localBluetoothManager,
                    fakeSystemClock,
                    logger,
                    listOf(createFactory({ true }, deviceItem2)),
                    emptyList(),
                    testScope.backgroundScope,
                    dispatcher,
                )
            val latest by collectLastValue(interactor.deviceItemUpdate)
            val latestShowSeeAll by collectLastValue(interactor.showSeeAllUpdate)
            interactor.updateDeviceItems(mContext, DeviceFetchTrigger.FIRST_LOAD)

            assertThat(latest).isEqualTo(listOf(deviceItem2, deviceItem2, deviceItem2))
            assertThat(latestShowSeeAll).isTrue()
        }
    }

    private fun createFactory(
        isFilterMatchFunc: (CachedBluetoothDevice) -> Boolean,
        deviceItem: DeviceItem,
    ): DeviceItemFactory {
        return object : DeviceItemFactory() {
            override fun isFilterMatched(
                context: Context,
                cachedDevice: CachedBluetoothDevice,
                audioManager: AudioManager,
                audioSharingAvailable: Boolean,
            ) = isFilterMatchFunc(cachedDevice)

            override fun create(context: Context, cachedDevice: CachedBluetoothDevice) = deviceItem
        }
    }
}
