/*
 *  Copyright (C) 2023 The Android Open Source Project
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package com.android.systemui.qs.tiles

import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import com.android.systemui.dagger.SysUISingleton
import java.util.function.Consumer
import javax.inject.Inject
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.collect
import com.android.app.tracing.coroutines.launchTraced as launch

/**
 * Utility for binding tiles to kotlin flows. Similar to [JavaAdapter] and usable for QS tiles. We
 * use [Lifecycle.State.RESUMED] here to match the implementation of [CallbackController.observe]
 */
@SysUISingleton
class TileJavaAdapter @Inject constructor() {
    fun <T> bind(
        lifecycleOwner: LifecycleOwner,
        flow: Flow<T>,
        consumer: Consumer<T>,
    ) {
        lifecycleOwner.lifecycleScope.launch {
            lifecycleOwner.repeatOnLifecycle(Lifecycle.State.RESUMED) {
                flow.collect { consumer.accept(it) }
            }
        }
    }
}
