/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package com.android.systemui.plugins.processor

import com.android.systemui.plugins.annotations.GeneratedImport
import com.android.systemui.plugins.annotations.ProtectedInterface
import com.android.systemui.plugins.annotations.ProtectedReturn
import com.android.systemui.plugins.annotations.SimpleProperty
import com.google.auto.service.AutoService
import javax.annotation.processing.AbstractProcessor
import javax.annotation.processing.ProcessingEnvironment
import javax.annotation.processing.RoundEnvironment
import javax.lang.model.element.Element
import javax.lang.model.element.ElementKind
import javax.lang.model.element.ExecutableElement
import javax.lang.model.element.PackageElement
import javax.lang.model.element.TypeElement
import javax.lang.model.type.TypeKind
import javax.lang.model.type.TypeMirror
import javax.tools.Diagnostic.Kind
import kotlin.collections.ArrayDeque

/**
 * [ProtectedPluginProcessor] generates a proxy implementation for interfaces annotated with
 * [ProtectedInterface] which catches [LinkageError]s generated by the proxied target. This protects
 * the plugin host from crashing due to out-of-date plugin code, where some call has changed so that
 * the [ClassLoader] can no longer resolve it correctly.
 *
 * [PluginInstance] observes these failures via [ProtectedMethodListener] and unloads the plugin in
 * question to prevent further issues. This persists through further load/unload requests.
 *
 * To centralize access to the proxy types, an additional type [PluginProtector] is also generated.
 * This class provides static methods which wrap an instance of the target interface in the proxy
 * type if it is not already an instance of the proxy.
 */
@AutoService(ProtectedPluginProcessor::class)
class ProtectedPluginProcessor : AbstractProcessor() {
    private lateinit var procEnv: ProcessingEnvironment

    override fun init(procEnv: ProcessingEnvironment) {
        this.procEnv = procEnv
    }

    override fun getSupportedAnnotationTypes(): Set<String> =
        setOf("com.android.systemui.plugins.annotations.ProtectedInterface")

    private data class TargetData(
        val attribute: TypeElement,
        val sourceType: Element,
        val sourcePkg: String,
        val sourceName: String,
        val outputName: String,
    )

    override fun process(annotations: Set<TypeElement>, roundEnv: RoundEnvironment): Boolean {
        val targets = mutableMapOf<String, TargetData>() // keyed by fully-qualified source name
        val additionalImports = mutableSetOf<String>()
        for (attr in annotations) {
            for (target in roundEnv.getElementsAnnotatedWith(attr)) {
                val sourceName = "${target.simpleName}"
                val outputName = "${sourceName}Protector"
                val pkg = (target.getEnclosingElement() as PackageElement).qualifiedName.toString()
                targets.put("$target", TargetData(attr, target, pkg, sourceName, outputName))

                // This creates excessive imports, but it should be fine
                additionalImports.add("$pkg.$sourceName")
                additionalImports.add("$pkg.$outputName")
            }
        }

        if (targets.size <= 0) return false
        for ((_, sourceType, sourcePkg, sourceName, outputName) in targets.values) {
            // Find all methods in this type and all super types to that need to be implemented
            val types = ArrayDeque<TypeMirror>().apply { addLast(sourceType.asType()) }
            val impAttrs = mutableListOf<GeneratedImport>()
            val methods = mutableListOf<ExecutableElement>()
            while (types.size > 0) {
                val typeMirror = types.removeLast()
                if (typeMirror.toString() == "java.lang.Object") continue
                val type = procEnv.typeUtils.asElement(typeMirror)
                for (member in type.enclosedElements) {
                    if (member.kind != ElementKind.METHOD) continue
                    methods.add(member as ExecutableElement)
                }

                impAttrs.addAll(type.getAnnotationsByType(GeneratedImport::class.java))
                types.addAll(procEnv.typeUtils.directSupertypes(typeMirror))
            }

            val file = procEnv.filer.createSourceFile("$outputName")
            TabbedWriter.writeTo(file.openWriter()) {
                line("package $sourcePkg;")
                line()

                // Imports used by the proxy implementation
                line("import android.util.Log;")
                line("import java.lang.LinkageError;")
                line("import com.android.systemui.plugins.PluginWrapper;")
                line("import com.android.systemui.plugins.ProtectedPluginListener;")
                line()

                // Imports of other generated types
                if (additionalImports.size > 0) {
                    for (impTarget in additionalImports) {
                        line("import $impTarget;")
                    }
                    line()
                }

                // Imports declared via @GeneratedImport
                if (impAttrs.size > 0) {
                    for (impAttr in impAttrs) {
                        line("import ${impAttr.extraImport};")
                    }
                    line()
                }

                val interfaces = "$sourceName, PluginWrapper<$sourceName>"
                braceBlock("public class $outputName implements $interfaces") {
                    line("private static final String CLASS = \"$sourceName\";")

                    // Static factory method to prevent wrapping the same object twice
                    parenBlock("public static $outputName protect") {
                        line("$sourceName instance,")
                        line("ProtectedPluginListener listener")
                    }
                    braceBlock {
                        line("if (instance instanceof $outputName)")
                        line("    return ($outputName)instance;")
                        line("return new $outputName(instance, listener);")
                    }
                    line()

                    // Member Fields
                    line("private $sourceName mInstance;")
                    line("private ProtectedPluginListener mListener;")
                    line("private boolean mHasError = false;")
                    line()

                    // Constructor
                    parenBlock("private $outputName") {
                        line("$sourceName instance,")
                        line("ProtectedPluginListener listener")
                    }
                    braceBlock {
                        line("mInstance = instance;")
                        line("mListener = listener;")
                    }
                    line()

                    // Wrapped instance getter for version checker
                    braceBlock("public $sourceName getPlugin()") { line("return mInstance;") }

                    // Method implementations
                    for (method in methods) {
                        val methodName = method.simpleName
                        val returnTypeName = method.returnType.toString()
                        val callArgs = StringBuilder()
                        var isFirst = true

                        line("@Override")
                        parenBlock("public $returnTypeName $methodName") {
                            // While copying the method signature for the proxy type, we also
                            // accumulate arguments for the nested callsite.
                            for (param in method.parameters) {
                                if (!isFirst) completeLine(",")
                                startLine("${param.asType()} ${param.simpleName}")
                                isFirst = false

                                if (callArgs.length > 0) callArgs.append(", ")
                                callArgs.append(param.simpleName)
                            }
                        }

                        val isVoid = method.returnType.kind == TypeKind.VOID
                        val nestedCall = "mInstance.$methodName($callArgs)"
                        val callStatement =
                            when {
                                isVoid -> "$nestedCall;"
                                targets.containsKey(returnTypeName) -> {
                                    val targetType = targets.get(returnTypeName)!!.outputName
                                    "return $targetType.protect($nestedCall, mListener);"
                                }
                                else -> "return $nestedCall;"
                            }

                        // Simple property methods forgo protection
                        val simpleAttr = method.getAnnotation(SimpleProperty::class.java)
                        if (simpleAttr != null) {
                            braceBlock {
                                line("final String METHOD = \"$methodName\";")
                                line(callStatement)
                            }
                            line()
                            continue
                        }

                        // Standard implementation wraps nested call in try-catch
                        braceBlock {
                            val retAttr = method.getAnnotation(ProtectedReturn::class.java)
                            val errorStatement =
                                when {
                                    retAttr != null -> retAttr.statement
                                    isVoid -> "return;"
                                    else -> {
                                        // Non-void methods must be annotated.
                                        procEnv.messager.printMessage(
                                            Kind.ERROR,
                                            "$outputName.$methodName must be annotated with " +
                                                "@ProtectedReturn or @SimpleProperty",
                                        )
                                        "throw ex;"
                                    }
                                }

                            line("final String METHOD = \"$methodName\";")

                            // Return immediately if any previous call has failed.
                            braceBlock("if (mHasError)") { line(errorStatement) }

                            // Protect callsite in try/catch block
                            braceBlock("try") { line(callStatement) }

                            // Notify listener when a LinkageError is caught
                            braceBlock("catch (LinkageError ex)") {
                                line("Log.wtf(CLASS, \"Failed to execute: \" + METHOD, ex);")
                                line("mHasError = mListener.onFail(CLASS, METHOD, ex);")
                                line(errorStatement)
                            }
                        }
                        line()
                    }
                }
            }
        }

        // Write a centralized static factory type to its own file. This is for convience so that
        // PluginInstance need not resolve each generated type at runtime as plugins are loaded.
        val factoryFile = procEnv.filer.createSourceFile("PluginProtector")
        TabbedWriter.writeTo(factoryFile.openWriter()) {
            line("package com.android.systemui.plugins;")
            line()

            line("import java.util.Map;")
            line("import java.util.ArrayList;")
            line("import java.util.HashSet;")
            line("import android.util.Log;")
            line("import static java.util.Map.entry;")
            line()

            for (impTarget in additionalImports) {
                line("import $impTarget;")
            }
            line()

            braceBlock("public final class PluginProtector") {
                line("private PluginProtector() { }")
                line()

                line("private static final String TAG = \"PluginProtector\";")
                line()

                // Untyped factory SAM, private to this type.
                braceBlock("private interface Factory") {
                    line("Object create(Object plugin, ProtectedPluginListener listener);")
                }
                line()

                // Store a reference to each `protect` method in a map by interface type.
                parenBlock("private static final Map<Class, Factory> sFactories = Map.ofEntries") {
                    var isFirst = true
                    for (target in targets.values) {
                        if (!isFirst) completeLine(",")
                        target.apply {
                            startLine("entry($sourceName.class, ")
                            appendLine("(p, h) -> $outputName.protect(($sourceName)p, h))")
                        }
                        isFirst = false
                    }
                }
                completeLine(";")
                line()

                // Lookup the relevant factory based on the instance type, if not found return null.
                parenBlock("public static <T> T tryProtect") {
                    line("T target,")
                    line("ProtectedPluginListener listener")
                }
                braceBlock {
                    // Accumulate interfaces from type and all base types
                    line("HashSet<Class> interfaces = new HashSet<Class>();")
                    line("Class current = target.getClass();")
                    braceBlock("while (current != null)") {
                        braceBlock("for (Class cls : current.getInterfaces())") {
                            line("interfaces.add(cls);")
                        }
                        line("current = current.getSuperclass();")
                    }
                    line()

                    // Check if any of the interfaces are marked protectable
                    line("int candidateCount = 0;")
                    line("Factory candidateFactory = null;")
                    braceBlock("for (Class cls : interfaces)") {
                        line("Factory factory = sFactories.get(cls);")
                        braceBlock("if (factory != null)") {
                            line("candidateFactory = factory;")
                            line("candidateCount++;")
                        }
                    }
                    line()

                    // No match, return null
                    braceBlock("if (candidateFactory == null)") {
                        line("Log.i(TAG, \"Wasn't able to wrap \" + target);")
                        line("return null;")
                    }

                    // Multiple matches, not supported
                    braceBlock("if (candidateCount >= 2)") {
                        var error = "Plugin implements more than one protected interface"
                        line("throw new UnsupportedOperationException(\"$error\");")
                    }

                    // Call the factory and wrap the target object
                    line("return (T)candidateFactory.create(target, listener);")
                }
                line()

                // Wraps the target with the appropriate generated proxy if it exists.
                parenBlock("public static <T> T protectIfAble") {
                    line("T target,")
                    line("ProtectedPluginListener listener")
                }
                braceBlock {
                    line("T result = tryProtect(target, listener);")
                    line("return result != null ? result : target;")
                }
                line()
            }
        }

        return true
    }
}
