/*
 * Copyright (C) 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.systemui.statusbar.pipeline.wifi.data.repository.prod

import android.content.Context
import android.content.pm.UserInfo
import android.net.wifi.ScanResult
import android.net.wifi.WifiManager
import android.net.wifi.WifiManager.UNKNOWN_SSID
import android.net.wifi.sharedconnectivity.app.NetworkProviderInfo
import android.os.UserHandle
import android.platform.test.annotations.DisableFlags
import android.platform.test.annotations.EnableFlags
import android.telephony.SubscriptionManager.INVALID_SUBSCRIPTION_ID
import android.testing.TestableLooper
import androidx.test.ext.junit.runners.AndroidJUnit4
import androidx.test.filters.SmallTest
import com.android.systemui.Flags.FLAG_MULTIUSER_WIFI_PICKER_TRACKER_SUPPORT
import com.android.systemui.SysuiTestCase
import com.android.systemui.coroutines.collectLastValue
import com.android.systemui.kosmos.testDispatcher
import com.android.systemui.kosmos.testScope
import com.android.systemui.kosmos.useUnconfinedTestDispatcher
import com.android.systemui.log.LogBuffer
import com.android.systemui.log.table.logcatTableLogBuffer
import com.android.systemui.statusbar.connectivity.WifiPickerTrackerFactory
import com.android.systemui.statusbar.pipeline.shared.data.model.DataActivityModel
import com.android.systemui.statusbar.pipeline.wifi.data.repository.prod.WifiRepositoryImpl.Companion.WIFI_NETWORK_DEFAULT
import com.android.systemui.statusbar.pipeline.wifi.shared.model.WifiNetworkModel
import com.android.systemui.statusbar.pipeline.wifi.shared.model.WifiScanEntry
import com.android.systemui.testKosmos
import com.android.systemui.user.data.repository.fakeUserRepository
import com.android.systemui.util.concurrency.FakeExecutor
import com.android.systemui.util.time.fakeSystemClock
import com.android.wifitrackerlib.HotspotNetworkEntry
import com.android.wifitrackerlib.HotspotNetworkEntry.DeviceType
import com.android.wifitrackerlib.MergedCarrierEntry
import com.android.wifitrackerlib.WifiEntry
import com.android.wifitrackerlib.WifiEntry.WIFI_LEVEL_MAX
import com.android.wifitrackerlib.WifiEntry.WIFI_LEVEL_MIN
import com.android.wifitrackerlib.WifiEntry.WIFI_LEVEL_UNREACHABLE
import com.android.wifitrackerlib.WifiPickerTracker
import com.google.common.truth.Truth.assertThat
import kotlinx.coroutines.test.runTest
import org.junit.Before
import org.junit.Test
import org.junit.runner.RunWith
import org.mockito.kotlin.any
import org.mockito.kotlin.argumentCaptor
import org.mockito.kotlin.mock
import org.mockito.kotlin.never
import org.mockito.kotlin.reset
import org.mockito.kotlin.verify
import org.mockito.kotlin.whenever

@SmallTest
@RunWith(AndroidJUnit4::class)
@TestableLooper.RunWithLooper(setAsMainLooper = true)
class WifiRepositoryImplTest : SysuiTestCase() {
    private val kosmos = testKosmos().useUnconfinedTestDispatcher()
    private val userRepository = kosmos.fakeUserRepository

    // Using lazy means that the class will only be constructed once it's fetched. Because the
    // repository internally sets some values on construction, we need to set up some test
    // parameters (like feature flags) *before* construction. Using lazy allows us to do that setup
    // inside each test case without needing to manually recreate the repository.
    private val underTest: WifiRepositoryImpl by lazy {
        WifiRepositoryImpl(
            mContext,
            userRepository,
            testScope.backgroundScope,
            executor,
            dispatcher,
            wifiPickerTrackerFactory,
            wifiManager,
            logger,
            tableLogger,
        )
    }

    private val executor = FakeExecutor(kosmos.fakeSystemClock)
    private val logger = LogBuffer("name", maxSize = 100, logcatEchoTracker = mock())
    private val tableLogger = logcatTableLogBuffer(kosmos, "WifiRepositoryImplTest")
    private val wifiManager =
        mock<WifiManager>().apply { whenever(this.maxSignalLevel).thenReturn(10) }
    private val wifiPickerTrackerFactory = mock<WifiPickerTrackerFactory>()
    private val wifiPickerTracker = mock<WifiPickerTracker>()

    private val callbackCaptor = argumentCaptor<WifiPickerTracker.WifiPickerTrackerCallback>()

    private val dispatcher = kosmos.testDispatcher
    private val testScope = kosmos.testScope

    @Before
    fun setUp() {
        userRepository.setUserInfos(listOf(PRIMARY_USER, ANOTHER_USER))
        whenever(wifiPickerTrackerFactory.create(any(), any(), callbackCaptor.capture(), any()))
            .thenReturn(wifiPickerTracker)
    }

    @Test
    fun wifiPickerTrackerCreation_scansDisabled() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            verify(wifiPickerTracker).disableScanning()
        }

    @Test
    fun isWifiEnabled_enabled_true() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiEnabled)

            whenever(wifiPickerTracker.wifiState).thenReturn(WifiManager.WIFI_STATE_ENABLED)
            getCallback().onWifiStateChanged()

            assertThat(latest).isTrue()
        }

    @Test
    fun isWifiEnabled_enabling_false() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiEnabled)

            whenever(wifiPickerTracker.wifiState).thenReturn(WifiManager.WIFI_STATE_ENABLING)
            getCallback().onWifiStateChanged()

            assertThat(latest).isFalse()
        }

    @Test
    fun isWifiEnabled_disabling_true() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiEnabled)

            whenever(wifiPickerTracker.wifiState).thenReturn(WifiManager.WIFI_STATE_DISABLING)
            getCallback().onWifiStateChanged()

            assertThat(latest).isFalse()
        }

    @Test
    fun isWifiEnabled_disabled_false() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiEnabled)

            whenever(wifiPickerTracker.wifiState).thenReturn(WifiManager.WIFI_STATE_DISABLED)
            getCallback().onWifiStateChanged()

            assertThat(latest).isFalse()
        }

    @Test
    fun isWifiEnabled_respondsToUpdates() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiEnabled)
            executor.runAllReady()

            whenever(wifiPickerTracker.wifiState).thenReturn(WifiManager.WIFI_STATE_ENABLED)
            getCallback().onWifiStateChanged()

            assertThat(latest).isTrue()

            whenever(wifiPickerTracker.wifiState).thenReturn(WifiManager.WIFI_STATE_DISABLED)
            getCallback().onWifiStateChanged()

            assertThat(latest).isFalse()
        }

    @Test
    fun isWifiDefault_initiallyGetsDefault() =
        testScope.runTest { assertThat(underTest.isWifiDefault.value).isFalse() }

    @Test
    fun isWifiDefault_wifiNetwork_isTrue() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiDefault)

            val wifiEntry =
                mock<WifiEntry>().apply { whenever(this.isDefaultNetwork).thenReturn(true) }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isTrue()
        }

    @Test
    fun isWifiDefault_carrierMerged_isTrue() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiDefault)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isDefaultNetwork).thenReturn(true)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isTrue()
        }

    @Test
    fun isWifiDefault_wifiNetworkNotDefault_isFalse() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiDefault)

            val wifiEntry =
                mock<WifiEntry>().apply { whenever(this.isDefaultNetwork).thenReturn(false) }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isFalse()
        }

    @Test
    fun isWifiDefault_carrierMergedNotDefault_isFalse() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiDefault)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isDefaultNetwork).thenReturn(false)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isFalse()
        }

    @Test
    fun isWifiDefault_noWifiNetwork_isFalse() =
        testScope.runTest {
            val latest by collectLastValue(underTest.isWifiDefault)

            // First, add a network
            val wifiEntry =
                mock<WifiEntry>().apply { whenever(this.isDefaultNetwork).thenReturn(true) }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isTrue()

            // WHEN the network is lost
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)
            getCallback().onWifiEntriesChanged()

            // THEN we update to false
            assertThat(latest).isFalse()
        }

    @Test
    fun wifiNetwork_initiallyGetsDefault() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            assertThat(latest).isEqualTo(WIFI_NETWORK_DEFAULT)
        }

    @Test
    fun wifiNetwork_primaryWifiNetworkAdded_flowHasNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(3)
                    whenever(this.title).thenReturn(TITLE)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest is WifiNetworkModel.Active).isTrue()
            val latestActive = latest as WifiNetworkModel.Active
            assertThat(latestActive.level).isEqualTo(3)
            assertThat(latestActive.ssid).isEqualTo(TITLE)
        }

    @Test
    fun wifiNetwork_unreachableLevel_inactiveNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(WIFI_LEVEL_UNREACHABLE)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isInstanceOf(WifiNetworkModel.Inactive::class.java)
            val inactiveReason = (latest as WifiNetworkModel.Inactive).inactiveReason
            assertThat(inactiveReason).contains("level")
            assertThat(inactiveReason).contains("$WIFI_LEVEL_UNREACHABLE")
        }

    @Test
    fun wifiNetwork_levelTooHigh_inactiveNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(WIFI_LEVEL_MAX + 1)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isInstanceOf(WifiNetworkModel.Inactive::class.java)
            val inactiveReason = (latest as WifiNetworkModel.Inactive).inactiveReason
            assertThat(inactiveReason).contains("level")
            assertThat(inactiveReason).contains("${WIFI_LEVEL_MAX + 1}")
        }

    @Test
    fun wifiNetwork_levelTooLow_inactiveNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(WIFI_LEVEL_MIN - 1)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isInstanceOf(WifiNetworkModel.Inactive::class.java)
            val inactiveReason = (latest as WifiNetworkModel.Inactive).inactiveReason
            assertThat(inactiveReason).contains("level")
            assertThat(inactiveReason).contains("${WIFI_LEVEL_MIN - 1}")
        }

    @Test
    fun wifiNetwork_levelIsMax_activeNetworkWithMaxLevel() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(WIFI_LEVEL_MAX)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isInstanceOf(WifiNetworkModel.Active::class.java)
            assertThat((latest as WifiNetworkModel.Active).level).isEqualTo(WIFI_LEVEL_MAX)
        }

    @Test
    fun wifiNetwork_levelIsMin_activeNetworkWithMinLevel() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(WIFI_LEVEL_MIN)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isInstanceOf(WifiNetworkModel.Active::class.java)
            assertThat((latest as WifiNetworkModel.Active).level).isEqualTo(WIFI_LEVEL_MIN)
        }

    @Test
    fun wifiNetwork_notHotspot_none() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply { whenever(this.isPrimaryNetwork).thenReturn(true) }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).hotspotDeviceType)
                .isEqualTo(WifiNetworkModel.HotspotDeviceType.NONE)
        }

    @Test
    fun wifiNetwork_hotspot_unknown() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry = createHotspotWithType(NetworkProviderInfo.DEVICE_TYPE_UNKNOWN)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).hotspotDeviceType)
                .isEqualTo(WifiNetworkModel.HotspotDeviceType.UNKNOWN)
        }

    @Test
    fun wifiNetwork_hotspot_phone() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry = createHotspotWithType(NetworkProviderInfo.DEVICE_TYPE_PHONE)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).hotspotDeviceType)
                .isEqualTo(WifiNetworkModel.HotspotDeviceType.PHONE)
        }

    @Test
    fun wifiNetwork_hotspot_tablet() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry = createHotspotWithType(NetworkProviderInfo.DEVICE_TYPE_TABLET)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).hotspotDeviceType)
                .isEqualTo(WifiNetworkModel.HotspotDeviceType.TABLET)
        }

    @Test
    fun wifiNetwork_hotspot_laptop() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry = createHotspotWithType(NetworkProviderInfo.DEVICE_TYPE_LAPTOP)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).hotspotDeviceType)
                .isEqualTo(WifiNetworkModel.HotspotDeviceType.LAPTOP)
        }

    @Test
    fun wifiNetwork_hotspot_watch() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry = createHotspotWithType(NetworkProviderInfo.DEVICE_TYPE_WATCH)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).hotspotDeviceType)
                .isEqualTo(WifiNetworkModel.HotspotDeviceType.WATCH)
        }

    @Test
    fun wifiNetwork_hotspot_auto() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry = createHotspotWithType(NetworkProviderInfo.DEVICE_TYPE_AUTO)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).hotspotDeviceType)
                .isEqualTo(WifiNetworkModel.HotspotDeviceType.AUTO)
        }

    @Test
    fun wifiNetwork_hotspot_invalid() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry = createHotspotWithType(1234)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).hotspotDeviceType)
                .isEqualTo(WifiNetworkModel.HotspotDeviceType.INVALID)
        }

    @Test
    fun wifiNetwork_isCarrierMerged_flowHasCarrierMerged() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(3)
                    whenever(this.subscriptionId).thenReturn(567)
                    whenever(this.isDefaultNetwork).thenReturn(true)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest is WifiNetworkModel.CarrierMerged).isTrue()
            val latestMerged = latest as WifiNetworkModel.CarrierMerged
            assertThat(latestMerged.level).isEqualTo(3)
            assertThat(latestMerged.subscriptionId).isEqualTo(567)
        }

    @Test
    fun wifiNetwork_isCarrierMerged_getsMaxSignalLevel() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.isDefaultNetwork).thenReturn(true)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)
            whenever(wifiManager.maxSignalLevel).thenReturn(5)

            getCallback().onWifiEntriesChanged()

            assertThat(latest is WifiNetworkModel.CarrierMerged).isTrue()
            val latestMerged = latest as WifiNetworkModel.CarrierMerged
            // numberOfLevels = maxSignalLevel + 1
            assertThat(latestMerged.numberOfLevels).isEqualTo(6)
        }

    @Test
    fun wifiNetwork_carrierMergedButInvalidSubId_flowHasInvalid() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.subscriptionId).thenReturn(INVALID_SUBSCRIPTION_ID)
                    whenever(this.isDefaultNetwork).thenReturn(true)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)

            getCallback().onWifiEntriesChanged()

            assertThat(latest).isInstanceOf(WifiNetworkModel.Invalid::class.java)
        }

    @Test
    fun wifiNetwork_carrierMergedButInvalidLevel_flowHasInvalid() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.subscriptionId).thenReturn(3)
                    whenever(this.isDefaultNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(WIFI_LEVEL_UNREACHABLE)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)

            getCallback().onWifiEntriesChanged()

            assertThat(latest).isInstanceOf(WifiNetworkModel.Invalid::class.java)
        }

    @Test
    fun wifiNetwork_notValidated_networkNotValidated() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.hasInternetAccess()).thenReturn(false)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).isValidated).isFalse()
        }

    @Test
    fun wifiNetwork_validated_networkValidated() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.hasInternetAccess()).thenReturn(true)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).isValidated).isTrue()
        }

    @Test
    fun wifiNetwork_nonPrimaryWifiNetworkAdded_flowHasNoNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply { whenever(this.isPrimaryNetwork).thenReturn(false) }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isEqualTo(WifiNetworkModel.Inactive())
        }

    @Test
    fun wifiNetwork_nonPrimaryCarrierMergedNetworkAdded_flowHasNoNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(false)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)
            getCallback().onWifiEntriesChanged()

            assertThat(latest).isEqualTo(WifiNetworkModel.Inactive())
        }

    @Test
    fun wifiNetwork_newPrimaryWifiNetwork_flowHasNewNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            // Start with the original network
            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(3)
                    whenever(this.title).thenReturn("AB")
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest is WifiNetworkModel.Active).isTrue()
            var latestActive = latest as WifiNetworkModel.Active
            assertThat(latestActive.level).isEqualTo(3)
            assertThat(latestActive.ssid).isEqualTo("AB")

            // WHEN we update to a new primary network
            val newWifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(4)
                    whenever(this.title).thenReturn("CD")
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(newWifiEntry)
            getCallback().onWifiEntriesChanged()

            // THEN we use the new network
            assertThat(latest is WifiNetworkModel.Active).isTrue()
            latestActive = latest as WifiNetworkModel.Active
            assertThat(latestActive.level).isEqualTo(4)
            assertThat(latestActive.ssid).isEqualTo("CD")
        }

    @Test
    fun wifiNetwork_noCurrentNetwork_networkLost_flowHasNoNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            // WHEN we receive a null network without any networks beforehand
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)
            getCallback().onWifiEntriesChanged()

            // THEN there's no crash and we still have no network
            assertThat(latest is WifiNetworkModel.Inactive).isTrue()
        }

    @Test
    fun wifiNetwork_currentActiveNetworkLost_flowHasNoNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.title).thenReturn(TITLE)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat((latest as WifiNetworkModel.Active).ssid).isEqualTo(TITLE)

            // WHEN we lose our current network
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)
            getCallback().onWifiEntriesChanged()

            // THEN we update to no network
            assertThat(latest is WifiNetworkModel.Inactive).isTrue()
        }

    /** Possible regression test for b/278618530. */
    @Test
    fun wifiNetwork_currentCarrierMergedNetworkLost_flowHasNoNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(3)
                    whenever(this.isDefaultNetwork).thenReturn(true)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest is WifiNetworkModel.CarrierMerged).isTrue()
            assertThat((latest as WifiNetworkModel.CarrierMerged).level).isEqualTo(3)

            // WHEN we lose our current network
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(null)
            getCallback().onWifiEntriesChanged()

            // THEN we update to no network
            assertThat(latest is WifiNetworkModel.Inactive).isTrue()
        }

    /** Regression test for b/244173280. */
    @Test
    fun wifiNetwork_multipleSubscribers_newSubscribersGetCurrentValue() =
        testScope.runTest {
            val latest1 by collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(1)
                    whenever(this.title).thenReturn(TITLE)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(latest1 is WifiNetworkModel.Active).isTrue()
            val latest1Active = latest1 as WifiNetworkModel.Active
            assertThat(latest1Active.level).isEqualTo(1)
            assertThat(latest1Active.ssid).isEqualTo(TITLE)

            // WHEN we add a second subscriber after having already emitted a value
            val latest2 by collectLastValue(underTest.wifiNetwork)

            // THEN the second subscribe receives the already-emitted value
            assertThat(latest2 is WifiNetworkModel.Active).isTrue()
            val latest2Active = latest2 as WifiNetworkModel.Active
            assertThat(latest2Active.level).isEqualTo(1)
            assertThat(latest2Active.ssid).isEqualTo(TITLE)
        }

    @Test
    fun wifiNetwork_carrierMerged_default_usesCarrierMergedInfo() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(3)
                    whenever(this.isDefaultNetwork).thenReturn(true)
                }
            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(1)
                    whenever(this.title).thenReturn(TITLE)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)

            getCallback().onWifiEntriesChanged()

            assertThat(latest is WifiNetworkModel.CarrierMerged).isTrue()
        }

    @Test
    fun wifiNetwork_carrierMerged_notDefault_usesConnectedInfo() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiNetwork)

            val mergedEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(3)
                    whenever(this.isDefaultNetwork).thenReturn(false)
                }
            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(1)
                    whenever(this.title).thenReturn(TITLE)
                }
            whenever(wifiPickerTracker.mergedCarrierEntry).thenReturn(mergedEntry)
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)

            getCallback().onWifiEntriesChanged()

            assertThat(latest is WifiNetworkModel.Active).isTrue()
        }

    @Test
    fun secondaryNetworks_activeEntriesEmpty_isEmpty() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            whenever(wifiPickerTracker.activeWifiEntries).thenReturn(listOf())

            getCallback().onWifiEntriesChanged()

            assertThat(latest).isEmpty()
        }

    @Test
    fun secondaryNetworks_oneActiveEntry_hasOne() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val wifiEntry = mock<WifiEntry>()
            whenever(wifiPickerTracker.activeWifiEntries).thenReturn(listOf(wifiEntry))

            getCallback().onWifiEntriesChanged()

            assertThat(latest).hasSize(1)
        }

    @Test
    fun secondaryNetworks_multipleActiveEntries_hasMultiple() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val wifiEntry1 = mock<WifiEntry>()
            val wifiEntry2 = mock<WifiEntry>()
            whenever(wifiPickerTracker.activeWifiEntries).thenReturn(listOf(wifiEntry1, wifiEntry2))

            getCallback().onWifiEntriesChanged()

            assertThat(latest).hasSize(2)
        }

    @Test
    fun secondaryNetworks_mapsToInactive() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val inactiveEntry =
                mock<WifiEntry>().apply { whenever(this.level).thenReturn(WIFI_LEVEL_UNREACHABLE) }
            whenever(wifiPickerTracker.activeWifiEntries).thenReturn(listOf(inactiveEntry))

            getCallback().onWifiEntriesChanged()

            assertThat(latest).hasSize(1)
            assertThat(latest!![0]).isInstanceOf(WifiNetworkModel.Inactive::class.java)
        }

    @Test
    fun secondaryNetworks_mapsToActive() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val activeEntry = mock<WifiEntry>().apply { whenever(this.level).thenReturn(2) }
            whenever(wifiPickerTracker.activeWifiEntries).thenReturn(listOf(activeEntry))

            getCallback().onWifiEntriesChanged()

            assertThat(latest).hasSize(1)
            assertThat(latest!![0]).isInstanceOf(WifiNetworkModel.Active::class.java)
            assertThat((latest!![0] as WifiNetworkModel.Active).level).isEqualTo(2)
        }

    @Test
    fun secondaryNetworks_mapsToCarrierMerged() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val carrierMergedEntry =
                mock<MergedCarrierEntry>().apply { whenever(this.level).thenReturn(3) }
            whenever(wifiPickerTracker.activeWifiEntries).thenReturn(listOf(carrierMergedEntry))

            getCallback().onWifiEntriesChanged()

            assertThat(latest).hasSize(1)
            assertThat(latest!![0]).isInstanceOf(WifiNetworkModel.CarrierMerged::class.java)
            assertThat((latest!![0] as WifiNetworkModel.CarrierMerged).level).isEqualTo(3)
        }

    @Test
    fun secondaryNetworks_mapsMultipleInOrder() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val activeEntry = mock<WifiEntry>().apply { whenever(this.level).thenReturn(2) }
            val carrierMergedEntry =
                mock<MergedCarrierEntry>().apply { whenever(this.level).thenReturn(3) }
            whenever(wifiPickerTracker.activeWifiEntries)
                .thenReturn(listOf(activeEntry, carrierMergedEntry))

            getCallback().onWifiEntriesChanged()

            assertThat(latest!![0]).isInstanceOf(WifiNetworkModel.Active::class.java)
            assertThat((latest!![0] as WifiNetworkModel.Active).level).isEqualTo(2)
            assertThat(latest!![1]).isInstanceOf(WifiNetworkModel.CarrierMerged::class.java)
            assertThat((latest!![1] as WifiNetworkModel.CarrierMerged).level).isEqualTo(3)
        }

    @Test
    fun secondaryNetworks_filtersOutConnectedEntry() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val connectedEntry = mock<WifiEntry>().apply { whenever(this.level).thenReturn(1) }
            val secondaryEntry1 = mock<WifiEntry>().apply { whenever(this.level).thenReturn(2) }
            val secondaryEntry2 = mock<WifiEntry>().apply { whenever(this.level).thenReturn(3) }
            // WHEN the active list has both a primary and secondary networks
            whenever(wifiPickerTracker.activeWifiEntries)
                .thenReturn(listOf(connectedEntry, secondaryEntry1, secondaryEntry2))
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(connectedEntry)

            getCallback().onWifiEntriesChanged()

            // THEN only the secondary networks are included
            assertThat(latest).hasSize(2)
            assertThat((latest!![0] as WifiNetworkModel.Active).level).isEqualTo(2)
            assertThat((latest!![1] as WifiNetworkModel.Active).level).isEqualTo(3)
        }

    @Test
    fun secondaryNetworks_noConnectedEntry_hasAllActiveEntries() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val secondaryEntry1 = mock<WifiEntry>().apply { whenever(this.level).thenReturn(2) }
            val secondaryEntry2 = mock<WifiEntry>().apply { whenever(this.level).thenReturn(3) }
            whenever(wifiPickerTracker.activeWifiEntries)
                .thenReturn(listOf(secondaryEntry1, secondaryEntry2))
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)

            getCallback().onWifiEntriesChanged()

            assertThat(latest).hasSize(2)
            assertThat((latest!![0] as WifiNetworkModel.Active).level).isEqualTo(2)
            assertThat((latest!![1] as WifiNetworkModel.Active).level).isEqualTo(3)
        }

    @Test
    fun secondaryNetworks_filtersOutPrimaryNetwork() =
        testScope.runTest {
            val latest by collectLastValue(underTest.secondaryNetworks)

            val primaryEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.level).thenReturn(1)
                }
            val secondaryEntry1 = mock<WifiEntry>().apply { whenever(this.level).thenReturn(2) }
            val secondaryEntry2 = mock<WifiEntry>().apply { whenever(this.level).thenReturn(3) }
            // WHEN the active list has both a primary and secondary networks
            whenever(wifiPickerTracker.activeWifiEntries)
                .thenReturn(listOf(secondaryEntry1, primaryEntry, secondaryEntry2))

            getCallback().onWifiEntriesChanged()

            // THEN only the secondary networks are included
            assertThat(latest).hasSize(2)
            assertThat((latest!![0] as WifiNetworkModel.Active).level).isEqualTo(2)
            assertThat((latest!![1] as WifiNetworkModel.Active).level).isEqualTo(3)
        }

    @Test
    fun isWifiConnectedWithValidSsid_inactiveNetwork_false() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)
            getCallback().onWifiEntriesChanged()

            assertThat(underTest.isWifiConnectedWithValidSsid()).isFalse()
        }

    @Test
    fun isWifiConnectedWithValidSsid_nonPrimaryNetwork_false() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply { whenever(this.isPrimaryNetwork).thenReturn(false) }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(underTest.isWifiConnectedWithValidSsid()).isFalse()
        }

    @Test
    fun isWifiConnectedWithValidSsid_carrierMergedNetwork_false() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(underTest.isWifiConnectedWithValidSsid()).isFalse()
        }

    @Test
    fun isWifiConnectedWithValidSsid_invalidNetwork_false() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<MergedCarrierEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.subscriptionId).thenReturn(INVALID_SUBSCRIPTION_ID)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(underTest.isWifiConnectedWithValidSsid()).isFalse()
        }

    @Test
    fun isWifiConnectedWithValidSsid_activeNetwork_nullTitle_false() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.title).thenReturn(null)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(underTest.isWifiConnectedWithValidSsid()).isFalse()
        }

    @Test
    fun isWifiConnectedWithValidSsid_activeNetwork_unknownTitle_false() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.title).thenReturn(UNKNOWN_SSID)
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(underTest.isWifiConnectedWithValidSsid()).isFalse()
        }

    @Test
    fun isWifiConnectedWithValidSsid_activeNetwork_validTitle_true() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.title).thenReturn("fakeSsid")
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(underTest.isWifiConnectedWithValidSsid()).isTrue()
        }

    @Test
    fun isWifiConnectedWithValidSsid_activeToInactive_trueToFalse() =
        testScope.runTest {
            collectLastValue(underTest.wifiNetwork)

            // Start with active
            val wifiEntry =
                mock<WifiEntry>().apply {
                    whenever(this.isPrimaryNetwork).thenReturn(true)
                    whenever(this.title).thenReturn("fakeSsid")
                }
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(wifiEntry)
            getCallback().onWifiEntriesChanged()

            assertThat(underTest.isWifiConnectedWithValidSsid()).isTrue()

            // WHEN the network is lost
            whenever(wifiPickerTracker.connectedWifiEntry).thenReturn(null)
            getCallback().onWifiEntriesChanged()

            // THEN the isWifiConnected updates
            assertThat(underTest.isWifiConnectedWithValidSsid()).isFalse()
        }

    @Test
    fun wifiActivity_callbackGivesNone_activityFlowHasNone() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiActivity)

            getTrafficStateCallback()
                .onStateChanged(WifiManager.TrafficStateCallback.DATA_ACTIVITY_NONE)

            assertThat(latest)
                .isEqualTo(DataActivityModel(hasActivityIn = false, hasActivityOut = false))
        }

    @Test
    fun wifiActivity_callbackGivesIn_activityFlowHasIn() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiActivity)

            getTrafficStateCallback()
                .onStateChanged(WifiManager.TrafficStateCallback.DATA_ACTIVITY_IN)

            assertThat(latest)
                .isEqualTo(DataActivityModel(hasActivityIn = true, hasActivityOut = false))
        }

    @Test
    fun wifiActivity_callbackGivesOut_activityFlowHasOut() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiActivity)

            getTrafficStateCallback()
                .onStateChanged(WifiManager.TrafficStateCallback.DATA_ACTIVITY_OUT)

            assertThat(latest)
                .isEqualTo(DataActivityModel(hasActivityIn = false, hasActivityOut = true))
        }

    @Test
    fun wifiActivity_callbackGivesInout_activityFlowHasInAndOut() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiActivity)

            getTrafficStateCallback()
                .onStateChanged(WifiManager.TrafficStateCallback.DATA_ACTIVITY_INOUT)

            assertThat(latest)
                .isEqualTo(DataActivityModel(hasActivityIn = true, hasActivityOut = true))
        }

    @Test
    fun wifiScanResults_containsSsidList() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiScanResults)

            val scanResults =
                listOf(
                    ScanResult().also { it.SSID = "ssid 1" },
                    ScanResult().also { it.SSID = "ssid 2" },
                    ScanResult().also { it.SSID = "ssid 3" },
                    ScanResult().also { it.SSID = "ssid 4" },
                    ScanResult().also { it.SSID = "ssid 5" },
                )
            whenever(wifiManager.scanResults).thenReturn(scanResults)
            getScanResultsCallback().onScanResultsAvailable()

            val expected =
                listOf(
                    WifiScanEntry(ssid = "ssid 1"),
                    WifiScanEntry(ssid = "ssid 2"),
                    WifiScanEntry(ssid = "ssid 3"),
                    WifiScanEntry(ssid = "ssid 4"),
                    WifiScanEntry(ssid = "ssid 5"),
                )

            assertThat(latest).isEqualTo(expected)
        }

    @Test
    fun wifiScanResults_updates() =
        testScope.runTest {
            val latest by collectLastValue(underTest.wifiScanResults)

            var scanResults =
                listOf(
                    ScanResult().also { it.SSID = "ssid 1" },
                    ScanResult().also { it.SSID = "ssid 2" },
                    ScanResult().also { it.SSID = "ssid 3" },
                    ScanResult().also { it.SSID = "ssid 4" },
                    ScanResult().also { it.SSID = "ssid 5" },
                )
            whenever(wifiManager.scanResults).thenReturn(scanResults)
            getScanResultsCallback().onScanResultsAvailable()

            // New scan representing no results
            scanResults = listOf()
            whenever(wifiManager.scanResults).thenReturn(scanResults)
            getScanResultsCallback().onScanResultsAvailable()

            assertThat(latest).isEmpty()
        }

    @Test
    @EnableFlags(FLAG_MULTIUSER_WIFI_PICKER_TRACKER_SUPPORT)
    fun oneUserVerifyCreatingWifiPickerTracker_multiuserFlagEnabled() =
        testScope.runTest {
            val primaryUserMockContext = mock<Context>()
            mContext.prepareCreateContextAsUser(
                UserHandle.of(PRIMARY_USER_ID),
                primaryUserMockContext,
            )

            userRepository.setSelectedUserInfo(PRIMARY_USER)

            collectLastValue(underTest.wifiNetwork)

            val contextCaptor = argumentCaptor<Context>()
            verify(wifiPickerTrackerFactory).create(contextCaptor.capture(), any(), any(), any())
            // If the flag is on, verify that we use the context from #createContextAsUser and we
            // do NOT use the top-level context
            assertThat(contextCaptor.firstValue).isEqualTo(primaryUserMockContext)
        }

    @Test
    @EnableFlags(FLAG_MULTIUSER_WIFI_PICKER_TRACKER_SUPPORT)
    fun changeUserVerifyCreatingWifiPickerTracker_multiuserEnabled() =
        testScope.runTest {
            val primaryUserMockContext = mock<Context>()
            mContext.prepareCreateContextAsUser(
                UserHandle.of(PRIMARY_USER_ID),
                primaryUserMockContext,
            )
            userRepository.setSelectedUserInfo(PRIMARY_USER)

            collectLastValue(underTest.wifiNetwork)

            val contextCaptor = argumentCaptor<Context>()
            verify(wifiPickerTrackerFactory).create(contextCaptor.capture(), any(), any(), any())
            assertThat(contextCaptor.firstValue).isEqualTo(primaryUserMockContext)

            reset(wifiPickerTrackerFactory)

            // WHEN we switch to a different user
            val otherUserMockContext = mock<Context>()
            mContext.prepareCreateContextAsUser(
                UserHandle.of(ANOTHER_USER_ID),
                otherUserMockContext,
            )
            userRepository.setSelectedUserInfo(ANOTHER_USER)

            // THEN we use the different user's context to create WifiPickerTracker
            val newCaptor = argumentCaptor<Context>()
            verify(wifiPickerTrackerFactory).create(newCaptor.capture(), any(), any(), any())
            assertThat(newCaptor.firstValue).isEqualTo(otherUserMockContext)
        }

    @Test
    @DisableFlags(FLAG_MULTIUSER_WIFI_PICKER_TRACKER_SUPPORT)
    fun changeUserVerifyCreatingWifiPickerTracker_multiuserDisabled() =
        testScope.runTest {
            val primaryUserMockContext = mock<Context>()
            mContext.prepareCreateContextAsUser(
                UserHandle.of(PRIMARY_USER_ID),
                primaryUserMockContext,
            )

            userRepository.setSelectedUserInfo(PRIMARY_USER)

            collectLastValue(underTest.wifiNetwork)

            val contextCaptor = argumentCaptor<Context>()
            verify(wifiPickerTrackerFactory).create(contextCaptor.capture(), any(), any(), any())
            // If the flag is off, verify that we do NOT use the context from #createContextAsUser
            // and we instead use the top-level context
            assertThat(contextCaptor.firstValue).isNotEqualTo(primaryUserMockContext)
            assertThat(contextCaptor.firstValue).isEqualTo(mContext)

            reset(wifiPickerTrackerFactory)

            // WHEN we switch to a different user
            val otherUserMockContext = mock<Context>()
            mContext.prepareCreateContextAsUser(
                UserHandle.of(ANOTHER_USER_ID),
                otherUserMockContext,
            )
            userRepository.setSelectedUserInfo(ANOTHER_USER)

            // THEN we do NOT re-create WifiPickerTracker because the multiuser flag is off
            verify(wifiPickerTrackerFactory, never()).create(any(), any(), any(), any())
        }

    private fun getCallback(): WifiPickerTracker.WifiPickerTrackerCallback {
        return callbackCaptor.firstValue
    }

    private fun getTrafficStateCallback(): WifiManager.TrafficStateCallback {
        val callbackCaptor = argumentCaptor<WifiManager.TrafficStateCallback>()
        verify(wifiManager).registerTrafficStateCallback(any(), callbackCaptor.capture())
        return callbackCaptor.firstValue
    }

    private fun createHotspotWithType(@DeviceType type: Int): HotspotNetworkEntry {
        return mock<HotspotNetworkEntry>().apply {
            whenever(this.isPrimaryNetwork).thenReturn(true)
            whenever(this.deviceType).thenReturn(type)
        }
    }

    private fun getScanResultsCallback(): WifiManager.ScanResultsCallback {
        val callbackCaptor = argumentCaptor<WifiManager.ScanResultsCallback>()
        verify(wifiManager).registerScanResultsCallback(any(), callbackCaptor.capture())
        return callbackCaptor.firstValue
    }

    private companion object {
        const val TITLE = "AB"
        private const val PRIMARY_USER_ID = 0
        private val PRIMARY_USER =
            UserInfo(
                /* id= */ PRIMARY_USER_ID,
                /* name= */ "primary user",
                /* flags= */ UserInfo.FLAG_PROFILE,
            )

        private const val ANOTHER_USER_ID = 1
        private val ANOTHER_USER =
            UserInfo(
                /* id= */ ANOTHER_USER_ID,
                /* name= */ "another user",
                /* flags= */ UserInfo.FLAG_PROFILE,
            )
    }
}
