/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.speech;

import android.annotation.NonNull;
import android.os.Parcelable;

import com.android.internal.util.DataClass;
import com.android.internal.util.Preconditions;

import java.util.ArrayList;
import java.util.List;

/**
 * List of alternative hypotheses for a specific span of a speech recognition result string.
 *
 * <p> A single {@link SpeechRecognizer} result is represented as a {@link String}. For a specific
 * span of the originally recognized result string, the recognizer may provide alternative
 * hypotheses of what it may have recognized. A span is specifically a substring and is thereby
 * defined by its start and end positions in the originally recognized string. Alternative
 * hypotheses are represented as strings which may replace that substring.
 *
 * <p> These alternatives can be used to enhance recognition by adding/re-ranking/applying or in
 * other ways manipulating the SpeechRecognizer results before powering dictation features.
 */
@DataClass(
        genEqualsHashCode = true,
        genParcelable = true,
        genToString = true
)
public final class AlternativeSpan implements Parcelable {
    /**
     * The start position of the span of the originally recognized string.
     *
     * <p> Must be set to a non-negative value before building.
     */
    private final int mStartPosition;

    /**
     * The exclusive end position of the span of the originally recognized string.
     *
     * <p> Must be set to a value greater than the start of the span before building.
     */
    private final int mEndPosition;

    /**
     * All the alternatives for the [mStart, mEnd) span.
     *
     * <p> Must not be empty. The object will only be created
     * if there are some alternatives for the given span.
     *
     * <p> The alternatives may be strings of different lengths than the span they can replace.
     */
    @NonNull
    @DataClass.PluralOf("alternative")
    private final List<String> mAlternatives;

    private void onConstructed() {
        Preconditions.checkArgumentNonnegative(mStartPosition,
                "The range start must be non-negative.");
        Preconditions.checkArgument(mStartPosition < mEndPosition,
                "Illegal range [%d, %d), must be start < end.", mStartPosition, mEndPosition);
        Preconditions.checkCollectionNotEmpty(mAlternatives,
                "List of alternative strings must not be empty.");
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/speech/AlternativeSpan.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /**
     * Creates a new AlternativeSpan.
     *
     * @param startPosition
     *   The start position of the span of the originally recognized string.
     *
     *   <p> Must be set to a non-negative value before building.
     * @param endPosition
     *   The exclusive end position of the span of the originally recognized string.
     *
     *   <p> Must be set to a value greater than the start of the span before building.
     * @param alternatives
     *   All the alternatives for the [mStart, mEnd) span.
     *
     *   <p> Must not be empty. The object will only be created
     *   if there are some alternatives for the given span.
     *
     *   <p> The alternatives may be strings of different lengths than the span they can replace.
     */
    @DataClass.Generated.Member
    public AlternativeSpan(
            int startPosition,
            int endPosition,
            @NonNull List<String> alternatives) {
        this.mStartPosition = startPosition;
        this.mEndPosition = endPosition;
        this.mAlternatives = alternatives;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mAlternatives);

        onConstructed();
    }

    /**
     * The start position of the span of the originally recognized string.
     *
     * <p> Must be set to a non-negative value before building.
     */
    @DataClass.Generated.Member
    public int getStartPosition() {
        return mStartPosition;
    }

    /**
     * The exclusive end position of the span of the originally recognized string.
     *
     * <p> Must be set to a value greater than the start of the span before building.
     */
    @DataClass.Generated.Member
    public int getEndPosition() {
        return mEndPosition;
    }

    /**
     * All the alternatives for the [mStart, mEnd) span.
     *
     * <p> Must not be empty. The object will only be created
     * if there are some alternatives for the given span.
     *
     * <p> The alternatives may be strings of different lengths than the span they can replace.
     */
    @DataClass.Generated.Member
    public @NonNull List<String> getAlternatives() {
        return mAlternatives;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "AlternativeSpan { " +
                "startPosition = " + mStartPosition + ", " +
                "endPosition = " + mEndPosition + ", " +
                "alternatives = " + mAlternatives +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public boolean equals(@android.annotation.Nullable Object o) {
        // You can override field equality logic by defining either of the methods like:
        // boolean fieldNameEquals(AlternativeSpan other) { ... }
        // boolean fieldNameEquals(FieldType otherValue) { ... }

        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        @SuppressWarnings("unchecked")
        AlternativeSpan that = (AlternativeSpan) o;
        //noinspection PointlessBooleanExpression
        return true
                && mStartPosition == that.mStartPosition
                && mEndPosition == that.mEndPosition
                && java.util.Objects.equals(mAlternatives, that.mAlternatives);
    }

    @Override
    @DataClass.Generated.Member
    public int hashCode() {
        // You can override field hashCode logic by defining methods like:
        // int fieldNameHashCode() { ... }

        int _hash = 1;
        _hash = 31 * _hash + mStartPosition;
        _hash = 31 * _hash + mEndPosition;
        _hash = 31 * _hash + java.util.Objects.hashCode(mAlternatives);
        return _hash;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull android.os.Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeInt(mStartPosition);
        dest.writeInt(mEndPosition);
        dest.writeStringList(mAlternatives);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ AlternativeSpan(@NonNull android.os.Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        int startPosition = in.readInt();
        int endPosition = in.readInt();
        List<String> alternatives = new ArrayList<>();
        in.readStringList(alternatives);

        this.mStartPosition = startPosition;
        this.mEndPosition = endPosition;
        this.mAlternatives = alternatives;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mAlternatives);

        onConstructed();
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<AlternativeSpan> CREATOR
            = new Parcelable.Creator<AlternativeSpan>() {
        @Override
        public AlternativeSpan[] newArray(int size) {
            return new AlternativeSpan[size];
        }

        @Override
        public AlternativeSpan createFromParcel(@NonNull android.os.Parcel in) {
            return new AlternativeSpan(in);
        }
    };

    @DataClass.Generated(
            time = 1656603431902L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/speech/AlternativeSpan.java",
            inputSignatures = "private final  int mStartPosition\nprivate final  int mEndPosition\nprivate final @android.annotation.NonNull @com.android.internal.util.DataClass.PluralOf(\"alternative\") java.util.List<java.lang.String> mAlternatives\nprivate  void onConstructed()\nclass AlternativeSpan extends java.lang.Object implements [android.os.Parcelable]\n@com.android.internal.util.DataClass(genEqualsHashCode=true, genParcelable=true, genToString=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
