/**
 * Copyright (c) 2020, The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.media;

import android.media.TranscodingSessionParcel;
import android.media.TranscodingRequestParcel;

/**
 * ITranscodingClient
 *
 * Interface for a client to communicate with MediaTranscodingService.
 *
 * {@hide}
 */
interface ITranscodingClient {
    /**
     * Submits a transcoding request to MediaTranscodingService.
     *
     * @param request a TranscodingRequest contains transcoding configuration.
     * @param session(output variable) a TranscodingSession generated by MediaTranscodingService.
     * @return true if success, false otherwise.
     */
    boolean submitRequest(in TranscodingRequestParcel request,
                          out TranscodingSessionParcel session);

    /**
     * Cancels a transcoding session.
     *
     * @param sessionId a TranscodingSession generated by the MediaTranscodingService.
     * @return true if succeeds, false otherwise.
     */
    boolean cancelSession(in int sessionId);

    /**
     * Queries the session detail associated with a sessionId.
     *
     * @param sessionId a TranscodingSession generated by the MediaTranscodingService.
     * @param session(output variable) the TranscodingSession associated with the sessionId.
     * @return true if succeeds, false otherwise.
     */
    boolean getSessionWithId(in int sessionId, out TranscodingSessionParcel session);

    /**
     * Add an additional client uid requesting a session.
     *
     * @sessionId the session id to which to add the additional client uid.
     * @clientUid the additional client uid to be added.
     * @return false if the session doesn't exist or the client is already requesting the
     * session, true otherwise.
     */
    boolean addClientUid(in int sessionId, int clientUid);

    /**
     * Retrieves the (unsorted) list of all clients requesting a session.
     *
     * Note that if a session was submitted with offline priority (
     * TranscodingSessionPriority::kUnspecified), it initially will not be considered requested
     * by any particular client, because the client could go away any time after the submission.
     * However, additional uids could be added via addClientUid() after the submission, which
     * essentially make the request a real-time request instead of an offline request.
     *
     * @sessionId the session id for which to retrieve the client uid list.
     * @clientUids array to hold the retrieved client uid list.
     * @return false if the session doesn't exist, true otherwise.
     */
    @nullable
    int[] getClientUids(in int sessionId);

    /**
    * Unregister the client with the MediaTranscodingService.
    *
    * Client will not be able to perform any more transcoding after unregister.
    */
    void unregister();
}
