// Copyright 2021 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package etc

// This file implements snapshot module of 'prebuilt_etc' type
// 'snapshot_etc' module defines android.PrebuiltInterface so it can be handled
// as prebuilt of 'prebuilt_etc' type.
// Properties of 'snapshot_etc' follows properties from snapshotJsonFlags type

import (
	"android/soong/android"
	"fmt"
	"strings"

	"github.com/google/blueprint"
	"github.com/google/blueprint/proptools"
)

func RegisterSnapshotEtcModule(ctx android.RegistrationContext) {
	ctx.RegisterModuleType("snapshot_etc", SnapshotEtcFactory)
}

func init() {
	RegisterSnapshotEtcModule(android.InitRegistrationContext)
}

// snapshot_etc is a prebuilt module type to be installed under etc which is auto-generated by
// development/vendor_snapshot/update.py. This module will override prebuilt_etc module with same
// name when 'prefer' property is true.
func SnapshotEtcFactory() android.Module {
	module := &SnapshotEtc{}
	module.AddProperties(&module.properties)

	var srcsSupplier = func(_ android.BaseModuleContext, prebuilt android.Module) []string {
		s, ok := prebuilt.(*SnapshotEtc)
		if !ok || s.properties.Src == nil {
			return []string{}
		}

		return []string{*s.properties.Src}
	}

	android.InitAndroidArchModule(module, android.DeviceSupported, android.MultilibFirst)
	android.InitPrebuiltModuleWithSrcSupplier(module, srcsSupplier, "src")
	return module
}

type snapshotEtcProperties struct {
	Src                   *string `android:"path,arch_variant"` // Source of snapshot_etc file
	Filename              *string `android:"arch_variant"`      // Target file name when it differs from module name
	Relative_install_path *string `android:"arch_variant"`      // Relative install path when it should be installed subdirectory of etc
}

type SnapshotEtc struct {
	android.ModuleBase
	prebuilt   android.Prebuilt
	properties snapshotEtcProperties

	outputFilePath android.OutputPath
	installDirPath android.InstallPath
}

func (s *SnapshotEtc) Prebuilt() *android.Prebuilt {
	return &s.prebuilt
}

func (s *SnapshotEtc) Name() string {
	return s.prebuilt.Name(s.BaseModuleName())
}

func (s *SnapshotEtc) GenerateAndroidBuildActions(ctx android.ModuleContext) {
	if s.properties.Src == nil {
		ctx.PropertyErrorf("src", "missing prebuilt source file")
		return
	}

	sourceFilePath := s.prebuilt.SingleSourcePath(ctx)

	// Determine the output file basename.
	// If Filename is set, use the name specified by the property.
	// Otherwise use the module name.
	filename := proptools.String(s.properties.Filename)
	if filename == "" {
		filename = ctx.ModuleName()
	}

	s.outputFilePath = android.PathForModuleOut(ctx, filename).OutputPath

	if strings.Contains(filename, "/") {
		ctx.PropertyErrorf("filename", "filename cannot contain separator '/'")
		return
	}

	subDir := ""
	if s.properties.Relative_install_path != nil {
		subDir = *s.properties.Relative_install_path
	}

	s.installDirPath = android.PathForModuleInstall(ctx, "etc", subDir)

	// This ensures that outputFilePath has the correct name for others to
	// use, as the source file may have a different name.
	ctx.Build(pctx, android.BuildParams{
		Rule:        android.Cp,
		Input:       sourceFilePath,
		Output:      s.outputFilePath,
		Description: "Install snapshot etc module " + s.BaseModuleName(),
	})

	ctx.InstallFile(s.installDirPath, s.outputFilePath.Base(), sourceFilePath)
}

func (p *SnapshotEtc) AndroidMkEntries() []android.AndroidMkEntries {
	return []android.AndroidMkEntries{{
		Class:      "ETC",
		OutputFile: android.OptionalPathForPath(p.outputFilePath),
		ExtraEntries: []android.AndroidMkExtraEntriesFunc{
			func(ctx android.AndroidMkExtraEntriesContext, entries *android.AndroidMkEntries) {
				entries.SetString("LOCAL_MODULE_TAGS", "optional")
				entries.SetString("LOCAL_MODULE_PATH", p.installDirPath.ToMakePath().String())
				entries.SetString("LOCAL_INSTALLED_MODULE_STEM", p.outputFilePath.Base())
			},
		},
	}}
}

type snapshotEtcDependencyTag struct {
	blueprint.DependencyTag
}

var tag = snapshotEtcDependencyTag{}

func (s *SnapshotEtc) CoreVariantNeeded(ctx android.BaseModuleContext) bool {
	return !s.ModuleBase.InstallInRecovery() && !s.ModuleBase.InstallInRamdisk() &&
		!s.ModuleBase.InstallInVendorRamdisk() && !s.ModuleBase.InstallInDebugRamdisk()
}

func (p *SnapshotEtc) RamdiskVariantNeeded(ctx android.BaseModuleContext) bool {
	return p.ModuleBase.InstallInRamdisk()
}

func (p *SnapshotEtc) VendorRamdiskVariantNeeded(ctx android.BaseModuleContext) bool {
	return p.ModuleBase.InstallInVendorRamdisk()
}

func (p *SnapshotEtc) DebugRamdiskVariantNeeded(ctx android.BaseModuleContext) bool {
	return p.ModuleBase.InstallInDebugRamdisk()
}

func (p *SnapshotEtc) RecoveryVariantNeeded(ctx android.BaseModuleContext) bool {
	return p.ModuleBase.InstallInRecovery()
}

func (p *SnapshotEtc) ExtraImageVariations(ctx android.BaseModuleContext) []string {
	return nil
}

func (p *SnapshotEtc) SetImageVariation(ctx android.BaseModuleContext, variation string, module android.Module) {
}

func (p *SnapshotEtc) ImageMutatorBegin(ctx android.BaseModuleContext) {}

func (p *SnapshotEtc) OutputFiles(tag string) (android.Paths, error) {
	switch tag {
	case "":
		return android.Paths{p.outputFilePath}, nil
	default:
		return nil, fmt.Errorf("unsupported module reference tag %q", tag)
	}

}

var _ android.PrebuiltInterface = (*SnapshotEtc)(nil)
var _ android.ImageInterface = (*SnapshotEtc)(nil)
var _ android.OutputFileProducer = (*SnapshotEtc)(nil)
