#!/usr/bin/env python
#
# Copyright (C) 2021 The Android Open Source Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Generate a set of signature patterns from the modular flags generated by a
bootclasspath_fragment that can be used to select a subset of monolithic flags
against which the modular flags can be compared.
"""

import argparse
import csv
import sys

def dict_reader(csvfile):
    return csv.DictReader(
        csvfile, delimiter=',', quotechar='|', fieldnames=['signature']
    )


def produce_patterns_from_file(file):
    with open(file, 'r') as f:
        return produce_patterns_from_stream(f)


def produce_patterns_from_stream(stream):
    # Read in all the signatures into a list and remove member names.
    patterns = set()
    for row in dict_reader(stream):
        signature = row['signature']
        text = signature.removeprefix("L")
        # Remove the class specific member signature
        pieces = text.split(";->")
        qualifiedClassName = pieces[0]
        # Remove inner class names as they cannot be separated
        # from the containing outer class.
        pieces = qualifiedClassName.split("$", maxsplit=1)
        pattern = pieces[0]
        patterns.add(pattern)

    patterns = list(patterns) #pylint: disable=redefined-variable-type
    patterns.sort()
    return patterns


def main(args):
    args_parser = argparse.ArgumentParser(
        description='Generate a set of signature patterns '
        'that select a subset of monolithic hidden API files.'
    )
    args_parser.add_argument(
        '--flags',
        help='The stub flags file which contains an entry for every dex member',
    )
    args_parser.add_argument('--output', help='Generated signature prefixes')
    args = args_parser.parse_args(args)

    # Read in all the patterns into a list.
    patterns = produce_patterns_from_file(args.flags)

    # Write out all the patterns.
    with open(args.output, 'w') as outputFile:
        for pattern in patterns:
            outputFile.write(pattern)
            outputFile.write("\n")


if __name__ == "__main__":
    main(sys.argv[1:])
