// Copyright 2024 Google Inc. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package release_config_lib

import (
	"os"
	"path/filepath"
	"testing"

	rc_proto "android/soong/cmd/release_config/release_config_proto"

	"google.golang.org/protobuf/proto"
)

type testCaseFlagValue struct {
	protoPath string
	name      string
	data      []byte
	expected  rc_proto.FlagValue
	err       error
}

func (tc testCaseFlagValue) assertProtoEqual(t *testing.T, expected, actual proto.Message) {
	if !proto.Equal(expected, actual) {
		t.Errorf("Expected %q found %q", expected, actual)
	}
}

func TestFlagValue(t *testing.T) {
	testCases := []testCaseFlagValue{
		{
			name:      "stringVal",
			protoPath: "build/release/flag_values/test/RELEASE_FOO.textproto",
			data:      []byte(`name: "RELEASE_FOO" value {string_value: "BAR"}`),
			expected: rc_proto.FlagValue{
				Name:  proto.String("RELEASE_FOO"),
				Value: &rc_proto.Value{Val: &rc_proto.Value_StringValue{"BAR"}},
			},
			err: nil,
		},
	}
	for _, tc := range testCases {
		var err error
		tempdir := t.TempDir()
		path := filepath.Join(tempdir, tc.protoPath)
		if err = os.MkdirAll(filepath.Dir(path), 0755); err != nil {
			t.Fatal(err)
		}
		if err = os.WriteFile(path, tc.data, 0644); err != nil {
			t.Fatal(err)
		}
		actual := FlagValueFactory(path)
		tc.assertProtoEqual(t, &tc.expected, &actual.proto)
	}
}
