// Copyright 2016 Google Inc. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package config

import (
	"os/exec"
	"path/filepath"
	"strings"

	"android/soong/android"
)

var (
	darwinCflags = []string{
		"-fdiagnostics-color",

		"-fPIC",
		"-funwind-tables",

		// Workaround differences in inttypes.h between host and target.
		//See bug 12708004.
		"-D__STDC_FORMAT_MACROS",
		"-D__STDC_CONSTANT_MACROS",

		"-isysroot ${macSdkRoot}",
		"-mmacosx-version-min=${macMinVersion}",
		"-DMACOSX_DEPLOYMENT_TARGET=${macMinVersion}",

		"-m64",
	}

	darwinLdflags = []string{
		"-isysroot ${macSdkRoot}",
		"-Wl,-syslibroot,${macSdkRoot}",
		"-mmacosx-version-min=${macMinVersion}",
		"-m64",
	}

	darwinClangCflags = append(ClangFilterUnknownCflags(darwinCflags), []string{
		"-integrated-as",
		"-fstack-protector-strong",
	}...)

	darwinClangLdflags = ClangFilterUnknownCflags(darwinLdflags)

	darwinClangLldflags = ClangFilterUnknownLldflags(darwinClangLdflags)

	darwinSupportedSdkVersions = []string{
		"10.10",
		"10.11",
		"10.12",
		"10.13",
		"10.14",
	}

	darwinAvailableLibraries = append(
		addPrefix([]string{
			"c",
			"dl",
			"m",
			"ncurses",
			"objc",
			"pthread",
		}, "-l"),
		"-framework AppKit",
		"-framework CoreFoundation",
		"-framework Foundation",
		"-framework IOKit",
		"-framework Security",
	)
)

const (
	darwinGccVersion = "4.2.1"
)

func init() {
	pctx.VariableFunc("macSdkPath", func(ctx android.PackageVarContext) string {
		xcodeselect := ctx.Config().HostSystemTool("xcode-select")
		bytes, err := exec.Command(xcodeselect, "--print-path").Output()
		if err != nil {
			ctx.Errorf("xcode-select failed with: %q", err.Error())
		}
		return strings.TrimSpace(string(bytes))
	})
	pctx.VariableFunc("macSdkRoot", func(ctx android.PackageVarContext) string {
		return xcrunSdk(ctx, "--show-sdk-path")
	})
	pctx.StaticVariable("macMinVersion", "10.8")
	pctx.VariableFunc("MacArPath", func(ctx android.PackageVarContext) string {
		return xcrun(ctx, "--find", "ar")
	})

	pctx.VariableFunc("MacStripPath", func(ctx android.PackageVarContext) string {
		return xcrun(ctx, "--find", "strip")
	})

	pctx.VariableFunc("MacToolPath", func(ctx android.PackageVarContext) string {
		return filepath.Dir(xcrun(ctx, "--find", "ld"))
	})

	pctx.StaticVariable("DarwinGccVersion", darwinGccVersion)
	pctx.SourcePathVariable("DarwinGccRoot",
		"prebuilts/gcc/${HostPrebuiltTag}/host/i686-apple-darwin-${DarwinGccVersion}")

	pctx.StaticVariable("DarwinGccTriple", "i686-apple-darwin11")

	pctx.StaticVariable("DarwinCflags", strings.Join(darwinCflags, " "))
	pctx.StaticVariable("DarwinLdflags", strings.Join(darwinLdflags, " "))

	pctx.StaticVariable("DarwinClangCflags", strings.Join(darwinClangCflags, " "))
	pctx.StaticVariable("DarwinClangLdflags", strings.Join(darwinClangLdflags, " "))
	pctx.StaticVariable("DarwinClangLldflags", strings.Join(darwinClangLldflags, " "))

	pctx.StaticVariable("DarwinYasmFlags", "-f macho -m amd64")
}

func xcrun(ctx android.PackageVarContext, args ...string) string {
	xcrun := ctx.Config().HostSystemTool("xcrun")
	bytes, err := exec.Command(xcrun, args...).Output()
	if err != nil {
		ctx.Errorf("xcrun failed with: %q", err.Error())
	}
	return strings.TrimSpace(string(bytes))
}

func xcrunSdk(ctx android.PackageVarContext, arg string) string {
	xcrun := ctx.Config().HostSystemTool("xcrun")
	if selected := ctx.Config().Getenv("MAC_SDK_VERSION"); selected != "" {
		if !inList(selected, darwinSupportedSdkVersions) {
			ctx.Errorf("MAC_SDK_VERSION %s isn't supported: %q", selected, darwinSupportedSdkVersions)
			return ""
		}

		bytes, err := exec.Command(xcrun, "--sdk", "macosx"+selected, arg).Output()
		if err != nil {
			ctx.Errorf("MAC_SDK_VERSION %s is not installed", selected)
		}
		return strings.TrimSpace(string(bytes))
	}

	for _, sdk := range darwinSupportedSdkVersions {
		bytes, err := exec.Command(xcrun, "--sdk", "macosx"+sdk, arg).Output()
		if err == nil {
			return strings.TrimSpace(string(bytes))
		}
	}
	ctx.Errorf("Could not find a supported mac sdk: %q", darwinSupportedSdkVersions)
	return ""
}

type toolchainDarwin struct {
	cFlags, ldFlags string
	toolchain64Bit
}

func (t *toolchainDarwin) Name() string {
	return "x86_64"
}

func (t *toolchainDarwin) GccRoot() string {
	return "${config.DarwinGccRoot}"
}

func (t *toolchainDarwin) GccTriple() string {
	return "${config.DarwinGccTriple}"
}

func (t *toolchainDarwin) GccVersion() string {
	return darwinGccVersion
}

func (t *toolchainDarwin) Cflags() string {
	return "${config.DarwinCflags}"
}

func (t *toolchainDarwin) Cppflags() string {
	return ""
}

func (t *toolchainDarwin) Ldflags() string {
	return "${config.DarwinLdflags}"
}

func (t *toolchainDarwin) IncludeFlags() string {
	return ""
}

func (t *toolchainDarwin) ClangTriple() string {
	return "x86_64-apple-darwin"
}

func (t *toolchainDarwin) ClangCflags() string {
	return "${config.DarwinClangCflags}"
}

func (t *toolchainDarwin) ClangCppflags() string {
	return ""
}

func (t *toolchainDarwin) ClangLdflags() string {
	return "${config.DarwinClangLdflags}"
}

func (t *toolchainDarwin) ClangLldflags() string {
	return "${config.DarwinClangLldflags}"
}

func (t *toolchainDarwin) YasmFlags() string {
	return "${config.DarwinYasmFlags}"
}

func (t *toolchainDarwin) ShlibSuffix() string {
	return ".dylib"
}

func (t *toolchainDarwin) AvailableLibraries() []string {
	return darwinAvailableLibraries
}

func (t *toolchainDarwin) Bionic() bool {
	return false
}

func (t *toolchainDarwin) ToolPath() string {
	return "${config.MacToolPath}"
}

var toolchainDarwinSingleton Toolchain = &toolchainDarwin{}

func darwinToolchainFactory(arch android.Arch) Toolchain {
	return toolchainDarwinSingleton
}

func init() {
	registerToolchainFactory(android.Darwin, android.X86_64, darwinToolchainFactory)
}
