/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <math.h>

#include <fenv.h>

#include <gtest/gtest.h>

#if defined(__BIONIC__)
typedef struct {
  double expected;
  double call_data;
} cos_intel_data_t;

static cos_intel_data_t g_cos_intel_data[] = {
  { // Entry 0
    0x1.c1a27ae836f128000000000000504e9bp-1,
    0x1.feb1f7920e248p-2
  },
  { // Entry 1
    0x1.c1a27ae836f128000000000000504e9bp-1,
    -0x1.feb1f7920e248p-2
  },
  { // Entry 2
    0x1.78daf01036d0cfffffffffffff9890d6p-1,
    0x1.7cb7648526f99p-1
  },
  { // Entry 3
    0x1.78daf01036d0cfffffffffffff9890d6p-1,
    -0x1.7cb7648526f99p-1
  },
  { // Entry 4
    0x1.ff8eb6a91ecb000000000000001f8f56p-1,
    0x1.549ec0c0c5afap-5
  },
  { // Entry 5
    0x1.ff8eb6a91ecb000000000000001f8f56p-1,
    -0x1.549ec0c0c5afap-5
  },
  { // Entry 6
    0x1.fed0476fc75c9800000000000020b13ep-1,
    0x1.16e534ee36580p-4
  },
  { // Entry 7
    0x1.fed0476fc75c9800000000000020b13ep-1,
    -0x1.16e534ee36580p-4
  },
  { // Entry 8
    0x1.f10fc61e2c78efffffffffffff87666ap-1,
    0x1.efeef61d39ac2p-3
  },
  { // Entry 9
    0x1.f10fc61e2c78efffffffffffff87666ap-1,
    -0x1.efeef61d39ac2p-3
  },
  { // Entry 10
    0x1.434a3645be2087ffffffffffff72d0aep-1,
    0x1.c65a170474549p-1
  },
  { // Entry 11
    0x1.434a3645be2087ffffffffffff72d0aep-1,
    -0x1.c65a170474549p-1
  },
  { // Entry 12
    0x1.337fc5b072c52800000000000017fe77p-3,
    0x1.6b8a6273d7c21p0
  },
  { // Entry 13
    0x1.337fc5b072c52800000000000017fe77p-3,
    -0x1.6b8a6273d7c21p0
  },
  { // Entry 14
    0x1.efa7cddb128fb8004356877b74ee8abdp-1,
    -0x1.036f4ba7e90aap-2
  },
  { // Entry 15
    0x1.efa7cddb128fb8004356877b74ee8abdp-1,
    0x1.036f4ba7e90aap-2
  },
  { // Entry 16
    0x1.fffffffffffffffb5117fee2eda7f13ep-1,
    -0x1.1500766c9df20p-31
  },
  { // Entry 17
    0x1.fffffffffffffffb5117fee2eda7f13ep-1,
    0x1.1500766c9df20p-31
  },
  { // Entry 18
    0x1.ec231802917bdffa627ab6a59abe3f7dp-1,
    -0x1.1e2a1563e068ep-2
  },
  { // Entry 19
    0x1.ec231802917bdffa627ab6a59abe3f7dp-1,
    0x1.1e2a1563e068ep-2
  },
  { // Entry 20
    0x1.dc044ac92b7fb9007913c87db76fa49fp-8,
    -0x1.2115aa73f8d05p5
  },
  { // Entry 21
    0x1.dc044ac92b7fb9007913c87db76fa49fp-8,
    0x1.2115aa73f8d05p5
  },
  { // Entry 22
    0x1.d1fa67c50dd527f6e9abbb0bd0664ab9p-4,
    -0x1.34e3bcdf8f69ap2
  },
  { // Entry 23
    0x1.d1fa67c50dd527f6e9abbb0bd0664ab9p-4,
    0x1.34e3bcdf8f69ap2
  },
  { // Entry 24
    0x1.e2f8d19fb8db88056dcddd76f5a05c20p-2,
    -0x1.380000000000bp7
  },
  { // Entry 25
    0x1.e2f8d19fb8db88056dcddd76f5a05c20p-2,
    0x1.380000000000bp7
  },
  { // Entry 26
    0x1.8da9c90c3eda17f5bd708cfc414ff1b8p-1,
    -0x1.440000004p6
  },
  { // Entry 27
    0x1.8da9c90c3eda17f5bd708cfc414ff1b8p-1,
    0x1.440000004p6
  },
  { // Entry 28
    0x1.b59b320603f837ff015e2a961e75946ep-1,
    -0x1.550c8ee67a4c4p29
  },
  { // Entry 29
    0x1.b59b320603f837ff015e2a961e75946ep-1,
    0x1.550c8ee67a4c4p29
  },
  { // Entry 30
    0x1.ffffff7af6c887e5f3186371a17dd81fp-1,
    -0x1.711789fdb2e8ap-13
  },
  { // Entry 31
    0x1.ffffff7af6c887e5f3186371a17dd81fp-1,
    0x1.711789fdb2e8ap-13
  },
  { // Entry 32
    0x1.c1b68ebb0b4fe82e8e99f9d425ee901ap-2,
    -0x1.77e000002p8
  },
  { // Entry 33
    0x1.c1b68ebb0b4fe82e8e99f9d425ee901ap-2,
    0x1.77e000002p8
  },
  { // Entry 34
    0x1.1161e1dad76dbfd55f08c4a4a90bd547p-4,
    -0x1.8106561931b43p0
  },
  { // Entry 35
    0x1.1161e1dad76dbfd55f08c4a4a90bd547p-4,
    0x1.8106561931b43p0
  },
  { // Entry 36
    0x1.f828c3226b3d77c374e789d7216a85b3p-5,
    -0x1.825be2461cad4p0
  },
  { // Entry 37
    0x1.f828c3226b3d77c374e789d7216a85b3p-5,
    0x1.825be2461cad4p0
  },
  { // Entry 38
    0x1.f2990d742e9fa80dd19483a2c0de428bp-5,
    -0x1.8288755803b08p0
  },
  { // Entry 39
    0x1.f2990d742e9fa80dd19483a2c0de428bp-5,
    0x1.8288755803b08p0
  },
  { // Entry 40
    -0x1.ff150dda7524ce678ab97ee460e8e961p-1,
    -0x1.8a75701f4ccd3p1
  },
  { // Entry 41
    -0x1.ff150dda7524ce678ab97ee460e8e961p-1,
    0x1.8a75701f4ccd3p1
  },
  { // Entry 42
    0x1.015c47c32b574802eb81deb0db46c207p-1,
    -0x1.b389316f37f37p3
  },
  { // Entry 43
    0x1.015c47c32b574802eb81deb0db46c207p-1,
    0x1.b389316f37f37p3
  },
  { // Entry 44
    0x1.d681a366a05347fa56b75d845b33c7bbp-1,
    -0x1.c602c465d7d27p6
  },
  { // Entry 45
    0x1.d681a366a05347fa56b75d845b33c7bbp-1,
    0x1.c602c465d7d27p6
  },
  { // Entry 46
    -0x1.84e896c7543d54c16d7be346e0f40017p-1,
    -0x1.cfb81fe69664cp4
  },
  { // Entry 47
    -0x1.84e896c7543d54c16d7be346e0f40017p-1,
    0x1.cfb81fe69664cp4
  },
  { // Entry 48
    0x1.fc5dcfddd54c07f1574ce114c1e5effep-1,
    -0x1.d08f2d86b12c6p13
  },
  { // Entry 49
    0x1.fc5dcfddd54c07f1574ce114c1e5effep-1,
    0x1.d08f2d86b12c6p13
  },
  { // Entry 50
    0x1.fe83235fbe015d6c361556ee5a0e2294p-3,
    -0x1.de13f0943c494p99
  },
  { // Entry 51
    0x1.fe83235fbe015d6c361556ee5a0e2294p-3,
    0x1.de13f0943c494p99
  },
  { // Entry 52
    -0x1.720321239ec4f38d55b40d8445099c2ap-1,
    -0x1.de3c1f1285e8bp3
  },
  { // Entry 53
    -0x1.720321239ec4f38d55b40d8445099c2ap-1,
    0x1.de3c1f1285e8bp3
  },
  { // Entry 54
    0x1.f7143c8bba406d6f834a3bca5e9c6a58p-4,
    -0x1.fffffffffff7fp1023
  },
  { // Entry 55
    0x1.f7143c8bba406d6f834a3bca5e9c6a58p-4,
    0x1.fffffffffff7fp1023
  },
  { // Entry 56
    -0x1.fffe62ecfab753c071b2680e1e26bbcep-1,
    -0x1.fffffffffffffp1023
  },
  { // Entry 57
    -0x1.fffe62ecfab753c071b2680e1e26bbcep-1,
    0x1.fffffffffffffp1023
  },
  { // Entry 58
    0x1.055e457ac122766a7a343ea4ac739b2ep-5,
    0x1.0000000000001p51
  },
  { // Entry 59
    0x1.055e457ac122766a7a343ea4ac739b2ep-5,
    -0x1.0000000000001p51
  },
  { // Entry 60
    0x1.c1528065b7d4e2d84d640301800ddd81p-1,
    0x1.0000000000003p-1
  },
  { // Entry 61
    0x1.c1528065b7d4e2d84d640301800ddd81p-1,
    -0x1.0000000000003p-1
  },
  { // Entry 62
    0x1.fffffffffffffffeffffffffffffa0p-1,
    0x1.0000000000003p-32
  },
  { // Entry 63
    0x1.fffffffffffffffeffffffffffffa0p-1,
    -0x1.0000000000003p-32
  },
  { // Entry 64
    0x1.fffea444bc05e0b2ec693f7226d056b8p-1,
    0x1.0000000000020p150
  },
  { // Entry 65
    0x1.fffea444bc05e0b2ec693f7226d056b8p-1,
    -0x1.0000000000020p150
  },
  { // Entry 66
    -0x1.ebddee876f4338cc885394d1039acfe1p-1,
    0x1.0000000000038p380
  },
  { // Entry 67
    -0x1.ebddee876f4338cc885394d1039acfe1p-1,
    -0x1.0000000000038p380
  },
  { // Entry 68
    -0x1.f2ffc51dc69681ded33362645b4e8513p-1,
    0x1.0000000000118p380
  },
  { // Entry 69
    -0x1.f2ffc51dc69681ded33362645b4e8513p-1,
    -0x1.0000000000118p380
  },
  { // Entry 70
    -0x1.f8fbb4d358b207f4f81c36cf21c4af97p-1,
    0x1.00000000003ffp641
  },
  { // Entry 71
    -0x1.f8fbb4d358b207f4f81c36cf21c4af97p-1,
    -0x1.00000000003ffp641
  },
  { // Entry 72
    -0x1.aa2265753e6687fde76269ee92a784b0p-2,
    0x1.00000000010p1
  },
  { // Entry 73
    -0x1.aa2265753e6687fde76269ee92a784b0p-2,
    -0x1.00000000010p1
  },
  { // Entry 74
    0x1.fd1242c25994d0cc5daf405fa510e539p-1,
    0x1.000000008p452
  },
  { // Entry 75
    0x1.fd1242c25994d0cc5daf405fa510e539p-1,
    -0x1.000000008p452
  },
  { // Entry 76
    0x1.fffc0001554da8000048828de97acccbp-1,
    0x1.00000000effafp-7
  },
  { // Entry 77
    0x1.fffc0001554da8000048828de97acccbp-1,
    -0x1.00000000effafp-7
  },
  { // Entry 78
    0x1.14a27f292552280000dd30fbf6b38481p-1,
    0x1.00000114fefe2p0
  },
  { // Entry 79
    0x1.14a27f292552280000dd30fbf6b38481p-1,
    -0x1.00000114fefe2p0
  },
  { // Entry 80
    0x1.bf81e0269c59c7ff753ca32147666c56p-3,
    0x1.0000070p40
  },
  { // Entry 81
    0x1.bf81e0269c59c7ff753ca32147666c56p-3,
    -0x1.0000070p40
  },
  { // Entry 82
    0x1.14a26ed1960d67ffffdf8d13b43fa2b4p-1,
    0x1.00000acadb3d3p0
  },
  { // Entry 83
    0x1.14a26ed1960d67ffffdf8d13b43fa2b4p-1,
    -0x1.00000acadb3d3p0
  },
  { // Entry 84
    0x1.ffffffffbfffe7ffff155655559a7d2dp-1,
    0x1.00003p-17
  },
  { // Entry 85
    0x1.ffffffffbfffe7ffff155655559a7d2dp-1,
    -0x1.00003p-17
  },
  { // Entry 86
    0x1.ffffffffeffff7ffff15f96ad3b29d94p-1,
    0x1.00003ffffffaep-18
  },
  { // Entry 87
    0x1.ffffffffeffff7ffff15f96ad3b29d94p-1,
    -0x1.00003ffffffaep-18
  },
  { // Entry 88
    0x1.ffffffffeffff7ffff15576aab329f45p-1,
    0x1.00003ffffffffp-18
  },
  { // Entry 89
    0x1.ffffffffeffff7ffff15576aab329f45p-1,
    -0x1.00003ffffffffp-18
  },
  { // Entry 90
    -0x1.dab7efeb35bad7fbdcb0efedb662097cp-2,
    0x1.00007ffffdea0p41
  },
  { // Entry 91
    -0x1.dab7efeb35bad7fbdcb0efedb662097cp-2,
    -0x1.00007ffffdea0p41
  },
  { // Entry 92
    0x1.fffffffffbfff8000001595aa9afffcfp-1,
    0x1.0000ffff8p-19
  },
  { // Entry 93
    0x1.fffffffffbfff8000001595aa9afffcfp-1,
    -0x1.0000ffff8p-19
  },
  { // Entry 94
    0x1.fffffffffefff80000004b36423050f6p-1,
    0x1.0003fff800051p-20
  },
  { // Entry 95
    0x1.fffffffffefff80000004b36423050f6p-1,
    -0x1.0003fff800051p-20
  },
  { // Entry 96
    0x1.fffffffffefff800000042961fb095f7p-1,
    0x1.0003fff800096p-20
  },
  { // Entry 97
    0x1.fffffffffefff800000042961fb095f7p-1,
    -0x1.0003fff800096p-20
  },
  { // Entry 98
    -0x1.fbf2b71a23a577fddd081d30bc24917fp-2,
    0x1.000fd20p334
  },
  { // Entry 99
    -0x1.fbf2b71a23a577fddd081d30bc24917fp-2,
    -0x1.000fd20p334
  },
  { // Entry 100
    0x1.fccc87eae7736d8d8a4776314f6b0ba7p-5,
    0x1.003p514
  },
  { // Entry 101
    0x1.fccc87eae7736d8d8a4776314f6b0ba7p-5,
    -0x1.003p514
  },
  { // Entry 102
    -0x1.a43f40d92b7ed7eef0fe6d1799442cfap-7,
    0x1.00600000015f4p41
  },
  { // Entry 103
    -0x1.a43f40d92b7ed7eef0fe6d1799442cfap-7,
    -0x1.00600000015f4p41
  },
  { // Entry 104
    0x1.c11cc38f40ab2827dd0ff8c6f3dd1dafp-1,
    0x1.007p-1
  },
  { // Entry 105
    0x1.c11cc38f40ab2827dd0ff8c6f3dd1dafp-1,
    -0x1.007p-1
  },
  { // Entry 106
    0x1.ffffffffffbfc7f3c0000157ac331d0cp-1,
    0x1.007p-21
  },
  { // Entry 107
    0x1.ffffffffffbfc7f3c0000157ac331d0cp-1,
    -0x1.007p-21
  },
  { // Entry 108
    0x1.e9ba98231f7346e0566694e1484a5b98p-8,
    0x1.00cp40
  },
  { // Entry 109
    0x1.e9ba98231f7346e0566694e1484a5b98p-8,
    -0x1.00cp40
  },
  { // Entry 110
    0x1.fefdf48ed649cd087e3e801219edf804p-1,
    0x1.011p-4
  },
  { // Entry 111
    0x1.fefdf48ed649cd087e3e801219edf804p-1,
    -0x1.011p-4
  },
  { // Entry 112
    -0x1.ffc16a0f12ff271ced72fc659ddcd382p-1,
    0x1.011p996
  },
  { // Entry 113
    -0x1.ffc16a0f12ff271ced72fc659ddcd382p-1,
    -0x1.011p996
  },
  { // Entry 114
    0x1.efd5b61a30a387fff1400519845fcb1ep-1,
    0x1.020p-2
  },
  { // Entry 115
    0x1.efd5b61a30a387fff1400519845fcb1ep-1,
    -0x1.020p-2
  },
  { // Entry 116
    0x1.c97b8161dc50a06cd4801deeb8e0715bp-2,
    0x1.0204260c18307p59
  },
  { // Entry 117
    0x1.c97b8161dc50a06cd4801deeb8e0715bp-2,
    -0x1.0204260c18307p59
  },
  { // Entry 118
    -0x1.bf26a3c9b9fbe83488449343f08b08d7p-2,
    0x1.02e78a321155ep1
  },
  { // Entry 119
    -0x1.bf26a3c9b9fbe83488449343f08b08d7p-2,
    -0x1.02e78a321155ep1
  },
  { // Entry 120
    0x1.fef806b1f84e4dfd5f71307a3c3eb602p-1,
    0x1.040p-4
  },
  { // Entry 121
    0x1.fef806b1f84e4dfd5f71307a3c3eb602p-1,
    -0x1.040p-4
  },
  { // Entry 122
    0x1.fe851fbf87d176fb4c69295c7f928402p-1,
    0x1.04bde8bb80258p98
  },
  { // Entry 123
    0x1.fe851fbf87d176fb4c69295c7f928402p-1,
    -0x1.04bde8bb80258p98
  },
  { // Entry 124
    0x1.70f6a51da8efec398e74169e771386eap-1,
    0x1.077e749e37ceep236
  },
  { // Entry 125
    0x1.70f6a51da8efec398e74169e771386eap-1,
    -0x1.077e749e37ceep236
  },
  { // Entry 126
    0x1.6b408c856bda57fe5829a58361fea112p-3,
    0x1.07f80p300
  },
  { // Entry 127
    0x1.6b408c856bda57fe5829a58361fea112p-3,
    -0x1.07f80p300
  },
  { // Entry 128
    0x1.2b2f965ae40fc7ffffbca0b7eaec5919p-1,
    0x1.07f9bea1b3546p27
  },
  { // Entry 129
    0x1.2b2f965ae40fc7ffffbca0b7eaec5919p-1,
    -0x1.07f9bea1b3546p27
  },
  { // Entry 130
    -0x1.4eed2f3fc76a77f803d8911eeddcc5e8p-1,
    0x1.090d18372f2d5p4
  },
  { // Entry 131
    -0x1.4eed2f3fc76a77f803d8911eeddcc5e8p-1,
    -0x1.090d18372f2d5p4
  },
  { // Entry 132
    0x1.fba59aecee5000315cee323079ed1427p-1,
    0x1.0b4p-3
  },
  { // Entry 133
    0x1.fba59aecee5000315cee323079ed1427p-1,
    -0x1.0b4p-3
  },
  { // Entry 134
    0x1.fffd1bcda7a7d5510aa3d0ed8addc785p-1,
    0x1.0c0d5c2af3c2ep346
  },
  { // Entry 135
    0x1.fffd1bcda7a7d5510aa3d0ed8addc785p-1,
    -0x1.0c0d5c2af3c2ep346
  },
  { // Entry 136
    -0x1.e4dfe831292867fdc1333c0a4c3f3f70p-1,
    0x1.0d30596ee91fdp216
  },
  { // Entry 137
    -0x1.e4dfe831292867fdc1333c0a4c3f3f70p-1,
    -0x1.0d30596ee91fdp216
  },
  { // Entry 138
    0x1.fb8432886a283b17acaa524e312ab1e3p-2,
    0x1.0d6p0
  },
  { // Entry 139
    0x1.fb8432886a283b17acaa524e312ab1e3p-2,
    -0x1.0d6p0
  },
  { // Entry 140
    0x1.ffffee202853ffbbc1684847019727a0p-1,
    0x1.0e9474c68831cp-10
  },
  { // Entry 141
    0x1.ffffee202853ffbbc1684847019727a0p-1,
    -0x1.0e9474c68831cp-10
  },
  { // Entry 142
    -0x1.b70d3d5584b1a83c68e254e523d37dfdp-2,
    0x1.113bae4049849p2
  },
  { // Entry 143
    -0x1.b70d3d5584b1a83c68e254e523d37dfdp-2,
    -0x1.113bae4049849p2
  },
  { // Entry 144
    0x1.fed8df58f625fdcbb1e40cf83caa2ca2p-1,
    0x1.12eb870976540p-4
  },
  { // Entry 145
    0x1.fed8df58f625fdcbb1e40cf83caa2ca2p-1,
    -0x1.12eb870976540p-4
  },
  { // Entry 146
    0x1.e536ae395dfce001457970c8aaac3b1fp-2,
    0x1.13cp0
  },
  { // Entry 147
    0x1.e536ae395dfce001457970c8aaac3b1fp-2,
    -0x1.13cp0
  },
  { // Entry 148
    0x1.fed0476fc75c9800000000000020b13ep-1,
    0x1.16e534ee36580p-4
  },
  { // Entry 149
    0x1.fed0476fc75c9800000000000020b13ep-1,
    -0x1.16e534ee36580p-4
  },
  { // Entry 150
    0x1.fb38e82e3193a7f19a17d231a2e8194ep-1,
    0x1.17fffffffea98p-3
  },
  { // Entry 151
    0x1.fb38e82e3193a7f19a17d231a2e8194ep-1,
    -0x1.17fffffffea98p-3
  },
  { // Entry 152
    0x1.fb38e82e3187fc65747ae443461eb434p-1,
    0x1.180p-3
  },
  { // Entry 153
    0x1.fb38e82e3187fc65747ae443461eb434p-1,
    -0x1.180p-3
  },
  { // Entry 154
    -0x1.e59647f1fe9c750059b9eafb88c2aceap-1,
    0x1.1a191ebbb4d7fp7
  },
  { // Entry 155
    -0x1.e59647f1fe9c750059b9eafb88c2aceap-1,
    -0x1.1a191ebbb4d7fp7
  },
  { // Entry 156
    -0x1.d0dca1f8715bd7f9ff373c763522db0ep-4,
    0x1.1da84f2b7b1d8p7
  },
  { // Entry 157
    -0x1.d0dca1f8715bd7f9ff373c763522db0ep-4,
    -0x1.1da84f2b7b1d8p7
  },
  { // Entry 158
    0x1.b917ebbc30e1d8364bc3c2c10ea13539p-2,
    0x1.201e973251302p0
  },
  { // Entry 159
    0x1.b917ebbc30e1d8364bc3c2c10ea13539p-2,
    -0x1.201e973251302p0
  },
  { // Entry 160
    0x1.fffadf12ff4147f6f6f35b44b0f6cb46p-1,
    0x1.21e02p-7
  },
  { // Entry 161
    0x1.fffadf12ff4147f6f6f35b44b0f6cb46p-1,
    -0x1.21e02p-7
  },
  { // Entry 162
    -0x1.598a4dab3de597d52faa736762a3f651p-1,
    0x1.27e29a4b985bfp1
  },
  { // Entry 163
    -0x1.598a4dab3de597d52faa736762a3f651p-1,
    -0x1.27e29a4b985bfp1
  },
  { // Entry 164
    0x1.fa95c1154abf56c4dbb77bfe5e4e8f42p-1,
    0x1.2a1f28dbfb6c0p-3
  },
  { // Entry 165
    0x1.fa95c1154abf56c4dbb77bfe5e4e8f42p-1,
    -0x1.2a1f28dbfb6c0p-3
  },
  { // Entry 166
    -0x1.6412293adb7bca97731e50a86312da2dp-1,
    0x1.2b8p1
  },
  { // Entry 167
    -0x1.6412293adb7bca97731e50a86312da2dp-1,
    -0x1.2b8p1
  },
  { // Entry 168
    0x1.fffa518a7d0e77f3b431fcd0ba353095p-1,
    0x1.31199def72f4dp-7
  },
  { // Entry 169
    0x1.fffa518a7d0e77f3b431fcd0ba353095p-1,
    -0x1.31199def72f4dp-7
  },
  { // Entry 170
    0x1.f36895fe177f77fe84798264ad4f33b1p-1,
    0x1.31260e1485014p4
  },
  { // Entry 171
    0x1.f36895fe177f77fe84798264ad4f33b1p-1,
    -0x1.31260e1485014p4
  },
  { // Entry 172
    0x1.d36207b4fee167f6b3185fdd9294a233p-4,
    0x1.34e964cd103bdp2
  },
  { // Entry 173
    0x1.d36207b4fee167f6b3185fdd9294a233p-4,
    -0x1.34e964cd103bdp2
  },
  { // Entry 174
    -0x1.84a37f4fa7616caad66b4d701ab53e94p-1,
    0x1.37618a0ba7850p1
  },
  { // Entry 175
    -0x1.84a37f4fa7616caad66b4d701ab53e94p-1,
    -0x1.37618a0ba7850p1
  },
  { // Entry 176
    -0x1.c830bbc99e2290f9d93aa62b2ddfec97p-39,
    0x1.379704f5f1eb3p24
  },
  { // Entry 177
    -0x1.c830bbc99e2290f9d93aa62b2ddfec97p-39,
    -0x1.379704f5f1eb3p24
  },
  { // Entry 178
    0x1.b5daaa233bd4f7fcf34b6d5d0280172fp-3,
    0x1.3b61dd166d470p2
  },
  { // Entry 179
    0x1.b5daaa233bd4f7fcf34b6d5d0280172fp-3,
    -0x1.3b61dd166d470p2
  },
  { // Entry 180
    -0x1.ffd00dc4db400d962e2d255a661e3ccbp-4,
    0x1.3c011022acbd0p37
  },
  { // Entry 181
    -0x1.ffd00dc4db400d962e2d255a661e3ccbp-4,
    -0x1.3c011022acbd0p37
  },
  { // Entry 182
    -0x1.14052b4016ff57fe5750ed3426bfd758p-1,
    0x1.3e7788e900b70p727
  },
  { // Entry 183
    -0x1.14052b4016ff57fe5750ed3426bfd758p-1,
    -0x1.3e7788e900b70p727
  },
  { // Entry 184
    0x1.ffffe6a5e4197f5fc50acb189eb85c94p-1,
    0x1.423eafdcc2779p-10
  },
  { // Entry 185
    0x1.ffffe6a5e4197f5fc50acb189eb85c94p-1,
    -0x1.423eafdcc2779p-10
  },
  { // Entry 186
    -0x1.fe09fc3d16fedd7508301df68857b756p-6,
    0x1.4321828c1b538p119
  },
  { // Entry 187
    -0x1.fe09fc3d16fedd7508301df68857b756p-6,
    -0x1.4321828c1b538p119
  },
  { // Entry 188
    0x1.b685d949a27a0143c4ac9b553541ebecp-14,
    0x1.43506cb22975dp22
  },
  { // Entry 189
    0x1.b685d949a27a0143c4ac9b553541ebecp-14,
    -0x1.43506cb22975dp22
  },
  { // Entry 190
    0x1.fe398090e203bc2f6fe6830d380facc4p-1,
    0x1.439f63495786ap67
  },
  { // Entry 191
    0x1.fe398090e203bc2f6fe6830d380facc4p-1,
    -0x1.439f63495786ap67
  },
  { // Entry 192
    0x1.fe6274e000973a26c1a0053e3fdeffc7p-1,
    0x1.457538a6bd073p-4
  },
  { // Entry 193
    0x1.fe6274e000973a26c1a0053e3fdeffc7p-1,
    -0x1.457538a6bd073p-4
  },
  { // Entry 194
    0x1.09fcb69359c09d5739f1f0255d4fce83p-1,
    0x1.478fc08p43
  },
  { // Entry 195
    0x1.09fcb69359c09d5739f1f0255d4fce83p-1,
    -0x1.478fc08p43
  },
  { // Entry 196
    -0x1.20c2158511e7899f1ec379e90c296780p-9,
    0x1.48a45797cbe63p61
  },
  { // Entry 197
    -0x1.20c2158511e7899f1ec379e90c296780p-9,
    -0x1.48a45797cbe63p61
  },
  { // Entry 198
    0x1.990d17aae25307fec4ae6d4e98f30093p-1,
    0x1.4a62e0e12c173p-1
  },
  { // Entry 199
    0x1.990d17aae25307fec4ae6d4e98f30093p-1,
    -0x1.4a62e0e12c173p-1
  },
  { // Entry 200
    0x1.fdd4f1e00b38700419f96d341905d045p-3,
    0x1.4c596642a9488p9
  },
  { // Entry 201
    0x1.fdd4f1e00b38700419f96d341905d045p-3,
    -0x1.4c596642a9488p9
  },
  { // Entry 202
    0x1.fe4f141032f377fe135b13e04436c5ffp-1,
    0x1.4d0p-4
  },
  { // Entry 203
    0x1.fe4f141032f377fe135b13e04436c5ffp-1,
    -0x1.4d0p-4
  },
  { // Entry 204
    0x1.94e9f45d43c137a3f345644240a2cef0p-2,
    0x1.4f0f308p488
  },
  { // Entry 205
    0x1.94e9f45d43c137a3f345644240a2cef0p-2,
    -0x1.4f0f308p488
  },
  { // Entry 206
    0x1.9355f69ad4326404827cd5ccff8a0c69p-2,
    0x1.52f00e0p793
  },
  { // Entry 207
    0x1.9355f69ad4326404827cd5ccff8a0c69p-2,
    -0x1.52f00e0p793
  },
  { // Entry 208
    0x1.1a19be8bea10a801d0b2d09d9509c2c4p-1,
    0x1.52f06c730ec02p2
  },
  { // Entry 209
    0x1.1a19be8bea10a801d0b2d09d9509c2c4p-1,
    -0x1.52f06c730ec02p2
  },
  { // Entry 210
    0x1.385d92ec0c7337fea39493f6f2674328p-1,
    0x1.53e7d5845fe3dp220
  },
  { // Entry 211
    0x1.385d92ec0c7337fea39493f6f2674328p-1,
    -0x1.53e7d5845fe3dp220
  },
  { // Entry 212
    0x1.fffffffffe2f0f000000465e3592bffbp-1,
    0x1.590p-20
  },
  { // Entry 213
    0x1.fffffffffe2f0f000000465e3592bffbp-1,
    -0x1.590p-20
  },
  { // Entry 214
    -0x1.ffd7bc28ded91ffa200d677109251801p-1,
    0x1.592f1176f0980p86
  },
  { // Entry 215
    -0x1.ffd7bc28ded91ffa200d677109251801p-1,
    -0x1.592f1176f0980p86
  },
  { // Entry 216
    -0x1.cee28b3d79799000009ea8081244116cp-1,
    0x1.5999999dc09dcp1
  },
  { // Entry 217
    -0x1.cee28b3d79799000009ea8081244116cp-1,
    -0x1.5999999dc09dcp1
  },
  { // Entry 218
    0x1.c1f1eb08c26040a32800f087952a383cp-1,
    0x1.5bea010p468
  },
  { // Entry 219
    0x1.c1f1eb08c26040a32800f087952a383cp-1,
    -0x1.5bea010p468
  },
  { // Entry 220
    0x1.fffe35ab09a653e266eada6f44055214p-1,
    0x1.5cb80a6135e5ap1000
  },
  { // Entry 221
    0x1.fffe35ab09a653e266eada6f44055214p-1,
    -0x1.5cb80a6135e5ap1000
  },
  { // Entry 222
    -0x1.07b85f606e75d765b1aa3bbccba6eaf8p-3,
    0x1.5d5be48730d2dp13
  },
  { // Entry 223
    -0x1.07b85f606e75d765b1aa3bbccba6eaf8p-3,
    -0x1.5d5be48730d2dp13
  },
  { // Entry 224
    0x1.ffffffffff86239c000004d581b055ebp-1,
    0x1.614p-21
  },
  { // Entry 225
    0x1.ffffffffff86239c000004d581b055ebp-1,
    -0x1.614p-21
  },
  { // Entry 226
    -0x1.dd3a806e89cf17fffff16ecd5397b486p-1,
    0x1.62adc8a660364p1
  },
  { // Entry 227
    -0x1.dd3a806e89cf17fffff16ecd5397b486p-1,
    -0x1.62adc8a660364p1
  },
  { // Entry 228
    -0x1.4308b14f4b6edc4c07ee6895f3188af3p-1,
    0x1.64ef438p142
  },
  { // Entry 229
    -0x1.4308b14f4b6edc4c07ee6895f3188af3p-1,
    -0x1.64ef438p142
  },
  { // Entry 230
    0x1.6623d2eb6add1ffc398a3c20447f9d06p-3,
    0x1.652p0
  },
  { // Entry 231
    0x1.6623d2eb6add1ffc398a3c20447f9d06p-3,
    -0x1.652p0
  },
  { // Entry 232
    0x1.fff832c50f471ff5599c23dc968eb8e4p-1,
    0x1.65865b2cb08a2p-7
  },
  { // Entry 233
    0x1.fff832c50f471ff5599c23dc968eb8e4p-1,
    -0x1.65865b2cb08a2p-7
  },
  { // Entry 234
    0x1.acc251be330228021bc31307bc81dc16p-1,
    0x1.6a937daabc20ep375
  },
  { // Entry 235
    0x1.acc251be330228021bc31307bc81dc16p-1,
    -0x1.6a937daabc20ep375
  },
  { // Entry 236
    -0x1.14ae72e6ba22ef4608875c41378eb052p-61,
    0x1.6ac5b262ca1ffp849
  },
  { // Entry 237
    -0x1.14ae72e6ba22ef4608875c41378eb052p-61,
    -0x1.6ac5b262ca1ffp849
  },
  { // Entry 238
    0x1.e0619960a11c6801e80ab0c9e25f89d0p-2,
    0x1.6f7bdef7bdef4p3
  },
  { // Entry 239
    0x1.e0619960a11c6801e80ab0c9e25f89d0p-2,
    -0x1.6f7bdef7bdef4p3
  },
  { // Entry 240
    0x1.8d23f97901a307fffd6c52adf83f0993p-1,
    0x1.739ce759ce738p200
  },
  { // Entry 241
    0x1.8d23f97901a307fffd6c52adf83f0993p-1,
    -0x1.739ce759ce738p200
  },
  { // Entry 242
    0x1.fffff78a14ba0f38a84b48dcdb63d44fp-1,
    0x1.7450c3f49d0b2p-11
  },
  { // Entry 243
    0x1.fffff78a14ba0f38a84b48dcdb63d44fp-1,
    -0x1.7450c3f49d0b2p-11
  },
  { // Entry 244
    0x1.d6f1c727fb2cb7f7e9cbee2e14246c08p-4,
    0x1.749fe53f963fdp0
  },
  { // Entry 245
    0x1.d6f1c727fb2cb7f7e9cbee2e14246c08p-4,
    -0x1.749fe53f963fdp0
  },
  { // Entry 246
    -0x1.f284b5028c1847fffffc9532fe892aadp-1,
    0x1.74af6725c6206p1
  },
  { // Entry 247
    -0x1.f284b5028c1847fffffc9532fe892aadp-1,
    -0x1.74af6725c6206p1
  },
  { // Entry 248
    -0x1.f3165a0b306b1ffcf8d11909fffba167p-1,
    0x1.7550d28ffccc4p1
  },
  { // Entry 249
    -0x1.f3165a0b306b1ffcf8d11909fffba167p-1,
    -0x1.7550d28ffccc4p1
  },
  { // Entry 250
    0x1.d66d2078ebdeb7ffedf8a48e25084c19p-1,
    0x1.775e397cd6aa0p6
  },
  { // Entry 251
    0x1.d66d2078ebdeb7ffedf8a48e25084c19p-1,
    -0x1.775e397cd6aa0p6
  },
  { // Entry 252
    0x1.7af9a13085f5382a87a541ae0b2e5965p-1,
    0x1.799302bf7f290p-1
  },
  { // Entry 253
    0x1.7af9a13085f5382a87a541ae0b2e5965p-1,
    -0x1.799302bf7f290p-1
  },
  { // Entry 254
    0x1.ffdd2fdac0c2483e7b116d6d488e7a4bp-1,
    0x1.799fffffffffdp-6
  },
  { // Entry 255
    0x1.ffdd2fdac0c2483e7b116d6d488e7a4bp-1,
    -0x1.799fffffffffdp-6
  },
  { // Entry 256
    0x1.fff744f185a737f256732ce87dbaf7a7p-1,
    0x1.7a3692ca94490p-7
  },
  { // Entry 257
    0x1.fff744f185a737f256732ce87dbaf7a7p-1,
    -0x1.7a3692ca94490p-7
  },
  { // Entry 258
    0x1.7a6b326b690fa831ee4d0dd72787fbc3p-1,
    0x1.7a66a638ac5b5p-1
  },
  { // Entry 259
    0x1.7a6b326b690fa831ee4d0dd72787fbc3p-1,
    -0x1.7a66a638ac5b5p-1
  },
  { // Entry 260
    0x1.671fdb64ffbed7e4952a9d967941e7dap-4,
    0x1.7ba65462b49a0p0
  },
  { // Entry 261
    0x1.671fdb64ffbed7e4952a9d967941e7dap-4,
    -0x1.7ba65462b49a0p0
  },
  { // Entry 262
    0x1.ffa55490f206e3b4323a02477b4149e8p-1,
    0x1.7cdf37cdf37c9p239
  },
  { // Entry 263
    0x1.ffa55490f206e3b4323a02477b4149e8p-1,
    -0x1.7cdf37cdf37c9p239
  },
  { // Entry 264
    0x1.4c5b5970a3a488233005742af5e6a95ep-4,
    0x1.7d542565f472ep0
  },
  { // Entry 265
    0x1.4c5b5970a3a488233005742af5e6a95ep-4,
    -0x1.7d542565f472ep0
  },
  { // Entry 266
    0x1.479a5667c63f57da26adf180a7eb2ffbp-4,
    0x1.7da0751649058p0
  },
  { // Entry 267
    0x1.479a5667c63f57da26adf180a7eb2ffbp-4,
    -0x1.7da0751649058p0
  },
  { // Entry 268
    0x1.fff717511dcb57f32d38f29c01379b98p-1,
    0x1.7e0ddcda6cc0dp-7
  },
  { // Entry 269
    0x1.fff717511dcb57f32d38f29c01379b98p-1,
    -0x1.7e0ddcda6cc0dp-7
  },
  { // Entry 270
    0x1.ffffffffff7077e7000006b4c7060c33p-1,
    0x1.7f6p-21
  },
  { // Entry 271
    0x1.ffffffffff7077e7000006b4c7060c33p-1,
    -0x1.7f6p-21
  },
  { // Entry 272
    -0x1.fff9e1554698017220917c3fb31188ddp-1,
    0x1.7f90117d44c74p100
  },
  { // Entry 273
    -0x1.fff9e1554698017220917c3fb31188ddp-1,
    -0x1.7f90117d44c74p100
  },
  { // Entry 274
    0x1.ffdc006bff7e984a8d5031a7152d36e5p-1,
    0x1.7ffffffffef7ap-6
  },
  { // Entry 275
    0x1.ffdc006bff7e984a8d5031a7152d36e5p-1,
    -0x1.7ffffffffef7ap-6
  },
  { // Entry 276
    0x1.ffdc006bff7e78474d66d7f25c3ac277p-1,
    0x1.7fffffffffa26p-6
  },
  { // Entry 277
    0x1.ffdc006bff7e78474d66d7f25c3ac277p-1,
    -0x1.7fffffffffa26p-6
  },
  { // Entry 278
    0x1.ffdc006bff7e6839adba2a91108d56afp-1,
    0x1.7ffffffffff80p-6
  },
  { // Entry 279
    0x1.ffdc006bff7e6839adba2a91108d56afp-1,
    -0x1.7ffffffffff80p-6
  },
  { // Entry 280
    0x1.760718ab443977fdd8f8f2221ca4db3cp-1,
    0x1.80ep-1
  },
  { // Entry 281
    0x1.760718ab443977fdd8f8f2221ca4db3cp-1,
    -0x1.80ep-1
  },
  { // Entry 282
    -0x1.fffffffffffffffffffffffffeb29d0bp-1,
    0x1.81ae0dffa3b33p959
  },
  { // Entry 283
    -0x1.fffffffffffffffffffffffffeb29d0bp-1,
    -0x1.81ae0dffa3b33p959
  },
  { // Entry 284
    -0x1.fbdc48125b34574cf1f91419e7edb75bp-1,
    0x1.81d612289c5cfp1
  },
  { // Entry 285
    -0x1.fbdc48125b34574cf1f91419e7edb75bp-1,
    -0x1.81d612289c5cfp1
  },
  { // Entry 286
    0x1.ff9e396651cc97e935b2e0ee73607d12p-5,
    0x1.8220192270a0ep0
  },
  { // Entry 287
    0x1.ff9e396651cc97e935b2e0ee73607d12p-5,
    -0x1.8220192270a0ep0
  },
  { // Entry 288
    0x1.fe2b26dddb5c882f30a885c1b291f945p-5,
    0x1.822bb780e9104p0
  },
  { // Entry 289
    0x1.fe2b26dddb5c882f30a885c1b291f945p-5,
    -0x1.822bb780e9104p0
  },
  { // Entry 290
    0x1.eb87cff7c91157a755818c01284b0e51p-5,
    0x1.82c119c4b8e49p0
  },
  { // Entry 291
    0x1.eb87cff7c91157a755818c01284b0e51p-5,
    -0x1.82c119c4b8e49p0
  },
  { // Entry 292
    0x1.eb87cff7a62b77b57a66ac275a3104c8p-5,
    0x1.82c119c4b9fc4p0
  },
  { // Entry 293
    0x1.eb87cff7a62b77b57a66ac275a3104c8p-5,
    -0x1.82c119c4b9fc4p0
  },
  { // Entry 294
    0x1.eb87cff795ab17a807864d8fb0ca43cap-5,
    0x1.82c119c4ba808p0
  },
  { // Entry 295
    0x1.eb87cff795ab17a807864d8fb0ca43cap-5,
    -0x1.82c119c4ba808p0
  },
  { // Entry 296
    0x1.de1d17ab0d6a48367523193d68568c1bp-5,
    0x1.832c9fc765270p0
  },
  { // Entry 297
    0x1.de1d17ab0d6a48367523193d68568c1bp-5,
    -0x1.832c9fc765270p0
  },
  { // Entry 298
    0x1.dc86e7bec0c44848cfc39b7f1b8204acp-5,
    0x1.833956ce7d1f9p0
  },
  { // Entry 299
    0x1.dc86e7bec0c44848cfc39b7f1b8204acp-5,
    -0x1.833956ce7d1f9p0
  },
  { // Entry 300
    0x1.db03cbb942a7a84974e904ecc896a5eep-5,
    0x1.834574eb1c099p0
  },
  { // Entry 301
    0x1.db03cbb942a7a84974e904ecc896a5eep-5,
    -0x1.834574eb1c099p0
  },
  { // Entry 302
    0x1.ce431710d15077b1e8170a532b68abdep-5,
    0x1.83aba5688e13ep0
  },
  { // Entry 303
    0x1.ce431710d15077b1e8170a532b68abdep-5,
    -0x1.83aba5688e13ep0
  },
  { // Entry 304
    0x1.cd46b3a77f6dd7ab0d312546aba7c0afp-5,
    0x1.83b38bbafd75bp0
  },
  { // Entry 305
    0x1.cd46b3a77f6dd7ab0d312546aba7c0afp-5,
    -0x1.83b38bbafd75bp0
  },
  { // Entry 306
    -0x1.ff29bc666bee6ea44d4db000e5f173bfp-1,
    0x1.86a017cb1c31cp16
  },
  { // Entry 307
    -0x1.ff29bc666bee6ea44d4db000e5f173bfp-1,
    -0x1.86a017cb1c31cp16
  },
  { // Entry 308
    -0x1.7968916e4c64630e8e3ae60276ba2d27p-2,
    0x1.8720588p392
  },
  { // Entry 309
    -0x1.7968916e4c64630e8e3ae60276ba2d27p-2,
    -0x1.8720588p392
  },
  { // Entry 310
    0x1.fb97c7e452917f59890dfb47e7c9b5e3p-1,
    0x1.88a2288a22888p9
  },
  { // Entry 311
    0x1.fb97c7e452917f59890dfb47e7c9b5e3p-1,
    -0x1.88a2288a22888p9
  },
  { // Entry 312
    -0x1.ae44a5f01bf633015e5aacccab4a7f3dp-1,
    0x1.8cf013991c308p1000
  },
  { // Entry 313
    -0x1.ae44a5f01bf633015e5aacccab4a7f3dp-1,
    -0x1.8cf013991c308p1000
  },
  { // Entry 314
    0x1.d96e82f71a9dc7fd86f57480e75491e3p-1,
    0x1.9p-2
  },
  { // Entry 315
    0x1.d96e82f71a9dc7fd86f57480e75491e3p-1,
    -0x1.9p-2
  },
  { // Entry 316
    0x1.0fd9d5c093df4f57fc49463e2f3f33e6p-7,
    0x1.9p0
  },
  { // Entry 317
    0x1.0fd9d5c093df4f57fc49463e2f3f33e6p-7,
    -0x1.9p0
  },
  { // Entry 318
    0x1.0fd9d5c05e5fc801a0f5cbfa33777ae4p-7,
    0x1.90000000006b0p0
  },
  { // Entry 319
    0x1.0fd9d5c05e5fc801a0f5cbfa33777ae4p-7,
    -0x1.90000000006b0p0
  },
  { // Entry 320
    0x1.bc8be725417d8800000cada42eeea04ep-1,
    0x1.900c206d44162p6
  },
  { // Entry 321
    0x1.bc8be725417d8800000cada42eeea04ep-1,
    -0x1.900c206d44162p6
  },
  { // Entry 322
    0x1.fffffffff63b67e4972d86a2f9d8209bp-1,
    0x1.900c2af7baef3p-19
  },
  { // Entry 323
    0x1.fffffffff63b67e4972d86a2f9d8209bp-1,
    -0x1.900c2af7baef3p-19
  },
  { // Entry 324
    0x1.bd464c9352d107ffffff8db9e0da71c0p-1,
    0x1.900f11bd8955dp6
  },
  { // Entry 325
    0x1.bd464c9352d107ffffff8db9e0da71c0p-1,
    -0x1.900f11bd8955dp6
  },
  { // Entry 326
    0x1.fffffda85cdd0d431e06f9aa74bc8ce0p-1,
    0x1.910b35c3253d4p100
  },
  { // Entry 327
    0x1.fffffda85cdd0d431e06f9aa74bc8ce0p-1,
    -0x1.910b35c3253d4p100
  },
  { // Entry 328
    0x1.1a62633145c06e0e6894812704419fa8p-54,
    0x1.921fb54442d18p0
  },
  { // Entry 329
    0x1.1a62633145c06e0e6894812704419fa8p-54,
    -0x1.921fb54442d18p0
  },
  { // Entry 330
    -0x1.2aeef4b9ea1ae4e64c71fccf1fd06f40p-18,
    0x1.922p0
  },
  { // Entry 331
    -0x1.2aeef4b9ea1ae4e64c71fccf1fd06f40p-18,
    -0x1.922p0
  },
  { // Entry 332
    -0x1.ffffffffd904847798c7acd46ad183cap-18,
    0x1.9220354442d18p0
  },
  { // Entry 333
    -0x1.ffffffffd904847798c7acd46ad183cap-18,
    -0x1.9220354442d18p0
  },
  { // Entry 334
    -0x1.ffffffffffffffffffff2ac8c3da72cbp-1,
    0x1.9251f93aeb59dp12
  },
  { // Entry 335
    -0x1.ffffffffffffffffffff2ac8c3da72cbp-1,
    -0x1.9251f93aeb59dp12
  },
  { // Entry 336
    0x1.ffb8c4d1f78a87fffffb23d171bd54d0p-1,
    0x1.943be221d909ap2
  },
  { // Entry 337
    0x1.ffb8c4d1f78a87fffffb23d171bd54d0p-1,
    -0x1.943be221d909ap2
  },
  { // Entry 338
    0x1.fff6011fdddab7f25c82e5384d46cc69p-1,
    0x1.94af699302875p-7
  },
  { // Entry 339
    0x1.fff6011fdddab7f25c82e5384d46cc69p-1,
    -0x1.94af699302875p-7
  },
  { // Entry 340
    0x1.d7954e7a3ee998000099470604329b68p-1,
    0x1.999999ab7b0edp-2
  },
  { // Entry 341
    0x1.d7954e7a3ee998000099470604329b68p-1,
    -0x1.999999ab7b0edp-2
  },
  { // Entry 342
    0x1.d7954e76c8e30fffff5a9bf455a06410p-1,
    0x1.999999bd4190bp-2
  },
  { // Entry 343
    0x1.d7954e76c8e30fffff5a9bf455a06410p-1,
    -0x1.999999bd4190bp-2
  },
  { // Entry 344
    0x1.fa23cfb82022440eb972c7e083933d20p-1,
    0x1.9bd0f19479a24p2
  },
  { // Entry 345
    0x1.fa23cfb82022440eb972c7e083933d20p-1,
    -0x1.9bd0f19479a24p2
  },
  { // Entry 346
    -0x1.6a09e667f3af07f2f4676b5f6f268c81p-1,
    0x1.9c55835e7e83ep8
  },
  { // Entry 347
    -0x1.6a09e667f3af07f2f4676b5f6f268c81p-1,
    -0x1.9c55835e7e83ep8
  },
  { // Entry 348
    0x1.fff59c12558097f2858f4a1326f2ddf4p-1,
    0x1.9c9942b14448dp-7
  },
  { // Entry 349
    0x1.fff59c12558097f2858f4a1326f2ddf4p-1,
    -0x1.9c9942b14448dp-7
  },
  { // Entry 350
    0x1.ffece5cab4ca5652ea2fc7ccce883304p-1,
    0x1.9d3d92485e2b5p523
  },
  { // Entry 351
    0x1.ffece5cab4ca5652ea2fc7ccce883304p-1,
    -0x1.9d3d92485e2b5p523
  },
  { // Entry 352
    -0x1.ff55301d3a780d19c28c22c82c1f5383p-5,
    0x1.a0d068341a080p1000
  },
  { // Entry 353
    -0x1.ff55301d3a780d19c28c22c82c1f5383p-5,
    -0x1.a0d068341a080p1000
  },
  { // Entry 354
    0x1.5a5615acd0dc09bf32e903149634f999p-1,
    0x1.a7ep-1
  },
  { // Entry 355
    0x1.5a5615acd0dc09bf32e903149634f999p-1,
    -0x1.a7ep-1
  },
  { // Entry 356
    0x1.766ad27a1de4fb1a5b667216bbe6bf68p-14,
    0x1.a858343863965p119
  },
  { // Entry 357
    0x1.766ad27a1de4fb1a5b667216bbe6bf68p-14,
    -0x1.a858343863965p119
  },
  { // Entry 358
    0x1.6bd4d5be7249325d8680606e6b9ea625p-1,
    0x1.ab190633d88eap3
  },
  { // Entry 359
    0x1.6bd4d5be7249325d8680606e6b9ea625p-1,
    -0x1.ab190633d88eap3
  },
  { // Entry 360
    0x1.ffffffffff4a57e64da87a5af47cfa18p-1,
    0x1.af4bd2f4bd2f0p-21
  },
  { // Entry 361
    0x1.ffffffffff4a57e64da87a5af47cfa18p-1,
    -0x1.af4bd2f4bd2f0p-21
  },
  { // Entry 362
    0x1.7ff2934ad29a74288b886124fead5842p-1,
    0x1.afa70300aee60p72
  },
  { // Entry 363
    0x1.7ff2934ad29a74288b886124fead5842p-1,
    -0x1.afa70300aee60p72
  },
  { // Entry 364
    0x1.ff866aebdce0a7fffffb6074d5199896p-1,
    0x1.b5ab427cffb4cp94
  },
  { // Entry 365
    0x1.ff866aebdce0a7fffffb6074d5199896p-1,
    -0x1.b5ab427cffb4cp94
  },
  { // Entry 366
    -0x1.f54f5227a4e83fbf939b2e96178f121dp-60,
    0x1.b951f1572eba5p23
  },
  { // Entry 367
    -0x1.f54f5227a4e83fbf939b2e96178f121dp-60,
    -0x1.b951f1572eba5p23
  },
  { // Entry 368
    0x1.fffd06d35579c7fe295dad17efbbbe97p-1,
    0x1.b96e5b96e5b91p-8
  },
  { // Entry 369
    0x1.fffd06d35579c7fe295dad17efbbbe97p-1,
    -0x1.b96e5b96e5b91p-8
  },
  { // Entry 370
    -0x1.7c4128e2aff4b2b78e147601fa658af5p-1,
    0x1.ba3b18395d17bp8
  },
  { // Entry 371
    -0x1.7c4128e2aff4b2b78e147601fa658af5p-1,
    -0x1.ba3b18395d17bp8
  },
  { // Entry 372
    -0x1.fffffffffffffffffffffffffefaff9dp-1,
    0x1.bab62ed655019p970
  },
  { // Entry 373
    -0x1.fffffffffffffffffffffffffefaff9dp-1,
    -0x1.bab62ed655019p970
  },
  { // Entry 374
    0x1.ffffff3e534467fffff37e509b7b792ep-1,
    0x1.bd55aa411ab46p-13
  },
  { // Entry 375
    0x1.ffffff3e534467fffff37e509b7b792ep-1,
    -0x1.bd55aa411ab46p-13
  },
  { // Entry 376
    -0x1.7fdb07b9f77e07ffff7207c4628d3f68p-1,
    0x1.bd616d4fe95cdp36
  },
  { // Entry 377
    -0x1.7fdb07b9f77e07ffff7207c4628d3f68p-1,
    -0x1.bd616d4fe95cdp36
  },
  { // Entry 378
    0x1.ffcf4da76222c889718239523341f4b5p-1,
    0x1.beap-6
  },
  { // Entry 379
    0x1.ffcf4da76222c889718239523341f4b5p-1,
    -0x1.beap-6
  },
  { // Entry 380
    -0x1.ddee13357ec6f7fcc9502399fccdc2f0p-1,
    0x1.c11516af585a4p1
  },
  { // Entry 381
    -0x1.ddee13357ec6f7fcc9502399fccdc2f0p-1,
    -0x1.c11516af585a4p1
  },
  { // Entry 382
    0x1.58cccec059da17d3f448a8b2b6e7c0e8p-1,
    0x1.c75e54de4c06ep2
  },
  { // Entry 383
    0x1.58cccec059da17d3f448a8b2b6e7c0e8p-1,
    -0x1.c75e54de4c06ep2
  },
  { // Entry 384
    -0x1.ffffffffffffffffffffffffffc8663ep-1,
    0x1.cb44e86bc192bp648
  },
  { // Entry 385
    -0x1.ffffffffffffffffffffffffffc8663ep-1,
    -0x1.cb44e86bc192bp648
  },
  { // Entry 386
    0x1.ffffffffffffffffffffffffff2198f9p-1,
    0x1.cb44e86bc192bp649
  },
  { // Entry 387
    0x1.ffffffffffffffffffffffffff2198f9p-1,
    -0x1.cb44e86bc192bp649
  },
  { // Entry 388
    -0x1.ca281d7fe44b07ffffd2b7d46ab5d361p-1,
    0x1.cd5a6f8762affp1
  },
  { // Entry 389
    -0x1.ca281d7fe44b07ffffd2b7d46ab5d361p-1,
    -0x1.cd5a6f8762affp1
  },
  { // Entry 390
    0x1.e80ad4fe54c71d4e604ede474cca0b19p-5,
    0x1.d0cb95f02ad77p464
  },
  { // Entry 391
    0x1.e80ad4fe54c71d4e604ede474cca0b19p-5,
    -0x1.d0cb95f02ad77p464
  },
  { // Entry 392
    0x1.0df8eb409efe37fffff925b5de2c80b6p-1,
    0x1.d31bd604903a0p2
  },
  { // Entry 393
    0x1.0df8eb409efe37fffff925b5de2c80b6p-1,
    -0x1.d31bd604903a0p2
  },
  { // Entry 394
    0x1.ff2ae968efe70ea4126849c3832c9cbdp-1,
    0x1.d32f4610180f6p-5
  },
  { // Entry 395
    0x1.ff2ae968efe70ea4126849c3832c9cbdp-1,
    -0x1.d32f4610180f6p-5
  },
  { // Entry 396
    -0x1.cec307a674d3ed2f8df47cf394aa88eap-3,
    0x1.d96e058p488
  },
  { // Entry 397
    -0x1.cec307a674d3ed2f8df47cf394aa88eap-3,
    -0x1.d96e058p488
  },
  { // Entry 398
    -0x1.ac8dbf9cdc95483577560b1814ea8895p-5,
    0x1.db0803c392b4cp15
  },
  { // Entry 399
    -0x1.ac8dbf9cdc95483577560b1814ea8895p-5,
    -0x1.db0803c392b4cp15
  },
  { // Entry 400
    -0x1.ac94870ca631684bd10b658b80cfcd42p-5,
    0x1.db0803c3ff51dp15
  },
  { // Entry 401
    -0x1.ac94870ca631684bd10b658b80cfcd42p-5,
    -0x1.db0803c3ff51dp15
  },
  { // Entry 402
    0x1.ff229073fd8b5e91d60dd095cfde5967p-1,
    0x1.dc4p-5
  },
  { // Entry 403
    0x1.ff229073fd8b5e91d60dd095cfde5967p-1,
    -0x1.dc4p-5
  },
  { // Entry 404
    0x1.ff21e5f975fffe83c2ae1c55a885f12fp-1,
    0x1.dcf73dcf73dccp-5
  },
  { // Entry 405
    0x1.ff21e5f975fffe83c2ae1c55a885f12fp-1,
    -0x1.dcf73dcf73dccp-5
  },
  { // Entry 406
    0x1.2f011326420e5002172db245fd9063e2p-1,
    0x1.dffffffffffffp-1
  },
  { // Entry 407
    0x1.2f011326420e5002172db245fd9063e2p-1,
    -0x1.dffffffffffffp-1
  },
  { // Entry 408
    0x1.f72c8e16dbc78b26afbf346185dccb48p-1,
    0x1.e123691a7c4bep26
  },
  { // Entry 409
    0x1.f72c8e16dbc78b26afbf346185dccb48p-1,
    -0x1.e123691a7c4bep26
  },
  { // Entry 410
    -0x1.4b0c6bb623f57fffff5e458203deef33p-2,
    0x1.e666666f9cf49p0
  },
  { // Entry 411
    -0x1.4b0c6bb623f57fffff5e458203deef33p-2,
    -0x1.e666666f9cf49p0
  },
  { // Entry 412
    0x1.fd74b5587588481884a92e83747f5c4ep-1,
    0x1.e83accfc50b70p995
  },
  { // Entry 413
    0x1.fd74b5587588481884a92e83747f5c4ep-1,
    -0x1.e83accfc50b70p995
  },
  { // Entry 414
    0x1.fff169b6ab7d17f43d59f6cf085accb0p-1,
    0x1.e8ep-7
  },
  { // Entry 415
    0x1.fff169b6ab7d17f43d59f6cf085accb0p-1,
    -0x1.e8ep-7
  },
  { // Entry 416
    0x1.7d39c9f1b0b3c0027a5fc9a76faee83dp-1,
    0x1.eaf5ea5317442p4
  },
  { // Entry 417
    0x1.7d39c9f1b0b3c0027a5fc9a76faee83dp-1,
    -0x1.eaf5ea5317442p4
  },
  { // Entry 418
    0x1.7f13af7081a6741660469fd60255fe49p-1,
    0x1.eb0c2b00b1b83p4
  },
  { // Entry 419
    0x1.7f13af7081a6741660469fd60255fe49p-1,
    -0x1.eb0c2b00b1b83p4
  },
  { // Entry 420
    -0x1.7ad7b88a1fe0f82b6f249c7c56dd8b5ap-1,
    0x1.ebc6b555311c4p15
  },
  { // Entry 421
    -0x1.7ad7b88a1fe0f82b6f249c7c56dd8b5ap-1,
    -0x1.ebc6b555311c4p15
  },
  { // Entry 422
    0x1.b06b2b58a2a23c98b12853415b5c83a1p-5,
    0x1.ef7bdef7bdef2p239
  },
  { // Entry 423
    0x1.b06b2b58a2a23c98b12853415b5c83a1p-5,
    -0x1.ef7bdef7bdef2p239
  },
  { // Entry 424
    0x1.fe6ded53172a6876790d3aab83a656f4p-1,
    0x1.efbbeefbbeef8p15
  },
  { // Entry 425
    0x1.fe6ded53172a6876790d3aab83a656f4p-1,
    -0x1.efbbeefbbeef8p15
  },
  { // Entry 426
    -0x1.fe2bcb87a7e158cffa2fe8d306cc7555p-1,
    0x1.f07c1f07c1ef7p239
  },
  { // Entry 427
    -0x1.fe2bcb87a7e158cffa2fe8d306cc7555p-1,
    -0x1.f07c1f07c1ef7p239
  },
  { // Entry 428
    -0x1.79d08d6b3a88282e0a0da2350464d0abp-1,
    0x1.f0f2b5e060b29p1
  },
  { // Entry 429
    -0x1.79d08d6b3a88282e0a0da2350464d0abp-1,
    -0x1.f0f2b5e060b29p1
  },
  { // Entry 430
    0x1.f0d11d321178d7ff15da48990d5983c2p-1,
    0x1.f40p-3
  },
  { // Entry 431
    0x1.f0d11d321178d7ff15da48990d5983c2p-1,
    -0x1.f40p-3
  },
  { // Entry 432
    0x1.e3ff5b15f723d7f7f7f5bb0dbce54d01p-4,
    0x1.f43d49f947e87p9
  },
  { // Entry 433
    0x1.e3ff5b15f723d7f7f7f5bb0dbce54d01p-4,
    -0x1.f43d49f947e87p9
  },
  { // Entry 434
    -0x1.6636c9f6a87a97f1cbdf708a2f1ad9bap-1,
    0x1.f7fffffffffffp1
  },
  { // Entry 435
    -0x1.6636c9f6a87a97f1cbdf708a2f1ad9bap-1,
    -0x1.f7fffffffffffp1
  },
  { // Entry 436
    0x1.ffc1be33092857ff26220f9981635bc7p-1,
    0x1.f8fffffffffffp-6
  },
  { // Entry 437
    0x1.ffc1be33092857ff26220f9981635bc7p-1,
    -0x1.f8fffffffffffp-6
  },
  { // Entry 438
    0x1.ffc1be33092857fb344affdd93d043a7p-1,
    0x1.f90p-6
  },
  { // Entry 439
    0x1.ffc1be33092857fb344affdd93d043a7p-1,
    -0x1.f90p-6
  },
  { // Entry 440
    -0x1.fffffffcab0d58220669dcfa421ccfa6p-1,
    0x1.fa0236523ce54p344
  },
  { // Entry 441
    -0x1.fffffffcab0d58220669dcfa421ccfa6p-1,
    -0x1.fa0236523ce54p344
  },
  { // Entry 442
    0x1.fc0d98ace2308800000212788a794eacp-1,
    0x1.fceab54d37da0p-4
  },
  { // Entry 443
    0x1.fc0d98ace2308800000212788a794eacp-1,
    -0x1.fceab54d37da0p-4
  },
  { // Entry 444
    -0x1.9589bca128b917fe59692a738c3791c9p-4,
    0x1.fd0072fffffffp2
  },
  { // Entry 445
    -0x1.9589bca128b917fe59692a738c3791c9p-4,
    -0x1.fd0072fffffffp2
  },
  { // Entry 446
    -0x1.4d304b07fc897cf1ade54fe97db7c8bdp-2,
    0x1.fe0f827673422p62
  },
  { // Entry 447
    -0x1.4d304b07fc897cf1ade54fe97db7c8bdp-2,
    -0x1.fe0f827673422p62
  },
  { // Entry 448
    0x1.c1a27ae836f128000000000000504e9bp-1,
    0x1.feb1f7920e248p-2
  },
  { // Entry 449
    0x1.c1a27ae836f128000000000000504e9bp-1,
    -0x1.feb1f7920e248p-2
  },
  { // Entry 450
    -0x1.936b64e955978d15aacfddf5821c6281p-1,
    0x1.feeffffffffc6p995
  },
  { // Entry 451
    -0x1.936b64e955978d15aacfddf5821c6281p-1,
    -0x1.feeffffffffc6p995
  },
  { // Entry 452
    0x1.fff007147ea577fb02130c68b335ef45p-1,
    0x1.ff8ffffffffffp-7
  },
  { // Entry 453
    0x1.fff007147ea577fb02130c68b335ef45p-1,
    -0x1.ff8ffffffffffp-7
  },
  { // Entry 454
    0x1.ffffc01bfe442b09cbec19f68af8fbf8p-1,
    0x1.ff8ffffffffffp-10
  },
  { // Entry 455
    0x1.ffffc01bfe442b09cbec19f68af8fbf8p-1,
    -0x1.ff8ffffffffffp-10
  },
  { // Entry 456
    0x1.7cc9fb75317ae93bf5ddee0e8b9c83cep-1,
    0x1.ff8ffffffffffp870
  },
  { // Entry 457
    0x1.7cc9fb75317ae93bf5ddee0e8b9c83cep-1,
    -0x1.ff8ffffffffffp870
  },
  { // Entry 458
    0x1.d6aea48015588e71983142804227fd84p-1,
    0x1.ffcfff8p19
  },
  { // Entry 459
    0x1.d6aea48015588e71983142804227fd84p-1,
    -0x1.ffcfff8p19
  },
  { // Entry 460
    -0x1.6a9972eee19badf9e34d36b0d1202091p-2,
    0x1.ffcfff8p365
  },
  { // Entry 461
    -0x1.6a9972eee19badf9e34d36b0d1202091p-2,
    -0x1.ffcfff8p365
  },
  { // Entry 462
    -0x1.3aaa15f7544b691a43e1fa1a639bdfc2p-1,
    0x1.ffcffffffff6cp720
  },
  { // Entry 463
    -0x1.3aaa15f7544b691a43e1fa1a639bdfc2p-1,
    -0x1.ffcffffffff6cp720
  },
  { // Entry 464
    0x1.3f164bce055c4c61b74a61f73ca73d3fp-1,
    0x1.ffcfffffffff9p320
  },
  { // Entry 465
    0x1.3f164bce055c4c61b74a61f73ca73d3fp-1,
    -0x1.ffcfffffffff9p320
  },
  { // Entry 466
    0x1.fffff002fff14d566ae8ec9d1edc3e3fp-1,
    0x1.ffcffffffffffp-11
  },
  { // Entry 467
    0x1.fffff002fff14d566ae8ec9d1edc3e3fp-1,
    -0x1.ffcffffffffffp-11
  },
  { // Entry 468
    -0x1.ffffff987f985d67944b867bff4ab857p-1,
    0x1.ffcffffffffffp405
  },
  { // Entry 469
    -0x1.ffffff987f985d67944b867bff4ab857p-1,
    -0x1.ffcffffffffffp405
  },
  { // Entry 470
    -0x1.ffff6235a25edb8c975b485c5c6f41f7p-1,
    0x1.ffcffffffffffp567
  },
  { // Entry 471
    -0x1.ffff6235a25edb8c975b485c5c6f41f7p-1,
    -0x1.ffcffffffffffp567
  },
  { // Entry 472
    0x1.fdf11ae4608b0894bab8786949aa6333p-3,
    0x1.ffefff8ffffffp16
  },
  { // Entry 473
    0x1.fdf11ae4608b0894bab8786949aa6333p-3,
    -0x1.ffefff8ffffffp16
  },
  { // Entry 474
    0x1.8f5525ab4583c064353aaad12c6cce6cp-1,
    0x1.ffeffffffffccp995
  },
  { // Entry 475
    0x1.8f5525ab4583c064353aaad12c6cce6cp-1,
    -0x1.ffeffffffffccp995
  },
  { // Entry 476
    0x1.a0af44a45c0569b72058cc34efd0e32ep-8,
    0x1.ffeffffffffffp77
  },
  { // Entry 477
    0x1.a0af44a45c0569b72058cc34efd0e32ep-8,
    -0x1.ffeffffffffffp77
  },
  { // Entry 478
    -0x1.df7546c31bf8cffef69c4859da055f33p-1,
    0x1.ffeffffffffffp122
  },
  { // Entry 479
    -0x1.df7546c31bf8cffef69c4859da055f33p-1,
    -0x1.ffeffffffffffp122
  },
  { // Entry 480
    -0x1.825a7bea27d5b1a598af6b684eb18478p-1,
    0x1.ffeffffffffffp179
  },
  { // Entry 481
    -0x1.825a7bea27d5b1a598af6b684eb18478p-1,
    -0x1.ffeffffffffffp179
  },
  { // Entry 482
    -0x1.1be2ab2078d547fff09932011fe16456p-1,
    0x1.ffeffffffffffp238
  },
  { // Entry 483
    -0x1.1be2ab2078d547fff09932011fe16456p-1,
    -0x1.ffeffffffffffp238
  },
  { // Entry 484
    -0x1.a4cc5f838f5297e0a7e749cb087c2f14p-7,
    0x1.fff0000002511p492
  },
  { // Entry 485
    -0x1.a4cc5f838f5297e0a7e749cb087c2f14p-7,
    -0x1.fff0000002511p492
  },
  { // Entry 486
    0x1.f16437d6119f89bfa73a2f14f377fd3ep-10,
    0x1.fff1fffffffffp41
  },
  { // Entry 487
    0x1.f16437d6119f89bfa73a2f14f377fd3ep-10,
    -0x1.fff1fffffffffp41
  },
  { // Entry 488
    0x1.898324c2f1cfc596e590b4a80d2508fbp-11,
    0x1.ffffc7fffffffp45
  },
  { // Entry 489
    0x1.898324c2f1cfc596e590b4a80d2508fbp-11,
    -0x1.ffffc7fffffffp45
  },
  { // Entry 490
    0x1.f0154c00688f87fcc96f14c8efb5914fp-1,
    0x1.ffffdf1ffffffp-3
  },
  { // Entry 491
    0x1.f0154c00688f87fcc96f14c8efb5914fp-1,
    -0x1.ffffdf1ffffffp-3
  },
  { // Entry 492
    0x1.ffc00157126a7d98216491df73d97cd3p-1,
    0x1.fffff8fffffffp-6
  },
  { // Entry 493
    0x1.ffc00157126a7d98216491df73d97cd3p-1,
    -0x1.fffff8fffffffp-6
  },
  { // Entry 494
    -0x1.e0d9f0f38c73f0069739e9de65191416p-2,
    0x1.fffffbfffffffp968
  },
  { // Entry 495
    -0x1.e0d9f0f38c73f0069739e9de65191416p-2,
    -0x1.fffffbfffffffp968
  },
  { // Entry 496
    0x1.fff4699dd560b5dbb88a029337b9ab86p-1,
    0x1.fffffcfffffffp40
  },
  { // Entry 497
    0x1.fff4699dd560b5dbb88a029337b9ab86p-1,
    -0x1.fffffcfffffffp40
  },
  { // Entry 498
    0x1.ff0015559f228802433732ae11942945p-1,
    0x1.ffffff0000040p-5
  },
  { // Entry 499
    0x1.ff0015559f228802433732ae11942945p-1,
    -0x1.ffffff0000040p-5
  },
  { // Entry 500
    -0x1.9c6951cccd39bf60d47db80be6fce34fp-2,
    0x1.ffffff8p119
  },
  { // Entry 501
    -0x1.9c6951cccd39bf60d47db80be6fce34fp-2,
    -0x1.ffffff8p119
  },
  { // Entry 502
    -0x1.f2c2263590034ec62522d45d2eeca285p-1,
    0x1.ffffff8p192
  },
  { // Entry 503
    -0x1.f2c2263590034ec62522d45d2eeca285p-1,
    -0x1.ffffff8p192
  },
  { // Entry 504
    0x1.c7884d6cfb5511a6b5111077fd0b1b72p-1,
    0x1.ffffff8p543
  },
  { // Entry 505
    0x1.c7884d6cfb5511a6b5111077fd0b1b72p-1,
    -0x1.ffffff8p543
  },
  { // Entry 506
    0x1.e66c79e776a1eff6b68f2d01289e08e8p-2,
    0x1.ffffffc3fffffp500
  },
  { // Entry 507
    0x1.e66c79e776a1eff6b68f2d01289e08e8p-2,
    -0x1.ffffffc3fffffp500
  },
  { // Entry 508
    0x1.c7c9a9c57c0b2009f18a6c2c07b52ea2p-3,
    0x1.ffffffe1fffffp700
  },
  { // Entry 509
    0x1.c7c9a9c57c0b2009f18a6c2c07b52ea2p-3,
    -0x1.ffffffe1fffffp700
  },
  { // Entry 510
    0x1.7bb28daf5f9ad3608dda8a16ea235cb4p-1,
    0x1.ffffffff0f0ffp400
  },
  { // Entry 511
    0x1.7bb28daf5f9ad3608dda8a16ea235cb4p-1,
    -0x1.ffffffff0f0ffp400
  },
  { // Entry 512
    0x1.fc015527d8bb37806e4976dcf7a7c98cp-1,
    0x1.ffffffff3ffffp-4
  },
  { // Entry 513
    0x1.fc015527d8bb37806e4976dcf7a7c98cp-1,
    -0x1.ffffffff3ffffp-4
  },
  { // Entry 514
    -0x1.ea5257eb66e3bffee900cd4447404c16p-1,
    0x1.ffffffff8ffffp3
  },
  { // Entry 515
    -0x1.ea5257eb66e3bffee900cd4447404c16p-1,
    -0x1.ffffffff8ffffp3
  },
  { // Entry 516
    -0x1.4eaa606dbef968000267b0375ded6872p-1,
    0x1.fffffffffbcffp1
  },
  { // Entry 517
    -0x1.4eaa606dbef968000267b0375ded6872p-1,
    -0x1.fffffffffbcffp1
  },
  { // Entry 518
    -0x1.fc9cd6b5f009482b0d5582e1c6cdf738p-1,
    0x1.fffffffffe0b5p720
  },
  { // Entry 519
    -0x1.fc9cd6b5f009482b0d5582e1c6cdf738p-1,
    -0x1.fffffffffe0b5p720
  },
  { // Entry 520
    0x1.e96ac045dd138d25741cb879b92afa48p-3,
    0x1.fffffffffe7ffp41
  },
  { // Entry 521
    0x1.e96ac045dd138d25741cb879b92afa48p-3,
    -0x1.fffffffffe7ffp41
  },
  { // Entry 522
    -0x1.fcaf39cfb94d48195d2b26060b30f822p-1,
    0x1.fffffffffee09p720
  },
  { // Entry 523
    -0x1.fcaf39cfb94d48195d2b26060b30f822p-1,
    -0x1.fffffffffee09p720
  },
  { // Entry 524
    0x1.8432232a6d1daa6ac8a94c0021e60d50p-1,
    0x1.ffffffffffdffp40
  },
  { // Entry 525
    0x1.8432232a6d1daa6ac8a94c0021e60d50p-1,
    -0x1.ffffffffffdffp40
  },
  { // Entry 526
    0x1.9e375143139d9a37b354ea33dd625cd6p-6,
    0x1.ffffffffffeffp41
  },
  { // Entry 527
    0x1.9e375143139d9a37b354ea33dd625cd6p-6,
    -0x1.ffffffffffeffp41
  },
  { // Entry 528
    0x1.fffc0001555528000049b10c26a1f539p-1,
    0x1.fffffffffff4ap-8
  },
  { // Entry 529
    0x1.fffc0001555528000049b10c26a1f539p-1,
    -0x1.fffffffffff4ap-8
  },
  { // Entry 530
    0x1.463a895c4ea5ce4e56e8f578388eed3ap-1,
    0x1.fffffffffff78p920
  },
  { // Entry 531
    0x1.463a895c4ea5ce4e56e8f578388eed3ap-1,
    -0x1.fffffffffff78p920
  },
  { // Entry 532
    0x1.3c1a48635cf380c8158d934c4d0dd87cp-1,
    0x1.fffffffffffd5p995
  },
  { // Entry 533
    0x1.3c1a48635cf380c8158d934c4d0dd87cp-1,
    -0x1.fffffffffffd5p995
  },
  { // Entry 534
    0x1.91c4e0708bd486217f5fc230f0416220p-1,
    0x1.fffffffffffe8p720
  },
  { // Entry 535
    0x1.91c4e0708bd486217f5fc230f0416220p-1,
    -0x1.fffffffffffe8p720
  },
  { // Entry 536
    -0x1.3e15cb849b5ea87bcc583f6344cbcc40p-1,
    0x1.fffffffffffebp920
  },
  { // Entry 537
    -0x1.3e15cb849b5ea87bcc583f6344cbcc40p-1,
    -0x1.fffffffffffebp920
  },
  { // Entry 538
    -0x1.816808349b80dd3c22cbe80b4c171d1fp-1,
    0x1.ffffffffffff1p245
  },
  { // Entry 539
    -0x1.816808349b80dd3c22cbe80b4c171d1fp-1,
    -0x1.ffffffffffff1p245
  },
  { // Entry 540
    0x1.4699c814c5f075bb0ed9472dfecc50a9p-1,
    0x1.ffffffffffff4p845
  },
  { // Entry 541
    0x1.4699c814c5f075bb0ed9472dfecc50a9p-1,
    -0x1.ffffffffffff4p845
  },
  { // Entry 542
    -0x1.815e92b7a2a019e74650a859968e0f29p-1,
    0x1.ffffffffffff4p1020
  },
  { // Entry 543
    -0x1.815e92b7a2a019e74650a859968e0f29p-1,
    -0x1.ffffffffffff4p1020
  },
  { // Entry 544
    -0x1.3e8d028153201ed272fc9549725fcb3fp-10,
    0x1.ffffffffffffcp45
  },
  { // Entry 545
    -0x1.3e8d028153201ed272fc9549725fcb3fp-10,
    -0x1.ffffffffffffcp45
  },
  { // Entry 546
    0x1.7d6765714c78532d3eb0f2a73c5d6126p-1,
    0x1.ffffffffffffep105
  },
  { // Entry 547
    0x1.7d6765714c78532d3eb0f2a73c5d6126p-1,
    -0x1.ffffffffffffep105
  },
  { // Entry 548
    -0x1.f869fb14d2568d67c37c90b0a038b240p-3,
    0x1.ffffffffffffep480
  },
  { // Entry 549
    -0x1.f869fb14d2568d67c37c90b0a038b240p-3,
    -0x1.ffffffffffffep480
  },
  { // Entry 550
    -0x1.80a75b369d3c3fd15b6060c6fb98f2d6p-1,
    0x1.ffffffffffffep970
  },
  { // Entry 551
    -0x1.80a75b369d3c3fd15b6060c6fb98f2d6p-1,
    -0x1.ffffffffffffep970
  },
  { // Entry 552
    -0x1.9dba69e853bd77fd883be3bb1171df55p-4,
    0x1.0000000000001p42
  },
  { // Entry 553
    -0x1.9dba69e853bd77fd883be3bb1171df55p-4,
    -0x1.0000000000001p42
  },
  { // Entry 554
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1074
  },
  { // Entry 555
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1074
  },
  { // Entry 556
    0x1.p0,
    -0.0
  },
  { // Entry 557
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1074
  },
  { // Entry 558
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1074
  },
  { // Entry 559
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0000000000001p-1022
  },
  { // Entry 560
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0000000000001p-1022
  },
  { // Entry 561
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1022
  },
  { // Entry 562
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1022
  },
  { // Entry 563
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.ffffffffffffep-1023
  },
  { // Entry 564
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.ffffffffffffep-1023
  },
  { // Entry 565
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.ffffffffffffep-1023
  },
  { // Entry 566
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.ffffffffffffep-1023
  },
  { // Entry 567
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1022
  },
  { // Entry 568
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1022
  },
  { // Entry 569
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0000000000001p-1022
  },
  { // Entry 570
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0000000000001p-1022
  },
  { // Entry 571
    0x1.ffffff5c28f5cb4c5272061281211120p-1,
    0x1.999999999999ap-13
  },
  { // Entry 572
    0x1.ffffff5c28f5cb4c5272061281211120p-1,
    -0x1.999999999999ap-13
  },
  { // Entry 573
    0x1.fffffd70a3d7960cd5695a06fdb80e74p-1,
    0x1.999999999999ap-12
  },
  { // Entry 574
    0x1.fffffd70a3d7960cd5695a06fdb80e74p-1,
    -0x1.999999999999ap-12
  },
  { // Entry 575
    0x1.fffffa3d70a69ad42b39d8696632f856p-1,
    0x1.3333333333334p-11
  },
  { // Entry 576
    0x1.fffffa3d70a69ad42b39d8696632f856p-1,
    -0x1.3333333333334p-11
  },
  { // Entry 577
    0x1.fffff5c28f64e5ec0da0a4f7f4388052p-1,
    0x1.999999999999ap-11
  },
  { // Entry 578
    0x1.fffff5c28f64e5ec0da0a4f7f4388052p-1,
    -0x1.999999999999ap-11
  },
  { // Entry 579
    0x1.fffff0000015555549f49f4d34d34ca0p-1,
    0x1.0p-10
  },
  { // Entry 580
    0x1.fffff0000015555549f49f4d34d34ca0p-1,
    -0x1.0p-10
  },
  { // Entry 581
    0x1.ffffe8f5c2bb98c7c103d2ff79f15d6ap-1,
    0x1.3333333333333p-10
  },
  { // Entry 582
    0x1.ffffe8f5c2bb98c7c103d2ff79f15d6ap-1,
    -0x1.3333333333333p-10
  },
  { // Entry 583
    0x1.ffffe0a3d75c31b26451166d6f398abdp-1,
    0x1.6666666666666p-10
  },
  { // Entry 584
    0x1.ffffe0a3d75c31b26451166d6f398abdp-1,
    -0x1.6666666666666p-10
  },
  { // Entry 585
    0x1.ffffd70a3dfc733b3331d8382b1e9df5p-1,
    0x1.9999999999999p-10
  },
  { // Entry 586
    0x1.ffffd70a3dfc733b3331d8382b1e9df5p-1,
    -0x1.9999999999999p-10
  },
  { // Entry 587
    0x1.ffffcc28f6a2823f3765b50659ecb0e2p-1,
    0x1.cccccccccccccp-10
  },
  { // Entry 588
    0x1.ffffcc28f6a2823f3765b50659ecb0e2p-1,
    -0x1.cccccccccccccp-10
  },
  { // Entry 589
    0x1.fffbcc2a6e86fef7d2af1580bd8e6699p-1,
    0x1.0666666666666p-7
  },
  { // Entry 590
    0x1.fffbcc2a6e86fef7d2af1580bd8e6699p-1,
    -0x1.0666666666666p-7
  },
  { // Entry 591
    0x1.fff30a4b6fcc1405e18fbf7335d2f789p-1,
    0x1.cccccccccccccp-7
  },
  { // Entry 592
    0x1.fff30a4b6fcc1405e18fbf7335d2f789p-1,
    -0x1.cccccccccccccp-7
  },
  { // Entry 593
    0x1.ffe57a780f38c0db37051fa8c8d60fbcp-1,
    0x1.4999999999999p-6
  },
  { // Entry 594
    0x1.ffe57a780f38c0db37051fa8c8d60fbcp-1,
    -0x1.4999999999999p-6
  },
  { // Entry 595
    0x1.ffd31cd0e1d62c05d2cded21add8bd33p-1,
    0x1.accccccccccccp-6
  },
  { // Entry 596
    0x1.ffd31cd0e1d62c05d2cded21add8bd33p-1,
    -0x1.accccccccccccp-6
  },
  { // Entry 597
    0x1.ffbbf18207542ef81390d73c3ba89c1ap-1,
    0x1.080p-5
  },
  { // Entry 598
    0x1.ffbbf18207542ef81390d73c3ba89c1ap-1,
    -0x1.080p-5
  },
  { // Entry 599
    0x1.ff9ff8c3299f54457bbaf8c12173b46bp-1,
    0x1.399999999999ap-5
  },
  { // Entry 600
    0x1.ff9ff8c3299f54457bbaf8c12173b46bp-1,
    -0x1.399999999999ap-5
  },
  { // Entry 601
    0x1.ff7f32d77c5b1c42f1660c9b6f2ef64fp-1,
    0x1.6b33333333334p-5
  },
  { // Entry 602
    0x1.ff7f32d77c5b1c42f1660c9b6f2ef64fp-1,
    -0x1.6b33333333334p-5
  },
  { // Entry 603
    0x1.ff59a00dbc40896bb5e4ac8ad293afb4p-1,
    0x1.9cccccccccccep-5
  },
  { // Entry 604
    0x1.ff59a00dbc40896bb5e4ac8ad293afb4p-1,
    -0x1.9cccccccccccep-5
  },
  { // Entry 605
    0x1.ff2f40c02e60f61d6dcfc39b6c2be087p-1,
    0x1.ce66666666666p-5
  },
  { // Entry 606
    0x1.ff2f40c02e60f61d6dcfc39b6c2be087p-1,
    -0x1.ce66666666666p-5
  },
  { // Entry 607
    0x1.8ca46c7d8975e57a1484f05c3738d83bp-1,
    0x1.5e7fc4369bdadp-1
  },
  { // Entry 608
    0x1.8ca46c7d8975e57a1484f05c3738d83bp-1,
    -0x1.5e7fc4369bdadp-1
  },
  { // Entry 609
    0x1.0b5d3802fc7991140168f294eedd7904p-2,
    0x1.4e7fc4369bdadp0
  },
  { // Entry 610
    0x1.0b5d3802fc7991140168f294eedd7904p-2,
    -0x1.4e7fc4369bdadp0
  },
  { // Entry 611
    -0x1.66b96f53323af1d7e31a7162ab18a75bp-2,
    0x1.edbfa651e9c84p0
  },
  { // Entry 612
    -0x1.66b96f53323af1d7e31a7162ab18a75bp-2,
    -0x1.edbfa651e9c84p0
  },
  { // Entry 613
    -0x1.a93554888c32fa57f22a9529a320c1cbp-1,
    0x1.467fc4369bdadp1
  },
  { // Entry 614
    -0x1.a93554888c32fa57f22a9529a320c1cbp-1,
    -0x1.467fc4369bdadp1
  },
  { // Entry 615
    -0x1.ffc00155527d2b9fda2ae89396e09727p-1,
    0x1.961fb54442d18p1
  },
  { // Entry 616
    -0x1.ffc00155527d2b9fda2ae89396e09727p-1,
    -0x1.961fb54442d18p1
  },
  { // Entry 617
    -0x1.96907c5c7c25b88e34addff1fbef66e4p-1,
    0x1.e5bfa651e9c83p1
  },
  { // Entry 618
    -0x1.96907c5c7c25b88e34addff1fbef66e4p-1,
    -0x1.e5bfa651e9c83p1
  },
  { // Entry 619
    -0x1.2a1e5a50f948cd487c5309682b110a53p-2,
    0x1.1aafcbafc85f7p2
  },
  { // Entry 620
    -0x1.2a1e5a50f948cd487c5309682b110a53p-2,
    -0x1.1aafcbafc85f7p2
  },
  { // Entry 621
    0x1.4894f695dc56bce8b273e5524f181264p-2,
    0x1.427fc4369bdadp2
  },
  { // Entry 622
    0x1.4894f695dc56bce8b273e5524f181264p-2,
    -0x1.427fc4369bdadp2
  },
  { // Entry 623
    0x1.a016ea3a692ce0c321b77f168de39122p-1,
    0x1.6a4fbcbd6f562p2
  },
  { // Entry 624
    0x1.a016ea3a692ce0c321b77f168de39122p-1,
    -0x1.6a4fbcbd6f562p2
  },
  { // Entry 625
    0x1.a30a69f5537ebc22f0870c2bd26ef284p-1,
    0x1.6af2eff0a2896p2
  },
  { // Entry 626
    0x1.a30a69f5537ebc22f0870c2bd26ef284p-1,
    -0x1.6af2eff0a2896p2
  },
  { // Entry 627
    0x1.5bd62e8b04ad5915e66242349b756e11p-2,
    0x1.43c62a9d02414p2
  },
  { // Entry 628
    0x1.5bd62e8b04ad5915e66242349b756e11p-2,
    -0x1.43c62a9d02414p2
  },
  { // Entry 629
    -0x1.0cb71f671e63410966e78d2009c0616fp-2,
    0x1.1c99654961f92p2
  },
  { // Entry 630
    -0x1.0cb71f671e63410966e78d2009c0616fp-2,
    -0x1.1c99654961f92p2
  },
  { // Entry 631
    -0x1.89d86aa8521c11b74f8b1954c08f9b36p-1,
    0x1.ead93feb8361fp1
  },
  { // Entry 632
    -0x1.89d86aa8521c11b74f8b1954c08f9b36p-1,
    -0x1.ead93feb8361fp1
  },
  { // Entry 633
    -0x1.fe51ac554a16ad8194f181085f8a17f2p-1,
    0x1.9c7fb54442d1ap1
  },
  { // Entry 634
    -0x1.fe51ac554a16ad8194f181085f8a17f2p-1,
    -0x1.9c7fb54442d1ap1
  },
  { // Entry 635
    -0x1.b97c04d08bc5d765b341a22b2c720b6fp-1,
    0x1.4e262a9d02415p1
  },
  { // Entry 636
    -0x1.b97c04d08bc5d765b341a22b2c720b6fp-1,
    -0x1.4e262a9d02415p1
  },
  { // Entry 637
    -0x1.a8ac8a3e58f6ca952390299d2e8b187fp-2,
    0x1.ff993feb83620p0
  },
  { // Entry 638
    -0x1.a8ac8a3e58f6ca952390299d2e8b187fp-2,
    -0x1.ff993feb83620p0
  },
  { // Entry 639
    0x1.77a8b9b3d254a9e39d02b3eb3e2390e7p-3,
    0x1.62e62a9d02416p0
  },
  { // Entry 640
    0x1.77a8b9b3d254a9e39d02b3eb3e2390e7p-3,
    -0x1.62e62a9d02416p0
  },
  { // Entry 641
    0x1.6e1061205dd79051c112d30a05097c61p-1,
    0x1.8c662a9d02419p-1
  },
  { // Entry 642
    0x1.6e1061205dd79051c112d30a05097c61p-1,
    -0x1.8c662a9d02419p-1
  },
  { // Entry 643
    -0x1.682f3cc3c7a08da2ce02a41cdc7bed86p-4,
    -0x1.a8aa1d11c44ffp0
  },
  { // Entry 644
    -0x1.682f3cc3c7a08da2ce02a41cdc7bed86p-4,
    0x1.a8aa1d11c44ffp0
  },
  { // Entry 645
    -0x1.e6669a270c36d4879b428ddba96cd87bp-7,
    -0x1.95ec8b9e03d54p0
  },
  { // Entry 646
    -0x1.e6669a270c36d4879b428ddba96cd87bp-7,
    0x1.95ec8b9e03d54p0
  },
  { // Entry 647
    0x1.ddd1ec25e209f1bbf7e17ef6c8450cd7p-5,
    -0x1.832efa2a435a9p0
  },
  { // Entry 648
    0x1.ddd1ec25e209f1bbf7e17ef6c8450cd7p-5,
    0x1.832efa2a435a9p0
  },
  { // Entry 649
    0x1.0cab9115640d993082a7343bb5affea2p-3,
    -0x1.707168b682dfep0
  },
  { // Entry 650
    0x1.0cab9115640d993082a7343bb5affea2p-3,
    0x1.707168b682dfep0
  },
  { // Entry 651
    0x1.a0723a95492edee5dc98394e45f96d88p-3,
    -0x1.5db3d742c2653p0
  },
  { // Entry 652
    0x1.a0723a95492edee5dc98394e45f96d88p-3,
    0x1.5db3d742c2653p0
  },
  { // Entry 653
    0x1.18fee96a1a585928a94cda7e3d916fe1p-2,
    -0x1.4af645cf01ea8p0
  },
  { // Entry 654
    0x1.18fee96a1a585928a94cda7e3d916fe1p-2,
    0x1.4af645cf01ea8p0
  },
  { // Entry 655
    0x1.6043621b13be2ff07085f8278598e566p-2,
    -0x1.3838b45b416fdp0
  },
  { // Entry 656
    0x1.6043621b13be2ff07085f8278598e566p-2,
    0x1.3838b45b416fdp0
  },
  { // Entry 657
    0x1.a5a4ccf40d9d9ba97faa4e23ecce9e3ap-2,
    -0x1.257b22e780f52p0
  },
  { // Entry 658
    0x1.a5a4ccf40d9d9ba97faa4e23ecce9e3ap-2,
    0x1.257b22e780f52p0
  },
  { // Entry 659
    0x1.e8c405f36f85b7f5d6a38dfd4a692341p-2,
    -0x1.12bd9173c07abp0
  },
  { // Entry 660
    0x1.e8c405f36f85b7f5d6a38dfd4a692341p-2,
    0x1.12bd9173c07abp0
  },
  { // Entry 661
    0x1.26976a6c4e0f86633327f1ceecb508aep-1,
    -0x1.ea5c3ed5b3850p-1
  },
  { // Entry 662
    0x1.26976a6c4e0f86633327f1ceecb508aep-1,
    0x1.ea5c3ed5b3850p-1
  },
  { // Entry 663
    0x1.3805a1882009f2843da808e959f17861p-1,
    -0x1.d4b87dab670a0p-1
  },
  { // Entry 664
    0x1.3805a1882009f2843da808e959f17861p-1,
    0x1.d4b87dab670a0p-1
  },
  { // Entry 665
    0x1.48e52e0a65bcb3cd46455c4d2338bdf2p-1,
    -0x1.bf14bc811a8f0p-1
  },
  { // Entry 666
    0x1.48e52e0a65bcb3cd46455c4d2338bdf2p-1,
    0x1.bf14bc811a8f0p-1
  },
  { // Entry 667
    0x1.592e58ea0a9eec0b357eb4e9a83b0ea5p-1,
    -0x1.a970fb56ce140p-1
  },
  { // Entry 668
    0x1.592e58ea0a9eec0b357eb4e9a83b0ea5p-1,
    0x1.a970fb56ce140p-1
  },
  { // Entry 669
    0x1.68d9afe052d1f0e9324ae876961bcdb1p-1,
    -0x1.93cd3a2c81990p-1
  },
  { // Entry 670
    0x1.68d9afe052d1f0e9324ae876961bcdb1p-1,
    0x1.93cd3a2c81990p-1
  },
  { // Entry 671
    0x1.77e008d0775e744eb16a2c4ec7184c43p-1,
    -0x1.7e297902351e0p-1
  },
  { // Entry 672
    0x1.77e008d0775e744eb16a2c4ec7184c43p-1,
    0x1.7e297902351e0p-1
  },
  { // Entry 673
    0x1.863a850e438fe029302aba0f5f127616p-1,
    -0x1.6885b7d7e8a30p-1
  },
  { // Entry 674
    0x1.863a850e438fe029302aba0f5f127616p-1,
    0x1.6885b7d7e8a30p-1
  },
  { // Entry 675
    0x1.93e2948233fce814439ed51fd2548920p-1,
    -0x1.52e1f6ad9c280p-1
  },
  { // Entry 676
    0x1.93e2948233fce814439ed51fd2548920p-1,
    0x1.52e1f6ad9c280p-1
  },
  { // Entry 677
    0x1.a0d1f8a9a791d4b5694ca68a42fe6c9bp-1,
    -0x1.3d3e35834fad0p-1
  },
  { // Entry 678
    0x1.a0d1f8a9a791d4b5694ca68a42fe6c9bp-1,
    0x1.3d3e35834fad0p-1
  },
  { // Entry 679
    0x1.bc6bd861e13de309428e00f7bef6c3ecp-1,
    -0x1.0a0b02501c799p-1
  },
  { // Entry 680
    0x1.bc6bd861e13de309428e00f7bef6c3ecp-1,
    0x1.0a0b02501c799p-1
  },
  { // Entry 681
    0x1.ca59c6fa3d9ce238a227393b6b075bc5p-1,
    -0x1.d8f7208e6b82cp-2
  },
  { // Entry 682
    0x1.ca59c6fa3d9ce238a227393b6b075bc5p-1,
    0x1.d8f7208e6b82cp-2
  },
  { // Entry 683
    0x1.d6c0b125791cffce83e32564712b78c6p-1,
    -0x1.9dd83c7c9e126p-2
  },
  { // Entry 684
    0x1.d6c0b125791cffce83e32564712b78c6p-1,
    0x1.9dd83c7c9e126p-2
  },
  { // Entry 685
    0x1.e1960261829858391645bbe12019e58ap-1,
    -0x1.62b9586ad0a20p-2
  },
  { // Entry 686
    0x1.e1960261829858391645bbe12019e58ap-1,
    0x1.62b9586ad0a20p-2
  },
  { // Entry 687
    0x1.ead07cc6356964e27a1036d2f8b158f7p-1,
    -0x1.279a74590331ap-2
  },
  { // Entry 688
    0x1.ead07cc6356964e27a1036d2f8b158f7p-1,
    0x1.279a74590331ap-2
  },
  { // Entry 689
    0x1.f26840e7b2188f7a0cc661a0ede3728bp-1,
    -0x1.d8f7208e6b829p-3
  },
  { // Entry 690
    0x1.f26840e7b2188f7a0cc661a0ede3728bp-1,
    0x1.d8f7208e6b829p-3
  },
  { // Entry 691
    0x1.f856d48db797dec0b79e1353409dc3f2p-1,
    -0x1.62b9586ad0a1ep-3
  },
  { // Entry 692
    0x1.f856d48db797dec0b79e1353409dc3f2p-1,
    0x1.62b9586ad0a1ep-3
  },
  { // Entry 693
    0x1.fc97283a424797215f8a8d1967736c9bp-1,
    -0x1.d8f7208e6b826p-4
  },
  { // Entry 694
    0x1.fc97283a424797215f8a8d1967736c9bp-1,
    0x1.d8f7208e6b826p-4
  },
  { // Entry 695
    0x1.ff259b7ab9f4f9a8cb9f1c333272e409p-1,
    -0x1.d8f7208e6b82dp-5
  },
  { // Entry 696
    0x1.ff259b7ab9f4f9a8cb9f1c333272e409p-1,
    0x1.d8f7208e6b82dp-5
  },
  { // Entry 697
    0x1.ff259b7ab9f4f9a8cb9f1c333272e409p-1,
    0x1.d8f7208e6b82dp-5
  },
  { // Entry 698
    0x1.ff259b7ab9f4f9a8cb9f1c333272e409p-1,
    -0x1.d8f7208e6b82dp-5
  },
  { // Entry 699
    0x1.fc97283a424795847294654a1d8a08edp-1,
    0x1.d8f7208e6b82dp-4
  },
  { // Entry 700
    0x1.fc97283a424795847294654a1d8a08edp-1,
    -0x1.d8f7208e6b82dp-4
  },
  { // Entry 701
    0x1.f856d48db797dbfecfa8b4cd3be44027p-1,
    0x1.62b9586ad0a22p-3
  },
  { // Entry 702
    0x1.f856d48db797dbfecfa8b4cd3be44027p-1,
    -0x1.62b9586ad0a22p-3
  },
  { // Entry 703
    0x1.f26840e7b2188bd0814e3dfc7f6f3f87p-1,
    0x1.d8f7208e6b82dp-3
  },
  { // Entry 704
    0x1.f26840e7b2188bd0814e3dfc7f6f3f87p-1,
    -0x1.d8f7208e6b82dp-3
  },
  { // Entry 705
    0x1.ead07cc6356960546ae634ef62621fb2p-1,
    0x1.279a74590331cp-2
  },
  { // Entry 706
    0x1.ead07cc6356960546ae634ef62621fb2p-1,
    -0x1.279a74590331cp-2
  },
  { // Entry 707
    0x1.e1960261829852ca662ca27d518c2fa9p-1,
    0x1.62b9586ad0a22p-2
  },
  { // Entry 708
    0x1.e1960261829852ca662ca27d518c2fa9p-1,
    -0x1.62b9586ad0a22p-2
  },
  { // Entry 709
    0x1.d6c0b125791cf983d53efaa7d45e291ep-1,
    0x1.9dd83c7c9e128p-2
  },
  { // Entry 710
    0x1.d6c0b125791cf983d53efaa7d45e291ep-1,
    -0x1.9dd83c7c9e128p-2
  },
  { // Entry 711
    0x1.ca59c6fa3d9cdb17530927aff1b33abbp-1,
    0x1.d8f7208e6b82ep-2
  },
  { // Entry 712
    0x1.ca59c6fa3d9cdb17530927aff1b33abbp-1,
    -0x1.d8f7208e6b82ep-2
  },
  { // Entry 713
    0x1.bc6bd861e13de309428e00f7bef6c3ecp-1,
    0x1.0a0b02501c799p-1
  },
  { // Entry 714
    0x1.bc6bd861e13de309428e00f7bef6c3ecp-1,
    -0x1.0a0b02501c799p-1
  },
  { // Entry 715
    0x1.a0d1f8a9a791f9dff5c993af4908264dp-1,
    0x1.3d3e35834faccp-1
  },
  { // Entry 716
    0x1.a0d1f8a9a791f9dff5c993af4908264dp-1,
    -0x1.3d3e35834faccp-1
  },
  { // Entry 717
    0x1.93e2948233fd0f69e3918982148f8265p-1,
    0x1.52e1f6ad9c27cp-1
  },
  { // Entry 718
    0x1.93e2948233fd0f69e3918982148f8265p-1,
    -0x1.52e1f6ad9c27cp-1
  },
  { // Entry 719
    0x1.863a850e43900997e76be80405437377p-1,
    0x1.6885b7d7e8a2cp-1
  },
  { // Entry 720
    0x1.863a850e43900997e76be80405437377p-1,
    -0x1.6885b7d7e8a2cp-1
  },
  { // Entry 721
    0x1.77e008d0775e9fc38e3f492f8e93ff51p-1,
    0x1.7e297902351dcp-1
  },
  { // Entry 722
    0x1.77e008d0775e9fc38e3f492f8e93ff51p-1,
    -0x1.7e297902351dcp-1
  },
  { // Entry 723
    0x1.68d9afe052d21e50560f9ffb6cc1b945p-1,
    0x1.93cd3a2c8198cp-1
  },
  { // Entry 724
    0x1.68d9afe052d21e50560f9ffb6cc1b945p-1,
    -0x1.93cd3a2c8198cp-1
  },
  { // Entry 725
    0x1.592e58ea0a9f1b4fddbaaf868fe47911p-1,
    0x1.a970fb56ce13cp-1
  },
  { // Entry 726
    0x1.592e58ea0a9f1b4fddbaaf868fe47911p-1,
    -0x1.a970fb56ce13cp-1
  },
  { // Entry 727
    0x1.48e52e0a65bce4d9d62a31293f7d41c1p-1,
    0x1.bf14bc811a8ecp-1
  },
  { // Entry 728
    0x1.48e52e0a65bce4d9d62a31293f7d41c1p-1,
    -0x1.bf14bc811a8ecp-1
  },
  { // Entry 729
    0x1.3805a188200a254247f30462c36acf6ap-1,
    0x1.d4b87dab6709cp-1
  },
  { // Entry 730
    0x1.3805a188200a254247f30462c36acf6ap-1,
    -0x1.d4b87dab6709cp-1
  },
  { // Entry 731
    0x1.26976a6c4e0fbabb84632bd99feec9c6p-1,
    0x1.ea5c3ed5b384cp-1
  },
  { // Entry 732
    0x1.26976a6c4e0fbabb84632bd99feec9c6p-1,
    -0x1.ea5c3ed5b384cp-1
  },
  { // Entry 733
    0x1.e8c405f36f85b7f5d6a38dfd4a692341p-2,
    0x1.12bd9173c07abp0
  },
  { // Entry 734
    0x1.e8c405f36f85b7f5d6a38dfd4a692341p-2,
    -0x1.12bd9173c07abp0
  },
  { // Entry 735
    0x1.a5a4ccf40d9cb25f16ad97e480c4b483p-2,
    0x1.257b22e780f56p0
  },
  { // Entry 736
    0x1.a5a4ccf40d9cb25f16ad97e480c4b483p-2,
    -0x1.257b22e780f56p0
  },
  { // Entry 737
    0x1.6043621b13bd3f904b3b876df5b2c6f4p-2,
    0x1.3838b45b41701p0
  },
  { // Entry 738
    0x1.6043621b13bd3f904b3b876df5b2c6f4p-2,
    -0x1.3838b45b41701p0
  },
  { // Entry 739
    0x1.18fee96a1a5762fc6770ff168e06ab3ep-2,
    0x1.4af645cf01eacp0
  },
  { // Entry 740
    0x1.18fee96a1a5762fc6770ff168e06ab3ep-2,
    -0x1.4af645cf01eacp0
  },
  { // Entry 741
    0x1.a0723a95492ce998457fb7a0d09a6385p-3,
    0x1.5db3d742c2657p0
  },
  { // Entry 742
    0x1.a0723a95492ce998457fb7a0d09a6385p-3,
    -0x1.5db3d742c2657p0
  },
  { // Entry 743
    0x1.0cab9115640b9d9d466723bbd5d589bep-3,
    0x1.707168b682e02p0
  },
  { // Entry 744
    0x1.0cab9115640b9d9d466723bbd5d589bep-3,
    -0x1.707168b682e02p0
  },
  { // Entry 745
    0x1.ddd1ec25e201f538925bf5bcf7c7df6ep-5,
    0x1.832efa2a435adp0
  },
  { // Entry 746
    0x1.ddd1ec25e201f538925bf5bcf7c7df6ep-5,
    -0x1.832efa2a435adp0
  },
  { // Entry 747
    -0x1.e6669a270c56d3a08d91cc2721f92fe1p-7,
    0x1.95ec8b9e03d58p0
  },
  { // Entry 748
    -0x1.e6669a270c56d3a08d91cc2721f92fe1p-7,
    -0x1.95ec8b9e03d58p0
  },
  { // Entry 749
    -0x1.682f3cc3c7a08da2ce02a41cdc7bed86p-4,
    0x1.a8aa1d11c44ffp0
  },
  { // Entry 750
    -0x1.682f3cc3c7a08da2ce02a41cdc7bed86p-4,
    -0x1.a8aa1d11c44ffp0
  },
  { // Entry 751
    0x1.0cb3469a29ea66d4031be769702aad5cp-1,
    0x1.04aff6d330942p0
  },
  { // Entry 752
    0x1.0cb3469a29ea66d4031be769702aad5cp-1,
    -0x1.04aff6d330942p0
  },
  { // Entry 753
    0x1.0cb228fa7f8117c82e61cf5393341c64p-1,
    0x1.04b09e98dcdb4p0
  },
  { // Entry 754
    0x1.0cb228fa7f8117c82e61cf5393341c64p-1,
    -0x1.04b09e98dcdb4p0
  },
  { // Entry 755
    0x1.0cb10b5a61b05a73e78a3e4447baf514p-1,
    0x1.04b1465e89226p0
  },
  { // Entry 756
    0x1.0cb10b5a61b05a73e78a3e4447baf514p-1,
    -0x1.04b1465e89226p0
  },
  { // Entry 757
    0x1.0cafedb9d078a984086928aa40d2e4a5p-1,
    0x1.04b1ee2435698p0
  },
  { // Entry 758
    0x1.0cafedb9d078a984086928aa40d2e4a5p-1,
    -0x1.04b1ee2435698p0
  },
  { // Entry 759
    0x1.0caed018cbda7fa59c631cd55b31aa8dp-1,
    0x1.04b295e9e1b0ap0
  },
  { // Entry 760
    0x1.0caed018cbda7fa59c631cd55b31aa8dp-1,
    -0x1.04b295e9e1b0ap0
  },
  { // Entry 761
    0x1.0cadb27753d65785e06d0e464006149ep-1,
    0x1.04b33daf8df7cp0
  },
  { // Entry 762
    0x1.0cadb27753d65785e06d0e464006149ep-1,
    -0x1.04b33daf8df7cp0
  },
  { // Entry 763
    0x1.0cac94d5686cabd2430c20fdf2855b47p-1,
    0x1.04b3e5753a3eep0
  },
  { // Entry 764
    0x1.0cac94d5686cabd2430c20fdf2855b47p-1,
    -0x1.04b3e5753a3eep0
  },
  { // Entry 765
    0x1.0cab7733099df738645574cd482ef4b2p-1,
    0x1.04b48d3ae6860p0
  },
  { // Entry 766
    0x1.0cab7733099df738645574cd482ef4b2p-1,
    -0x1.04b48d3ae6860p0
  },
  { // Entry 767
    0x1.0caa5990376b061ec1cf3890f1b8e1e3p-1,
    0x1.04b5350092ccfp0
  },
  { // Entry 768
    0x1.0caa5990376b061ec1cf3890f1b8e1e3p-1,
    -0x1.04b5350092ccfp0
  },
  { // Entry 769
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1074
  },
  { // Entry 770
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1074
  },
  { // Entry 771
    0x1.p0,
    -0.0
  },
  { // Entry 772
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1074
  },
  { // Entry 773
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1074
  },
  { // Entry 774
    0x1.ad02c771c35ed5f01089a00c6a27e0cfp-1,
    0x1.279a74590331bp-1
  },
  { // Entry 775
    0x1.ad02c771c35ed5f01089a00c6a27e0cfp-1,
    -0x1.279a74590331bp-1
  },
  { // Entry 776
    0x1.ad02c771c35ecd3471de9021e6c3b92bp-1,
    0x1.279a74590331cp-1
  },
  { // Entry 777
    0x1.ad02c771c35ecd3471de9021e6c3b92bp-1,
    -0x1.279a74590331cp-1
  },
  { // Entry 778
    0x1.ad02c771c35ec478d333803762f450d5p-1,
    0x1.279a74590331dp-1
  },
  { // Entry 779
    0x1.ad02c771c35ec478d333803762f450d5p-1,
    -0x1.279a74590331dp-1
  },
  { // Entry 780
    -0x1.48d1ddd2b2b3f8c21b9421e65b380735p-3,
    0x1.bb67ae8584ca9p0
  },
  { // Entry 781
    -0x1.48d1ddd2b2b3f8c21b9421e65b380735p-3,
    -0x1.bb67ae8584ca9p0
  },
  { // Entry 782
    -0x1.48d1ddd2b2b47718ff3063b6bd981099p-3,
    0x1.bb67ae8584caap0
  },
  { // Entry 783
    -0x1.48d1ddd2b2b47718ff3063b6bd981099p-3,
    -0x1.bb67ae8584caap0
  },
  { // Entry 784
    -0x1.48d1ddd2b2b4f56fe2cca5871eaf4820p-3,
    0x1.bb67ae8584cabp0
  },
  { // Entry 785
    -0x1.48d1ddd2b2b4f56fe2cca5871eaf4820p-3,
    -0x1.bb67ae8584cabp0
  },
  { // Entry 786
    0x1.cfc6cfa52ad9f9911db4ca0d45fdb0b3p-1,
    0x1.bffffffffffffp-2
  },
  { // Entry 787
    0x1.cfc6cfa52ad9f9911db4ca0d45fdb0b3p-1,
    -0x1.bffffffffffffp-2
  },
  { // Entry 788
    0x1.cfc6cfa52ad9f62d6d5423ca8339a00ap-1,
    0x1.cp-2
  },
  { // Entry 789
    0x1.cfc6cfa52ad9f62d6d5423ca8339a00ap-1,
    -0x1.cp-2
  },
  { // Entry 790
    0x1.cfc6cfa52ad9f2c9bcf37d87c05892f5p-1,
    0x1.c000000000001p-2
  },
  { // Entry 791
    0x1.cfc6cfa52ad9f2c9bcf37d87c05892f5p-1,
    -0x1.c000000000001p-2
  },
  { // Entry 792
    0x1.8bb105a5dc90104051d08cb965631807p-1,
    0x1.5ffffffffffffp-1
  },
  { // Entry 793
    0x1.8bb105a5dc90104051d08cb965631807p-1,
    -0x1.5ffffffffffffp-1
  },
  { // Entry 794
    0x1.8bb105a5dc900618f80fa51d303c69p-1,
    0x1.6p-1
  },
  { // Entry 795
    0x1.8bb105a5dc900618f80fa51d303c69p-1,
    -0x1.6p-1
  },
  { // Entry 796
    0x1.8bb105a5dc8ffbf19e4ebd80fab2cdb8p-1,
    0x1.6000000000001p-1
  },
  { // Entry 797
    0x1.8bb105a5dc8ffbf19e4ebd80fab2cdb8p-1,
    -0x1.6000000000001p-1
  },
  { // Entry 798
    0x1.7ef4842f0bcd11686aaf6f21c9aa8354p-2,
    0x1.2ffffffffffffp0
  },
  { // Entry 799
    0x1.7ef4842f0bcd11686aaf6f21c9aa8354p-2,
    -0x1.2ffffffffffffp0
  },
  { // Entry 800
    0x1.7ef4842f0bccd60d4a501dc8bc4b57b3p-2,
    0x1.3p0
  },
  { // Entry 801
    0x1.7ef4842f0bccd60d4a501dc8bc4b57b3p-2,
    -0x1.3p0
  },
  { // Entry 802
    0x1.7ef4842f0bcc9ab229f0cc6fad6d378dp-2,
    0x1.3000000000001p0
  },
  { // Entry 803
    0x1.7ef4842f0bcc9ab229f0cc6fad6d378dp-2,
    -0x1.3000000000001p0
  },
  { // Entry 804
    -0x1.863efa361dc2294e929b9515fb34f9bap-1,
    0x1.37fffffffffffp1
  },
  { // Entry 805
    -0x1.863efa361dc2294e929b9515fb34f9bap-1,
    -0x1.37fffffffffffp1
  },
  { // Entry 806
    -0x1.863efa361dc252bca1eaeed39749bed7p-1,
    0x1.380p1
  },
  { // Entry 807
    -0x1.863efa361dc252bca1eaeed39749bed7p-1,
    -0x1.380p1
  },
  { // Entry 808
    -0x1.863efa361dc27c2ab13a48912d45880bp-1,
    0x1.3800000000001p1
  },
  { // Entry 809
    -0x1.863efa361dc27c2ab13a48912d45880bp-1,
    -0x1.3800000000001p1
  },
  { // Entry 810
    0x1.fef2b2d21cf6c106e86ff9395f8204a0p-1,
    0x1.069c8b46b3792p-4
  },
  { // Entry 811
    0x1.fef2b2d21cf6c106e86ff9395f8204a0p-1,
    -0x1.069c8b46b3792p-4
  },
  { // Entry 812
    0x1.fbcbe693bd8ec85723b6cb55e4f5e78fp-1,
    0x1.069c8b46b3792p-3
  },
  { // Entry 813
    0x1.fbcbe693bd8ec85723b6cb55e4f5e78fp-1,
    -0x1.069c8b46b3792p-3
  },
  { // Entry 814
    0x1.f68eebfcbb5e841900e2542f7c24bab0p-1,
    0x1.89ead0ea0d35bp-3
  },
  { // Entry 815
    0x1.f68eebfcbb5e841900e2542f7c24bab0p-1,
    -0x1.89ead0ea0d35bp-3
  },
  { // Entry 816
    0x1.ef4145b4aecffbdaaffb78ffb49ac9bdp-1,
    0x1.069c8b46b3792p-2
  },
  { // Entry 817
    0x1.ef4145b4aecffbdaaffb78ffb49ac9bdp-1,
    -0x1.069c8b46b3792p-2
  },
  { // Entry 818
    0x1.e5eaa286fbbc670dbf6392d7c98ab0a0p-1,
    0x1.4843ae1860576p-2
  },
  { // Entry 819
    0x1.e5eaa286fbbc670dbf6392d7c98ab0a0p-1,
    -0x1.4843ae1860576p-2
  },
  { // Entry 820
    0x1.da94d54dd4c0876e18ee97c70d1a4a94p-1,
    0x1.89ead0ea0d35ap-2
  },
  { // Entry 821
    0x1.da94d54dd4c0876e18ee97c70d1a4a94p-1,
    -0x1.89ead0ea0d35ap-2
  },
  { // Entry 822
    0x1.cd4bca9cb5c715302001e446cc93a7bcp-1,
    0x1.cb91f3bbba13ep-2
  },
  { // Entry 823
    0x1.cd4bca9cb5c715302001e446cc93a7bcp-1,
    -0x1.cb91f3bbba13ep-2
  },
  { // Entry 824
    0x1.be1d7c3534c40331fddf243d8a0a56b0p-1,
    0x1.069c8b46b3791p-1
  },
  { // Entry 825
    0x1.be1d7c3534c40331fddf243d8a0a56b0p-1,
    -0x1.069c8b46b3791p-1
  },
  { // Entry 826
    0x1.ad19e2535aa9678bd5fdafc68817700bp-1,
    0x1.27701caf89e83p-1
  },
  { // Entry 827
    0x1.ad19e2535aa9678bd5fdafc68817700bp-1,
    -0x1.27701caf89e83p-1
  },
  { // Entry 828
    0x1.9a52e2e0fbcb3a3f4bde6f6ec27767a9p-1,
    0x1.4843ae1860575p-1
  },
  { // Entry 829
    0x1.9a52e2e0fbcb3a3f4bde6f6ec27767a9p-1,
    -0x1.4843ae1860575p-1
  },
  { // Entry 830
    0x1.85dc3ea1bbce9a8085f66593a87b7e2ep-1,
    0x1.69173f8136c67p-1
  },
  { // Entry 831
    0x1.85dc3ea1bbce9a8085f66593a87b7e2ep-1,
    -0x1.69173f8136c67p-1
  },
  { // Entry 832
    0x1.6fcb7c6b8b919af3dda53094c9a27aabp-1,
    0x1.89ead0ea0d359p-1
  },
  { // Entry 833
    0x1.6fcb7c6b8b919af3dda53094c9a27aabp-1,
    -0x1.89ead0ea0d359p-1
  },
  { // Entry 834
    0x1.5837d2817cf303ef6dae69faeb0f015ep-1,
    0x1.aabe6252e3a4bp-1
  },
  { // Entry 835
    0x1.5837d2817cf303ef6dae69faeb0f015ep-1,
    -0x1.aabe6252e3a4bp-1
  },
  { // Entry 836
    0x1.3f3a0e28bedd40445858f823e150264bp-1,
    0x1.cb91f3bbba13dp-1
  },
  { // Entry 837
    0x1.3f3a0e28bedd40445858f823e150264bp-1,
    -0x1.cb91f3bbba13dp-1
  },
  { // Entry 838
    0x1.24ec799171642dbd24d259005822bd25p-1,
    0x1.ec6585249082fp-1
  },
  { // Entry 839
    0x1.24ec799171642dbd24d259005822bd25p-1,
    -0x1.ec6585249082fp-1
  },
  { // Entry 840
    0x1.096ac02ec42c85b7b10afed9202785b9p-1,
    0x1.069c8b46b3791p0
  },
  { // Entry 841
    0x1.096ac02ec42c85b7b10afed9202785b9p-1,
    -0x1.069c8b46b3791p0
  },
  { // Entry 842
    0x1.d9a3a336edb7613df062e86a32d09fe1p-2,
    0x1.170653fb1eb0ap0
  },
  { // Entry 843
    0x1.d9a3a336edb7613df062e86a32d09fe1p-2,
    -0x1.170653fb1eb0ap0
  },
  { // Entry 844
    0x1.9e7f8652b47582afd29744293170c07cp-2,
    0x1.27701caf89e83p0
  },
  { // Entry 845
    0x1.9e7f8652b47582afd29744293170c07cp-2,
    -0x1.27701caf89e83p0
  },
  { // Entry 846
    0x1.61a76077aee07bb349ca76cf700913d1p-2,
    0x1.37d9e563f51fcp0
  },
  { // Entry 847
    0x1.61a76077aee07bb349ca76cf700913d1p-2,
    -0x1.37d9e563f51fcp0
  },
  { // Entry 848
    0x1.235b331d8f748e20fb6ddb6a708dba10p-2,
    0x1.4843ae1860575p0
  },
  { // Entry 849
    0x1.235b331d8f748e20fb6ddb6a708dba10p-2,
    -0x1.4843ae1860575p0
  },
  { // Entry 850
    0x1.c7b90e3024593da8449963cfe08dde85p-3,
    0x1.58ad76cccb8eep0
  },
  { // Entry 851
    0x1.c7b90e3024593da8449963cfe08dde85p-3,
    -0x1.58ad76cccb8eep0
  },
  { // Entry 852
    0x1.46dc4f4ce83da727ea048cc7d2f276d1p-3,
    0x1.69173f8136c67p0
  },
  { // Entry 853
    0x1.46dc4f4ce83da727ea048cc7d2f276d1p-3,
    -0x1.69173f8136c67p0
  },
  { // Entry 854
    0x1.894f70befbb99ab7df9d1790a28f48adp-4,
    0x1.79810835a1fe0p0
  },
  { // Entry 855
    0x1.894f70befbb99ab7df9d1790a28f48adp-4,
    -0x1.79810835a1fe0p0
  },
  { // Entry 856
    0x1.069107ae9332f95fa2c5ceeadfb29f77p-5,
    0x1.89ead0ea0d359p0
  },
  { // Entry 857
    0x1.069107ae9332f95fa2c5ceeadfb29f77p-5,
    -0x1.89ead0ea0d359p0
  },
  { // Entry 858
    -0x1.069107ae9327e0731a748c21f03b5efcp-5,
    0x1.9a54999e786d2p0
  },
  { // Entry 859
    -0x1.069107ae9327e0731a748c21f03b5efcp-5,
    -0x1.9a54999e786d2p0
  },
  { // Entry 860
    -0x1.894f70befbb41417dff843e81fac388bp-4,
    0x1.aabe6252e3a4bp0
  },
  { // Entry 861
    -0x1.894f70befbb41417dff843e81fac388bp-4,
    -0x1.aabe6252e3a4bp0
  },
  { // Entry 862
    -0x1.46dc4f4ce83ae9ab1cc1b2367cb753ebp-3,
    0x1.bb282b074edc4p0
  },
  { // Entry 863
    -0x1.46dc4f4ce83ae9ab1cc1b2367cb753ebp-3,
    -0x1.bb282b074edc4p0
  },
  { // Entry 864
    -0x1.c7b90e30245688e099860e8d4fff601cp-3,
    0x1.cb91f3bbba13dp0
  },
  { // Entry 865
    -0x1.c7b90e30245688e099860e8d4fff601cp-3,
    -0x1.cb91f3bbba13dp0
  },
  { // Entry 866
    -0x1.235b331d8f7339841a517312d0d347fbp-2,
    0x1.dbfbbc70254b6p0
  },
  { // Entry 867
    -0x1.235b331d8f7339841a517312d0d347fbp-2,
    -0x1.dbfbbc70254b6p0
  },
  { // Entry 868
    -0x1.61a76077aedf2e43aca418f7a2e1324dp-2,
    0x1.ec6585249082fp0
  },
  { // Entry 869
    -0x1.61a76077aedf2e43aca418f7a2e1324dp-2,
    -0x1.ec6585249082fp0
  },
  { // Entry 870
    -0x1.9e7f8652b4743dcc3c3568baff8bf9ebp-2,
    0x1.fccf4dd8fbba8p0
  },
  { // Entry 871
    -0x1.9e7f8652b4743dcc3c3568baff8bf9ebp-2,
    -0x1.fccf4dd8fbba8p0
  },
  { // Entry 872
    -0x1.d9a3a336edb65efa30e1a6679aa064c2p-2,
    0x1.069c8b46b3791p1
  },
  { // Entry 873
    -0x1.d9a3a336edb65efa30e1a6679aa064c2p-2,
    -0x1.069c8b46b3791p1
  },
  { // Entry 874
    -0x1.096ac02ec42c24880a5951788cb383c8p-1,
    0x1.0ed16fa0e914ep1
  },
  { // Entry 875
    -0x1.096ac02ec42c24880a5951788cb383c8p-1,
    -0x1.0ed16fa0e914ep1
  },
  { // Entry 876
    -0x1.24ec79917163dda65afd8109f59cb465p-1,
    0x1.170653fb1eb0bp1
  },
  { // Entry 877
    -0x1.24ec79917163dda65afd8109f59cb465p-1,
    -0x1.170653fb1eb0bp1
  },
  { // Entry 878
    -0x1.3f3a0e28bedd0cf0c4bfbd8c82a3baafp-1,
    0x1.1f3b3855544c8p1
  },
  { // Entry 879
    -0x1.3f3a0e28bedd0cf0c4bfbd8c82a3baafp-1,
    -0x1.1f3b3855544c8p1
  },
  { // Entry 880
    -0x1.5837d2817cf2eb069035552dc3ae834cp-1,
    0x1.27701caf89e85p1
  },
  { // Entry 881
    -0x1.5837d2817cf2eb069035552dc3ae834cp-1,
    -0x1.27701caf89e85p1
  },
  { // Entry 882
    -0x1.6fcb7c6b8b9199ce2f17dd3ee86b3b9ap-1,
    0x1.2fa50109bf842p1
  },
  { // Entry 883
    -0x1.6fcb7c6b8b9199ce2f17dd3ee86b3b9ap-1,
    -0x1.2fa50109bf842p1
  },
  { // Entry 884
    -0x1.85dc3ea1bbceae2d294421e8c7350f8cp-1,
    0x1.37d9e563f51ffp1
  },
  { // Entry 885
    -0x1.85dc3ea1bbceae2d294421e8c7350f8cp-1,
    -0x1.37d9e563f51ffp1
  },
  { // Entry 886
    -0x1.9a52e2e0fbcb5f8a3f55c274f9ec754bp-1,
    0x1.400ec9be2abbcp1
  },
  { // Entry 887
    -0x1.9a52e2e0fbcb5f8a3f55c274f9ec754bp-1,
    -0x1.400ec9be2abbcp1
  },
  { // Entry 888
    -0x1.ad19e2535aa99b049ac0b5858c5d381fp-1,
    0x1.4843ae1860579p1
  },
  { // Entry 889
    -0x1.ad19e2535aa99b049ac0b5858c5d381fp-1,
    -0x1.4843ae1860579p1
  },
  { // Entry 890
    -0x1.be1d7c3534c44132ab1c4130cbe9dfa0p-1,
    0x1.5078927295f36p1
  },
  { // Entry 891
    -0x1.be1d7c3534c44132ab1c4130cbe9dfa0p-1,
    -0x1.5078927295f36p1
  },
  { // Entry 892
    -0x1.cd4bca9cb5c759e4d6dc8601ec3d84b6p-1,
    0x1.58ad76cccb8f3p1
  },
  { // Entry 893
    -0x1.cd4bca9cb5c759e4d6dc8601ec3d84b6p-1,
    -0x1.58ad76cccb8f3p1
  },
  { // Entry 894
    -0x1.da94d54dd4c0cedccd73684994422740p-1,
    0x1.60e25b27012b0p1
  },
  { // Entry 895
    -0x1.da94d54dd4c0cedccd73684994422740p-1,
    -0x1.60e25b27012b0p1
  },
  { // Entry 896
    -0x1.e5eaa286fbbcad1e4a6373392e679669p-1,
    0x1.69173f8136c6dp1
  },
  { // Entry 897
    -0x1.e5eaa286fbbcad1e4a6373392e679669p-1,
    -0x1.69173f8136c6dp1
  },
  { // Entry 898
    -0x1.ef4145b4aed03c5f1d39763b1eee6ed8p-1,
    0x1.714c23db6c62ap1
  },
  { // Entry 899
    -0x1.ef4145b4aed03c5f1d39763b1eee6ed8p-1,
    -0x1.714c23db6c62ap1
  },
  { // Entry 900
    -0x1.f68eebfcbb5eba124d8cc48fd1beb04dp-1,
    0x1.79810835a1fe7p1
  },
  { // Entry 901
    -0x1.f68eebfcbb5eba124d8cc48fd1beb04dp-1,
    -0x1.79810835a1fe7p1
  },
  { // Entry 902
    -0x1.fbcbe693bd8ef006f5ff02210dfe0619p-1,
    0x1.81b5ec8fd79a4p1
  },
  { // Entry 903
    -0x1.fbcbe693bd8ef006f5ff02210dfe0619p-1,
    -0x1.81b5ec8fd79a4p1
  },
  { // Entry 904
    -0x1.fef2b2d21cf6be1a2c7ea665ef1f874ep-1,
    0x1.89ead0ea0d35bp1
  },
  { // Entry 905
    -0x1.fef2b2d21cf6be1a2c7ea665ef1f874ep-1,
    -0x1.89ead0ea0d35bp1
  },
  { // Entry 906
    0x1.ef4145b4aecff6f58edecf24955428c1p-1,
    -0x1.81b5ec8fd799fp2
  },
  { // Entry 907
    0x1.ef4145b4aecff6f58edecf24955428c1p-1,
    0x1.81b5ec8fd799fp2
  },
  { // Entry 908
    0x1.be1d7c3534c3f9b9b35619280049de85p-1,
    -0x1.714c23db6c626p2
  },
  { // Entry 909
    0x1.be1d7c3534c3f9b9b35619280049de85p-1,
    0x1.714c23db6c626p2
  },
  { // Entry 910
    0x1.6fcb7c6b8b918d86fc83d612a6587eddp-1,
    -0x1.60e25b27012adp2
  },
  { // Entry 911
    0x1.6fcb7c6b8b918d86fc83d612a6587eddp-1,
    0x1.60e25b27012adp2
  },
  { // Entry 912
    0x1.096ac02ec42c82e5b225185bd6c757d5p-1,
    -0x1.5078927295f34p2
  },
  { // Entry 913
    0x1.096ac02ec42c82e5b225185bd6c757d5p-1,
    0x1.5078927295f34p2
  },
  { // Entry 914
    0x1.235b331d8f7487ce2db97819fae7777cp-2,
    -0x1.400ec9be2abbbp2
  },
  { // Entry 915
    0x1.235b331d8f7487ce2db97819fae7777cp-2,
    0x1.400ec9be2abbbp2
  },
  { // Entry 916
    0x1.069107ae9332c4a1cd2dc033b8d50598p-5,
    -0x1.2fa50109bf842p2
  },
  { // Entry 917
    0x1.069107ae9332c4a1cd2dc033b8d50598p-5,
    0x1.2fa50109bf842p2
  },
  { // Entry 918
    -0x1.c7b90e30245695bd1ec170f45feeb1ffp-3,
    -0x1.1f3b3855544c9p2
  },
  { // Entry 919
    -0x1.c7b90e30245695bd1ec170f45feeb1ffp-3,
    0x1.1f3b3855544c9p2
  },
  { // Entry 920
    -0x1.d9a3a336edb62c1541b8584cd6c00f87p-2,
    -0x1.0ed16fa0e9150p2
  },
  { // Entry 921
    -0x1.d9a3a336edb62c1541b8584cd6c00f87p-2,
    0x1.0ed16fa0e9150p2
  },
  { // Entry 922
    -0x1.5837d2817cf27705cac7881fb569ffc7p-1,
    -0x1.fccf4dd8fbbaep1
  },
  { // Entry 923
    -0x1.5837d2817cf27705cac7881fb569ffc7p-1,
    0x1.fccf4dd8fbbaep1
  },
  { // Entry 924
    -0x1.ad19e2535aa8ffb40066d78aef71fabdp-1,
    -0x1.dbfbbc70254bcp1
  },
  { // Entry 925
    -0x1.ad19e2535aa8ffb40066d78aef71fabdp-1,
    0x1.dbfbbc70254bcp1
  },
  { // Entry 926
    -0x1.e5eaa286fbbc2b129238160df30ce704p-1,
    -0x1.bb282b074edcap1
  },
  { // Entry 927
    -0x1.e5eaa286fbbc2b129238160df30ce704p-1,
    0x1.bb282b074edcap1
  },
  { // Entry 928
    -0x1.fef2b2d21cf6b40ff3b530ce8dc0d8a7p-1,
    -0x1.9a54999e786d8p1
  },
  { // Entry 929
    -0x1.fef2b2d21cf6b40ff3b530ce8dc0d8a7p-1,
    0x1.9a54999e786d8p1
  },
  { // Entry 930
    -0x1.f68eebfcbb5eadd65c261cd803990ae1p-1,
    -0x1.79810835a1fe6p1
  },
  { // Entry 931
    -0x1.f68eebfcbb5eadd65c261cd803990ae1p-1,
    0x1.79810835a1fe6p1
  },
  { // Entry 932
    -0x1.cd4bca9cb5c775a99729f7ad95b7dce3p-1,
    -0x1.58ad76cccb8f4p1
  },
  { // Entry 933
    -0x1.cd4bca9cb5c775a99729f7ad95b7dce3p-1,
    0x1.58ad76cccb8f4p1
  },
  { // Entry 934
    -0x1.85dc3ea1bbcf2aa2e21ec586d5497e35p-1,
    -0x1.37d9e563f5202p1
  },
  { // Entry 935
    -0x1.85dc3ea1bbcf2aa2e21ec586d5497e35p-1,
    0x1.37d9e563f5202p1
  },
  { // Entry 936
    -0x1.24ec79917164e41addd4bacd4420f9fbp-1,
    -0x1.170653fb1eb10p1
  },
  { // Entry 937
    -0x1.24ec79917164e41addd4bacd4420f9fbp-1,
    0x1.170653fb1eb10p1
  },
  { // Entry 938
    -0x1.61a76077aee23b11f0c673f638003b0ap-2,
    -0x1.ec6585249083cp0
  },
  { // Entry 939
    -0x1.61a76077aee23b11f0c673f638003b0ap-2,
    0x1.ec6585249083cp0
  },
  { // Entry 940
    -0x1.894f70befbc104b706e85cf4c1c96a52p-4,
    -0x1.aabe6252e3a58p0
  },
  { // Entry 941
    -0x1.894f70befbc104b706e85cf4c1c96a52p-4,
    0x1.aabe6252e3a58p0
  },
  { // Entry 942
    0x1.46dc4f4ce8373c7c44f13b57363edd3bp-3,
    -0x1.69173f8136c74p0
  },
  { // Entry 943
    0x1.46dc4f4ce8373c7c44f13b57363edd3bp-3,
    0x1.69173f8136c74p0
  },
  { // Entry 944
    0x1.9e7f8652b47289e53fccd54955db4552p-2,
    -0x1.27701caf89e90p0
  },
  { // Entry 945
    0x1.9e7f8652b47289e53fccd54955db4552p-2,
    0x1.27701caf89e90p0
  },
  { // Entry 946
    0x1.3f3a0e28bedbfb066b67abd9c338409ep-1,
    -0x1.cb91f3bbba157p-1
  },
  { // Entry 947
    0x1.3f3a0e28bedbfb066b67abd9c338409ep-1,
    0x1.cb91f3bbba157p-1
  },
  { // Entry 948
    0x1.9a52e2e0fbca4b00c72daa3cdaca257cp-1,
    -0x1.4843ae186058ep-1
  },
  { // Entry 949
    0x1.9a52e2e0fbca4b00c72daa3cdaca257cp-1,
    0x1.4843ae186058ep-1
  },
  { // Entry 950
    0x1.da94d54dd4bff753d988c1755e2ffc04p-1,
    -0x1.89ead0ea0d38ap-2
  },
  { // Entry 951
    0x1.da94d54dd4bff753d988c1755e2ffc04p-1,
    0x1.89ead0ea0d38ap-2
  },
  { // Entry 952
    0x1.fbcbe693bd8e98423207e36587d942b7p-1,
    -0x1.069c8b46b37f0p-3
  },
  { // Entry 953
    0x1.fbcbe693bd8e98423207e36587d942b7p-1,
    0x1.069c8b46b37f0p-3
  },
  { // Entry 954
    0x1.fbcbe693bd8ef86c1565b3453036e55ep-1,
    0x1.069c8b46b3734p-3
  },
  { // Entry 955
    0x1.fbcbe693bd8ef86c1565b3453036e55ep-1,
    -0x1.069c8b46b3734p-3
  },
  { // Entry 956
    0x1.da94d54dd4c11187405ada7f04e5b171p-1,
    0x1.89ead0ea0d32cp-2
  },
  { // Entry 957
    0x1.da94d54dd4c11187405ada7f04e5b171p-1,
    -0x1.89ead0ea0d32cp-2
  },
  { // Entry 958
    0x1.9a52e2e0fbcc0cc83b843bae58c6cdf8p-1,
    0x1.4843ae186055fp-1
  },
  { // Entry 959
    0x1.9a52e2e0fbcc0cc83b843bae58c6cdf8p-1,
    -0x1.4843ae186055fp-1
  },
  { // Entry 960
    0x1.3f3a0e28bede46f65ca5b5c19ad99dd7p-1,
    0x1.cb91f3bbba128p-1
  },
  { // Entry 961
    0x1.3f3a0e28bede46f65ca5b5c19ad99dd7p-1,
    -0x1.cb91f3bbba128p-1
  },
  { // Entry 962
    0x1.9e7f8652b478066eec563f835097f148p-2,
    0x1.27701caf89e78p0
  },
  { // Entry 963
    0x1.9e7f8652b478066eec563f835097f148p-2,
    -0x1.27701caf89e78p0
  },
  { // Entry 964
    0x1.46dc4f4ce843151b9d14e561879e5fe3p-3,
    0x1.69173f8136c5cp0
  },
  { // Entry 965
    0x1.46dc4f4ce843151b9d14e561879e5fe3p-3,
    -0x1.69173f8136c5cp0
  },
  { // Entry 966
    -0x1.894f70befba9211b0dcaa4dca450670fp-4,
    0x1.aabe6252e3a40p0
  },
  { // Entry 967
    -0x1.894f70befba9211b0dcaa4dca450670fp-4,
    -0x1.aabe6252e3a40p0
  },
  { // Entry 968
    -0x1.61a76077aedc99952438421f820a2befp-2,
    0x1.ec65852490824p0
  },
  { // Entry 969
    -0x1.61a76077aedc99952438421f820a2befp-2,
    -0x1.ec65852490824p0
  },
  { // Entry 970
    -0x1.24ec799171626e36709cfcf7c7752332p-1,
    0x1.170653fb1eb04p1
  },
  { // Entry 971
    -0x1.24ec799171626e36709cfcf7c7752332p-1,
    -0x1.170653fb1eb04p1
  },
  { // Entry 972
    -0x1.85dc3ea1bbcd38cbfeb4370d5405eebap-1,
    0x1.37d9e563f51f6p1
  },
  { // Entry 973
    -0x1.85dc3ea1bbcd38cbfeb4370d5405eebap-1,
    -0x1.37d9e563f51f6p1
  },
  { // Entry 974
    -0x1.cd4bca9cb5c628709388a39fc84591d1p-1,
    0x1.58ad76cccb8e8p1
  },
  { // Entry 975
    -0x1.cd4bca9cb5c628709388a39fc84591d1p-1,
    -0x1.58ad76cccb8e8p1
  },
  { // Entry 976
    -0x1.f68eebfcbb5e1b070b564037f5571a39p-1,
    0x1.79810835a1fdap1
  },
  { // Entry 977
    -0x1.f68eebfcbb5e1b070b564037f5571a39p-1,
    -0x1.79810835a1fdap1
  },
  { // Entry 978
    -0x1.fef2b2d21cf6e544ab7795aed10d9fa7p-1,
    0x1.9a54999e786ccp1
  },
  { // Entry 979
    -0x1.fef2b2d21cf6e544ab7795aed10d9fa7p-1,
    -0x1.9a54999e786ccp1
  },
  { // Entry 980
    -0x1.e5eaa286fbbd1d135e216c49a9f7e5dap-1,
    0x1.bb282b074edbep1
  },
  { // Entry 981
    -0x1.e5eaa286fbbd1d135e216c49a9f7e5dap-1,
    -0x1.bb282b074edbep1
  },
  { // Entry 982
    -0x1.ad19e2535aaaa2ac87056b6d7776e97ap-1,
    0x1.dbfbbc70254b0p1
  },
  { // Entry 983
    -0x1.ad19e2535aaaa2ac87056b6d7776e97ap-1,
    -0x1.dbfbbc70254b0p1
  },
  { // Entry 984
    -0x1.5837d2817cf4af8e5e59b13b4aa9b5e3p-1,
    0x1.fccf4dd8fbba2p1
  },
  { // Entry 985
    -0x1.5837d2817cf4af8e5e59b13b4aa9b5e3p-1,
    -0x1.fccf4dd8fbba2p1
  },
  { // Entry 986
    -0x1.d9a3a336edbb7de64a2183cb27be4b5bp-2,
    0x1.0ed16fa0e914ap2
  },
  { // Entry 987
    -0x1.d9a3a336edbb7de64a2183cb27be4b5bp-2,
    -0x1.0ed16fa0e914ap2
  },
  { // Entry 988
    -0x1.c7b90e30246248b7a0c2c87a3dd25224p-3,
    0x1.1f3b3855544c3p2
  },
  { // Entry 989
    -0x1.c7b90e30246248b7a0c2c87a3dd25224p-3,
    -0x1.1f3b3855544c3p2
  },
  { // Entry 990
    0x1.069107ae9302caf2068b48842afdf051p-5,
    0x1.2fa50109bf83cp2
  },
  { // Entry 991
    0x1.069107ae9302caf2068b48842afdf051p-5,
    -0x1.2fa50109bf83cp2
  },
  { // Entry 992
    0x1.235b331d8f6ec74aa3de5aed15fa3f68p-2,
    0x1.400ec9be2abb5p2
  },
  { // Entry 993
    0x1.235b331d8f6ec74aa3de5aed15fa3f68p-2,
    -0x1.400ec9be2abb5p2
  },
  { // Entry 994
    0x1.096ac02ec429f225c99b89bb4c9e5d3ep-1,
    0x1.5078927295f2ep2
  },
  { // Entry 995
    0x1.096ac02ec429f225c99b89bb4c9e5d3ep-1,
    -0x1.5078927295f2ep2
  },
  { // Entry 996
    0x1.6fcb7c6b8b8f773e3b421dded6fc1f26p-1,
    0x1.60e25b27012a7p2
  },
  { // Entry 997
    0x1.6fcb7c6b8b8f773e3b421dded6fc1f26p-1,
    -0x1.60e25b27012a7p2
  },
  { // Entry 998
    0x1.be1d7c3534c280dab43dced670330b63p-1,
    0x1.714c23db6c620p2
  },
  { // Entry 999
    0x1.be1d7c3534c280dab43dced670330b63p-1,
    -0x1.714c23db6c620p2
  },
  { // Entry 1000
    0x1.ef4145b4aecf342709a3b19320d1b194p-1,
    0x1.81b5ec8fd7999p2
  },
  { // Entry 1001
    0x1.ef4145b4aecf342709a3b19320d1b194p-1,
    -0x1.81b5ec8fd7999p2
  },
  { // Entry 1002
    0x1.ff0fd2c96adfbae576981ee4b34769dep-1,
    0x1.effffffffffffp-5
  },
  { // Entry 1003
    0x1.ff0fd2c96adfbae576981ee4b34769dep-1,
    -0x1.effffffffffffp-5
  },
  { // Entry 1004
    0x1.ff0fd2c96adfbad5f904a71b2d210a2ap-1,
    0x1.fp-5
  },
  { // Entry 1005
    0x1.ff0fd2c96adfbad5f904a71b2d210a2ap-1,
    -0x1.fp-5
  },
  { // Entry 1006
    0x1.ff0fd2c96adfbac67b712f51a6fa2ab3p-1,
    0x1.f000000000001p-5
  },
  { // Entry 1007
    0x1.ff0fd2c96adfbac67b712f51a6fa2ab3p-1,
    -0x1.f000000000001p-5
  },
  { // Entry 1008
    0x1.fc210055467fe5c8f76e75fd7083818cp-1,
    0x1.f7fffffffffffp-4
  },
  { // Entry 1009
    0x1.fc210055467fe5c8f76e75fd7083818cp-1,
    -0x1.f7fffffffffffp-4
  },
  { // Entry 1010
    0x1.fc210055467fe58a20193399b3bc0dd2p-1,
    0x1.f80p-4
  },
  { // Entry 1011
    0x1.fc210055467fe58a20193399b3bc0dd2p-1,
    -0x1.f80p-4
  },
  { // Entry 1012
    0x1.fc210055467fe54b48c3f135f6f29df7p-1,
    0x1.f800000000001p-4
  },
  { // Entry 1013
    0x1.fc210055467fe54b48c3f135f6f29df7p-1,
    -0x1.f800000000001p-4
  },
  { // Entry 1014
    0x1.f94984b2552e19e7329413b8c2e8dc51p-1,
    0x1.4bfffffffffffp-3
  },
  { // Entry 1015
    0x1.f94984b2552e19e7329413b8c2e8dc51p-1,
    -0x1.4bfffffffffffp-3
  },
  { // Entry 1016
    0x1.f94984b2552e1941ec766c6a82ece4a3p-1,
    0x1.4c0p-3
  },
  { // Entry 1017
    0x1.f94984b2552e1941ec766c6a82ece4a3p-1,
    -0x1.4c0p-3
  },
  { // Entry 1018
    0x1.f94984b2552e189ca658c51c42e907cep-1,
    0x1.4c00000000001p-3
  },
  { // Entry 1019
    0x1.f94984b2552e189ca658c51c42e907cep-1,
    -0x1.4c00000000001p-3
  },
  { // Entry 1020
    0x1.e921dd42f09ba868603ea376f6e2d012p-1,
    0x1.3333333333332p-2
  },
  { // Entry 1021
    0x1.e921dd42f09ba868603ea376f6e2d012p-1,
    -0x1.3333333333332p-2
  },
  { // Entry 1022
    0x1.e921dd42f09ba60b268bec1fb0878a42p-1,
    0x1.3333333333333p-2
  },
  { // Entry 1023
    0x1.e921dd42f09ba60b268bec1fb0878a42p-1,
    -0x1.3333333333333p-2
  },
  { // Entry 1024
    0x1.e921dd42f09ba3adecd934c86a0db254p-1,
    0x1.3333333333334p-2
  },
  { // Entry 1025
    0x1.e921dd42f09ba3adecd934c86a0db254p-1,
    -0x1.3333333333334p-2
  },
  { // Entry 1026
    0x1.8feedb86bf0ef3158f8a1dcbef49d123p-1,
    0x1.594317acc4ef8p-1
  },
  { // Entry 1027
    0x1.8feedb86bf0ef3158f8a1dcbef49d123p-1,
    -0x1.594317acc4ef8p-1
  },
  { // Entry 1028
    0x1.8feedb86bf0ee91817a64b28b79e5119p-1,
    0x1.594317acc4ef9p-1
  },
  { // Entry 1029
    0x1.8feedb86bf0ee91817a64b28b79e5119p-1,
    -0x1.594317acc4ef9p-1
  },
  { // Entry 1030
    0x1.8feedb86bf0edf1a9fc278857f8ed559p-1,
    0x1.594317acc4efap-1
  },
  { // Entry 1031
    0x1.8feedb86bf0edf1a9fc278857f8ed559p-1,
    -0x1.594317acc4efap-1
  },
  { // Entry 1032
    0x1.6b898fa9efb5dd6f9e17e3442d59b241p-1,
    0x1.8ffffffffffffp-1
  },
  { // Entry 1033
    0x1.6b898fa9efb5dd6f9e17e3442d59b241p-1,
    -0x1.8ffffffffffffp-1
  },
  { // Entry 1034
    0x1.6b898fa9efb5d22b58f0d99e9634931ap-1,
    0x1.9p-1
  },
  { // Entry 1035
    0x1.6b898fa9efb5d22b58f0d99e9634931ap-1,
    -0x1.9p-1
  },
  { // Entry 1036
    0x1.6b898fa9efb5c6e713c9cff8feb4918fp-1,
    0x1.9000000000001p-1
  },
  { // Entry 1037
    0x1.6b898fa9efb5c6e713c9cff8feb4918fp-1,
    -0x1.9000000000001p-1
  },
  { // Entry 1038
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1074
  },
  { // Entry 1039
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1074
  },
  { // Entry 1040
    0x1.p0,
    -0.0
  },
  { // Entry 1041
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1074
  },
  { // Entry 1042
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1074
  },
  { // Entry 1043
    0x1.ff621e3796d7de4a8ab7d7cd8488f499p-1,
    0x1.921fb54442d17p-5
  },
  { // Entry 1044
    0x1.ff621e3796d7de4a8ab7d7cd8488f499p-1,
    -0x1.921fb54442d17p-5
  },
  { // Entry 1045
    0x1.ff621e3796d7de3dfb04df46987f6450p-1,
    0x1.921fb54442d18p-5
  },
  { // Entry 1046
    0x1.ff621e3796d7de3dfb04df46987f6450p-1,
    -0x1.921fb54442d18p-5
  },
  { // Entry 1047
    0x1.ff621e3796d7de316b51e6bfac75542fp-1,
    0x1.921fb54442d19p-5
  },
  { // Entry 1048
    0x1.ff621e3796d7de316b51e6bfac75542fp-1,
    -0x1.921fb54442d19p-5
  },
  { // Entry 1049
    0x1.fd88da3d12525a208898cf58ded0eeb3p-1,
    0x1.921fb54442d17p-4
  },
  { // Entry 1050
    0x1.fd88da3d12525a208898cf58ded0eeb3p-1,
    -0x1.921fb54442d17p-4
  },
  { // Entry 1051
    0x1.fd88da3d125259ee594b5705767ab649p-1,
    0x1.921fb54442d18p-4
  },
  { // Entry 1052
    0x1.fd88da3d125259ee594b5705767ab649p-1,
    -0x1.921fb54442d18p-4
  },
  { // Entry 1053
    0x1.fd88da3d125259bc29fddeb20e228056p-1,
    0x1.921fb54442d19p-4
  },
  { // Entry 1054
    0x1.fd88da3d125259bc29fddeb20e228056p-1,
    -0x1.921fb54442d19p-4
  },
  { // Entry 1055
    0x1.f6297cff75cb03ab1fced6337e35a245p-1,
    0x1.921fb54442d17p-3
  },
  { // Entry 1056
    0x1.f6297cff75cb03ab1fced6337e35a245p-1,
    -0x1.921fb54442d17p-3
  },
  { // Entry 1057
    0x1.f6297cff75cb02e35a0cf2e64de60626p-1,
    0x1.921fb54442d18p-3
  },
  { // Entry 1058
    0x1.f6297cff75cb02e35a0cf2e64de60626p-1,
    -0x1.921fb54442d18p-3
  },
  { // Entry 1059
    0x1.f6297cff75cb021b944b0f991d8e9161p-1,
    0x1.921fb54442d19p-3
  },
  { // Entry 1060
    0x1.f6297cff75cb021b944b0f991d8e9161p-1,
    -0x1.921fb54442d19p-3
  },
  { // Entry 1061
    0x1.d906bcf328d46672d9c1a865898e5966p-1,
    0x1.921fb54442d17p-2
  },
  { // Entry 1062
    0x1.d906bcf328d46672d9c1a865898e5966p-1,
    -0x1.921fb54442d17p-2
  },
  { // Entry 1063
    0x1.d906bcf328d463631d6cd2905d4b13f5p-1,
    0x1.921fb54442d18p-2
  },
  { // Entry 1064
    0x1.d906bcf328d463631d6cd2905d4b13f5p-1,
    -0x1.921fb54442d18p-2
  },
  { // Entry 1065
    0x1.d906bcf328d460536117fcbb30ea3e17p-1,
    0x1.921fb54442d19p-2
  },
  { // Entry 1066
    0x1.d906bcf328d460536117fcbb30ea3e17p-1,
    -0x1.921fb54442d19p-2
  },
  { // Entry 1067
    0x1.6a09e667f3bcd777b6461376ab523242p-1,
    0x1.921fb54442d17p-1
  },
  { // Entry 1068
    0x1.6a09e667f3bcd777b6461376ab523242p-1,
    -0x1.921fb54442d17p-1
  },
  { // Entry 1069
    0x1.6a09e667f3bccc276712d3d8c5502387p-1,
    0x1.921fb54442d18p-1
  },
  { // Entry 1070
    0x1.6a09e667f3bccc276712d3d8c5502387p-1,
    -0x1.921fb54442d18p-1
  },
  { // Entry 1071
    0x1.6a09e667f3bcc0d717df943adef39253p-1,
    0x1.921fb54442d19p-1
  },
  { // Entry 1072
    0x1.6a09e667f3bcc0d717df943adef39253p-1,
    -0x1.921fb54442d19p-1
  },
  { // Entry 1073
    0x1.469898cc51701b839a252049c0b8b50bp-52,
    0x1.921fb54442d17p0
  },
  { // Entry 1074
    0x1.469898cc51701b839a252049c0b8b50bp-52,
    -0x1.921fb54442d17p0
  },
  { // Entry 1075
    0x1.1a62633145c06e0e6894812704419fa8p-54,
    0x1.921fb54442d18p0
  },
  { // Entry 1076
    0x1.1a62633145c06e0e6894812704419fa8p-54,
    -0x1.921fb54442d18p0
  },
  { // Entry 1077
    -0x1.72cece675d1fc8f8cbb5bf6c7dbcfba0p-53,
    0x1.921fb54442d19p0
  },
  { // Entry 1078
    -0x1.72cece675d1fc8f8cbb5bf6c7dbcfba0p-53,
    -0x1.921fb54442d19p0
  },
  { // Entry 1079
    -0x1.fffffffffffffffffffffffff97d5bffp-1,
    0x1.921fb54442d17p1
  },
  { // Entry 1080
    -0x1.fffffffffffffffffffffffff97d5bffp-1,
    -0x1.921fb54442d17p1
  },
  { // Entry 1081
    -0x1.ffffffffffffffffffffffffffb220c5p-1,
    0x1.921fb54442d18p1
  },
  { // Entry 1082
    -0x1.ffffffffffffffffffffffffffb220c5p-1,
    -0x1.921fb54442d18p1
  },
  { // Entry 1083
    -0x1.fffffffffffffffffffffffffde6e58cp-1,
    0x1.921fb54442d19p1
  },
  { // Entry 1084
    -0x1.fffffffffffffffffffffffffde6e58cp-1,
    -0x1.921fb54442d19p1
  },
  { // Entry 1085
    0x1.ffffffffffffffffffffffffe5f56ffep-1,
    0x1.921fb54442d17p2
  },
  { // Entry 1086
    0x1.ffffffffffffffffffffffffe5f56ffep-1,
    -0x1.921fb54442d17p2
  },
  { // Entry 1087
    0x1.fffffffffffffffffffffffffec88317p-1,
    0x1.921fb54442d18p2
  },
  { // Entry 1088
    0x1.fffffffffffffffffffffffffec88317p-1,
    -0x1.921fb54442d18p2
  },
  { // Entry 1089
    0x1.fffffffffffffffffffffffff79b9631p-1,
    0x1.921fb54442d19p2
  },
  { // Entry 1090
    0x1.fffffffffffffffffffffffff79b9631p-1,
    -0x1.921fb54442d19p2
  },
  { // Entry 1091
    0x1.ffffffffffffffffffffffff97d5bff8p-1,
    0x1.921fb54442d17p3
  },
  { // Entry 1092
    0x1.ffffffffffffffffffffffff97d5bff8p-1,
    -0x1.921fb54442d17p3
  },
  { // Entry 1093
    0x1.fffffffffffffffffffffffffb220c5ep-1,
    0x1.921fb54442d18p3
  },
  { // Entry 1094
    0x1.fffffffffffffffffffffffffb220c5ep-1,
    -0x1.921fb54442d18p3
  },
  { // Entry 1095
    0x1.ffffffffffffffffffffffffde6e58c4p-1,
    0x1.921fb54442d19p3
  },
  { // Entry 1096
    0x1.ffffffffffffffffffffffffde6e58c4p-1,
    -0x1.921fb54442d19p3
  },
  { // Entry 1097
    0x1.fffffffffffffffffffffffe5f56ffe1p-1,
    0x1.921fb54442d17p4
  },
  { // Entry 1098
    0x1.fffffffffffffffffffffffe5f56ffe1p-1,
    -0x1.921fb54442d17p4
  },
  { // Entry 1099
    0x1.ffffffffffffffffffffffffec88317ap-1,
    0x1.921fb54442d18p4
  },
  { // Entry 1100
    0x1.ffffffffffffffffffffffffec88317ap-1,
    -0x1.921fb54442d18p4
  },
  { // Entry 1101
    0x1.ffffffffffffffffffffffff79b96313p-1,
    0x1.921fb54442d19p4
  },
  { // Entry 1102
    0x1.ffffffffffffffffffffffff79b96313p-1,
    -0x1.921fb54442d19p4
  },
  { // Entry 1103
    0x1.fffffffffffffffffffffff97d5bff87p-1,
    0x1.921fb54442d17p5
  },
  { // Entry 1104
    0x1.fffffffffffffffffffffff97d5bff87p-1,
    -0x1.921fb54442d17p5
  },
  { // Entry 1105
    0x1.ffffffffffffffffffffffffb220c5e9p-1,
    0x1.921fb54442d18p5
  },
  { // Entry 1106
    0x1.ffffffffffffffffffffffffb220c5e9p-1,
    -0x1.921fb54442d18p5
  },
  { // Entry 1107
    0x1.fffffffffffffffffffffffde6e58c4cp-1,
    0x1.921fb54442d19p5
  },
  { // Entry 1108
    0x1.fffffffffffffffffffffffde6e58c4cp-1,
    -0x1.921fb54442d19p5
  },
  { // Entry 1109
    0x1.ffffffffffffffffffffffe5f56ffe1dp-1,
    0x1.921fb54442d17p6
  },
  { // Entry 1110
    0x1.ffffffffffffffffffffffe5f56ffe1dp-1,
    -0x1.921fb54442d17p6
  },
  { // Entry 1111
    0x1.fffffffffffffffffffffffec88317a7p-1,
    0x1.921fb54442d18p6
  },
  { // Entry 1112
    0x1.fffffffffffffffffffffffec88317a7p-1,
    -0x1.921fb54442d18p6
  },
  { // Entry 1113
    0x1.fffffffffffffffffffffff79b963131p-1,
    0x1.921fb54442d19p6
  },
  { // Entry 1114
    0x1.fffffffffffffffffffffff79b963131p-1,
    -0x1.921fb54442d19p6
  },
  { // Entry 1115
    0x1.ffffffffffffffffffffff97d5bff874p-1,
    0x1.921fb54442d17p7
  },
  { // Entry 1116
    0x1.ffffffffffffffffffffff97d5bff874p-1,
    -0x1.921fb54442d17p7
  },
  { // Entry 1117
    0x1.fffffffffffffffffffffffb220c5e9dp-1,
    0x1.921fb54442d18p7
  },
  { // Entry 1118
    0x1.fffffffffffffffffffffffb220c5e9dp-1,
    -0x1.921fb54442d18p7
  },
  { // Entry 1119
    0x1.ffffffffffffffffffffffde6e58c4c6p-1,
    0x1.921fb54442d19p7
  },
  { // Entry 1120
    0x1.ffffffffffffffffffffffde6e58c4c6p-1,
    -0x1.921fb54442d19p7
  },
  { // Entry 1121
    -0x1.6a09e667f3bc926b59e6d399bd1b906cp-1,
    0x1.2d97c7f3321d1p1
  },
  { // Entry 1122
    -0x1.6a09e667f3bc926b59e6d399bd1b906cp-1,
    -0x1.2d97c7f3321d1p1
  },
  { // Entry 1123
    -0x1.6a09e667f3bcbfac96b3d2115a3c3e21p-1,
    0x1.2d97c7f3321d2p1
  },
  { // Entry 1124
    -0x1.6a09e667f3bcbfac96b3d2115a3c3e21p-1,
    -0x1.2d97c7f3321d2p1
  },
  { // Entry 1125
    -0x1.6a09e667f3bcecedd380d088f1b4c43cp-1,
    0x1.2d97c7f3321d3p1
  },
  { // Entry 1126
    -0x1.6a09e667f3bcecedd380d088f1b4c43cp-1,
    -0x1.2d97c7f3321d3p1
  },
  { // Entry 1127
    -0x1.6a09e667f3bd05e3743ed417c44fba46p-1,
    0x1.f6a7a2955385dp1
  },
  { // Entry 1128
    -0x1.6a09e667f3bd05e3743ed417c44fba46p-1,
    -0x1.f6a7a2955385dp1
  },
  { // Entry 1129
    -0x1.6a09e667f3bcd8a23771d5a02ff5e843p-1,
    0x1.f6a7a2955385ep1
  },
  { // Entry 1130
    -0x1.6a09e667f3bcd8a23771d5a02ff5e843p-1,
    -0x1.f6a7a2955385ep1
  },
  { // Entry 1131
    -0x1.6a09e667f3bcab60faa4d72895f3eea5p-1,
    0x1.f6a7a2955385fp1
  },
  { // Entry 1132
    -0x1.6a09e667f3bcab60faa4d72895f3eea5p-1,
    -0x1.f6a7a2955385fp1
  },
  { // Entry 1133
    -0x1.34f272993d1414a2b39bd8374c1d1631p-50,
    0x1.2d97c7f3321d1p2
  },
  { // Entry 1134
    -0x1.34f272993d1414a2b39bd8374c1d1631p-50,
    -0x1.2d97c7f3321d1p2
  },
  { // Entry 1135
    -0x1.a79394c9e8a0a5159cdec1ba86377c92p-53,
    0x1.2d97c7f3321d2p2
  },
  { // Entry 1136
    -0x1.a79394c9e8a0a5159cdec1ba86377c92p-53,
    -0x1.2d97c7f3321d2p2
  },
  { // Entry 1137
    0x1.961b1acd85d7d6ba98c84f915bbcbc6cp-51,
    0x1.2d97c7f3321d3p2
  },
  { // Entry 1138
    0x1.961b1acd85d7d6ba98c84f915bbcbc6cp-51,
    -0x1.2d97c7f3321d3p2
  },
  { // Entry 1139
    0x1.6a09e667f3bc58af4cbad35aabb200f4p-1,
    0x1.5fdbbe9bba774p2
  },
  { // Entry 1140
    0x1.6a09e667f3bc58af4cbad35aabb200f4p-1,
    -0x1.5fdbbe9bba774p2
  },
  { // Entry 1141
    0x1.6a09e667f3bcb331c654d049eeba380fp-1,
    0x1.5fdbbe9bba775p2
  },
  { // Entry 1142
    0x1.6a09e667f3bcb331c654d049eeba380fp-1,
    -0x1.5fdbbe9bba775p2
  },
  { // Entry 1143
    0x1.6a09e667f3bd0db43feecd391b21d0c4p-1,
    0x1.5fdbbe9bba776p2
  },
  { // Entry 1144
    0x1.6a09e667f3bd0db43feecd391b21d0c4p-1,
    -0x1.5fdbbe9bba776p2
  },
  { // Entry 1145
    0x1.6a09e667f3bd3f9f816ad456ba1a54a9p-1,
    0x1.c463abeccb2bap2
  },
  { // Entry 1146
    0x1.6a09e667f3bd3f9f816ad456ba1a54a9p-1,
    -0x1.c463abeccb2bap2
  },
  { // Entry 1147
    0x1.6a09e667f3bce51d07d0d7679a2d8c53p-1,
    0x1.c463abeccb2bbp2
  },
  { // Entry 1148
    0x1.6a09e667f3bce51d07d0d7679a2d8c53p-1,
    -0x1.c463abeccb2bbp2
  },
  { // Entry 1149
    0x1.6a09e667f3bc8a9a8e36da7863a02597p-1,
    0x1.c463abeccb2bcp2
  },
  { // Entry 1150
    0x1.6a09e667f3bc8a9a8e36da7863a02597p-1,
    -0x1.c463abeccb2bcp2
  },
  { // Entry 1151
    0x1.583ebeff65cc226480ae685c2ad9afdap-50,
    0x1.f6a7a2955385dp2
  },
  { // Entry 1152
    0x1.583ebeff65cc226480ae685c2ad9afdap-50,
    -0x1.f6a7a2955385dp2
  },
  { // Entry 1153
    0x1.60fafbfd9730899202b9a170c4e6a849p-52,
    0x1.f6a7a2955385ep2
  },
  { // Entry 1154
    0x1.60fafbfd9730899202b9a170c4e6a849p-52,
    -0x1.f6a7a2955385ep2
  },
  { // Entry 1155
    -0x1.4f8282013467bb36fea32f479bd48f4ap-51,
    0x1.f6a7a2955385fp2
  },
  { // Entry 1156
    -0x1.4f8282013467bb36fea32f479bd48f4ap-51,
    -0x1.f6a7a2955385fp2
  },
  { // Entry 1157
    -0x1.6a09e667f3bb972f8927d7b46c737485p-1,
    0x1.1475cc9eedeffp3
  },
  { // Entry 1158
    -0x1.6a09e667f3bb972f8927d7b46c737485p-1,
    -0x1.1475cc9eedeffp3
  },
  { // Entry 1159
    -0x1.6a09e667f3bc4c347c5bd1933ca3261fp-1,
    0x1.1475cc9eedfp3
  },
  { // Entry 1160
    -0x1.6a09e667f3bc4c347c5bd1933ca3261fp-1,
    -0x1.1475cc9eedfp3
  },
  { // Entry 1161
    -0x1.6a09e667f3bd01396f8fcb71b2505e1fp-1,
    0x1.1475cc9eedf01p3
  },
  { // Entry 1162
    -0x1.6a09e667f3bd01396f8fcb71b2505e1fp-1,
    -0x1.1475cc9eedf01p3
  },
  { // Entry 1163
    -0x1.ffffffffffffffffffffffffa2c9eda8p-1,
    0x1.2d97c7f3321d1p3
  },
  { // Entry 1164
    -0x1.ffffffffffffffffffffffffa2c9eda8p-1,
    -0x1.2d97c7f3321d1p3
  },
  { // Entry 1165
    -0x1.fffffffffffffffffffffffffd4326f5p-1,
    0x1.2d97c7f3321d2p3
  },
  { // Entry 1166
    -0x1.fffffffffffffffffffffffffd4326f5p-1,
    -0x1.2d97c7f3321d2p3
  },
  { // Entry 1167
    -0x1.ffffffffffffffffffffffffd7bc6041p-1,
    0x1.2d97c7f3321d3p3
  },
  { // Entry 1168
    -0x1.ffffffffffffffffffffffffd7bc6041p-1,
    -0x1.2d97c7f3321d3p3
  },
  { // Entry 1169
    -0x1.6a09e667f3be011f44fdcffc167f7140p-1,
    0x1.46b9c347764a2p3
  },
  { // Entry 1170
    -0x1.6a09e667f3be011f44fdcffc167f7140p-1,
    -0x1.46b9c347764a2p3
  },
  { // Entry 1171
    -0x1.6a09e667f3bd4c1a51c9d61e20c523f7p-1,
    0x1.46b9c347764a3p3
  },
  { // Entry 1172
    -0x1.6a09e667f3bd4c1a51c9d61e20c523f7p-1,
    -0x1.46b9c347764a3p3
  },
  { // Entry 1173
    -0x1.6a09e667f3bc97155e95dc3fd0885d14p-1,
    0x1.46b9c347764a4p3
  },
  { // Entry 1174
    -0x1.6a09e667f3bc97155e95dc3fd0885d14p-1,
    -0x1.46b9c347764a4p3
  },
  { // Entry 1175
    -0x1.3dc585b2c742181326e07c40748873bbp-49,
    0x1.5fdbbe9bba774p3
  },
  { // Entry 1176
    -0x1.3dc585b2c742181326e07c40748873bbp-49,
    -0x1.5fdbbe9bba774p3
  },
  { // Entry 1177
    -0x1.ee2c2d963a10c0993703e20446463301p-52,
    0x1.5fdbbe9bba775p3
  },
  { // Entry 1178
    -0x1.ee2c2d963a10c0993703e20446463301p-52,
    -0x1.5fdbbe9bba775p3
  },
  { // Entry 1179
    0x1.8474f49a717bcfd9b23f077ee4d090cfp-50,
    0x1.5fdbbe9bba776p3
  },
  { // Entry 1180
    0x1.8474f49a717bcfd9b23f077ee4d090cfp-50,
    -0x1.5fdbbe9bba776p3
  },
  { // Entry 1181
    0x1.6a09e667f3bb8ab4b8c8d5ecf6b910d6p-1,
    0x1.78fdb9effea45p3
  },
  { // Entry 1182
    0x1.6a09e667f3bb8ab4b8c8d5ecf6b910d6p-1,
    -0x1.78fdb9effea45p3
  },
  { // Entry 1183
    0x1.6a09e667f3bc3fb9abfccfcbcd262aa0p-1,
    0x1.78fdb9effea46p3
  },
  { // Entry 1184
    0x1.6a09e667f3bc3fb9abfccfcbcd262aa0p-1,
    -0x1.78fdb9effea46p3
  },
  { // Entry 1185
    0x1.6a09e667f3bcf4be9f30c9aa4910cacfp-1,
    0x1.78fdb9effea47p3
  },
  { // Entry 1186
    0x1.6a09e667f3bcf4be9f30c9aa4910cacfp-1,
    -0x1.78fdb9effea47p3
  },
  { // Entry 1187
    0x1.6a09e667f3be0d9a155cd1c3767eb7b3p-1,
    0x1.ab41b09886fe8p3
  },
  { // Entry 1188
    0x1.6a09e667f3be0d9a155cd1c3767eb7b3p-1,
    -0x1.ab41b09886fe8p3
  },
  { // Entry 1189
    0x1.6a09e667f3bd58952228d7e58701d299p-1,
    0x1.ab41b09886fe9p3
  },
  { // Entry 1190
    0x1.6a09e667f3bd58952228d7e58701d299p-1,
    -0x1.ab41b09886fe9p3
  },
  { // Entry 1191
    0x1.6a09e667f3bca3902ef4de073d0273e6p-1,
    0x1.ab41b09886feap3
  },
  { // Entry 1192
    0x1.6a09e667f3bca3902ef4de073d0273e6p-1,
    -0x1.ab41b09886feap3
  },
  { // Entry 1193
    0x1.4f6babe5db9e1ef40d69c452e135591dp-49,
    0x1.c463abeccb2bap3
  },
  { // Entry 1194
    0x1.4f6babe5db9e1ef40d69c452e135591dp-49,
    -0x1.c463abeccb2bap3
  },
  { // Entry 1195
    0x1.3daeaf976e787bd035a7114be387b5c3p-51,
    0x1.c463abeccb2bbp3
  },
  { // Entry 1196
    0x1.3daeaf976e787bd035a7114be387b5c3p-51,
    -0x1.c463abeccb2bbp3
  },
  { // Entry 1197
    -0x1.6128a83448c3c217e52c775a0698d26ap-50,
    0x1.c463abeccb2bcp3
  },
  { // Entry 1198
    -0x1.6128a83448c3c217e52c775a0698d26ap-50,
    -0x1.c463abeccb2bcp3
  },
  { // Entry 1199
    -0x1.6a09e667f3bb7e39e869d42580908c7cp-1,
    0x1.dd85a7410f58bp3
  },
  { // Entry 1200
    -0x1.6a09e667f3bb7e39e869d42580908c7cp-1,
    -0x1.dd85a7410f58bp3
  },
  { // Entry 1201
    -0x1.6a09e667f3bc333edb9dce045d3b0e75p-1,
    0x1.dd85a7410f58cp3
  },
  { // Entry 1202
    -0x1.6a09e667f3bc333edb9dce045d3b0e75p-1,
    -0x1.dd85a7410f58cp3
  },
  { // Entry 1203
    -0x1.6a09e667f3bce843ced1c7e2df6316d4p-1,
    0x1.dd85a7410f58dp3
  },
  { // Entry 1204
    -0x1.6a09e667f3bce843ced1c7e2df6316d4p-1,
    -0x1.dd85a7410f58dp3
  },
  { // Entry 1205
    -0x1.ffffffffffffffffffffffff8c45d3d4p-1,
    0x1.f6a7a2955385dp3
  },
  { // Entry 1206
    -0x1.ffffffffffffffffffffffff8c45d3d4p-1,
    -0x1.f6a7a2955385dp3
  },
  { // Entry 1207
    -0x1.fffffffffffffffffffffffff8653353p-1,
    0x1.f6a7a2955385ep3
  },
  { // Entry 1208
    -0x1.fffffffffffffffffffffffff8653353p-1,
    -0x1.f6a7a2955385ep3
  },
  { // Entry 1209
    -0x1.ffffffffffffffffffffffffe48492d3p-1,
    0x1.f6a7a2955385fp3
  },
  { // Entry 1210
    -0x1.ffffffffffffffffffffffffe48492d3p-1,
    -0x1.f6a7a2955385fp3
  },
  { // Entry 1211
    -0x1.6a09e667f3bf841ecc23c74599076a81p-1,
    0x1.07e4cef4cbd96p4
  },
  { // Entry 1212
    -0x1.6a09e667f3bf841ecc23c74599076a81p-1,
    -0x1.07e4cef4cbd96p4
  },
  { // Entry 1213
    -0x1.6a09e667f3be1a14e5bbd38ad60fdd7bp-1,
    0x1.07e4cef4cbd97p4
  },
  { // Entry 1214
    -0x1.6a09e667f3be1a14e5bbd38ad60fdd7bp-1,
    -0x1.07e4cef4cbd97p4
  },
  { // Entry 1215
    -0x1.6a09e667f3bcb00aff53dfcea90e6a0cp-1,
    0x1.07e4cef4cbd98p4
  },
  { // Entry 1216
    -0x1.6a09e667f3bcb00aff53dfcea90e6a0cp-1,
    -0x1.07e4cef4cbd98p4
  },
  { // Entry 1217
    -0x1.b088e90c77fd12ea79f98631e6f0b74bp-48,
    0x1.1475cc9eedeffp4
  },
  { // Entry 1218
    -0x1.b088e90c77fd12ea79f98631e6f0b74bp-48,
    -0x1.1475cc9eedeffp4
  },
  { // Entry 1219
    -0x1.6111d218effa25d4f3f30c654d7c36a1p-49,
    0x1.1475cc9eedfp4
  },
  { // Entry 1220
    -0x1.6111d218effa25d4f3f30c654d7c36a1p-49,
    -0x1.1475cc9eedfp4
  },
  { // Entry 1221
    0x1.3ddc5bce200bb4561819e73527f5a6d7p-50,
    0x1.1475cc9eedf01p4
  },
  { // Entry 1222
    0x1.3ddc5bce200bb4561819e73527f5a6d7p-50,
    -0x1.1475cc9eedf01p4
  },
  { // Entry 1223
    0x1.6a09e667f3babcba24d6d87ecc8f83b4p-1,
    0x1.2106ca4910068p4
  },
  { // Entry 1224
    0x1.6a09e667f3babcba24d6d87ecc8f83b4p-1,
    -0x1.2106ca4910068p4
  },
  { // Entry 1225
    0x1.6a09e667f3bc26c40b3ecc3cece1d1a0p-1,
    0x1.2106ca4910069p4
  },
  { // Entry 1226
    0x1.6a09e667f3bc26c40b3ecc3cece1d1a0p-1,
    -0x1.2106ca4910069p4
  },
  { // Entry 1227
    0x1.6a09e667f3bd90cdf1a6bff9a32a3923p-1,
    0x1.2106ca491006ap4
  },
  { // Entry 1228
    0x1.6a09e667f3bd90cdf1a6bff9a32a3923p-1,
    -0x1.2106ca491006ap4
  },
  { // Entry 1229
    0x1.fffffffffffffffffffffffe8b27b6a2p-1,
    0x1.2d97c7f3321d1p4
  },
  { // Entry 1230
    0x1.fffffffffffffffffffffffe8b27b6a2p-1,
    -0x1.2d97c7f3321d1p4
  },
  { // Entry 1231
    0x1.fffffffffffffffffffffffff50c9bd4p-1,
    0x1.2d97c7f3321d2p4
  },
  { // Entry 1232
    0x1.fffffffffffffffffffffffff50c9bd4p-1,
    -0x1.2d97c7f3321d2p4
  },
  { // Entry 1233
    0x1.ffffffffffffffffffffffff5ef18107p-1,
    0x1.2d97c7f3321d3p4
  },
  { // Entry 1234
    0x1.ffffffffffffffffffffffff5ef18107p-1,
    -0x1.2d97c7f3321d3p4
  },
  { // Entry 1235
    0x1.6a09e667f3bf90999c82c90cebaf9f3fp-1,
    0x1.3a28c59d54339p4
  },
  { // Entry 1236
    0x1.6a09e667f3bf90999c82c90cebaf9f3fp-1,
    -0x1.3a28c59d54339p4
  },
  { // Entry 1237
    0x1.6a09e667f3be268fb61ad5523532e298p-1,
    0x1.3a28c59d5433ap4
  },
  { // Entry 1238
    0x1.6a09e667f3be268fb61ad5523532e298p-1,
    -0x1.3a28c59d5433ap4
  },
  { // Entry 1239
    0x1.6a09e667f3bcbc85cfb2e19614ac3f88p-1,
    0x1.3a28c59d5433bp4
  },
  { // Entry 1240
    0x1.6a09e667f3bcbc85cfb2e19614ac3f88p-1,
    -0x1.3a28c59d5433bp4
  },
  { // Entry 1241
    0x1.b95bfc26022b165aed3e2a3b12382479p-48,
    0x1.46b9c347764a2p4
  },
  { // Entry 1242
    0x1.b95bfc26022b165aed3e2a3b12382479p-48,
    -0x1.46b9c347764a2p4
  },
  { // Entry 1243
    0x1.72b7f84c04562cb5da7c5477b957adebp-49,
    0x1.46b9c347764a3p4
  },
  { // Entry 1244
    0x1.72b7f84c04562cb5da7c5477b957adebp-49,
    -0x1.46b9c347764a3p4
  },
  { // Entry 1245
    -0x1.1a900f67f753a6944b07571048f1cad2p-50,
    0x1.46b9c347764a4p4
  },
  { // Entry 1246
    -0x1.1a900f67f753a6944b07571048f1cad2p-50,
    -0x1.46b9c347764a4p4
  },
  { // Entry 1247
    -0x1.6a09e667f3bab03f5477d6b74f4d55d5p-1,
    0x1.534ac0f19860bp4
  },
  { // Entry 1248
    -0x1.6a09e667f3bab03f5477d6b74f4d55d5p-1,
    -0x1.534ac0f19860bp4
  },
  { // Entry 1249
    -0x1.6a09e667f3bc1a493adfca757c1a741fp-1,
    0x1.534ac0f19860cp4
  },
  { // Entry 1250
    -0x1.6a09e667f3bc1a493adfca757c1a741fp-1,
    -0x1.534ac0f19860cp4
  },
  { // Entry 1251
    -0x1.6a09e667f3bd84532147be323eddac01p-1,
    0x1.534ac0f19860dp4
  },
  { // Entry 1252
    -0x1.6a09e667f3bd84532147be323eddac01p-1,
    -0x1.534ac0f19860dp4
  },
  { // Entry 1253
    -0x1.fffffffffffffffffffffffe758d3a7cp-1,
    0x1.5fdbbe9bba774p4
  },
  { // Entry 1254
    -0x1.fffffffffffffffffffffffe758d3a7cp-1,
    -0x1.5fdbbe9bba774p4
  },
  { // Entry 1255
    -0x1.fffffffffffffffffffffffff11845e1p-1,
    0x1.5fdbbe9bba775p4
  },
  { // Entry 1256
    -0x1.fffffffffffffffffffffffff11845e1p-1,
    -0x1.5fdbbe9bba775p4
  },
  { // Entry 1257
    -0x1.ffffffffffffffffffffffff6ca35147p-1,
    0x1.5fdbbe9bba776p4
  },
  { // Entry 1258
    -0x1.ffffffffffffffffffffffff6ca35147p-1,
    -0x1.5fdbbe9bba776p4
  },
  { // Entry 1259
    -0x1.6a09e667f3bf9d146ce1cad43de9b352p-1,
    0x1.6c6cbc45dc8dcp4
  },
  { // Entry 1260
    -0x1.6a09e667f3bf9d146ce1cad43de9b352p-1,
    -0x1.6c6cbc45dc8dcp4
  },
  { // Entry 1261
    -0x1.6a09e667f3be330a8679d71993e7c709p-1,
    0x1.6c6cbc45dc8ddp4
  },
  { // Entry 1262
    -0x1.6a09e667f3be330a8679d71993e7c709p-1,
    -0x1.6c6cbc45dc8ddp4
  },
  { // Entry 1263
    -0x1.6a09e667f3bcc900a011e35d7fdbf459p-1,
    0x1.6c6cbc45dc8dep4
  },
  { // Entry 1264
    -0x1.6a09e667f3bcc900a011e35d7fdbf459p-1,
    -0x1.6c6cbc45dc8dep4
  },
  { // Entry 1265
    -0x1.c22f0f3f8c5919cb6082ce443cf95022p-48,
    0x1.78fdb9effea45p4
  },
  { // Entry 1266
    -0x1.c22f0f3f8c5919cb6082ce443cf95022p-48,
    -0x1.78fdb9effea45p4
  },
  { // Entry 1267
    -0x1.845e1e7f18b23396c1059c8a24c2609cp-49,
    0x1.78fdb9effea46p4
  },
  { // Entry 1268
    -0x1.845e1e7f18b23396c1059c8a24c2609cp-49,
    -0x1.78fdb9effea46p4
  },
  { // Entry 1269
    0x1.ee8786039d3731a4fbe98dd6d32ff62ap-51,
    0x1.78fdb9effea47p4
  },
  { // Entry 1270
    0x1.ee8786039d3731a4fbe98dd6d32ff62ap-51,
    -0x1.78fdb9effea47p4
  },
  { // Entry 1271
    0x1.6a09e667f3baa3c48418d4efd19d074ap-1,
    0x1.858eb79a20baep4
  },
  { // Entry 1272
    0x1.6a09e667f3baa3c48418d4efd19d074ap-1,
    -0x1.858eb79a20baep4
  },
  { // Entry 1273
    0x1.6a09e667f3bc0dce6a80c8ae0ae4f5f3p-1,
    0x1.858eb79a20bafp4
  },
  { // Entry 1274
    0x1.6a09e667f3bc0dce6a80c8ae0ae4f5f3p-1,
    -0x1.858eb79a20bafp4
  },
  { // Entry 1275
    0x1.6a09e667f3bd77d850e8bc6ada22fe34p-1,
    0x1.858eb79a20bb0p4
  },
  { // Entry 1276
    0x1.6a09e667f3bd77d850e8bc6ada22fe34p-1,
    -0x1.858eb79a20bb0p4
  },
  { // Entry 1277
    -0x1.2ccaf641d4261ea3ccd8d2a102dc3066p-3,
    0x1.fffffffffffffp62
  },
  { // Entry 1278
    -0x1.2ccaf641d4261ea3ccd8d2a102dc3066p-3,
    -0x1.fffffffffffffp62
  },
  { // Entry 1279
    0x1.82aa375b3c33e70663731bab4beb6ed3p-7,
    0x1.0p63
  },
  { // Entry 1280
    0x1.82aa375b3c33e70663731bab4beb6ed3p-7,
    -0x1.0p63
  },
  { // Entry 1281
    0x1.4c0622a6e35ddefd5f2b7700716a581fp-2,
    0x1.0000000000001p63
  },
  { // Entry 1282
    0x1.4c0622a6e35ddefd5f2b7700716a581fp-2,
    -0x1.0000000000001p63
  },
  { // Entry 1283
    0x1.4ab650b8c60734385375e22603fcdb10p-1,
    0x1.fffffffffffffp26
  },
  { // Entry 1284
    0x1.4ab650b8c60734385375e22603fcdb10p-1,
    -0x1.fffffffffffffp26
  },
  { // Entry 1285
    0x1.4ab6511a7d39ad3cc88ded1e775ca147p-1,
    0x1.0p27
  },
  { // Entry 1286
    0x1.4ab6511a7d39ad3cc88ded1e775ca147p-1,
    -0x1.0p27
  },
  { // Entry 1287
    0x1.4ab651ddeb9e61438382f02167ade4afp-1,
    0x1.0000000000001p27
  },
  { // Entry 1288
    0x1.4ab651ddeb9e61438382f02167ade4afp-1,
    -0x1.0000000000001p27
  },
  { // Entry 1289
    0x1.40ad67e777b1df0195189f50a8c26345p-1,
    0x1.fffffffffffffp23
  },
  { // Entry 1290
    0x1.40ad67e777b1df0195189f50a8c26345p-1,
    -0x1.fffffffffffffp23
  },
  { // Entry 1291
    0x1.40ad67f3f0c9a143963c9c96dbce3f8ap-1,
    0x1.0p24
  },
  { // Entry 1292
    0x1.40ad67f3f0c9a143963c9c96dbce3f8ap-1,
    -0x1.0p24
  },
  { // Entry 1293
    0x1.40ad680ce2f924d716769d1064bc8defp-1,
    0x1.0000000000001p24
  },
  { // Entry 1294
    0x1.40ad680ce2f924d716769d1064bc8defp-1,
    -0x1.0000000000001p24
  },
  { // Entry 1295
    -0x1.4eaa606db24c3cb5e15d7b19d5ee73a1p-1,
    0x1.fffffffffffffp1
  },
  { // Entry 1296
    -0x1.4eaa606db24c3cb5e15d7b19d5ee73a1p-1,
    -0x1.fffffffffffffp1
  },
  { // Entry 1297
    -0x1.4eaa606db24c0c466da1c2dc7baa2b32p-1,
    0x1.0p2
  },
  { // Entry 1298
    -0x1.4eaa606db24c0c466da1c2dc7baa2b32p-1,
    -0x1.0p2
  },
  { // Entry 1299
    -0x1.4eaa606db24bab67862a5261b7719dcfp-1,
    0x1.0000000000001p2
  },
  { // Entry 1300
    -0x1.4eaa606db24bab67862a5261b7719dcfp-1,
    -0x1.0000000000001p2
  },
  { // Entry 1301
    -0x1.aa2265753720101145230952e61c6f2cp-2,
    0x1.fffffffffffffp0
  },
  { // Entry 1302
    -0x1.aa2265753720101145230952e61c6f2cp-2,
    -0x1.fffffffffffffp0
  },
  { // Entry 1303
    -0x1.aa22657537204a4332f8acbb72b0d768p-2,
    0x1.0p1
  },
  { // Entry 1304
    -0x1.aa22657537204a4332f8acbb72b0d768p-2,
    -0x1.0p1
  },
  { // Entry 1305
    -0x1.aa2265753720bea70ea3f38c86db40afp-2,
    0x1.0000000000001p1
  },
  { // Entry 1306
    -0x1.aa2265753720bea70ea3f38c86db40afp-2,
    -0x1.0000000000001p1
  },
  { // Entry 1307
    0x1.14a280fb5068c69a2ed45fb547c2f271p-1,
    0x1.fffffffffffffp-1
  },
  { // Entry 1308
    0x1.14a280fb5068c69a2ed45fb547c2f271p-1,
    -0x1.fffffffffffffp-1
  },
  { // Entry 1309
    0x1.14a280fb5068b923848cdb2ed0e37a53p-1,
    0x1.0p0
  },
  { // Entry 1310
    0x1.14a280fb5068b923848cdb2ed0e37a53p-1,
    -0x1.0p0
  },
  { // Entry 1311
    0x1.14a280fb50689e362ffdd221e2551035p-1,
    0x1.0000000000001p0
  },
  { // Entry 1312
    0x1.14a280fb50689e362ffdd221e2551035p-1,
    -0x1.0000000000001p0
  },
  { // Entry 1313
    0x1.c1528065b7d4fdb158c9c4a6b3e4b740p-1,
    0x1.fffffffffffffp-2
  },
  { // Entry 1314
    0x1.c1528065b7d4fdb158c9c4a6b3e4b740p-1,
    -0x1.fffffffffffffp-2
  },
  { // Entry 1315
    0x1.c1528065b7d4f9db7bbb3b45f5f5b30ap-1,
    0x1.0p-1
  },
  { // Entry 1316
    0x1.c1528065b7d4f9db7bbb3b45f5f5b30ap-1,
    -0x1.0p-1
  },
  { // Entry 1317
    0x1.c1528065b7d4f22fc19e288479c36b27p-1,
    0x1.0000000000001p-1
  },
  { // Entry 1318
    0x1.c1528065b7d4f22fc19e288479c36b27p-1,
    -0x1.0000000000001p-1
  },
  { // Entry 1319
    0x1.f01549f7deea184c5f1d210b6adbe56cp-1,
    0x1.fffffffffffffp-3
  },
  { // Entry 1320
    0x1.f01549f7deea184c5f1d210b6adbe56cp-1,
    -0x1.fffffffffffffp-3
  },
  { // Entry 1321
    0x1.f01549f7deea174f07a67972bf29f148p-1,
    0x1.0p-2
  },
  { // Entry 1322
    0x1.f01549f7deea174f07a67972bf29f148p-1,
    -0x1.0p-2
  },
  { // Entry 1323
    0x1.f01549f7deea155458b92a4167aec7ffp-1,
    0x1.0000000000001p-2
  },
  { // Entry 1324
    0x1.f01549f7deea155458b92a4167aec7ffp-1,
    -0x1.0000000000001p-2
  },
  { // Entry 1325
    0x1.fc015527d5bd371a12320249ca4dafd7p-1,
    0x1.fffffffffffffp-4
  },
  { // Entry 1326
    0x1.fc015527d5bd371a12320249ca4dafd7p-1,
    -0x1.fffffffffffffp-4
  },
  { // Entry 1327
    0x1.fc015527d5bd36da3cd4253bede319cap-1,
    0x1.0p-3
  },
  { // Entry 1328
    0x1.fc015527d5bd36da3cd4253bede319cap-1,
    -0x1.0p-3
  },
  { // Entry 1329
    0x1.fc015527d5bd365a92186b203507f9adp-1,
    0x1.0000000000001p-3
  },
  { // Entry 1330
    0x1.fc015527d5bd365a92186b203507f9adp-1,
    -0x1.0000000000001p-3
  },
  { // Entry 1331
    0x1.ff0015549f4d34da0b745dc7433145efp-1,
    0x1.fffffffffffffp-5
  },
  { // Entry 1332
    0x1.ff0015549f4d34da0b745dc7433145efp-1,
    -0x1.fffffffffffffp-5
  },
  { // Entry 1333
    0x1.ff0015549f4d34ca0e1ee6509bc42b71p-1,
    0x1.0p-4
  },
  { // Entry 1334
    0x1.ff0015549f4d34ca0e1ee6509bc42b71p-1,
    -0x1.0p-4
  },
  { // Entry 1335
    0x1.ff0015549f4d34aa1373f7634ce87737p-1,
    0x1.0000000000001p-4
  },
  { // Entry 1336
    0x1.ff0015549f4d34aa1373f7634ce87737p-1,
    -0x1.0000000000001p-4
  },
  { // Entry 1337
    0x1.ffc00155527d2b16aeb09fb70636a10ap-1,
    0x1.fffffffffffffp-6
  },
  { // Entry 1338
    0x1.ffc00155527d2b16aeb09fb70636a10ap-1,
    -0x1.fffffffffffffp-6
  },
  { // Entry 1339
    0x1.ffc00155527d2b12aedb49d92928df72p-1,
    0x1.0p-5
  },
  { // Entry 1340
    0x1.ffc00155527d2b12aedb49d92928df72p-1,
    -0x1.0p-5
  },
  { // Entry 1341
    0x1.ffc00155527d2b0aaf309e1d6f0cfc4ep-1,
    0x1.0000000000001p-5
  },
  { // Entry 1342
    0x1.ffc00155527d2b0aaf309e1d6f0cfc4ep-1,
    -0x1.0000000000001p-5
  },
  { // Entry 1343
    0x1.fff000155549f4a38a2563ef344c3ff4p-1,
    0x1.fffffffffffffp-7
  },
  { // Entry 1344
    0x1.fff000155549f4a38a2563ef344c3ff4p-1,
    -0x1.fffffffffffffp-7
  },
  { // Entry 1345
    0x1.fff000155549f4a28a280e97bcd59c8ap-1,
    0x1.0p-6
  },
  { // Entry 1346
    0x1.fff000155549f4a28a280e97bcd59c8ap-1,
    -0x1.0p-6
  },
  { // Entry 1347
    0x1.fff000155549f4a08a2d63e8cde83db5p-1,
    0x1.0000000000001p-6
  },
  { // Entry 1348
    0x1.fff000155549f4a08a2d63e8cde83db5p-1,
    -0x1.0000000000001p-6
  },
  { // Entry 1349
    0x1.fffffff00000001555565549f49c9f4dp-1,
    0x1.fffffffffffffp-15
  },
  { // Entry 1350
    0x1.fffffff00000001555565549f49c9f4dp-1,
    -0x1.fffffffffffffp-15
  },
  { // Entry 1351
    0x1.fffffff00000001555555549f49f49f7p-1,
    0x1.0p-14
  },
  { // Entry 1352
    0x1.fffffff00000001555555549f49f49f7p-1,
    -0x1.0p-14
  },
  { // Entry 1353
    0x1.fffffff00000001555535549f4a49f4dp-1,
    0x1.0000000000001p-14
  },
  { // Entry 1354
    0x1.fffffff00000001555535549f4a49f4dp-1,
    -0x1.0000000000001p-14
  },
  { // Entry 1355
    0x1.fffffffffffffc000000000000415555p-1,
    0x1.fffffffffffffp-28
  },
  { // Entry 1356
    0x1.fffffffffffffc000000000000415555p-1,
    -0x1.fffffffffffffp-28
  },
  { // Entry 1357
    0x1.fffffffffffffc000000000000015555p-1,
    0x1.0p-27
  },
  { // Entry 1358
    0x1.fffffffffffffc000000000000015555p-1,
    -0x1.0p-27
  },
  { // Entry 1359
    0x1.fffffffffffffbffffffffffff815555p-1,
    0x1.0000000000001p-27
  },
  { // Entry 1360
    0x1.fffffffffffffbffffffffffff815555p-1,
    -0x1.0000000000001p-27
  },
  { // Entry 1361
    0x1.fffffffffffffff00000000000010015p-1,
    0x1.fffffffffffffp-31
  },
  { // Entry 1362
    0x1.fffffffffffffff00000000000010015p-1,
    -0x1.fffffffffffffp-31
  },
  { // Entry 1363
    0x1.fffffffffffffff00000000000000015p-1,
    0x1.0p-30
  },
  { // Entry 1364
    0x1.fffffffffffffff00000000000000015p-1,
    -0x1.0p-30
  },
  { // Entry 1365
    0x1.ffffffffffffffeffffffffffffe0015p-1,
    0x1.0000000000001p-30
  },
  { // Entry 1366
    0x1.ffffffffffffffeffffffffffffe0015p-1,
    -0x1.0000000000001p-30
  },
  { // Entry 1367
    -0x1.fffe62ecfab753c071b2680e1e26bbcep-1,
    -0x1.fffffffffffffp1023
  },
  { // Entry 1368
    -0x1.fffe62ecfab753c071b2680e1e26bbcep-1,
    0x1.fffffffffffffp1023
  },
  { // Entry 1369
    -0x1.fffe62ecfab753c071b2680e1e26bbcep-1,
    0x1.fffffffffffffp1023
  },
  { // Entry 1370
    -0x1.fffe62ecfab753c071b2680e1e26bbcep-1,
    -0x1.fffffffffffffp1023
  },
  { // Entry 1371
    -0x1.fffe62ecfab753c071b2680e1e26bbcep-1,
    0x1.fffffffffffffp1023
  },
  { // Entry 1372
    -0x1.fffe62ecfab753c071b2680e1e26bbcep-1,
    -0x1.fffffffffffffp1023
  },
  { // Entry 1373
    0x1.7ffdfb4c5308f777cf774c733b256695p-2,
    0x1.ffffffffffffep1023
  },
  { // Entry 1374
    0x1.7ffdfb4c5308f777cf774c733b256695p-2,
    -0x1.ffffffffffffep1023
  },
  { // Entry 1375
    -0x1.ffffffffffffffffffffffffffb220c5p-1,
    0x1.921fb54442d18p1
  },
  { // Entry 1376
    -0x1.ffffffffffffffffffffffffffb220c5p-1,
    -0x1.921fb54442d18p1
  },
  { // Entry 1377
    0x1.1a62633145c06e0e6894812704419fa8p-54,
    0x1.921fb54442d18p0
  },
  { // Entry 1378
    0x1.1a62633145c06e0e6894812704419fa8p-54,
    -0x1.921fb54442d18p0
  },
  { // Entry 1379
    0x1.14a280fb50689e362ffdd221e2551035p-1,
    0x1.0000000000001p0
  },
  { // Entry 1380
    0x1.14a280fb50689e362ffdd221e2551035p-1,
    -0x1.0000000000001p0
  },
  { // Entry 1381
    0x1.14a280fb5068b923848cdb2ed0e37a53p-1,
    0x1.0p0
  },
  { // Entry 1382
    0x1.14a280fb5068b923848cdb2ed0e37a53p-1,
    -0x1.0p0
  },
  { // Entry 1383
    0x1.14a280fb5068c69a2ed45fb547c2f271p-1,
    0x1.fffffffffffffp-1
  },
  { // Entry 1384
    0x1.14a280fb5068c69a2ed45fb547c2f271p-1,
    -0x1.fffffffffffffp-1
  },
  { // Entry 1385
    0x1.6a09e667f3bccc276712d3d8c5502387p-1,
    0x1.921fb54442d18p-1
  },
  { // Entry 1386
    0x1.6a09e667f3bccc276712d3d8c5502387p-1,
    -0x1.921fb54442d18p-1
  },
  { // Entry 1387
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0000000000001p-1022
  },
  { // Entry 1388
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0000000000001p-1022
  },
  { // Entry 1389
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1022
  },
  { // Entry 1390
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1022
  },
  { // Entry 1391
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.ffffffffffffep-1023
  },
  { // Entry 1392
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.ffffffffffffep-1023
  },
  { // Entry 1393
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.ffffffffffffcp-1023
  },
  { // Entry 1394
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.ffffffffffffcp-1023
  },
  { // Entry 1395
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1073
  },
  { // Entry 1396
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1073
  },
  { // Entry 1397
    0x1.ffffffffffffffffffffffffffffffffp-1,
    0x1.0p-1074
  },
  { // Entry 1398
    0x1.ffffffffffffffffffffffffffffffffp-1,
    -0x1.0p-1074
  },
  { // Entry 1399
    0x1.p0,
    0.0
  },
  { // Entry 1400
    0x1.p0,
    -0.0
  },
};
#endif // __BIONIC__

TEST(math_cos, cos_intel) {
#if defined(__BIONIC__)
  fesetenv(FE_DFL_ENV);
  for (size_t i = 0; i < sizeof(g_cos_intel_data)/sizeof(cos_intel_data_t); i++) {
    EXPECT_DOUBLE_EQ(g_cos_intel_data[i].expected, cos(g_cos_intel_data[i].call_data)) << "Failed on element " << i;
  }
#else // __BIONIC__
  GTEST_LOG_(INFO) << "This test does nothing.";
#endif // __BIONIC__
}
