/*-
 * Copyright (c) 2008 Stephen L. Moshier <steve@moshier.net>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*							cpowf
 *
 *	Complex power function
 *
 *
 *
 * SYNOPSIS:
 *
 * float complex cpowf();
 * float complex a, z, w;
 *
 * w = cpowf (a, z);
 *
 *
 *
 * DESCRIPTION:
 *
 * Raises complex A to the complex Zth power.
 * Definition is per AMS55 # 4.2.8,
 * analytically equivalent to cpow(a,z) = cexp(z clog(a)).
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -10,+10     30000       9.4e-15     1.5e-15
 *
 */

#include <complex.h>
#include <math.h>
#include "math_private.h"

float complex
cpowf(float complex a, float complex z)
{
	float complex w;
	float x, y, r, theta, absa, arga;

	x = crealf(z);
	y = cimagf(z);
	absa = cabsf (a);
	if (absa == 0.0f) {
		return (CMPLXF(0.0f, 0.0f));
	}
	arga = cargf (a);
	r = powf (absa, x);
	theta = x * arga;
	if (y != 0.0f) {
		r = r * expf (-y * arga);
		theta = theta + y * logf (absa);
	}
	w = CMPLXF(r * cosf (theta), r * sinf (theta));
	return (w);
}
