/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <fcntl.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <regex>
#include <string>
#include <vector>

#include <android-base/file.h>
#include <gtest/gtest.h>

class SystemTests : public ::testing::Test {
 protected:
  void SetUp() override {
    raw_output_ = "";
    sanitized_output_ = "";
    exitcode_ = 0;
  }

  void SanitizeOutput();

  void Exec(std::vector<const char*> args);
  void RunTest(int expected_exitcode, std::vector<const char*> extra_args = {});
  void Verify(const std::string& expected_output, int expected_exitcode,
              std::vector<const char*> extra_args = {}, bool sanitize = true);

  std::string raw_output_;
  std::string sanitized_output_;
  int exitcode_;
  pid_t pid_;
  int fd_;
};

static std::string GetBenchmarkExe() {
  return android::base::GetExecutableDirectory() + "/bionic-benchmarks";
}

static std::string GetBionicXmlArg(const char* xml_file) {
  return "--bionic_xml=" + android::base::GetExecutableDirectory() + "/test_suites/" + xml_file;
}

void SystemTests::SanitizeOutput() {
  // Cut off anything after the arguments, since that varies with time.
  sanitized_output_ = std::regex_replace(raw_output_, std::regex(".*(BM_\\S+)\\s+.+"), "$1");

  // Remove everything before the header.
  sanitized_output_.erase(0, sanitized_output_.find("------------------------------------------------"));

  // Remove the header.
  sanitized_output_.erase(0, sanitized_output_.find("BM_"));

  // Remove any hanging output.
  sanitized_output_.erase(sanitized_output_.find_last_of("BM_\\S+\n") + 1);
}

void SystemTests::Exec(std::vector<const char*> args) {
  int fds[2];
  ASSERT_NE(-1, pipe(fds));
  ASSERT_NE(-1, fcntl(fds[0], F_SETFL, O_NONBLOCK));

  if ((pid_ = fork()) == 0) {
    // Run the test.
    close(fds[0]);
    close(STDIN_FILENO);
    close(STDOUT_FILENO);
    close(STDERR_FILENO);
    ASSERT_NE(0, dup2(fds[1], STDOUT_FILENO));
    ASSERT_NE(0, dup2(fds[1], STDERR_FILENO));
    close(fds[1]);

    const std::string exe(GetBenchmarkExe());
    args.insert(args.begin(), exe.c_str());
    args.push_back(nullptr);
    execv(args[0], reinterpret_cast<char* const*>(const_cast<char**>(args.data())));
    exit(1);
  }
  ASSERT_NE(-1, pid_);

  close(fds[1]);
  fd_ = fds[0];
}

void SystemTests::Verify(const std::string& expected_output,
                         int expected_exitcode, std::vector<const char*> extra_args, bool sanitize) {
  RunTest(expected_exitcode, extra_args);

  if (sanitize) {
    SanitizeOutput();
    ASSERT_EQ(expected_output, sanitized_output_);
  } else {
    ASSERT_EQ(expected_output, raw_output_);
  }
}

void SystemTests::RunTest(int expected_exitcode, std::vector<const char*> extra_args) {
  std::vector<const char*> args;
  for (const auto& arg : extra_args) {
    args.push_back(arg);
  }

  Exec(args);

  raw_output_ = "";
  while (true) {
    char buffer[4097];
    ssize_t bytes = TEMP_FAILURE_RETRY(read(fd_, buffer, sizeof(buffer) - 1));
    if (bytes == -1 && errno == EAGAIN) {
      continue;
    }
    ASSERT_NE(-1, bytes);
    if (bytes == 0) {
      break;
    }
    buffer[bytes] = '\0';
    raw_output_ += buffer;
  }
  close(fd_);

  int status;
  ASSERT_EQ(pid_, TEMP_FAILURE_RETRY(waitpid(pid_, &status, 0))) << "Test output:\n" << raw_output_;
  exitcode_ = WEXITSTATUS(status);
  ASSERT_EQ(expected_exitcode, exitcode_) << "Test output:\n" << raw_output_;
}

TEST_F(SystemTests, check_benchmark_exe) {
  // Verify that the benchmark exe is present.
  const std::string exe(GetBenchmarkExe());

  struct stat st;
  ASSERT_NE(-1, stat(exe.c_str(), &st)) << "Stat of " << exe << " failed";
  ASSERT_TRUE(S_ISREG(st.st_mode)) << exe << " is not a file, or doesn't exist.";
}

TEST_F(SystemTests, help) {
  std::string expected =
    "Usage:\n"
    "bionic_benchmarks [--bionic_cpu=<cpu_to_isolate>]\n"
    "                  [--bionic_xml=<path_to_xml>]\n"
    "                  [--bionic_iterations=<num_iter>]\n"
    "                  [--bionic_extra=\"<fn_name> <arg1> <arg 2> ...\"]\n"
    "                  [<Google benchmark flags>]\n"
    "Google benchmark flags:\n"
    "benchmark [--benchmark_list_tests={true|false}]\n"
    "          [--benchmark_filter=<regex>]\n"
    "          [--benchmark_min_time=`<integer>x` OR `<float>s` ]\n"
    "          [--benchmark_min_warmup_time=<min_warmup_time>]\n"
    "          [--benchmark_repetitions=<num_repetitions>]\n"
    "          [--benchmark_enable_random_interleaving={true|false}]\n"
    "          [--benchmark_report_aggregates_only={true|false}]\n"
    "          [--benchmark_display_aggregates_only={true|false}]\n"
    "          [--benchmark_format=<console|json|csv>]\n"
    "          [--benchmark_out=<filename>]\n"
    "          [--benchmark_out_format=<json|console|csv>]\n"
    "          [--benchmark_color={auto|true|false}]\n"
    "          [--benchmark_counters_tabular={true|false}]\n"
    "          [--benchmark_context=<key>=<value>,...]\n"
    "          [--benchmark_time_unit={ns|us|ms|s}]\n"
    "          [--v=<verbosity>]\n";
  Verify(expected, 0, std::vector<const char*>{"--help"}, false);
}

TEST_F(SystemTests, all_benchmarks) {
  RunTest(0, std::vector<const char*>{"--bionic_iterations=1"});
  // Count the number of benchmarks.
  // Rather than make every person who adds a benchmark add one here, just
  // make sure the count is not zero and some large number to verify
  // that something has not gone horribly wrong.
  std::regex benchmark_re("BM_\\S+/iterations:1");
  std::smatch sm;
  size_t num_tests = 0;
  std::string output(raw_output_);
  while (std::regex_search(output, sm, benchmark_re)) {
    num_tests++;
    output = sm.suffix();
  }
  ASSERT_LT(400U, num_tests) << "Only " << num_tests << " found:\n" << raw_output_;
}

TEST_F(SystemTests, small) {
  std::string expected =
      "BM_string_memcmp/8/8/8/iterations:1\n"
      "BM_math_sqrt/iterations:1\n"
      "BM_property_get/1/iterations:1\n"
      "BM_property_get/2/iterations:1\n"
      "BM_property_get/3/iterations:1\n";
  Verify(expected, 0, std::vector<const char*>{GetBionicXmlArg("test_small.xml").c_str(),
                                               "--bionic_iterations=1"});
}

TEST_F(SystemTests, medium) {
  std::string expected =
      "BM_string_memcmp/8/0/0/iterations:1\n"
      "BM_string_memcmp/16/0/0/iterations:1\n"
      "BM_string_memcmp/32/0/0/iterations:1\n"
      "BM_string_memcmp/64/0/0/iterations:1\n"
      "BM_string_memcmp/512/0/0/iterations:1\n"
      "BM_string_memcmp/1024/0/0/iterations:1\n"
      "BM_string_memcmp/8192/0/0/iterations:1\n"
      "BM_string_memcmp/16384/0/0/iterations:1\n"
      "BM_string_memcmp/32768/0/0/iterations:1\n"
      "BM_string_memcmp/65536/0/0/iterations:1\n"
      "BM_string_memcmp/131072/0/0/iterations:1\n"
      "BM_math_sqrt/iterations:1\n"
      "BM_string_memcpy/512/4/4/iterations:25\n"
      "BM_property_get/1/iterations:1\n";
  Verify(expected, 0, std::vector<const char*>{GetBionicXmlArg("test_medium.xml").c_str(),
                                               "--bionic_iterations=1"});
}

TEST_F(SystemTests, from_each) {
  std::string expected =
    "BM_atomic_empty/iterations:1\n"
    "BM_math_sqrt/iterations:1\n"
    "BM_property_get/1/iterations:1\n"
    "BM_pthread_self/iterations:1\n"
    "BM_semaphore_sem_getvalue/iterations:1\n"
    "BM_stdio_fread/64/iterations:1\n"
    "BM_string_memcpy/512/4/4/iterations:1\n"
    "BM_time_clock_gettime/iterations:1\n"
    "BM_unistd_getpid/iterations:1\n";
  Verify(expected, 0, std::vector<const char*>{GetBionicXmlArg("test_from_each.xml").c_str(),
                                               "--bionic_iterations=1"});
}

TEST_F(SystemTests, cmd_args) {
  std::string expected =
    "BM_string_memcpy/8/8/8/iterations:1\n"
    "BM_math_log10/iterations:1\n";
  Verify(expected, 0,
         std::vector<const char*>{"--bionic_extra=BM_string_memcpy 8/8/8",
                                  "--bionic_extra=BM_math_log10", "--bionic_iterations=1"});
}

TEST_F(SystemTests, cmd_args_no_iter) {
  std::string expected =
    "BM_string_memcpy/8/8/8\n"
    "BM_math_log10\n";
  Verify(expected, 0,
         std::vector<const char*>{"--bionic_extra=BM_string_memcpy 8/8/8",
                                  "--bionic_extra=BM_math_log10"});
}

TEST_F(SystemTests, xml_and_args) {
  std::string expected =
      "BM_string_memcmp/8/0/0/iterations:1\n"
      "BM_string_memcmp/16/0/0/iterations:1\n"
      "BM_string_memcmp/32/0/0/iterations:1\n"
      "BM_string_memcmp/64/0/0/iterations:1\n"
      "BM_string_memcmp/512/0/0/iterations:1\n"
      "BM_string_memcmp/1024/0/0/iterations:1\n"
      "BM_string_memcmp/8192/0/0/iterations:1\n"
      "BM_string_memcmp/16384/0/0/iterations:1\n"
      "BM_string_memcmp/32768/0/0/iterations:1\n"
      "BM_string_memcmp/65536/0/0/iterations:1\n"
      "BM_string_memcmp/131072/0/0/iterations:1\n"
      "BM_math_sqrt/iterations:1\n"
      "BM_string_memcpy/512/4/4/iterations:25\n"
      "BM_property_get/1/iterations:1\n"
      "BM_string_memcpy/8/0/0/iterations:1\n"
      "BM_string_memcpy/16/0/0/iterations:1\n"
      "BM_string_memcpy/32/0/0/iterations:1\n"
      "BM_string_memcpy/64/0/0/iterations:1\n"
      "BM_string_memcpy/512/0/0/iterations:1\n"
      "BM_string_memcpy/1024/0/0/iterations:1\n"
      "BM_string_memcpy/8192/0/0/iterations:1\n"
      "BM_string_memcpy/16384/0/0/iterations:1\n"
      "BM_string_memcpy/32768/0/0/iterations:1\n"
      "BM_string_memcpy/65536/0/0/iterations:1\n"
      "BM_string_memcpy/131072/0/0/iterations:1\n"
      "BM_math_log10/iterations:1\n";
  Verify(expected, 0, std::vector<const char*>{"--bionic_extra=BM_string_memcpy AT_ALIGNED_TWOBUF",
                                               "--bionic_extra=BM_math_log10",
                                               "--bionic_cpu=0",
                                               GetBionicXmlArg("test_medium.xml").c_str(),
                                               "--bionic_iterations=1"});
}

TEST_F(SystemTests, sizes) {
  std::string expected =
      "BM_stdio_fwrite/8/iterations:1\n"
      "BM_stdio_fwrite/16/iterations:1\n"
      "BM_stdio_fwrite/32/iterations:1\n"
      "BM_stdio_fwrite/64/iterations:1\n"
      "BM_stdio_fwrite/512/iterations:1\n"
      "BM_stdio_fwrite/1024/iterations:1\n"
      "BM_stdio_fwrite/8192/iterations:1\n"
      "BM_stdio_fwrite/16384/iterations:1\n"
      "BM_stdio_fwrite/32768/iterations:1\n"
      "BM_stdio_fwrite/65536/iterations:1\n"
      "BM_stdio_fwrite/131072/iterations:1\n"
      "BM_stdio_fread/1/iterations:1\n"
      "BM_stdio_fread/2/iterations:1\n"
      "BM_stdio_fread/3/iterations:1\n"
      "BM_stdio_fread/4/iterations:1\n"
      "BM_stdio_fread/5/iterations:1\n"
      "BM_stdio_fread/6/iterations:1\n"
      "BM_stdio_fread/7/iterations:1\n"
      "BM_stdio_fread/8/iterations:1\n"
      "BM_stdio_fread/9/iterations:1\n"
      "BM_stdio_fread/10/iterations:1\n"
      "BM_stdio_fread/11/iterations:1\n"
      "BM_stdio_fread/12/iterations:1\n"
      "BM_stdio_fread/13/iterations:1\n"
      "BM_stdio_fread/14/iterations:1\n"
      "BM_stdio_fread/15/iterations:1\n"
      "BM_stdio_fread/16/iterations:1\n"
      "BM_stdio_fread/24/iterations:1\n"
      "BM_stdio_fread/32/iterations:1\n"
      "BM_stdio_fread/40/iterations:1\n"
      "BM_stdio_fread/48/iterations:1\n"
      "BM_stdio_fread/56/iterations:1\n"
      "BM_stdio_fread/64/iterations:1\n"
      "BM_stdio_fread/72/iterations:1\n"
      "BM_stdio_fread/80/iterations:1\n"
      "BM_stdio_fread/88/iterations:1\n"
      "BM_stdio_fread/96/iterations:1\n"
      "BM_stdio_fread/104/iterations:1\n"
      "BM_stdio_fread/112/iterations:1\n"
      "BM_stdio_fread/120/iterations:1\n"
      "BM_stdio_fread/128/iterations:1\n"
      "BM_stdio_fread/136/iterations:1\n"
      "BM_stdio_fread/144/iterations:1\n"
      "BM_stdio_fread/160/iterations:1\n"
      "BM_stdio_fread/176/iterations:1\n"
      "BM_stdio_fread/192/iterations:1\n"
      "BM_stdio_fread/208/iterations:1\n"
      "BM_stdio_fread/224/iterations:1\n"
      "BM_stdio_fread/240/iterations:1\n"
      "BM_stdio_fread/256/iterations:1\n"
      "BM_stdio_fwrite/512/iterations:1\n"
      "BM_stdio_fwrite/1024/iterations:1\n"
      "BM_stdio_fwrite/8192/iterations:1\n"
      "BM_stdio_fwrite/16384/iterations:1\n"
      "BM_stdio_fwrite/32768/iterations:1\n"
      "BM_stdio_fwrite/65536/iterations:1\n"
      "BM_stdio_fwrite/131072/iterations:1\n"
      "BM_stdio_fread/262144/iterations:1\n"
      "BM_stdio_fread/524288/iterations:1\n"
      "BM_stdio_fread/1048576/iterations:1\n"
      "BM_stdio_fread/2097152/iterations:1\n"
      "BM_stdio_fwrite/1/iterations:1\n"
      "BM_stdio_fwrite/2/iterations:1\n"
      "BM_stdio_fwrite/3/iterations:1\n"
      "BM_stdio_fwrite/4/iterations:1\n"
      "BM_stdio_fwrite/5/iterations:1\n"
      "BM_stdio_fwrite/6/iterations:1\n"
      "BM_stdio_fwrite/7/iterations:1\n"
      "BM_stdio_fwrite/8/iterations:1\n"
      "BM_stdio_fwrite/9/iterations:1\n"
      "BM_stdio_fwrite/10/iterations:1\n"
      "BM_stdio_fwrite/11/iterations:1\n"
      "BM_stdio_fwrite/12/iterations:1\n"
      "BM_stdio_fwrite/13/iterations:1\n"
      "BM_stdio_fwrite/14/iterations:1\n"
      "BM_stdio_fwrite/15/iterations:1\n"
      "BM_stdio_fwrite/16/iterations:1\n"
      "BM_stdio_fwrite/24/iterations:1\n"
      "BM_stdio_fwrite/32/iterations:1\n"
      "BM_stdio_fwrite/40/iterations:1\n"
      "BM_stdio_fwrite/48/iterations:1\n"
      "BM_stdio_fwrite/56/iterations:1\n"
      "BM_stdio_fwrite/64/iterations:1\n"
      "BM_stdio_fwrite/72/iterations:1\n"
      "BM_stdio_fwrite/80/iterations:1\n"
      "BM_stdio_fwrite/88/iterations:1\n"
      "BM_stdio_fwrite/96/iterations:1\n"
      "BM_stdio_fwrite/104/iterations:1\n"
      "BM_stdio_fwrite/112/iterations:1\n"
      "BM_stdio_fwrite/120/iterations:1\n"
      "BM_stdio_fwrite/128/iterations:1\n"
      "BM_stdio_fwrite/136/iterations:1\n"
      "BM_stdio_fwrite/144/iterations:1\n"
      "BM_stdio_fwrite/160/iterations:1\n"
      "BM_stdio_fwrite/176/iterations:1\n"
      "BM_stdio_fwrite/192/iterations:1\n"
      "BM_stdio_fwrite/208/iterations:1\n"
      "BM_stdio_fwrite/224/iterations:1\n"
      "BM_stdio_fwrite/240/iterations:1\n"
      "BM_stdio_fwrite/256/iterations:1\n"
      "BM_stdio_fwrite/512/iterations:1\n"
      "BM_stdio_fwrite/1024/iterations:1\n"
      "BM_stdio_fwrite/8192/iterations:1\n"
      "BM_stdio_fwrite/16384/iterations:1\n"
      "BM_stdio_fwrite/32768/iterations:1\n"
      "BM_stdio_fwrite/65536/iterations:1\n"
      "BM_stdio_fwrite/131072/iterations:1\n"
      "BM_stdio_fwrite/262144/iterations:1\n"
      "BM_stdio_fwrite/524288/iterations:1\n"
      "BM_stdio_fwrite/1048576/iterations:1\n"
      "BM_stdio_fwrite/2097152/iterations:1\n";

  Verify(expected, 0, std::vector<const char*>{GetBionicXmlArg("test_size.xml").c_str()});
}

TEST_F(SystemTests, alignment_onebuf) {
  std::string expected =
      "BM_string_strlen/8/0/iterations:1\n"
      "BM_string_strlen/16/0/iterations:1\n"
      "BM_string_strlen/32/0/iterations:1\n"
      "BM_string_strlen/64/0/iterations:1\n"
      "BM_string_strlen/512/0/iterations:1\n"
      "BM_string_strlen/1024/0/iterations:1\n"
      "BM_string_strlen/8192/0/iterations:1\n"
      "BM_string_strlen/16384/0/iterations:1\n"
      "BM_string_strlen/32768/0/iterations:1\n"
      "BM_string_strlen/65536/0/iterations:1\n"
      "BM_string_strlen/131072/0/iterations:1\n"
      "BM_string_memset/1/0/iterations:1\n"
      "BM_string_memset/2/0/iterations:1\n"
      "BM_string_memset/3/0/iterations:1\n"
      "BM_string_memset/4/0/iterations:1\n"
      "BM_string_memset/5/0/iterations:1\n"
      "BM_string_memset/6/0/iterations:1\n"
      "BM_string_memset/7/0/iterations:1\n"
      "BM_string_memset/8/0/iterations:1\n"
      "BM_string_memset/9/0/iterations:1\n"
      "BM_string_memset/10/0/iterations:1\n"
      "BM_string_memset/11/0/iterations:1\n"
      "BM_string_memset/12/0/iterations:1\n"
      "BM_string_memset/13/0/iterations:1\n"
      "BM_string_memset/14/0/iterations:1\n"
      "BM_string_memset/15/0/iterations:1\n"
      "BM_string_memset/16/0/iterations:1\n"
      "BM_string_memset/24/0/iterations:1\n"
      "BM_string_memset/32/0/iterations:1\n"
      "BM_string_memset/40/0/iterations:1\n"
      "BM_string_memset/48/0/iterations:1\n"
      "BM_string_memset/56/0/iterations:1\n"
      "BM_string_memset/64/0/iterations:1\n"
      "BM_string_memset/72/0/iterations:1\n"
      "BM_string_memset/80/0/iterations:1\n"
      "BM_string_memset/88/0/iterations:1\n"
      "BM_string_memset/96/0/iterations:1\n"
      "BM_string_memset/104/0/iterations:1\n"
      "BM_string_memset/112/0/iterations:1\n"
      "BM_string_memset/120/0/iterations:1\n"
      "BM_string_memset/128/0/iterations:1\n"
      "BM_string_memset/136/0/iterations:1\n"
      "BM_string_memset/144/0/iterations:1\n"
      "BM_string_memset/160/0/iterations:1\n"
      "BM_string_memset/176/0/iterations:1\n"
      "BM_string_memset/192/0/iterations:1\n"
      "BM_string_memset/208/0/iterations:1\n"
      "BM_string_memset/224/0/iterations:1\n"
      "BM_string_memset/240/0/iterations:1\n"
      "BM_string_memset/256/0/iterations:1\n"
      "BM_string_strlen/512/0/iterations:1\n"
      "BM_string_strlen/1024/0/iterations:1\n"
      "BM_string_strlen/8192/0/iterations:1\n"
      "BM_string_strlen/16384/0/iterations:1\n"
      "BM_string_strlen/32768/0/iterations:1\n"
      "BM_string_strlen/65536/0/iterations:1\n"
      "BM_string_strlen/131072/0/iterations:1\n"
      "BM_string_memset/262144/0/iterations:1\n"
      "BM_string_memset/524288/0/iterations:1\n"
      "BM_string_memset/1048576/0/iterations:1\n"
      "BM_string_memset/2097152/0/iterations:1\n"
      "BM_string_strlen/1/0/iterations:1\n"
      "BM_string_strlen/2/0/iterations:1\n"
      "BM_string_strlen/3/0/iterations:1\n"
      "BM_string_strlen/4/0/iterations:1\n"
      "BM_string_strlen/5/0/iterations:1\n"
      "BM_string_strlen/6/0/iterations:1\n"
      "BM_string_strlen/7/0/iterations:1\n"
      "BM_string_strlen/8/0/iterations:1\n"
      "BM_string_strlen/9/0/iterations:1\n"
      "BM_string_strlen/10/0/iterations:1\n"
      "BM_string_strlen/11/0/iterations:1\n"
      "BM_string_strlen/12/0/iterations:1\n"
      "BM_string_strlen/13/0/iterations:1\n"
      "BM_string_strlen/14/0/iterations:1\n"
      "BM_string_strlen/15/0/iterations:1\n"
      "BM_string_strlen/16/0/iterations:1\n"
      "BM_string_strlen/24/0/iterations:1\n"
      "BM_string_strlen/32/0/iterations:1\n"
      "BM_string_strlen/40/0/iterations:1\n"
      "BM_string_strlen/48/0/iterations:1\n"
      "BM_string_strlen/56/0/iterations:1\n"
      "BM_string_strlen/64/0/iterations:1\n"
      "BM_string_strlen/72/0/iterations:1\n"
      "BM_string_strlen/80/0/iterations:1\n"
      "BM_string_strlen/88/0/iterations:1\n"
      "BM_string_strlen/96/0/iterations:1\n"
      "BM_string_strlen/104/0/iterations:1\n"
      "BM_string_strlen/112/0/iterations:1\n"
      "BM_string_strlen/120/0/iterations:1\n"
      "BM_string_strlen/128/0/iterations:1\n"
      "BM_string_strlen/136/0/iterations:1\n"
      "BM_string_strlen/144/0/iterations:1\n"
      "BM_string_strlen/160/0/iterations:1\n"
      "BM_string_strlen/176/0/iterations:1\n"
      "BM_string_strlen/192/0/iterations:1\n"
      "BM_string_strlen/208/0/iterations:1\n"
      "BM_string_strlen/224/0/iterations:1\n"
      "BM_string_strlen/240/0/iterations:1\n"
      "BM_string_strlen/256/0/iterations:1\n"
      "BM_string_strlen/512/0/iterations:1\n"
      "BM_string_strlen/1024/0/iterations:1\n"
      "BM_string_strlen/8192/0/iterations:1\n"
      "BM_string_strlen/16384/0/iterations:1\n"
      "BM_string_strlen/32768/0/iterations:1\n"
      "BM_string_strlen/65536/0/iterations:1\n"
      "BM_string_strlen/131072/0/iterations:1\n"
      "BM_string_strlen/262144/0/iterations:1\n"
      "BM_string_strlen/524288/0/iterations:1\n"
      "BM_string_strlen/1048576/0/iterations:1\n"
      "BM_string_strlen/2097152/0/iterations:1\n"
      "BM_string_memset/1/0/iterations:1\n"
      "BM_string_memset/1/1/iterations:1\n"
      "BM_string_memset/1/2/iterations:1\n"
      "BM_string_memset/1/4/iterations:1\n"
      "BM_string_memset/1/8/iterations:1\n"
      "BM_string_memset/1/16/iterations:1\n"
      "BM_string_memset/1/32/iterations:1\n"
      "BM_string_memset/2/0/iterations:1\n"
      "BM_string_memset/2/1/iterations:1\n"
      "BM_string_memset/2/2/iterations:1\n"
      "BM_string_memset/2/4/iterations:1\n"
      "BM_string_memset/2/8/iterations:1\n"
      "BM_string_memset/2/16/iterations:1\n"
      "BM_string_memset/2/32/iterations:1\n"
      "BM_string_memset/3/0/iterations:1\n"
      "BM_string_memset/3/1/iterations:1\n"
      "BM_string_memset/3/2/iterations:1\n"
      "BM_string_memset/3/4/iterations:1\n"
      "BM_string_memset/3/8/iterations:1\n"
      "BM_string_memset/3/16/iterations:1\n"
      "BM_string_memset/3/32/iterations:1\n"
      "BM_string_memset/4/0/iterations:1\n"
      "BM_string_memset/4/1/iterations:1\n"
      "BM_string_memset/4/2/iterations:1\n"
      "BM_string_memset/4/4/iterations:1\n"
      "BM_string_memset/4/8/iterations:1\n"
      "BM_string_memset/4/16/iterations:1\n"
      "BM_string_memset/4/32/iterations:1\n"
      "BM_string_memset/5/0/iterations:1\n"
      "BM_string_memset/5/1/iterations:1\n"
      "BM_string_memset/5/2/iterations:1\n"
      "BM_string_memset/5/4/iterations:1\n"
      "BM_string_memset/5/8/iterations:1\n"
      "BM_string_memset/5/16/iterations:1\n"
      "BM_string_memset/5/32/iterations:1\n"
      "BM_string_memset/6/0/iterations:1\n"
      "BM_string_memset/6/1/iterations:1\n"
      "BM_string_memset/6/2/iterations:1\n"
      "BM_string_memset/6/4/iterations:1\n"
      "BM_string_memset/6/8/iterations:1\n"
      "BM_string_memset/6/16/iterations:1\n"
      "BM_string_memset/6/32/iterations:1\n"
      "BM_string_memset/7/0/iterations:1\n"
      "BM_string_memset/7/1/iterations:1\n"
      "BM_string_memset/7/2/iterations:1\n"
      "BM_string_memset/7/4/iterations:1\n"
      "BM_string_memset/7/8/iterations:1\n"
      "BM_string_memset/7/16/iterations:1\n"
      "BM_string_memset/7/32/iterations:1\n"
      "BM_string_memset/8/0/iterations:1\n"
      "BM_string_memset/8/1/iterations:1\n"
      "BM_string_memset/8/2/iterations:1\n"
      "BM_string_memset/8/4/iterations:1\n"
      "BM_string_memset/8/8/iterations:1\n"
      "BM_string_memset/8/16/iterations:1\n"
      "BM_string_memset/8/32/iterations:1\n"
      "BM_string_memset/9/0/iterations:1\n"
      "BM_string_memset/9/1/iterations:1\n"
      "BM_string_memset/9/2/iterations:1\n"
      "BM_string_memset/9/4/iterations:1\n"
      "BM_string_memset/9/8/iterations:1\n"
      "BM_string_memset/9/16/iterations:1\n"
      "BM_string_memset/9/32/iterations:1\n"
      "BM_string_memset/10/0/iterations:1\n"
      "BM_string_memset/10/1/iterations:1\n"
      "BM_string_memset/10/2/iterations:1\n"
      "BM_string_memset/10/4/iterations:1\n"
      "BM_string_memset/10/8/iterations:1\n"
      "BM_string_memset/10/16/iterations:1\n"
      "BM_string_memset/10/32/iterations:1\n"
      "BM_string_memset/11/0/iterations:1\n"
      "BM_string_memset/11/1/iterations:1\n"
      "BM_string_memset/11/2/iterations:1\n"
      "BM_string_memset/11/4/iterations:1\n"
      "BM_string_memset/11/8/iterations:1\n"
      "BM_string_memset/11/16/iterations:1\n"
      "BM_string_memset/11/32/iterations:1\n"
      "BM_string_memset/12/0/iterations:1\n"
      "BM_string_memset/12/1/iterations:1\n"
      "BM_string_memset/12/2/iterations:1\n"
      "BM_string_memset/12/4/iterations:1\n"
      "BM_string_memset/12/8/iterations:1\n"
      "BM_string_memset/12/16/iterations:1\n"
      "BM_string_memset/12/32/iterations:1\n"
      "BM_string_memset/13/0/iterations:1\n"
      "BM_string_memset/13/1/iterations:1\n"
      "BM_string_memset/13/2/iterations:1\n"
      "BM_string_memset/13/4/iterations:1\n"
      "BM_string_memset/13/8/iterations:1\n"
      "BM_string_memset/13/16/iterations:1\n"
      "BM_string_memset/13/32/iterations:1\n"
      "BM_string_memset/14/0/iterations:1\n"
      "BM_string_memset/14/1/iterations:1\n"
      "BM_string_memset/14/2/iterations:1\n"
      "BM_string_memset/14/4/iterations:1\n"
      "BM_string_memset/14/8/iterations:1\n"
      "BM_string_memset/14/16/iterations:1\n"
      "BM_string_memset/14/32/iterations:1\n"
      "BM_string_memset/15/0/iterations:1\n"
      "BM_string_memset/15/1/iterations:1\n"
      "BM_string_memset/15/2/iterations:1\n"
      "BM_string_memset/15/4/iterations:1\n"
      "BM_string_memset/15/8/iterations:1\n"
      "BM_string_memset/15/16/iterations:1\n"
      "BM_string_memset/15/32/iterations:1\n"
      "BM_string_memset/16/0/iterations:1\n"
      "BM_string_memset/16/1/iterations:1\n"
      "BM_string_memset/16/2/iterations:1\n"
      "BM_string_memset/16/4/iterations:1\n"
      "BM_string_memset/16/8/iterations:1\n"
      "BM_string_memset/16/16/iterations:1\n"
      "BM_string_memset/16/32/iterations:1\n"
      "BM_string_memset/24/0/iterations:1\n"
      "BM_string_memset/24/1/iterations:1\n"
      "BM_string_memset/24/2/iterations:1\n"
      "BM_string_memset/24/4/iterations:1\n"
      "BM_string_memset/24/8/iterations:1\n"
      "BM_string_memset/24/16/iterations:1\n"
      "BM_string_memset/24/32/iterations:1\n"
      "BM_string_memset/32/0/iterations:1\n"
      "BM_string_memset/32/1/iterations:1\n"
      "BM_string_memset/32/2/iterations:1\n"
      "BM_string_memset/32/4/iterations:1\n"
      "BM_string_memset/32/8/iterations:1\n"
      "BM_string_memset/32/16/iterations:1\n"
      "BM_string_memset/32/32/iterations:1\n"
      "BM_string_memset/40/0/iterations:1\n"
      "BM_string_memset/40/1/iterations:1\n"
      "BM_string_memset/40/2/iterations:1\n"
      "BM_string_memset/40/4/iterations:1\n"
      "BM_string_memset/40/8/iterations:1\n"
      "BM_string_memset/40/16/iterations:1\n"
      "BM_string_memset/40/32/iterations:1\n"
      "BM_string_memset/48/0/iterations:1\n"
      "BM_string_memset/48/1/iterations:1\n"
      "BM_string_memset/48/2/iterations:1\n"
      "BM_string_memset/48/4/iterations:1\n"
      "BM_string_memset/48/8/iterations:1\n"
      "BM_string_memset/48/16/iterations:1\n"
      "BM_string_memset/48/32/iterations:1\n"
      "BM_string_memset/56/0/iterations:1\n"
      "BM_string_memset/56/1/iterations:1\n"
      "BM_string_memset/56/2/iterations:1\n"
      "BM_string_memset/56/4/iterations:1\n"
      "BM_string_memset/56/8/iterations:1\n"
      "BM_string_memset/56/16/iterations:1\n"
      "BM_string_memset/56/32/iterations:1\n"
      "BM_string_memset/64/0/iterations:1\n"
      "BM_string_memset/64/1/iterations:1\n"
      "BM_string_memset/64/2/iterations:1\n"
      "BM_string_memset/64/4/iterations:1\n"
      "BM_string_memset/64/8/iterations:1\n"
      "BM_string_memset/64/16/iterations:1\n"
      "BM_string_memset/64/32/iterations:1\n"
      "BM_string_memset/72/0/iterations:1\n"
      "BM_string_memset/72/1/iterations:1\n"
      "BM_string_memset/72/2/iterations:1\n"
      "BM_string_memset/72/4/iterations:1\n"
      "BM_string_memset/72/8/iterations:1\n"
      "BM_string_memset/72/16/iterations:1\n"
      "BM_string_memset/72/32/iterations:1\n"
      "BM_string_memset/80/0/iterations:1\n"
      "BM_string_memset/80/1/iterations:1\n"
      "BM_string_memset/80/2/iterations:1\n"
      "BM_string_memset/80/4/iterations:1\n"
      "BM_string_memset/80/8/iterations:1\n"
      "BM_string_memset/80/16/iterations:1\n"
      "BM_string_memset/80/32/iterations:1\n"
      "BM_string_memset/88/0/iterations:1\n"
      "BM_string_memset/88/1/iterations:1\n"
      "BM_string_memset/88/2/iterations:1\n"
      "BM_string_memset/88/4/iterations:1\n"
      "BM_string_memset/88/8/iterations:1\n"
      "BM_string_memset/88/16/iterations:1\n"
      "BM_string_memset/88/32/iterations:1\n"
      "BM_string_memset/96/0/iterations:1\n"
      "BM_string_memset/96/1/iterations:1\n"
      "BM_string_memset/96/2/iterations:1\n"
      "BM_string_memset/96/4/iterations:1\n"
      "BM_string_memset/96/8/iterations:1\n"
      "BM_string_memset/96/16/iterations:1\n"
      "BM_string_memset/96/32/iterations:1\n"
      "BM_string_memset/104/0/iterations:1\n"
      "BM_string_memset/104/1/iterations:1\n"
      "BM_string_memset/104/2/iterations:1\n"
      "BM_string_memset/104/4/iterations:1\n"
      "BM_string_memset/104/8/iterations:1\n"
      "BM_string_memset/104/16/iterations:1\n"
      "BM_string_memset/104/32/iterations:1\n"
      "BM_string_memset/112/0/iterations:1\n"
      "BM_string_memset/112/1/iterations:1\n"
      "BM_string_memset/112/2/iterations:1\n"
      "BM_string_memset/112/4/iterations:1\n"
      "BM_string_memset/112/8/iterations:1\n"
      "BM_string_memset/112/16/iterations:1\n"
      "BM_string_memset/112/32/iterations:1\n"
      "BM_string_memset/120/0/iterations:1\n"
      "BM_string_memset/120/1/iterations:1\n"
      "BM_string_memset/120/2/iterations:1\n"
      "BM_string_memset/120/4/iterations:1\n"
      "BM_string_memset/120/8/iterations:1\n"
      "BM_string_memset/120/16/iterations:1\n"
      "BM_string_memset/120/32/iterations:1\n"
      "BM_string_memset/128/0/iterations:1\n"
      "BM_string_memset/128/1/iterations:1\n"
      "BM_string_memset/128/2/iterations:1\n"
      "BM_string_memset/128/4/iterations:1\n"
      "BM_string_memset/128/8/iterations:1\n"
      "BM_string_memset/128/16/iterations:1\n"
      "BM_string_memset/128/32/iterations:1\n"
      "BM_string_memset/136/0/iterations:1\n"
      "BM_string_memset/136/1/iterations:1\n"
      "BM_string_memset/136/2/iterations:1\n"
      "BM_string_memset/136/4/iterations:1\n"
      "BM_string_memset/136/8/iterations:1\n"
      "BM_string_memset/136/16/iterations:1\n"
      "BM_string_memset/136/32/iterations:1\n"
      "BM_string_memset/144/0/iterations:1\n"
      "BM_string_memset/144/1/iterations:1\n"
      "BM_string_memset/144/2/iterations:1\n"
      "BM_string_memset/144/4/iterations:1\n"
      "BM_string_memset/144/8/iterations:1\n"
      "BM_string_memset/144/16/iterations:1\n"
      "BM_string_memset/144/32/iterations:1\n"
      "BM_string_memset/160/0/iterations:1\n"
      "BM_string_memset/160/1/iterations:1\n"
      "BM_string_memset/160/2/iterations:1\n"
      "BM_string_memset/160/4/iterations:1\n"
      "BM_string_memset/160/8/iterations:1\n"
      "BM_string_memset/160/16/iterations:1\n"
      "BM_string_memset/160/32/iterations:1\n"
      "BM_string_memset/176/0/iterations:1\n"
      "BM_string_memset/176/1/iterations:1\n"
      "BM_string_memset/176/2/iterations:1\n"
      "BM_string_memset/176/4/iterations:1\n"
      "BM_string_memset/176/8/iterations:1\n"
      "BM_string_memset/176/16/iterations:1\n"
      "BM_string_memset/176/32/iterations:1\n"
      "BM_string_memset/192/0/iterations:1\n"
      "BM_string_memset/192/1/iterations:1\n"
      "BM_string_memset/192/2/iterations:1\n"
      "BM_string_memset/192/4/iterations:1\n"
      "BM_string_memset/192/8/iterations:1\n"
      "BM_string_memset/192/16/iterations:1\n"
      "BM_string_memset/192/32/iterations:1\n"
      "BM_string_memset/208/0/iterations:1\n"
      "BM_string_memset/208/1/iterations:1\n"
      "BM_string_memset/208/2/iterations:1\n"
      "BM_string_memset/208/4/iterations:1\n"
      "BM_string_memset/208/8/iterations:1\n"
      "BM_string_memset/208/16/iterations:1\n"
      "BM_string_memset/208/32/iterations:1\n"
      "BM_string_memset/224/0/iterations:1\n"
      "BM_string_memset/224/1/iterations:1\n"
      "BM_string_memset/224/2/iterations:1\n"
      "BM_string_memset/224/4/iterations:1\n"
      "BM_string_memset/224/8/iterations:1\n"
      "BM_string_memset/224/16/iterations:1\n"
      "BM_string_memset/224/32/iterations:1\n"
      "BM_string_memset/240/0/iterations:1\n"
      "BM_string_memset/240/1/iterations:1\n"
      "BM_string_memset/240/2/iterations:1\n"
      "BM_string_memset/240/4/iterations:1\n"
      "BM_string_memset/240/8/iterations:1\n"
      "BM_string_memset/240/16/iterations:1\n"
      "BM_string_memset/240/32/iterations:1\n"
      "BM_string_memset/256/0/iterations:1\n"
      "BM_string_memset/256/1/iterations:1\n"
      "BM_string_memset/256/2/iterations:1\n"
      "BM_string_memset/256/4/iterations:1\n"
      "BM_string_memset/256/8/iterations:1\n"
      "BM_string_memset/256/16/iterations:1\n"
      "BM_string_memset/256/32/iterations:1\n"
      "BM_string_memset/512/0/iterations:1\n"
      "BM_string_memset/512/1/iterations:1\n"
      "BM_string_memset/512/2/iterations:1\n"
      "BM_string_memset/512/4/iterations:1\n"
      "BM_string_memset/512/8/iterations:1\n"
      "BM_string_memset/512/16/iterations:1\n"
      "BM_string_memset/512/32/iterations:1\n"
      "BM_string_memset/1024/0/iterations:1\n"
      "BM_string_memset/1024/1/iterations:1\n"
      "BM_string_memset/1024/2/iterations:1\n"
      "BM_string_memset/1024/4/iterations:1\n"
      "BM_string_memset/1024/8/iterations:1\n"
      "BM_string_memset/1024/16/iterations:1\n"
      "BM_string_memset/1024/32/iterations:1\n"
      "BM_string_memset/8192/0/iterations:1\n"
      "BM_string_memset/8192/1/iterations:1\n"
      "BM_string_memset/8192/2/iterations:1\n"
      "BM_string_memset/8192/4/iterations:1\n"
      "BM_string_memset/8192/8/iterations:1\n"
      "BM_string_memset/8192/16/iterations:1\n"
      "BM_string_memset/8192/32/iterations:1\n"
      "BM_string_memset/16384/0/iterations:1\n"
      "BM_string_memset/16384/1/iterations:1\n"
      "BM_string_memset/16384/2/iterations:1\n"
      "BM_string_memset/16384/4/iterations:1\n"
      "BM_string_memset/16384/8/iterations:1\n"
      "BM_string_memset/16384/16/iterations:1\n"
      "BM_string_memset/16384/32/iterations:1\n"
      "BM_string_memset/32768/0/iterations:1\n"
      "BM_string_memset/32768/1/iterations:1\n"
      "BM_string_memset/32768/2/iterations:1\n"
      "BM_string_memset/32768/4/iterations:1\n"
      "BM_string_memset/32768/8/iterations:1\n"
      "BM_string_memset/32768/16/iterations:1\n"
      "BM_string_memset/32768/32/iterations:1\n"
      "BM_string_memset/65536/0/iterations:1\n"
      "BM_string_memset/65536/1/iterations:1\n"
      "BM_string_memset/65536/2/iterations:1\n"
      "BM_string_memset/65536/4/iterations:1\n"
      "BM_string_memset/65536/8/iterations:1\n"
      "BM_string_memset/65536/16/iterations:1\n"
      "BM_string_memset/65536/32/iterations:1\n"
      "BM_string_memset/131072/0/iterations:1\n"
      "BM_string_memset/131072/1/iterations:1\n"
      "BM_string_memset/131072/2/iterations:1\n"
      "BM_string_memset/131072/4/iterations:1\n"
      "BM_string_memset/131072/8/iterations:1\n"
      "BM_string_memset/131072/16/iterations:1\n"
      "BM_string_memset/131072/32/iterations:1\n"
      "BM_string_memset/262144/0/iterations:1\n"
      "BM_string_memset/262144/1/iterations:1\n"
      "BM_string_memset/262144/2/iterations:1\n"
      "BM_string_memset/262144/4/iterations:1\n"
      "BM_string_memset/262144/8/iterations:1\n"
      "BM_string_memset/262144/16/iterations:1\n"
      "BM_string_memset/262144/32/iterations:1\n"
      "BM_string_memset/524288/0/iterations:1\n"
      "BM_string_memset/524288/1/iterations:1\n"
      "BM_string_memset/524288/2/iterations:1\n"
      "BM_string_memset/524288/4/iterations:1\n"
      "BM_string_memset/524288/8/iterations:1\n"
      "BM_string_memset/524288/16/iterations:1\n"
      "BM_string_memset/524288/32/iterations:1\n"
      "BM_string_memset/1048576/0/iterations:1\n"
      "BM_string_memset/1048576/1/iterations:1\n"
      "BM_string_memset/1048576/2/iterations:1\n"
      "BM_string_memset/1048576/4/iterations:1\n"
      "BM_string_memset/1048576/8/iterations:1\n"
      "BM_string_memset/1048576/16/iterations:1\n"
      "BM_string_memset/1048576/32/iterations:1\n"
      "BM_string_memset/2097152/0/iterations:1\n"
      "BM_string_memset/2097152/1/iterations:1\n"
      "BM_string_memset/2097152/2/iterations:1\n"
      "BM_string_memset/2097152/4/iterations:1\n"
      "BM_string_memset/2097152/8/iterations:1\n"
      "BM_string_memset/2097152/16/iterations:1\n"
      "BM_string_memset/2097152/32/iterations:1\n";

  Verify(expected, 0,
         std::vector<const char*>{GetBionicXmlArg("test_alignment_onebuf.xml").c_str()});
}

TEST_F(SystemTests, alignment_onebuf_manual) {
  std::string expected =
    "BM_string_strlen/115/0/iterations:1\n"
    "BM_string_memset/1024/1/iterations:1\n"
    "BM_string_strlen/2048/4/iterations:1\n"
    "BM_string_memset/4000/32/iterations:1\n"
    "BM_string_strlen/512/4/iterations:1\n"
    "BM_string_strlen/1024/4/iterations:1\n"
    "BM_string_strlen/8192/4/iterations:1\n"
    "BM_string_strlen/16384/4/iterations:1\n"
    "BM_string_strlen/32768/4/iterations:1\n"
    "BM_string_strlen/65536/4/iterations:1\n"
    "BM_string_strlen/131072/4/iterations:1\n"
    "BM_string_memset/262144/32/iterations:1\n"
    "BM_string_memset/524288/32/iterations:1\n"
    "BM_string_memset/1048576/32/iterations:1\n"
    "BM_string_memset/2097152/32/iterations:1\n";

  Verify(expected, 0,
         std::vector<const char*>{GetBionicXmlArg("test_alignment_onebuf_manual.xml").c_str()});
}

TEST_F(SystemTests, alignment_twobuf) {
  std::string expected =
      "BM_string_strcpy/8/0/0/iterations:1\n"
      "BM_string_strcpy/16/0/0/iterations:1\n"
      "BM_string_strcpy/32/0/0/iterations:1\n"
      "BM_string_strcpy/64/0/0/iterations:1\n"
      "BM_string_strcpy/512/0/0/iterations:1\n"
      "BM_string_strcpy/1024/0/0/iterations:1\n"
      "BM_string_strcpy/8192/0/0/iterations:1\n"
      "BM_string_strcpy/16384/0/0/iterations:1\n"
      "BM_string_strcpy/32768/0/0/iterations:1\n"
      "BM_string_strcpy/65536/0/0/iterations:1\n"
      "BM_string_strcpy/131072/0/0/iterations:1\n"
      "BM_string_memcpy/1/0/0/iterations:1\n"
      "BM_string_memcpy/2/0/0/iterations:1\n"
      "BM_string_memcpy/3/0/0/iterations:1\n"
      "BM_string_memcpy/4/0/0/iterations:1\n"
      "BM_string_memcpy/5/0/0/iterations:1\n"
      "BM_string_memcpy/6/0/0/iterations:1\n"
      "BM_string_memcpy/7/0/0/iterations:1\n"
      "BM_string_memcpy/8/0/0/iterations:1\n"
      "BM_string_memcpy/9/0/0/iterations:1\n"
      "BM_string_memcpy/10/0/0/iterations:1\n"
      "BM_string_memcpy/11/0/0/iterations:1\n"
      "BM_string_memcpy/12/0/0/iterations:1\n"
      "BM_string_memcpy/13/0/0/iterations:1\n"
      "BM_string_memcpy/14/0/0/iterations:1\n"
      "BM_string_memcpy/15/0/0/iterations:1\n"
      "BM_string_memcpy/16/0/0/iterations:1\n"
      "BM_string_memcpy/24/0/0/iterations:1\n"
      "BM_string_memcpy/32/0/0/iterations:1\n"
      "BM_string_memcpy/40/0/0/iterations:1\n"
      "BM_string_memcpy/48/0/0/iterations:1\n"
      "BM_string_memcpy/56/0/0/iterations:1\n"
      "BM_string_memcpy/64/0/0/iterations:1\n"
      "BM_string_memcpy/72/0/0/iterations:1\n"
      "BM_string_memcpy/80/0/0/iterations:1\n"
      "BM_string_memcpy/88/0/0/iterations:1\n"
      "BM_string_memcpy/96/0/0/iterations:1\n"
      "BM_string_memcpy/104/0/0/iterations:1\n"
      "BM_string_memcpy/112/0/0/iterations:1\n"
      "BM_string_memcpy/120/0/0/iterations:1\n"
      "BM_string_memcpy/128/0/0/iterations:1\n"
      "BM_string_memcpy/136/0/0/iterations:1\n"
      "BM_string_memcpy/144/0/0/iterations:1\n"
      "BM_string_memcpy/160/0/0/iterations:1\n"
      "BM_string_memcpy/176/0/0/iterations:1\n"
      "BM_string_memcpy/192/0/0/iterations:1\n"
      "BM_string_memcpy/208/0/0/iterations:1\n"
      "BM_string_memcpy/224/0/0/iterations:1\n"
      "BM_string_memcpy/240/0/0/iterations:1\n"
      "BM_string_memcpy/256/0/0/iterations:1\n"
      "BM_string_strcpy/512/0/0/iterations:1\n"
      "BM_string_strcpy/1024/0/0/iterations:1\n"
      "BM_string_strcpy/8192/0/0/iterations:1\n"
      "BM_string_strcpy/16384/0/0/iterations:1\n"
      "BM_string_strcpy/32768/0/0/iterations:1\n"
      "BM_string_strcpy/65536/0/0/iterations:1\n"
      "BM_string_strcpy/131072/0/0/iterations:1\n"
      "BM_string_memcpy/262144/0/0/iterations:1\n"
      "BM_string_memcpy/524288/0/0/iterations:1\n"
      "BM_string_memcpy/1048576/0/0/iterations:1\n"
      "BM_string_memcpy/2097152/0/0/iterations:1\n"
      "BM_string_strcpy/1/0/0/iterations:1\n"
      "BM_string_strcpy/2/0/0/iterations:1\n"
      "BM_string_strcpy/3/0/0/iterations:1\n"
      "BM_string_strcpy/4/0/0/iterations:1\n"
      "BM_string_strcpy/5/0/0/iterations:1\n"
      "BM_string_strcpy/6/0/0/iterations:1\n"
      "BM_string_strcpy/7/0/0/iterations:1\n"
      "BM_string_strcpy/8/0/0/iterations:1\n"
      "BM_string_strcpy/9/0/0/iterations:1\n"
      "BM_string_strcpy/10/0/0/iterations:1\n"
      "BM_string_strcpy/11/0/0/iterations:1\n"
      "BM_string_strcpy/12/0/0/iterations:1\n"
      "BM_string_strcpy/13/0/0/iterations:1\n"
      "BM_string_strcpy/14/0/0/iterations:1\n"
      "BM_string_strcpy/15/0/0/iterations:1\n"
      "BM_string_strcpy/16/0/0/iterations:1\n"
      "BM_string_strcpy/24/0/0/iterations:1\n"
      "BM_string_strcpy/32/0/0/iterations:1\n"
      "BM_string_strcpy/40/0/0/iterations:1\n"
      "BM_string_strcpy/48/0/0/iterations:1\n"
      "BM_string_strcpy/56/0/0/iterations:1\n"
      "BM_string_strcpy/64/0/0/iterations:1\n"
      "BM_string_strcpy/72/0/0/iterations:1\n"
      "BM_string_strcpy/80/0/0/iterations:1\n"
      "BM_string_strcpy/88/0/0/iterations:1\n"
      "BM_string_strcpy/96/0/0/iterations:1\n"
      "BM_string_strcpy/104/0/0/iterations:1\n"
      "BM_string_strcpy/112/0/0/iterations:1\n"
      "BM_string_strcpy/120/0/0/iterations:1\n"
      "BM_string_strcpy/128/0/0/iterations:1\n"
      "BM_string_strcpy/136/0/0/iterations:1\n"
      "BM_string_strcpy/144/0/0/iterations:1\n"
      "BM_string_strcpy/160/0/0/iterations:1\n"
      "BM_string_strcpy/176/0/0/iterations:1\n"
      "BM_string_strcpy/192/0/0/iterations:1\n"
      "BM_string_strcpy/208/0/0/iterations:1\n"
      "BM_string_strcpy/224/0/0/iterations:1\n"
      "BM_string_strcpy/240/0/0/iterations:1\n"
      "BM_string_strcpy/256/0/0/iterations:1\n"
      "BM_string_strcpy/512/0/0/iterations:1\n"
      "BM_string_strcpy/1024/0/0/iterations:1\n"
      "BM_string_strcpy/8192/0/0/iterations:1\n"
      "BM_string_strcpy/16384/0/0/iterations:1\n"
      "BM_string_strcpy/32768/0/0/iterations:1\n"
      "BM_string_strcpy/65536/0/0/iterations:1\n"
      "BM_string_strcpy/131072/0/0/iterations:1\n"
      "BM_string_strcpy/262144/0/0/iterations:1\n"
      "BM_string_strcpy/524288/0/0/iterations:1\n"
      "BM_string_strcpy/1048576/0/0/iterations:1\n"
      "BM_string_strcpy/2097152/0/0/iterations:1\n"
      "BM_string_memcpy/1/0/0/iterations:1\n"
      "BM_string_memcpy/1/1/1/iterations:1\n"
      "BM_string_memcpy/1/1/2/iterations:1\n"
      "BM_string_memcpy/1/1/4/iterations:1\n"
      "BM_string_memcpy/1/1/8/iterations:1\n"
      "BM_string_memcpy/1/1/16/iterations:1\n"
      "BM_string_memcpy/1/1/32/iterations:1\n"
      "BM_string_memcpy/1/2/1/iterations:1\n"
      "BM_string_memcpy/1/2/2/iterations:1\n"
      "BM_string_memcpy/1/2/4/iterations:1\n"
      "BM_string_memcpy/1/2/8/iterations:1\n"
      "BM_string_memcpy/1/2/16/iterations:1\n"
      "BM_string_memcpy/1/2/32/iterations:1\n"
      "BM_string_memcpy/1/4/1/iterations:1\n"
      "BM_string_memcpy/1/4/2/iterations:1\n"
      "BM_string_memcpy/1/4/4/iterations:1\n"
      "BM_string_memcpy/1/4/8/iterations:1\n"
      "BM_string_memcpy/1/4/16/iterations:1\n"
      "BM_string_memcpy/1/4/32/iterations:1\n"
      "BM_string_memcpy/1/8/1/iterations:1\n"
      "BM_string_memcpy/1/8/2/iterations:1\n"
      "BM_string_memcpy/1/8/4/iterations:1\n"
      "BM_string_memcpy/1/8/8/iterations:1\n"
      "BM_string_memcpy/1/8/16/iterations:1\n"
      "BM_string_memcpy/1/8/32/iterations:1\n"
      "BM_string_memcpy/1/16/1/iterations:1\n"
      "BM_string_memcpy/1/16/2/iterations:1\n"
      "BM_string_memcpy/1/16/4/iterations:1\n"
      "BM_string_memcpy/1/16/8/iterations:1\n"
      "BM_string_memcpy/1/16/16/iterations:1\n"
      "BM_string_memcpy/1/16/32/iterations:1\n"
      "BM_string_memcpy/1/32/1/iterations:1\n"
      "BM_string_memcpy/1/32/2/iterations:1\n"
      "BM_string_memcpy/1/32/4/iterations:1\n"
      "BM_string_memcpy/1/32/8/iterations:1\n"
      "BM_string_memcpy/1/32/16/iterations:1\n"
      "BM_string_memcpy/1/32/32/iterations:1\n"
      "BM_string_memcpy/2/0/0/iterations:1\n"
      "BM_string_memcpy/2/1/1/iterations:1\n"
      "BM_string_memcpy/2/1/2/iterations:1\n"
      "BM_string_memcpy/2/1/4/iterations:1\n"
      "BM_string_memcpy/2/1/8/iterations:1\n"
      "BM_string_memcpy/2/1/16/iterations:1\n"
      "BM_string_memcpy/2/1/32/iterations:1\n"
      "BM_string_memcpy/2/2/1/iterations:1\n"
      "BM_string_memcpy/2/2/2/iterations:1\n"
      "BM_string_memcpy/2/2/4/iterations:1\n"
      "BM_string_memcpy/2/2/8/iterations:1\n"
      "BM_string_memcpy/2/2/16/iterations:1\n"
      "BM_string_memcpy/2/2/32/iterations:1\n"
      "BM_string_memcpy/2/4/1/iterations:1\n"
      "BM_string_memcpy/2/4/2/iterations:1\n"
      "BM_string_memcpy/2/4/4/iterations:1\n"
      "BM_string_memcpy/2/4/8/iterations:1\n"
      "BM_string_memcpy/2/4/16/iterations:1\n"
      "BM_string_memcpy/2/4/32/iterations:1\n"
      "BM_string_memcpy/2/8/1/iterations:1\n"
      "BM_string_memcpy/2/8/2/iterations:1\n"
      "BM_string_memcpy/2/8/4/iterations:1\n"
      "BM_string_memcpy/2/8/8/iterations:1\n"
      "BM_string_memcpy/2/8/16/iterations:1\n"
      "BM_string_memcpy/2/8/32/iterations:1\n"
      "BM_string_memcpy/2/16/1/iterations:1\n"
      "BM_string_memcpy/2/16/2/iterations:1\n"
      "BM_string_memcpy/2/16/4/iterations:1\n"
      "BM_string_memcpy/2/16/8/iterations:1\n"
      "BM_string_memcpy/2/16/16/iterations:1\n"
      "BM_string_memcpy/2/16/32/iterations:1\n"
      "BM_string_memcpy/2/32/1/iterations:1\n"
      "BM_string_memcpy/2/32/2/iterations:1\n"
      "BM_string_memcpy/2/32/4/iterations:1\n"
      "BM_string_memcpy/2/32/8/iterations:1\n"
      "BM_string_memcpy/2/32/16/iterations:1\n"
      "BM_string_memcpy/2/32/32/iterations:1\n"
      "BM_string_memcpy/3/0/0/iterations:1\n"
      "BM_string_memcpy/3/1/1/iterations:1\n"
      "BM_string_memcpy/3/1/2/iterations:1\n"
      "BM_string_memcpy/3/1/4/iterations:1\n"
      "BM_string_memcpy/3/1/8/iterations:1\n"
      "BM_string_memcpy/3/1/16/iterations:1\n"
      "BM_string_memcpy/3/1/32/iterations:1\n"
      "BM_string_memcpy/3/2/1/iterations:1\n"
      "BM_string_memcpy/3/2/2/iterations:1\n"
      "BM_string_memcpy/3/2/4/iterations:1\n"
      "BM_string_memcpy/3/2/8/iterations:1\n"
      "BM_string_memcpy/3/2/16/iterations:1\n"
      "BM_string_memcpy/3/2/32/iterations:1\n"
      "BM_string_memcpy/3/4/1/iterations:1\n"
      "BM_string_memcpy/3/4/2/iterations:1\n"
      "BM_string_memcpy/3/4/4/iterations:1\n"
      "BM_string_memcpy/3/4/8/iterations:1\n"
      "BM_string_memcpy/3/4/16/iterations:1\n"
      "BM_string_memcpy/3/4/32/iterations:1\n"
      "BM_string_memcpy/3/8/1/iterations:1\n"
      "BM_string_memcpy/3/8/2/iterations:1\n"
      "BM_string_memcpy/3/8/4/iterations:1\n"
      "BM_string_memcpy/3/8/8/iterations:1\n"
      "BM_string_memcpy/3/8/16/iterations:1\n"
      "BM_string_memcpy/3/8/32/iterations:1\n"
      "BM_string_memcpy/3/16/1/iterations:1\n"
      "BM_string_memcpy/3/16/2/iterations:1\n"
      "BM_string_memcpy/3/16/4/iterations:1\n"
      "BM_string_memcpy/3/16/8/iterations:1\n"
      "BM_string_memcpy/3/16/16/iterations:1\n"
      "BM_string_memcpy/3/16/32/iterations:1\n"
      "BM_string_memcpy/3/32/1/iterations:1\n"
      "BM_string_memcpy/3/32/2/iterations:1\n"
      "BM_string_memcpy/3/32/4/iterations:1\n"
      "BM_string_memcpy/3/32/8/iterations:1\n"
      "BM_string_memcpy/3/32/16/iterations:1\n"
      "BM_string_memcpy/3/32/32/iterations:1\n"
      "BM_string_memcpy/4/0/0/iterations:1\n"
      "BM_string_memcpy/4/1/1/iterations:1\n"
      "BM_string_memcpy/4/1/2/iterations:1\n"
      "BM_string_memcpy/4/1/4/iterations:1\n"
      "BM_string_memcpy/4/1/8/iterations:1\n"
      "BM_string_memcpy/4/1/16/iterations:1\n"
      "BM_string_memcpy/4/1/32/iterations:1\n"
      "BM_string_memcpy/4/2/1/iterations:1\n"
      "BM_string_memcpy/4/2/2/iterations:1\n"
      "BM_string_memcpy/4/2/4/iterations:1\n"
      "BM_string_memcpy/4/2/8/iterations:1\n"
      "BM_string_memcpy/4/2/16/iterations:1\n"
      "BM_string_memcpy/4/2/32/iterations:1\n"
      "BM_string_memcpy/4/4/1/iterations:1\n"
      "BM_string_memcpy/4/4/2/iterations:1\n"
      "BM_string_memcpy/4/4/4/iterations:1\n"
      "BM_string_memcpy/4/4/8/iterations:1\n"
      "BM_string_memcpy/4/4/16/iterations:1\n"
      "BM_string_memcpy/4/4/32/iterations:1\n"
      "BM_string_memcpy/4/8/1/iterations:1\n"
      "BM_string_memcpy/4/8/2/iterations:1\n"
      "BM_string_memcpy/4/8/4/iterations:1\n"
      "BM_string_memcpy/4/8/8/iterations:1\n"
      "BM_string_memcpy/4/8/16/iterations:1\n"
      "BM_string_memcpy/4/8/32/iterations:1\n"
      "BM_string_memcpy/4/16/1/iterations:1\n"
      "BM_string_memcpy/4/16/2/iterations:1\n"
      "BM_string_memcpy/4/16/4/iterations:1\n"
      "BM_string_memcpy/4/16/8/iterations:1\n"
      "BM_string_memcpy/4/16/16/iterations:1\n"
      "BM_string_memcpy/4/16/32/iterations:1\n"
      "BM_string_memcpy/4/32/1/iterations:1\n"
      "BM_string_memcpy/4/32/2/iterations:1\n"
      "BM_string_memcpy/4/32/4/iterations:1\n"
      "BM_string_memcpy/4/32/8/iterations:1\n"
      "BM_string_memcpy/4/32/16/iterations:1\n"
      "BM_string_memcpy/4/32/32/iterations:1\n"
      "BM_string_memcpy/5/0/0/iterations:1\n"
      "BM_string_memcpy/5/1/1/iterations:1\n"
      "BM_string_memcpy/5/1/2/iterations:1\n"
      "BM_string_memcpy/5/1/4/iterations:1\n"
      "BM_string_memcpy/5/1/8/iterations:1\n"
      "BM_string_memcpy/5/1/16/iterations:1\n"
      "BM_string_memcpy/5/1/32/iterations:1\n"
      "BM_string_memcpy/5/2/1/iterations:1\n"
      "BM_string_memcpy/5/2/2/iterations:1\n"
      "BM_string_memcpy/5/2/4/iterations:1\n"
      "BM_string_memcpy/5/2/8/iterations:1\n"
      "BM_string_memcpy/5/2/16/iterations:1\n"
      "BM_string_memcpy/5/2/32/iterations:1\n"
      "BM_string_memcpy/5/4/1/iterations:1\n"
      "BM_string_memcpy/5/4/2/iterations:1\n"
      "BM_string_memcpy/5/4/4/iterations:1\n"
      "BM_string_memcpy/5/4/8/iterations:1\n"
      "BM_string_memcpy/5/4/16/iterations:1\n"
      "BM_string_memcpy/5/4/32/iterations:1\n"
      "BM_string_memcpy/5/8/1/iterations:1\n"
      "BM_string_memcpy/5/8/2/iterations:1\n"
      "BM_string_memcpy/5/8/4/iterations:1\n"
      "BM_string_memcpy/5/8/8/iterations:1\n"
      "BM_string_memcpy/5/8/16/iterations:1\n"
      "BM_string_memcpy/5/8/32/iterations:1\n"
      "BM_string_memcpy/5/16/1/iterations:1\n"
      "BM_string_memcpy/5/16/2/iterations:1\n"
      "BM_string_memcpy/5/16/4/iterations:1\n"
      "BM_string_memcpy/5/16/8/iterations:1\n"
      "BM_string_memcpy/5/16/16/iterations:1\n"
      "BM_string_memcpy/5/16/32/iterations:1\n"
      "BM_string_memcpy/5/32/1/iterations:1\n"
      "BM_string_memcpy/5/32/2/iterations:1\n"
      "BM_string_memcpy/5/32/4/iterations:1\n"
      "BM_string_memcpy/5/32/8/iterations:1\n"
      "BM_string_memcpy/5/32/16/iterations:1\n"
      "BM_string_memcpy/5/32/32/iterations:1\n"
      "BM_string_memcpy/6/0/0/iterations:1\n"
      "BM_string_memcpy/6/1/1/iterations:1\n"
      "BM_string_memcpy/6/1/2/iterations:1\n"
      "BM_string_memcpy/6/1/4/iterations:1\n"
      "BM_string_memcpy/6/1/8/iterations:1\n"
      "BM_string_memcpy/6/1/16/iterations:1\n"
      "BM_string_memcpy/6/1/32/iterations:1\n"
      "BM_string_memcpy/6/2/1/iterations:1\n"
      "BM_string_memcpy/6/2/2/iterations:1\n"
      "BM_string_memcpy/6/2/4/iterations:1\n"
      "BM_string_memcpy/6/2/8/iterations:1\n"
      "BM_string_memcpy/6/2/16/iterations:1\n"
      "BM_string_memcpy/6/2/32/iterations:1\n"
      "BM_string_memcpy/6/4/1/iterations:1\n"
      "BM_string_memcpy/6/4/2/iterations:1\n"
      "BM_string_memcpy/6/4/4/iterations:1\n"
      "BM_string_memcpy/6/4/8/iterations:1\n"
      "BM_string_memcpy/6/4/16/iterations:1\n"
      "BM_string_memcpy/6/4/32/iterations:1\n"
      "BM_string_memcpy/6/8/1/iterations:1\n"
      "BM_string_memcpy/6/8/2/iterations:1\n"
      "BM_string_memcpy/6/8/4/iterations:1\n"
      "BM_string_memcpy/6/8/8/iterations:1\n"
      "BM_string_memcpy/6/8/16/iterations:1\n"
      "BM_string_memcpy/6/8/32/iterations:1\n"
      "BM_string_memcpy/6/16/1/iterations:1\n"
      "BM_string_memcpy/6/16/2/iterations:1\n"
      "BM_string_memcpy/6/16/4/iterations:1\n"
      "BM_string_memcpy/6/16/8/iterations:1\n"
      "BM_string_memcpy/6/16/16/iterations:1\n"
      "BM_string_memcpy/6/16/32/iterations:1\n"
      "BM_string_memcpy/6/32/1/iterations:1\n"
      "BM_string_memcpy/6/32/2/iterations:1\n"
      "BM_string_memcpy/6/32/4/iterations:1\n"
      "BM_string_memcpy/6/32/8/iterations:1\n"
      "BM_string_memcpy/6/32/16/iterations:1\n"
      "BM_string_memcpy/6/32/32/iterations:1\n"
      "BM_string_memcpy/7/0/0/iterations:1\n"
      "BM_string_memcpy/7/1/1/iterations:1\n"
      "BM_string_memcpy/7/1/2/iterations:1\n"
      "BM_string_memcpy/7/1/4/iterations:1\n"
      "BM_string_memcpy/7/1/8/iterations:1\n"
      "BM_string_memcpy/7/1/16/iterations:1\n"
      "BM_string_memcpy/7/1/32/iterations:1\n"
      "BM_string_memcpy/7/2/1/iterations:1\n"
      "BM_string_memcpy/7/2/2/iterations:1\n"
      "BM_string_memcpy/7/2/4/iterations:1\n"
      "BM_string_memcpy/7/2/8/iterations:1\n"
      "BM_string_memcpy/7/2/16/iterations:1\n"
      "BM_string_memcpy/7/2/32/iterations:1\n"
      "BM_string_memcpy/7/4/1/iterations:1\n"
      "BM_string_memcpy/7/4/2/iterations:1\n"
      "BM_string_memcpy/7/4/4/iterations:1\n"
      "BM_string_memcpy/7/4/8/iterations:1\n"
      "BM_string_memcpy/7/4/16/iterations:1\n"
      "BM_string_memcpy/7/4/32/iterations:1\n"
      "BM_string_memcpy/7/8/1/iterations:1\n"
      "BM_string_memcpy/7/8/2/iterations:1\n"
      "BM_string_memcpy/7/8/4/iterations:1\n"
      "BM_string_memcpy/7/8/8/iterations:1\n"
      "BM_string_memcpy/7/8/16/iterations:1\n"
      "BM_string_memcpy/7/8/32/iterations:1\n"
      "BM_string_memcpy/7/16/1/iterations:1\n"
      "BM_string_memcpy/7/16/2/iterations:1\n"
      "BM_string_memcpy/7/16/4/iterations:1\n"
      "BM_string_memcpy/7/16/8/iterations:1\n"
      "BM_string_memcpy/7/16/16/iterations:1\n"
      "BM_string_memcpy/7/16/32/iterations:1\n"
      "BM_string_memcpy/7/32/1/iterations:1\n"
      "BM_string_memcpy/7/32/2/iterations:1\n"
      "BM_string_memcpy/7/32/4/iterations:1\n"
      "BM_string_memcpy/7/32/8/iterations:1\n"
      "BM_string_memcpy/7/32/16/iterations:1\n"
      "BM_string_memcpy/7/32/32/iterations:1\n"
      "BM_string_memcpy/8/0/0/iterations:1\n"
      "BM_string_memcpy/8/1/1/iterations:1\n"
      "BM_string_memcpy/8/1/2/iterations:1\n"
      "BM_string_memcpy/8/1/4/iterations:1\n"
      "BM_string_memcpy/8/1/8/iterations:1\n"
      "BM_string_memcpy/8/1/16/iterations:1\n"
      "BM_string_memcpy/8/1/32/iterations:1\n"
      "BM_string_memcpy/8/2/1/iterations:1\n"
      "BM_string_memcpy/8/2/2/iterations:1\n"
      "BM_string_memcpy/8/2/4/iterations:1\n"
      "BM_string_memcpy/8/2/8/iterations:1\n"
      "BM_string_memcpy/8/2/16/iterations:1\n"
      "BM_string_memcpy/8/2/32/iterations:1\n"
      "BM_string_memcpy/8/4/1/iterations:1\n"
      "BM_string_memcpy/8/4/2/iterations:1\n"
      "BM_string_memcpy/8/4/4/iterations:1\n"
      "BM_string_memcpy/8/4/8/iterations:1\n"
      "BM_string_memcpy/8/4/16/iterations:1\n"
      "BM_string_memcpy/8/4/32/iterations:1\n"
      "BM_string_memcpy/8/8/1/iterations:1\n"
      "BM_string_memcpy/8/8/2/iterations:1\n"
      "BM_string_memcpy/8/8/4/iterations:1\n"
      "BM_string_memcpy/8/8/8/iterations:1\n"
      "BM_string_memcpy/8/8/16/iterations:1\n"
      "BM_string_memcpy/8/8/32/iterations:1\n"
      "BM_string_memcpy/8/16/1/iterations:1\n"
      "BM_string_memcpy/8/16/2/iterations:1\n"
      "BM_string_memcpy/8/16/4/iterations:1\n"
      "BM_string_memcpy/8/16/8/iterations:1\n"
      "BM_string_memcpy/8/16/16/iterations:1\n"
      "BM_string_memcpy/8/16/32/iterations:1\n"
      "BM_string_memcpy/8/32/1/iterations:1\n"
      "BM_string_memcpy/8/32/2/iterations:1\n"
      "BM_string_memcpy/8/32/4/iterations:1\n"
      "BM_string_memcpy/8/32/8/iterations:1\n"
      "BM_string_memcpy/8/32/16/iterations:1\n"
      "BM_string_memcpy/8/32/32/iterations:1\n"
      "BM_string_memcpy/9/0/0/iterations:1\n"
      "BM_string_memcpy/9/1/1/iterations:1\n"
      "BM_string_memcpy/9/1/2/iterations:1\n"
      "BM_string_memcpy/9/1/4/iterations:1\n"
      "BM_string_memcpy/9/1/8/iterations:1\n"
      "BM_string_memcpy/9/1/16/iterations:1\n"
      "BM_string_memcpy/9/1/32/iterations:1\n"
      "BM_string_memcpy/9/2/1/iterations:1\n"
      "BM_string_memcpy/9/2/2/iterations:1\n"
      "BM_string_memcpy/9/2/4/iterations:1\n"
      "BM_string_memcpy/9/2/8/iterations:1\n"
      "BM_string_memcpy/9/2/16/iterations:1\n"
      "BM_string_memcpy/9/2/32/iterations:1\n"
      "BM_string_memcpy/9/4/1/iterations:1\n"
      "BM_string_memcpy/9/4/2/iterations:1\n"
      "BM_string_memcpy/9/4/4/iterations:1\n"
      "BM_string_memcpy/9/4/8/iterations:1\n"
      "BM_string_memcpy/9/4/16/iterations:1\n"
      "BM_string_memcpy/9/4/32/iterations:1\n"
      "BM_string_memcpy/9/8/1/iterations:1\n"
      "BM_string_memcpy/9/8/2/iterations:1\n"
      "BM_string_memcpy/9/8/4/iterations:1\n"
      "BM_string_memcpy/9/8/8/iterations:1\n"
      "BM_string_memcpy/9/8/16/iterations:1\n"
      "BM_string_memcpy/9/8/32/iterations:1\n"
      "BM_string_memcpy/9/16/1/iterations:1\n"
      "BM_string_memcpy/9/16/2/iterations:1\n"
      "BM_string_memcpy/9/16/4/iterations:1\n"
      "BM_string_memcpy/9/16/8/iterations:1\n"
      "BM_string_memcpy/9/16/16/iterations:1\n"
      "BM_string_memcpy/9/16/32/iterations:1\n"
      "BM_string_memcpy/9/32/1/iterations:1\n"
      "BM_string_memcpy/9/32/2/iterations:1\n"
      "BM_string_memcpy/9/32/4/iterations:1\n"
      "BM_string_memcpy/9/32/8/iterations:1\n"
      "BM_string_memcpy/9/32/16/iterations:1\n"
      "BM_string_memcpy/9/32/32/iterations:1\n"
      "BM_string_memcpy/10/0/0/iterations:1\n"
      "BM_string_memcpy/10/1/1/iterations:1\n"
      "BM_string_memcpy/10/1/2/iterations:1\n"
      "BM_string_memcpy/10/1/4/iterations:1\n"
      "BM_string_memcpy/10/1/8/iterations:1\n"
      "BM_string_memcpy/10/1/16/iterations:1\n"
      "BM_string_memcpy/10/1/32/iterations:1\n"
      "BM_string_memcpy/10/2/1/iterations:1\n"
      "BM_string_memcpy/10/2/2/iterations:1\n"
      "BM_string_memcpy/10/2/4/iterations:1\n"
      "BM_string_memcpy/10/2/8/iterations:1\n"
      "BM_string_memcpy/10/2/16/iterations:1\n"
      "BM_string_memcpy/10/2/32/iterations:1\n"
      "BM_string_memcpy/10/4/1/iterations:1\n"
      "BM_string_memcpy/10/4/2/iterations:1\n"
      "BM_string_memcpy/10/4/4/iterations:1\n"
      "BM_string_memcpy/10/4/8/iterations:1\n"
      "BM_string_memcpy/10/4/16/iterations:1\n"
      "BM_string_memcpy/10/4/32/iterations:1\n"
      "BM_string_memcpy/10/8/1/iterations:1\n"
      "BM_string_memcpy/10/8/2/iterations:1\n"
      "BM_string_memcpy/10/8/4/iterations:1\n"
      "BM_string_memcpy/10/8/8/iterations:1\n"
      "BM_string_memcpy/10/8/16/iterations:1\n"
      "BM_string_memcpy/10/8/32/iterations:1\n"
      "BM_string_memcpy/10/16/1/iterations:1\n"
      "BM_string_memcpy/10/16/2/iterations:1\n"
      "BM_string_memcpy/10/16/4/iterations:1\n"
      "BM_string_memcpy/10/16/8/iterations:1\n"
      "BM_string_memcpy/10/16/16/iterations:1\n"
      "BM_string_memcpy/10/16/32/iterations:1\n"
      "BM_string_memcpy/10/32/1/iterations:1\n"
      "BM_string_memcpy/10/32/2/iterations:1\n"
      "BM_string_memcpy/10/32/4/iterations:1\n"
      "BM_string_memcpy/10/32/8/iterations:1\n"
      "BM_string_memcpy/10/32/16/iterations:1\n"
      "BM_string_memcpy/10/32/32/iterations:1\n"
      "BM_string_memcpy/11/0/0/iterations:1\n"
      "BM_string_memcpy/11/1/1/iterations:1\n"
      "BM_string_memcpy/11/1/2/iterations:1\n"
      "BM_string_memcpy/11/1/4/iterations:1\n"
      "BM_string_memcpy/11/1/8/iterations:1\n"
      "BM_string_memcpy/11/1/16/iterations:1\n"
      "BM_string_memcpy/11/1/32/iterations:1\n"
      "BM_string_memcpy/11/2/1/iterations:1\n"
      "BM_string_memcpy/11/2/2/iterations:1\n"
      "BM_string_memcpy/11/2/4/iterations:1\n"
      "BM_string_memcpy/11/2/8/iterations:1\n"
      "BM_string_memcpy/11/2/16/iterations:1\n"
      "BM_string_memcpy/11/2/32/iterations:1\n"
      "BM_string_memcpy/11/4/1/iterations:1\n"
      "BM_string_memcpy/11/4/2/iterations:1\n"
      "BM_string_memcpy/11/4/4/iterations:1\n"
      "BM_string_memcpy/11/4/8/iterations:1\n"
      "BM_string_memcpy/11/4/16/iterations:1\n"
      "BM_string_memcpy/11/4/32/iterations:1\n"
      "BM_string_memcpy/11/8/1/iterations:1\n"
      "BM_string_memcpy/11/8/2/iterations:1\n"
      "BM_string_memcpy/11/8/4/iterations:1\n"
      "BM_string_memcpy/11/8/8/iterations:1\n"
      "BM_string_memcpy/11/8/16/iterations:1\n"
      "BM_string_memcpy/11/8/32/iterations:1\n"
      "BM_string_memcpy/11/16/1/iterations:1\n"
      "BM_string_memcpy/11/16/2/iterations:1\n"
      "BM_string_memcpy/11/16/4/iterations:1\n"
      "BM_string_memcpy/11/16/8/iterations:1\n"
      "BM_string_memcpy/11/16/16/iterations:1\n"
      "BM_string_memcpy/11/16/32/iterations:1\n"
      "BM_string_memcpy/11/32/1/iterations:1\n"
      "BM_string_memcpy/11/32/2/iterations:1\n"
      "BM_string_memcpy/11/32/4/iterations:1\n"
      "BM_string_memcpy/11/32/8/iterations:1\n"
      "BM_string_memcpy/11/32/16/iterations:1\n"
      "BM_string_memcpy/11/32/32/iterations:1\n"
      "BM_string_memcpy/12/0/0/iterations:1\n"
      "BM_string_memcpy/12/1/1/iterations:1\n"
      "BM_string_memcpy/12/1/2/iterations:1\n"
      "BM_string_memcpy/12/1/4/iterations:1\n"
      "BM_string_memcpy/12/1/8/iterations:1\n"
      "BM_string_memcpy/12/1/16/iterations:1\n"
      "BM_string_memcpy/12/1/32/iterations:1\n"
      "BM_string_memcpy/12/2/1/iterations:1\n"
      "BM_string_memcpy/12/2/2/iterations:1\n"
      "BM_string_memcpy/12/2/4/iterations:1\n"
      "BM_string_memcpy/12/2/8/iterations:1\n"
      "BM_string_memcpy/12/2/16/iterations:1\n"
      "BM_string_memcpy/12/2/32/iterations:1\n"
      "BM_string_memcpy/12/4/1/iterations:1\n"
      "BM_string_memcpy/12/4/2/iterations:1\n"
      "BM_string_memcpy/12/4/4/iterations:1\n"
      "BM_string_memcpy/12/4/8/iterations:1\n"
      "BM_string_memcpy/12/4/16/iterations:1\n"
      "BM_string_memcpy/12/4/32/iterations:1\n"
      "BM_string_memcpy/12/8/1/iterations:1\n"
      "BM_string_memcpy/12/8/2/iterations:1\n"
      "BM_string_memcpy/12/8/4/iterations:1\n"
      "BM_string_memcpy/12/8/8/iterations:1\n"
      "BM_string_memcpy/12/8/16/iterations:1\n"
      "BM_string_memcpy/12/8/32/iterations:1\n"
      "BM_string_memcpy/12/16/1/iterations:1\n"
      "BM_string_memcpy/12/16/2/iterations:1\n"
      "BM_string_memcpy/12/16/4/iterations:1\n"
      "BM_string_memcpy/12/16/8/iterations:1\n"
      "BM_string_memcpy/12/16/16/iterations:1\n"
      "BM_string_memcpy/12/16/32/iterations:1\n"
      "BM_string_memcpy/12/32/1/iterations:1\n"
      "BM_string_memcpy/12/32/2/iterations:1\n"
      "BM_string_memcpy/12/32/4/iterations:1\n"
      "BM_string_memcpy/12/32/8/iterations:1\n"
      "BM_string_memcpy/12/32/16/iterations:1\n"
      "BM_string_memcpy/12/32/32/iterations:1\n"
      "BM_string_memcpy/13/0/0/iterations:1\n"
      "BM_string_memcpy/13/1/1/iterations:1\n"
      "BM_string_memcpy/13/1/2/iterations:1\n"
      "BM_string_memcpy/13/1/4/iterations:1\n"
      "BM_string_memcpy/13/1/8/iterations:1\n"
      "BM_string_memcpy/13/1/16/iterations:1\n"
      "BM_string_memcpy/13/1/32/iterations:1\n"
      "BM_string_memcpy/13/2/1/iterations:1\n"
      "BM_string_memcpy/13/2/2/iterations:1\n"
      "BM_string_memcpy/13/2/4/iterations:1\n"
      "BM_string_memcpy/13/2/8/iterations:1\n"
      "BM_string_memcpy/13/2/16/iterations:1\n"
      "BM_string_memcpy/13/2/32/iterations:1\n"
      "BM_string_memcpy/13/4/1/iterations:1\n"
      "BM_string_memcpy/13/4/2/iterations:1\n"
      "BM_string_memcpy/13/4/4/iterations:1\n"
      "BM_string_memcpy/13/4/8/iterations:1\n"
      "BM_string_memcpy/13/4/16/iterations:1\n"
      "BM_string_memcpy/13/4/32/iterations:1\n"
      "BM_string_memcpy/13/8/1/iterations:1\n"
      "BM_string_memcpy/13/8/2/iterations:1\n"
      "BM_string_memcpy/13/8/4/iterations:1\n"
      "BM_string_memcpy/13/8/8/iterations:1\n"
      "BM_string_memcpy/13/8/16/iterations:1\n"
      "BM_string_memcpy/13/8/32/iterations:1\n"
      "BM_string_memcpy/13/16/1/iterations:1\n"
      "BM_string_memcpy/13/16/2/iterations:1\n"
      "BM_string_memcpy/13/16/4/iterations:1\n"
      "BM_string_memcpy/13/16/8/iterations:1\n"
      "BM_string_memcpy/13/16/16/iterations:1\n"
      "BM_string_memcpy/13/16/32/iterations:1\n"
      "BM_string_memcpy/13/32/1/iterations:1\n"
      "BM_string_memcpy/13/32/2/iterations:1\n"
      "BM_string_memcpy/13/32/4/iterations:1\n"
      "BM_string_memcpy/13/32/8/iterations:1\n"
      "BM_string_memcpy/13/32/16/iterations:1\n"
      "BM_string_memcpy/13/32/32/iterations:1\n"
      "BM_string_memcpy/14/0/0/iterations:1\n"
      "BM_string_memcpy/14/1/1/iterations:1\n"
      "BM_string_memcpy/14/1/2/iterations:1\n"
      "BM_string_memcpy/14/1/4/iterations:1\n"
      "BM_string_memcpy/14/1/8/iterations:1\n"
      "BM_string_memcpy/14/1/16/iterations:1\n"
      "BM_string_memcpy/14/1/32/iterations:1\n"
      "BM_string_memcpy/14/2/1/iterations:1\n"
      "BM_string_memcpy/14/2/2/iterations:1\n"
      "BM_string_memcpy/14/2/4/iterations:1\n"
      "BM_string_memcpy/14/2/8/iterations:1\n"
      "BM_string_memcpy/14/2/16/iterations:1\n"
      "BM_string_memcpy/14/2/32/iterations:1\n"
      "BM_string_memcpy/14/4/1/iterations:1\n"
      "BM_string_memcpy/14/4/2/iterations:1\n"
      "BM_string_memcpy/14/4/4/iterations:1\n"
      "BM_string_memcpy/14/4/8/iterations:1\n"
      "BM_string_memcpy/14/4/16/iterations:1\n"
      "BM_string_memcpy/14/4/32/iterations:1\n"
      "BM_string_memcpy/14/8/1/iterations:1\n"
      "BM_string_memcpy/14/8/2/iterations:1\n"
      "BM_string_memcpy/14/8/4/iterations:1\n"
      "BM_string_memcpy/14/8/8/iterations:1\n"
      "BM_string_memcpy/14/8/16/iterations:1\n"
      "BM_string_memcpy/14/8/32/iterations:1\n"
      "BM_string_memcpy/14/16/1/iterations:1\n"
      "BM_string_memcpy/14/16/2/iterations:1\n"
      "BM_string_memcpy/14/16/4/iterations:1\n"
      "BM_string_memcpy/14/16/8/iterations:1\n"
      "BM_string_memcpy/14/16/16/iterations:1\n"
      "BM_string_memcpy/14/16/32/iterations:1\n"
      "BM_string_memcpy/14/32/1/iterations:1\n"
      "BM_string_memcpy/14/32/2/iterations:1\n"
      "BM_string_memcpy/14/32/4/iterations:1\n"
      "BM_string_memcpy/14/32/8/iterations:1\n"
      "BM_string_memcpy/14/32/16/iterations:1\n"
      "BM_string_memcpy/14/32/32/iterations:1\n"
      "BM_string_memcpy/15/0/0/iterations:1\n"
      "BM_string_memcpy/15/1/1/iterations:1\n"
      "BM_string_memcpy/15/1/2/iterations:1\n"
      "BM_string_memcpy/15/1/4/iterations:1\n"
      "BM_string_memcpy/15/1/8/iterations:1\n"
      "BM_string_memcpy/15/1/16/iterations:1\n"
      "BM_string_memcpy/15/1/32/iterations:1\n"
      "BM_string_memcpy/15/2/1/iterations:1\n"
      "BM_string_memcpy/15/2/2/iterations:1\n"
      "BM_string_memcpy/15/2/4/iterations:1\n"
      "BM_string_memcpy/15/2/8/iterations:1\n"
      "BM_string_memcpy/15/2/16/iterations:1\n"
      "BM_string_memcpy/15/2/32/iterations:1\n"
      "BM_string_memcpy/15/4/1/iterations:1\n"
      "BM_string_memcpy/15/4/2/iterations:1\n"
      "BM_string_memcpy/15/4/4/iterations:1\n"
      "BM_string_memcpy/15/4/8/iterations:1\n"
      "BM_string_memcpy/15/4/16/iterations:1\n"
      "BM_string_memcpy/15/4/32/iterations:1\n"
      "BM_string_memcpy/15/8/1/iterations:1\n"
      "BM_string_memcpy/15/8/2/iterations:1\n"
      "BM_string_memcpy/15/8/4/iterations:1\n"
      "BM_string_memcpy/15/8/8/iterations:1\n"
      "BM_string_memcpy/15/8/16/iterations:1\n"
      "BM_string_memcpy/15/8/32/iterations:1\n"
      "BM_string_memcpy/15/16/1/iterations:1\n"
      "BM_string_memcpy/15/16/2/iterations:1\n"
      "BM_string_memcpy/15/16/4/iterations:1\n"
      "BM_string_memcpy/15/16/8/iterations:1\n"
      "BM_string_memcpy/15/16/16/iterations:1\n"
      "BM_string_memcpy/15/16/32/iterations:1\n"
      "BM_string_memcpy/15/32/1/iterations:1\n"
      "BM_string_memcpy/15/32/2/iterations:1\n"
      "BM_string_memcpy/15/32/4/iterations:1\n"
      "BM_string_memcpy/15/32/8/iterations:1\n"
      "BM_string_memcpy/15/32/16/iterations:1\n"
      "BM_string_memcpy/15/32/32/iterations:1\n"
      "BM_string_memcpy/16/0/0/iterations:1\n"
      "BM_string_memcpy/16/1/1/iterations:1\n"
      "BM_string_memcpy/16/1/2/iterations:1\n"
      "BM_string_memcpy/16/1/4/iterations:1\n"
      "BM_string_memcpy/16/1/8/iterations:1\n"
      "BM_string_memcpy/16/1/16/iterations:1\n"
      "BM_string_memcpy/16/1/32/iterations:1\n"
      "BM_string_memcpy/16/2/1/iterations:1\n"
      "BM_string_memcpy/16/2/2/iterations:1\n"
      "BM_string_memcpy/16/2/4/iterations:1\n"
      "BM_string_memcpy/16/2/8/iterations:1\n"
      "BM_string_memcpy/16/2/16/iterations:1\n"
      "BM_string_memcpy/16/2/32/iterations:1\n"
      "BM_string_memcpy/16/4/1/iterations:1\n"
      "BM_string_memcpy/16/4/2/iterations:1\n"
      "BM_string_memcpy/16/4/4/iterations:1\n"
      "BM_string_memcpy/16/4/8/iterations:1\n"
      "BM_string_memcpy/16/4/16/iterations:1\n"
      "BM_string_memcpy/16/4/32/iterations:1\n"
      "BM_string_memcpy/16/8/1/iterations:1\n"
      "BM_string_memcpy/16/8/2/iterations:1\n"
      "BM_string_memcpy/16/8/4/iterations:1\n"
      "BM_string_memcpy/16/8/8/iterations:1\n"
      "BM_string_memcpy/16/8/16/iterations:1\n"
      "BM_string_memcpy/16/8/32/iterations:1\n"
      "BM_string_memcpy/16/16/1/iterations:1\n"
      "BM_string_memcpy/16/16/2/iterations:1\n"
      "BM_string_memcpy/16/16/4/iterations:1\n"
      "BM_string_memcpy/16/16/8/iterations:1\n"
      "BM_string_memcpy/16/16/16/iterations:1\n"
      "BM_string_memcpy/16/16/32/iterations:1\n"
      "BM_string_memcpy/16/32/1/iterations:1\n"
      "BM_string_memcpy/16/32/2/iterations:1\n"
      "BM_string_memcpy/16/32/4/iterations:1\n"
      "BM_string_memcpy/16/32/8/iterations:1\n"
      "BM_string_memcpy/16/32/16/iterations:1\n"
      "BM_string_memcpy/16/32/32/iterations:1\n"
      "BM_string_memcpy/24/0/0/iterations:1\n"
      "BM_string_memcpy/24/1/1/iterations:1\n"
      "BM_string_memcpy/24/1/2/iterations:1\n"
      "BM_string_memcpy/24/1/4/iterations:1\n"
      "BM_string_memcpy/24/1/8/iterations:1\n"
      "BM_string_memcpy/24/1/16/iterations:1\n"
      "BM_string_memcpy/24/1/32/iterations:1\n"
      "BM_string_memcpy/24/2/1/iterations:1\n"
      "BM_string_memcpy/24/2/2/iterations:1\n"
      "BM_string_memcpy/24/2/4/iterations:1\n"
      "BM_string_memcpy/24/2/8/iterations:1\n"
      "BM_string_memcpy/24/2/16/iterations:1\n"
      "BM_string_memcpy/24/2/32/iterations:1\n"
      "BM_string_memcpy/24/4/1/iterations:1\n"
      "BM_string_memcpy/24/4/2/iterations:1\n"
      "BM_string_memcpy/24/4/4/iterations:1\n"
      "BM_string_memcpy/24/4/8/iterations:1\n"
      "BM_string_memcpy/24/4/16/iterations:1\n"
      "BM_string_memcpy/24/4/32/iterations:1\n"
      "BM_string_memcpy/24/8/1/iterations:1\n"
      "BM_string_memcpy/24/8/2/iterations:1\n"
      "BM_string_memcpy/24/8/4/iterations:1\n"
      "BM_string_memcpy/24/8/8/iterations:1\n"
      "BM_string_memcpy/24/8/16/iterations:1\n"
      "BM_string_memcpy/24/8/32/iterations:1\n"
      "BM_string_memcpy/24/16/1/iterations:1\n"
      "BM_string_memcpy/24/16/2/iterations:1\n"
      "BM_string_memcpy/24/16/4/iterations:1\n"
      "BM_string_memcpy/24/16/8/iterations:1\n"
      "BM_string_memcpy/24/16/16/iterations:1\n"
      "BM_string_memcpy/24/16/32/iterations:1\n"
      "BM_string_memcpy/24/32/1/iterations:1\n"
      "BM_string_memcpy/24/32/2/iterations:1\n"
      "BM_string_memcpy/24/32/4/iterations:1\n"
      "BM_string_memcpy/24/32/8/iterations:1\n"
      "BM_string_memcpy/24/32/16/iterations:1\n"
      "BM_string_memcpy/24/32/32/iterations:1\n"
      "BM_string_memcpy/32/0/0/iterations:1\n"
      "BM_string_memcpy/32/1/1/iterations:1\n"
      "BM_string_memcpy/32/1/2/iterations:1\n"
      "BM_string_memcpy/32/1/4/iterations:1\n"
      "BM_string_memcpy/32/1/8/iterations:1\n"
      "BM_string_memcpy/32/1/16/iterations:1\n"
      "BM_string_memcpy/32/1/32/iterations:1\n"
      "BM_string_memcpy/32/2/1/iterations:1\n"
      "BM_string_memcpy/32/2/2/iterations:1\n"
      "BM_string_memcpy/32/2/4/iterations:1\n"
      "BM_string_memcpy/32/2/8/iterations:1\n"
      "BM_string_memcpy/32/2/16/iterations:1\n"
      "BM_string_memcpy/32/2/32/iterations:1\n"
      "BM_string_memcpy/32/4/1/iterations:1\n"
      "BM_string_memcpy/32/4/2/iterations:1\n"
      "BM_string_memcpy/32/4/4/iterations:1\n"
      "BM_string_memcpy/32/4/8/iterations:1\n"
      "BM_string_memcpy/32/4/16/iterations:1\n"
      "BM_string_memcpy/32/4/32/iterations:1\n"
      "BM_string_memcpy/32/8/1/iterations:1\n"
      "BM_string_memcpy/32/8/2/iterations:1\n"
      "BM_string_memcpy/32/8/4/iterations:1\n"
      "BM_string_memcpy/32/8/8/iterations:1\n"
      "BM_string_memcpy/32/8/16/iterations:1\n"
      "BM_string_memcpy/32/8/32/iterations:1\n"
      "BM_string_memcpy/32/16/1/iterations:1\n"
      "BM_string_memcpy/32/16/2/iterations:1\n"
      "BM_string_memcpy/32/16/4/iterations:1\n"
      "BM_string_memcpy/32/16/8/iterations:1\n"
      "BM_string_memcpy/32/16/16/iterations:1\n"
      "BM_string_memcpy/32/16/32/iterations:1\n"
      "BM_string_memcpy/32/32/1/iterations:1\n"
      "BM_string_memcpy/32/32/2/iterations:1\n"
      "BM_string_memcpy/32/32/4/iterations:1\n"
      "BM_string_memcpy/32/32/8/iterations:1\n"
      "BM_string_memcpy/32/32/16/iterations:1\n"
      "BM_string_memcpy/32/32/32/iterations:1\n"
      "BM_string_memcpy/40/0/0/iterations:1\n"
      "BM_string_memcpy/40/1/1/iterations:1\n"
      "BM_string_memcpy/40/1/2/iterations:1\n"
      "BM_string_memcpy/40/1/4/iterations:1\n"
      "BM_string_memcpy/40/1/8/iterations:1\n"
      "BM_string_memcpy/40/1/16/iterations:1\n"
      "BM_string_memcpy/40/1/32/iterations:1\n"
      "BM_string_memcpy/40/2/1/iterations:1\n"
      "BM_string_memcpy/40/2/2/iterations:1\n"
      "BM_string_memcpy/40/2/4/iterations:1\n"
      "BM_string_memcpy/40/2/8/iterations:1\n"
      "BM_string_memcpy/40/2/16/iterations:1\n"
      "BM_string_memcpy/40/2/32/iterations:1\n"
      "BM_string_memcpy/40/4/1/iterations:1\n"
      "BM_string_memcpy/40/4/2/iterations:1\n"
      "BM_string_memcpy/40/4/4/iterations:1\n"
      "BM_string_memcpy/40/4/8/iterations:1\n"
      "BM_string_memcpy/40/4/16/iterations:1\n"
      "BM_string_memcpy/40/4/32/iterations:1\n"
      "BM_string_memcpy/40/8/1/iterations:1\n"
      "BM_string_memcpy/40/8/2/iterations:1\n"
      "BM_string_memcpy/40/8/4/iterations:1\n"
      "BM_string_memcpy/40/8/8/iterations:1\n"
      "BM_string_memcpy/40/8/16/iterations:1\n"
      "BM_string_memcpy/40/8/32/iterations:1\n"
      "BM_string_memcpy/40/16/1/iterations:1\n"
      "BM_string_memcpy/40/16/2/iterations:1\n"
      "BM_string_memcpy/40/16/4/iterations:1\n"
      "BM_string_memcpy/40/16/8/iterations:1\n"
      "BM_string_memcpy/40/16/16/iterations:1\n"
      "BM_string_memcpy/40/16/32/iterations:1\n"
      "BM_string_memcpy/40/32/1/iterations:1\n"
      "BM_string_memcpy/40/32/2/iterations:1\n"
      "BM_string_memcpy/40/32/4/iterations:1\n"
      "BM_string_memcpy/40/32/8/iterations:1\n"
      "BM_string_memcpy/40/32/16/iterations:1\n"
      "BM_string_memcpy/40/32/32/iterations:1\n"
      "BM_string_memcpy/48/0/0/iterations:1\n"
      "BM_string_memcpy/48/1/1/iterations:1\n"
      "BM_string_memcpy/48/1/2/iterations:1\n"
      "BM_string_memcpy/48/1/4/iterations:1\n"
      "BM_string_memcpy/48/1/8/iterations:1\n"
      "BM_string_memcpy/48/1/16/iterations:1\n"
      "BM_string_memcpy/48/1/32/iterations:1\n"
      "BM_string_memcpy/48/2/1/iterations:1\n"
      "BM_string_memcpy/48/2/2/iterations:1\n"
      "BM_string_memcpy/48/2/4/iterations:1\n"
      "BM_string_memcpy/48/2/8/iterations:1\n"
      "BM_string_memcpy/48/2/16/iterations:1\n"
      "BM_string_memcpy/48/2/32/iterations:1\n"
      "BM_string_memcpy/48/4/1/iterations:1\n"
      "BM_string_memcpy/48/4/2/iterations:1\n"
      "BM_string_memcpy/48/4/4/iterations:1\n"
      "BM_string_memcpy/48/4/8/iterations:1\n"
      "BM_string_memcpy/48/4/16/iterations:1\n"
      "BM_string_memcpy/48/4/32/iterations:1\n"
      "BM_string_memcpy/48/8/1/iterations:1\n"
      "BM_string_memcpy/48/8/2/iterations:1\n"
      "BM_string_memcpy/48/8/4/iterations:1\n"
      "BM_string_memcpy/48/8/8/iterations:1\n"
      "BM_string_memcpy/48/8/16/iterations:1\n"
      "BM_string_memcpy/48/8/32/iterations:1\n"
      "BM_string_memcpy/48/16/1/iterations:1\n"
      "BM_string_memcpy/48/16/2/iterations:1\n"
      "BM_string_memcpy/48/16/4/iterations:1\n"
      "BM_string_memcpy/48/16/8/iterations:1\n"
      "BM_string_memcpy/48/16/16/iterations:1\n"
      "BM_string_memcpy/48/16/32/iterations:1\n"
      "BM_string_memcpy/48/32/1/iterations:1\n"
      "BM_string_memcpy/48/32/2/iterations:1\n"
      "BM_string_memcpy/48/32/4/iterations:1\n"
      "BM_string_memcpy/48/32/8/iterations:1\n"
      "BM_string_memcpy/48/32/16/iterations:1\n"
      "BM_string_memcpy/48/32/32/iterations:1\n"
      "BM_string_memcpy/56/0/0/iterations:1\n"
      "BM_string_memcpy/56/1/1/iterations:1\n"
      "BM_string_memcpy/56/1/2/iterations:1\n"
      "BM_string_memcpy/56/1/4/iterations:1\n"
      "BM_string_memcpy/56/1/8/iterations:1\n"
      "BM_string_memcpy/56/1/16/iterations:1\n"
      "BM_string_memcpy/56/1/32/iterations:1\n"
      "BM_string_memcpy/56/2/1/iterations:1\n"
      "BM_string_memcpy/56/2/2/iterations:1\n"
      "BM_string_memcpy/56/2/4/iterations:1\n"
      "BM_string_memcpy/56/2/8/iterations:1\n"
      "BM_string_memcpy/56/2/16/iterations:1\n"
      "BM_string_memcpy/56/2/32/iterations:1\n"
      "BM_string_memcpy/56/4/1/iterations:1\n"
      "BM_string_memcpy/56/4/2/iterations:1\n"
      "BM_string_memcpy/56/4/4/iterations:1\n"
      "BM_string_memcpy/56/4/8/iterations:1\n"
      "BM_string_memcpy/56/4/16/iterations:1\n"
      "BM_string_memcpy/56/4/32/iterations:1\n"
      "BM_string_memcpy/56/8/1/iterations:1\n"
      "BM_string_memcpy/56/8/2/iterations:1\n"
      "BM_string_memcpy/56/8/4/iterations:1\n"
      "BM_string_memcpy/56/8/8/iterations:1\n"
      "BM_string_memcpy/56/8/16/iterations:1\n"
      "BM_string_memcpy/56/8/32/iterations:1\n"
      "BM_string_memcpy/56/16/1/iterations:1\n"
      "BM_string_memcpy/56/16/2/iterations:1\n"
      "BM_string_memcpy/56/16/4/iterations:1\n"
      "BM_string_memcpy/56/16/8/iterations:1\n"
      "BM_string_memcpy/56/16/16/iterations:1\n"
      "BM_string_memcpy/56/16/32/iterations:1\n"
      "BM_string_memcpy/56/32/1/iterations:1\n"
      "BM_string_memcpy/56/32/2/iterations:1\n"
      "BM_string_memcpy/56/32/4/iterations:1\n"
      "BM_string_memcpy/56/32/8/iterations:1\n"
      "BM_string_memcpy/56/32/16/iterations:1\n"
      "BM_string_memcpy/56/32/32/iterations:1\n"
      "BM_string_memcpy/64/0/0/iterations:1\n"
      "BM_string_memcpy/64/1/1/iterations:1\n"
      "BM_string_memcpy/64/1/2/iterations:1\n"
      "BM_string_memcpy/64/1/4/iterations:1\n"
      "BM_string_memcpy/64/1/8/iterations:1\n"
      "BM_string_memcpy/64/1/16/iterations:1\n"
      "BM_string_memcpy/64/1/32/iterations:1\n"
      "BM_string_memcpy/64/2/1/iterations:1\n"
      "BM_string_memcpy/64/2/2/iterations:1\n"
      "BM_string_memcpy/64/2/4/iterations:1\n"
      "BM_string_memcpy/64/2/8/iterations:1\n"
      "BM_string_memcpy/64/2/16/iterations:1\n"
      "BM_string_memcpy/64/2/32/iterations:1\n"
      "BM_string_memcpy/64/4/1/iterations:1\n"
      "BM_string_memcpy/64/4/2/iterations:1\n"
      "BM_string_memcpy/64/4/4/iterations:1\n"
      "BM_string_memcpy/64/4/8/iterations:1\n"
      "BM_string_memcpy/64/4/16/iterations:1\n"
      "BM_string_memcpy/64/4/32/iterations:1\n"
      "BM_string_memcpy/64/8/1/iterations:1\n"
      "BM_string_memcpy/64/8/2/iterations:1\n"
      "BM_string_memcpy/64/8/4/iterations:1\n"
      "BM_string_memcpy/64/8/8/iterations:1\n"
      "BM_string_memcpy/64/8/16/iterations:1\n"
      "BM_string_memcpy/64/8/32/iterations:1\n"
      "BM_string_memcpy/64/16/1/iterations:1\n"
      "BM_string_memcpy/64/16/2/iterations:1\n"
      "BM_string_memcpy/64/16/4/iterations:1\n"
      "BM_string_memcpy/64/16/8/iterations:1\n"
      "BM_string_memcpy/64/16/16/iterations:1\n"
      "BM_string_memcpy/64/16/32/iterations:1\n"
      "BM_string_memcpy/64/32/1/iterations:1\n"
      "BM_string_memcpy/64/32/2/iterations:1\n"
      "BM_string_memcpy/64/32/4/iterations:1\n"
      "BM_string_memcpy/64/32/8/iterations:1\n"
      "BM_string_memcpy/64/32/16/iterations:1\n"
      "BM_string_memcpy/64/32/32/iterations:1\n"
      "BM_string_memcpy/72/0/0/iterations:1\n"
      "BM_string_memcpy/72/1/1/iterations:1\n"
      "BM_string_memcpy/72/1/2/iterations:1\n"
      "BM_string_memcpy/72/1/4/iterations:1\n"
      "BM_string_memcpy/72/1/8/iterations:1\n"
      "BM_string_memcpy/72/1/16/iterations:1\n"
      "BM_string_memcpy/72/1/32/iterations:1\n"
      "BM_string_memcpy/72/2/1/iterations:1\n"
      "BM_string_memcpy/72/2/2/iterations:1\n"
      "BM_string_memcpy/72/2/4/iterations:1\n"
      "BM_string_memcpy/72/2/8/iterations:1\n"
      "BM_string_memcpy/72/2/16/iterations:1\n"
      "BM_string_memcpy/72/2/32/iterations:1\n"
      "BM_string_memcpy/72/4/1/iterations:1\n"
      "BM_string_memcpy/72/4/2/iterations:1\n"
      "BM_string_memcpy/72/4/4/iterations:1\n"
      "BM_string_memcpy/72/4/8/iterations:1\n"
      "BM_string_memcpy/72/4/16/iterations:1\n"
      "BM_string_memcpy/72/4/32/iterations:1\n"
      "BM_string_memcpy/72/8/1/iterations:1\n"
      "BM_string_memcpy/72/8/2/iterations:1\n"
      "BM_string_memcpy/72/8/4/iterations:1\n"
      "BM_string_memcpy/72/8/8/iterations:1\n"
      "BM_string_memcpy/72/8/16/iterations:1\n"
      "BM_string_memcpy/72/8/32/iterations:1\n"
      "BM_string_memcpy/72/16/1/iterations:1\n"
      "BM_string_memcpy/72/16/2/iterations:1\n"
      "BM_string_memcpy/72/16/4/iterations:1\n"
      "BM_string_memcpy/72/16/8/iterations:1\n"
      "BM_string_memcpy/72/16/16/iterations:1\n"
      "BM_string_memcpy/72/16/32/iterations:1\n"
      "BM_string_memcpy/72/32/1/iterations:1\n"
      "BM_string_memcpy/72/32/2/iterations:1\n"
      "BM_string_memcpy/72/32/4/iterations:1\n"
      "BM_string_memcpy/72/32/8/iterations:1\n"
      "BM_string_memcpy/72/32/16/iterations:1\n"
      "BM_string_memcpy/72/32/32/iterations:1\n"
      "BM_string_memcpy/80/0/0/iterations:1\n"
      "BM_string_memcpy/80/1/1/iterations:1\n"
      "BM_string_memcpy/80/1/2/iterations:1\n"
      "BM_string_memcpy/80/1/4/iterations:1\n"
      "BM_string_memcpy/80/1/8/iterations:1\n"
      "BM_string_memcpy/80/1/16/iterations:1\n"
      "BM_string_memcpy/80/1/32/iterations:1\n"
      "BM_string_memcpy/80/2/1/iterations:1\n"
      "BM_string_memcpy/80/2/2/iterations:1\n"
      "BM_string_memcpy/80/2/4/iterations:1\n"
      "BM_string_memcpy/80/2/8/iterations:1\n"
      "BM_string_memcpy/80/2/16/iterations:1\n"
      "BM_string_memcpy/80/2/32/iterations:1\n"
      "BM_string_memcpy/80/4/1/iterations:1\n"
      "BM_string_memcpy/80/4/2/iterations:1\n"
      "BM_string_memcpy/80/4/4/iterations:1\n"
      "BM_string_memcpy/80/4/8/iterations:1\n"
      "BM_string_memcpy/80/4/16/iterations:1\n"
      "BM_string_memcpy/80/4/32/iterations:1\n"
      "BM_string_memcpy/80/8/1/iterations:1\n"
      "BM_string_memcpy/80/8/2/iterations:1\n"
      "BM_string_memcpy/80/8/4/iterations:1\n"
      "BM_string_memcpy/80/8/8/iterations:1\n"
      "BM_string_memcpy/80/8/16/iterations:1\n"
      "BM_string_memcpy/80/8/32/iterations:1\n"
      "BM_string_memcpy/80/16/1/iterations:1\n"
      "BM_string_memcpy/80/16/2/iterations:1\n"
      "BM_string_memcpy/80/16/4/iterations:1\n"
      "BM_string_memcpy/80/16/8/iterations:1\n"
      "BM_string_memcpy/80/16/16/iterations:1\n"
      "BM_string_memcpy/80/16/32/iterations:1\n"
      "BM_string_memcpy/80/32/1/iterations:1\n"
      "BM_string_memcpy/80/32/2/iterations:1\n"
      "BM_string_memcpy/80/32/4/iterations:1\n"
      "BM_string_memcpy/80/32/8/iterations:1\n"
      "BM_string_memcpy/80/32/16/iterations:1\n"
      "BM_string_memcpy/80/32/32/iterations:1\n"
      "BM_string_memcpy/88/0/0/iterations:1\n"
      "BM_string_memcpy/88/1/1/iterations:1\n"
      "BM_string_memcpy/88/1/2/iterations:1\n"
      "BM_string_memcpy/88/1/4/iterations:1\n"
      "BM_string_memcpy/88/1/8/iterations:1\n"
      "BM_string_memcpy/88/1/16/iterations:1\n"
      "BM_string_memcpy/88/1/32/iterations:1\n"
      "BM_string_memcpy/88/2/1/iterations:1\n"
      "BM_string_memcpy/88/2/2/iterations:1\n"
      "BM_string_memcpy/88/2/4/iterations:1\n"
      "BM_string_memcpy/88/2/8/iterations:1\n"
      "BM_string_memcpy/88/2/16/iterations:1\n"
      "BM_string_memcpy/88/2/32/iterations:1\n"
      "BM_string_memcpy/88/4/1/iterations:1\n"
      "BM_string_memcpy/88/4/2/iterations:1\n"
      "BM_string_memcpy/88/4/4/iterations:1\n"
      "BM_string_memcpy/88/4/8/iterations:1\n"
      "BM_string_memcpy/88/4/16/iterations:1\n"
      "BM_string_memcpy/88/4/32/iterations:1\n"
      "BM_string_memcpy/88/8/1/iterations:1\n"
      "BM_string_memcpy/88/8/2/iterations:1\n"
      "BM_string_memcpy/88/8/4/iterations:1\n"
      "BM_string_memcpy/88/8/8/iterations:1\n"
      "BM_string_memcpy/88/8/16/iterations:1\n"
      "BM_string_memcpy/88/8/32/iterations:1\n"
      "BM_string_memcpy/88/16/1/iterations:1\n"
      "BM_string_memcpy/88/16/2/iterations:1\n"
      "BM_string_memcpy/88/16/4/iterations:1\n"
      "BM_string_memcpy/88/16/8/iterations:1\n"
      "BM_string_memcpy/88/16/16/iterations:1\n"
      "BM_string_memcpy/88/16/32/iterations:1\n"
      "BM_string_memcpy/88/32/1/iterations:1\n"
      "BM_string_memcpy/88/32/2/iterations:1\n"
      "BM_string_memcpy/88/32/4/iterations:1\n"
      "BM_string_memcpy/88/32/8/iterations:1\n"
      "BM_string_memcpy/88/32/16/iterations:1\n"
      "BM_string_memcpy/88/32/32/iterations:1\n"
      "BM_string_memcpy/96/0/0/iterations:1\n"
      "BM_string_memcpy/96/1/1/iterations:1\n"
      "BM_string_memcpy/96/1/2/iterations:1\n"
      "BM_string_memcpy/96/1/4/iterations:1\n"
      "BM_string_memcpy/96/1/8/iterations:1\n"
      "BM_string_memcpy/96/1/16/iterations:1\n"
      "BM_string_memcpy/96/1/32/iterations:1\n"
      "BM_string_memcpy/96/2/1/iterations:1\n"
      "BM_string_memcpy/96/2/2/iterations:1\n"
      "BM_string_memcpy/96/2/4/iterations:1\n"
      "BM_string_memcpy/96/2/8/iterations:1\n"
      "BM_string_memcpy/96/2/16/iterations:1\n"
      "BM_string_memcpy/96/2/32/iterations:1\n"
      "BM_string_memcpy/96/4/1/iterations:1\n"
      "BM_string_memcpy/96/4/2/iterations:1\n"
      "BM_string_memcpy/96/4/4/iterations:1\n"
      "BM_string_memcpy/96/4/8/iterations:1\n"
      "BM_string_memcpy/96/4/16/iterations:1\n"
      "BM_string_memcpy/96/4/32/iterations:1\n"
      "BM_string_memcpy/96/8/1/iterations:1\n"
      "BM_string_memcpy/96/8/2/iterations:1\n"
      "BM_string_memcpy/96/8/4/iterations:1\n"
      "BM_string_memcpy/96/8/8/iterations:1\n"
      "BM_string_memcpy/96/8/16/iterations:1\n"
      "BM_string_memcpy/96/8/32/iterations:1\n"
      "BM_string_memcpy/96/16/1/iterations:1\n"
      "BM_string_memcpy/96/16/2/iterations:1\n"
      "BM_string_memcpy/96/16/4/iterations:1\n"
      "BM_string_memcpy/96/16/8/iterations:1\n"
      "BM_string_memcpy/96/16/16/iterations:1\n"
      "BM_string_memcpy/96/16/32/iterations:1\n"
      "BM_string_memcpy/96/32/1/iterations:1\n"
      "BM_string_memcpy/96/32/2/iterations:1\n"
      "BM_string_memcpy/96/32/4/iterations:1\n"
      "BM_string_memcpy/96/32/8/iterations:1\n"
      "BM_string_memcpy/96/32/16/iterations:1\n"
      "BM_string_memcpy/96/32/32/iterations:1\n"
      "BM_string_memcpy/104/0/0/iterations:1\n"
      "BM_string_memcpy/104/1/1/iterations:1\n"
      "BM_string_memcpy/104/1/2/iterations:1\n"
      "BM_string_memcpy/104/1/4/iterations:1\n"
      "BM_string_memcpy/104/1/8/iterations:1\n"
      "BM_string_memcpy/104/1/16/iterations:1\n"
      "BM_string_memcpy/104/1/32/iterations:1\n"
      "BM_string_memcpy/104/2/1/iterations:1\n"
      "BM_string_memcpy/104/2/2/iterations:1\n"
      "BM_string_memcpy/104/2/4/iterations:1\n"
      "BM_string_memcpy/104/2/8/iterations:1\n"
      "BM_string_memcpy/104/2/16/iterations:1\n"
      "BM_string_memcpy/104/2/32/iterations:1\n"
      "BM_string_memcpy/104/4/1/iterations:1\n"
      "BM_string_memcpy/104/4/2/iterations:1\n"
      "BM_string_memcpy/104/4/4/iterations:1\n"
      "BM_string_memcpy/104/4/8/iterations:1\n"
      "BM_string_memcpy/104/4/16/iterations:1\n"
      "BM_string_memcpy/104/4/32/iterations:1\n"
      "BM_string_memcpy/104/8/1/iterations:1\n"
      "BM_string_memcpy/104/8/2/iterations:1\n"
      "BM_string_memcpy/104/8/4/iterations:1\n"
      "BM_string_memcpy/104/8/8/iterations:1\n"
      "BM_string_memcpy/104/8/16/iterations:1\n"
      "BM_string_memcpy/104/8/32/iterations:1\n"
      "BM_string_memcpy/104/16/1/iterations:1\n"
      "BM_string_memcpy/104/16/2/iterations:1\n"
      "BM_string_memcpy/104/16/4/iterations:1\n"
      "BM_string_memcpy/104/16/8/iterations:1\n"
      "BM_string_memcpy/104/16/16/iterations:1\n"
      "BM_string_memcpy/104/16/32/iterations:1\n"
      "BM_string_memcpy/104/32/1/iterations:1\n"
      "BM_string_memcpy/104/32/2/iterations:1\n"
      "BM_string_memcpy/104/32/4/iterations:1\n"
      "BM_string_memcpy/104/32/8/iterations:1\n"
      "BM_string_memcpy/104/32/16/iterations:1\n"
      "BM_string_memcpy/104/32/32/iterations:1\n"
      "BM_string_memcpy/112/0/0/iterations:1\n"
      "BM_string_memcpy/112/1/1/iterations:1\n"
      "BM_string_memcpy/112/1/2/iterations:1\n"
      "BM_string_memcpy/112/1/4/iterations:1\n"
      "BM_string_memcpy/112/1/8/iterations:1\n"
      "BM_string_memcpy/112/1/16/iterations:1\n"
      "BM_string_memcpy/112/1/32/iterations:1\n"
      "BM_string_memcpy/112/2/1/iterations:1\n"
      "BM_string_memcpy/112/2/2/iterations:1\n"
      "BM_string_memcpy/112/2/4/iterations:1\n"
      "BM_string_memcpy/112/2/8/iterations:1\n"
      "BM_string_memcpy/112/2/16/iterations:1\n"
      "BM_string_memcpy/112/2/32/iterations:1\n"
      "BM_string_memcpy/112/4/1/iterations:1\n"
      "BM_string_memcpy/112/4/2/iterations:1\n"
      "BM_string_memcpy/112/4/4/iterations:1\n"
      "BM_string_memcpy/112/4/8/iterations:1\n"
      "BM_string_memcpy/112/4/16/iterations:1\n"
      "BM_string_memcpy/112/4/32/iterations:1\n"
      "BM_string_memcpy/112/8/1/iterations:1\n"
      "BM_string_memcpy/112/8/2/iterations:1\n"
      "BM_string_memcpy/112/8/4/iterations:1\n"
      "BM_string_memcpy/112/8/8/iterations:1\n"
      "BM_string_memcpy/112/8/16/iterations:1\n"
      "BM_string_memcpy/112/8/32/iterations:1\n"
      "BM_string_memcpy/112/16/1/iterations:1\n"
      "BM_string_memcpy/112/16/2/iterations:1\n"
      "BM_string_memcpy/112/16/4/iterations:1\n"
      "BM_string_memcpy/112/16/8/iterations:1\n"
      "BM_string_memcpy/112/16/16/iterations:1\n"
      "BM_string_memcpy/112/16/32/iterations:1\n"
      "BM_string_memcpy/112/32/1/iterations:1\n"
      "BM_string_memcpy/112/32/2/iterations:1\n"
      "BM_string_memcpy/112/32/4/iterations:1\n"
      "BM_string_memcpy/112/32/8/iterations:1\n"
      "BM_string_memcpy/112/32/16/iterations:1\n"
      "BM_string_memcpy/112/32/32/iterations:1\n"
      "BM_string_memcpy/120/0/0/iterations:1\n"
      "BM_string_memcpy/120/1/1/iterations:1\n"
      "BM_string_memcpy/120/1/2/iterations:1\n"
      "BM_string_memcpy/120/1/4/iterations:1\n"
      "BM_string_memcpy/120/1/8/iterations:1\n"
      "BM_string_memcpy/120/1/16/iterations:1\n"
      "BM_string_memcpy/120/1/32/iterations:1\n"
      "BM_string_memcpy/120/2/1/iterations:1\n"
      "BM_string_memcpy/120/2/2/iterations:1\n"
      "BM_string_memcpy/120/2/4/iterations:1\n"
      "BM_string_memcpy/120/2/8/iterations:1\n"
      "BM_string_memcpy/120/2/16/iterations:1\n"
      "BM_string_memcpy/120/2/32/iterations:1\n"
      "BM_string_memcpy/120/4/1/iterations:1\n"
      "BM_string_memcpy/120/4/2/iterations:1\n"
      "BM_string_memcpy/120/4/4/iterations:1\n"
      "BM_string_memcpy/120/4/8/iterations:1\n"
      "BM_string_memcpy/120/4/16/iterations:1\n"
      "BM_string_memcpy/120/4/32/iterations:1\n"
      "BM_string_memcpy/120/8/1/iterations:1\n"
      "BM_string_memcpy/120/8/2/iterations:1\n"
      "BM_string_memcpy/120/8/4/iterations:1\n"
      "BM_string_memcpy/120/8/8/iterations:1\n"
      "BM_string_memcpy/120/8/16/iterations:1\n"
      "BM_string_memcpy/120/8/32/iterations:1\n"
      "BM_string_memcpy/120/16/1/iterations:1\n"
      "BM_string_memcpy/120/16/2/iterations:1\n"
      "BM_string_memcpy/120/16/4/iterations:1\n"
      "BM_string_memcpy/120/16/8/iterations:1\n"
      "BM_string_memcpy/120/16/16/iterations:1\n"
      "BM_string_memcpy/120/16/32/iterations:1\n"
      "BM_string_memcpy/120/32/1/iterations:1\n"
      "BM_string_memcpy/120/32/2/iterations:1\n"
      "BM_string_memcpy/120/32/4/iterations:1\n"
      "BM_string_memcpy/120/32/8/iterations:1\n"
      "BM_string_memcpy/120/32/16/iterations:1\n"
      "BM_string_memcpy/120/32/32/iterations:1\n"
      "BM_string_memcpy/128/0/0/iterations:1\n"
      "BM_string_memcpy/128/1/1/iterations:1\n"
      "BM_string_memcpy/128/1/2/iterations:1\n"
      "BM_string_memcpy/128/1/4/iterations:1\n"
      "BM_string_memcpy/128/1/8/iterations:1\n"
      "BM_string_memcpy/128/1/16/iterations:1\n"
      "BM_string_memcpy/128/1/32/iterations:1\n"
      "BM_string_memcpy/128/2/1/iterations:1\n"
      "BM_string_memcpy/128/2/2/iterations:1\n"
      "BM_string_memcpy/128/2/4/iterations:1\n"
      "BM_string_memcpy/128/2/8/iterations:1\n"
      "BM_string_memcpy/128/2/16/iterations:1\n"
      "BM_string_memcpy/128/2/32/iterations:1\n"
      "BM_string_memcpy/128/4/1/iterations:1\n"
      "BM_string_memcpy/128/4/2/iterations:1\n"
      "BM_string_memcpy/128/4/4/iterations:1\n"
      "BM_string_memcpy/128/4/8/iterations:1\n"
      "BM_string_memcpy/128/4/16/iterations:1\n"
      "BM_string_memcpy/128/4/32/iterations:1\n"
      "BM_string_memcpy/128/8/1/iterations:1\n"
      "BM_string_memcpy/128/8/2/iterations:1\n"
      "BM_string_memcpy/128/8/4/iterations:1\n"
      "BM_string_memcpy/128/8/8/iterations:1\n"
      "BM_string_memcpy/128/8/16/iterations:1\n"
      "BM_string_memcpy/128/8/32/iterations:1\n"
      "BM_string_memcpy/128/16/1/iterations:1\n"
      "BM_string_memcpy/128/16/2/iterations:1\n"
      "BM_string_memcpy/128/16/4/iterations:1\n"
      "BM_string_memcpy/128/16/8/iterations:1\n"
      "BM_string_memcpy/128/16/16/iterations:1\n"
      "BM_string_memcpy/128/16/32/iterations:1\n"
      "BM_string_memcpy/128/32/1/iterations:1\n"
      "BM_string_memcpy/128/32/2/iterations:1\n"
      "BM_string_memcpy/128/32/4/iterations:1\n"
      "BM_string_memcpy/128/32/8/iterations:1\n"
      "BM_string_memcpy/128/32/16/iterations:1\n"
      "BM_string_memcpy/128/32/32/iterations:1\n"
      "BM_string_memcpy/136/0/0/iterations:1\n"
      "BM_string_memcpy/136/1/1/iterations:1\n"
      "BM_string_memcpy/136/1/2/iterations:1\n"
      "BM_string_memcpy/136/1/4/iterations:1\n"
      "BM_string_memcpy/136/1/8/iterations:1\n"
      "BM_string_memcpy/136/1/16/iterations:1\n"
      "BM_string_memcpy/136/1/32/iterations:1\n"
      "BM_string_memcpy/136/2/1/iterations:1\n"
      "BM_string_memcpy/136/2/2/iterations:1\n"
      "BM_string_memcpy/136/2/4/iterations:1\n"
      "BM_string_memcpy/136/2/8/iterations:1\n"
      "BM_string_memcpy/136/2/16/iterations:1\n"
      "BM_string_memcpy/136/2/32/iterations:1\n"
      "BM_string_memcpy/136/4/1/iterations:1\n"
      "BM_string_memcpy/136/4/2/iterations:1\n"
      "BM_string_memcpy/136/4/4/iterations:1\n"
      "BM_string_memcpy/136/4/8/iterations:1\n"
      "BM_string_memcpy/136/4/16/iterations:1\n"
      "BM_string_memcpy/136/4/32/iterations:1\n"
      "BM_string_memcpy/136/8/1/iterations:1\n"
      "BM_string_memcpy/136/8/2/iterations:1\n"
      "BM_string_memcpy/136/8/4/iterations:1\n"
      "BM_string_memcpy/136/8/8/iterations:1\n"
      "BM_string_memcpy/136/8/16/iterations:1\n"
      "BM_string_memcpy/136/8/32/iterations:1\n"
      "BM_string_memcpy/136/16/1/iterations:1\n"
      "BM_string_memcpy/136/16/2/iterations:1\n"
      "BM_string_memcpy/136/16/4/iterations:1\n"
      "BM_string_memcpy/136/16/8/iterations:1\n"
      "BM_string_memcpy/136/16/16/iterations:1\n"
      "BM_string_memcpy/136/16/32/iterations:1\n"
      "BM_string_memcpy/136/32/1/iterations:1\n"
      "BM_string_memcpy/136/32/2/iterations:1\n"
      "BM_string_memcpy/136/32/4/iterations:1\n"
      "BM_string_memcpy/136/32/8/iterations:1\n"
      "BM_string_memcpy/136/32/16/iterations:1\n"
      "BM_string_memcpy/136/32/32/iterations:1\n"
      "BM_string_memcpy/144/0/0/iterations:1\n"
      "BM_string_memcpy/144/1/1/iterations:1\n"
      "BM_string_memcpy/144/1/2/iterations:1\n"
      "BM_string_memcpy/144/1/4/iterations:1\n"
      "BM_string_memcpy/144/1/8/iterations:1\n"
      "BM_string_memcpy/144/1/16/iterations:1\n"
      "BM_string_memcpy/144/1/32/iterations:1\n"
      "BM_string_memcpy/144/2/1/iterations:1\n"
      "BM_string_memcpy/144/2/2/iterations:1\n"
      "BM_string_memcpy/144/2/4/iterations:1\n"
      "BM_string_memcpy/144/2/8/iterations:1\n"
      "BM_string_memcpy/144/2/16/iterations:1\n"
      "BM_string_memcpy/144/2/32/iterations:1\n"
      "BM_string_memcpy/144/4/1/iterations:1\n"
      "BM_string_memcpy/144/4/2/iterations:1\n"
      "BM_string_memcpy/144/4/4/iterations:1\n"
      "BM_string_memcpy/144/4/8/iterations:1\n"
      "BM_string_memcpy/144/4/16/iterations:1\n"
      "BM_string_memcpy/144/4/32/iterations:1\n"
      "BM_string_memcpy/144/8/1/iterations:1\n"
      "BM_string_memcpy/144/8/2/iterations:1\n"
      "BM_string_memcpy/144/8/4/iterations:1\n"
      "BM_string_memcpy/144/8/8/iterations:1\n"
      "BM_string_memcpy/144/8/16/iterations:1\n"
      "BM_string_memcpy/144/8/32/iterations:1\n"
      "BM_string_memcpy/144/16/1/iterations:1\n"
      "BM_string_memcpy/144/16/2/iterations:1\n"
      "BM_string_memcpy/144/16/4/iterations:1\n"
      "BM_string_memcpy/144/16/8/iterations:1\n"
      "BM_string_memcpy/144/16/16/iterations:1\n"
      "BM_string_memcpy/144/16/32/iterations:1\n"
      "BM_string_memcpy/144/32/1/iterations:1\n"
      "BM_string_memcpy/144/32/2/iterations:1\n"
      "BM_string_memcpy/144/32/4/iterations:1\n"
      "BM_string_memcpy/144/32/8/iterations:1\n"
      "BM_string_memcpy/144/32/16/iterations:1\n"
      "BM_string_memcpy/144/32/32/iterations:1\n"
      "BM_string_memcpy/160/0/0/iterations:1\n"
      "BM_string_memcpy/160/1/1/iterations:1\n"
      "BM_string_memcpy/160/1/2/iterations:1\n"
      "BM_string_memcpy/160/1/4/iterations:1\n"
      "BM_string_memcpy/160/1/8/iterations:1\n"
      "BM_string_memcpy/160/1/16/iterations:1\n"
      "BM_string_memcpy/160/1/32/iterations:1\n"
      "BM_string_memcpy/160/2/1/iterations:1\n"
      "BM_string_memcpy/160/2/2/iterations:1\n"
      "BM_string_memcpy/160/2/4/iterations:1\n"
      "BM_string_memcpy/160/2/8/iterations:1\n"
      "BM_string_memcpy/160/2/16/iterations:1\n"
      "BM_string_memcpy/160/2/32/iterations:1\n"
      "BM_string_memcpy/160/4/1/iterations:1\n"
      "BM_string_memcpy/160/4/2/iterations:1\n"
      "BM_string_memcpy/160/4/4/iterations:1\n"
      "BM_string_memcpy/160/4/8/iterations:1\n"
      "BM_string_memcpy/160/4/16/iterations:1\n"
      "BM_string_memcpy/160/4/32/iterations:1\n"
      "BM_string_memcpy/160/8/1/iterations:1\n"
      "BM_string_memcpy/160/8/2/iterations:1\n"
      "BM_string_memcpy/160/8/4/iterations:1\n"
      "BM_string_memcpy/160/8/8/iterations:1\n"
      "BM_string_memcpy/160/8/16/iterations:1\n"
      "BM_string_memcpy/160/8/32/iterations:1\n"
      "BM_string_memcpy/160/16/1/iterations:1\n"
      "BM_string_memcpy/160/16/2/iterations:1\n"
      "BM_string_memcpy/160/16/4/iterations:1\n"
      "BM_string_memcpy/160/16/8/iterations:1\n"
      "BM_string_memcpy/160/16/16/iterations:1\n"
      "BM_string_memcpy/160/16/32/iterations:1\n"
      "BM_string_memcpy/160/32/1/iterations:1\n"
      "BM_string_memcpy/160/32/2/iterations:1\n"
      "BM_string_memcpy/160/32/4/iterations:1\n"
      "BM_string_memcpy/160/32/8/iterations:1\n"
      "BM_string_memcpy/160/32/16/iterations:1\n"
      "BM_string_memcpy/160/32/32/iterations:1\n"
      "BM_string_memcpy/176/0/0/iterations:1\n"
      "BM_string_memcpy/176/1/1/iterations:1\n"
      "BM_string_memcpy/176/1/2/iterations:1\n"
      "BM_string_memcpy/176/1/4/iterations:1\n"
      "BM_string_memcpy/176/1/8/iterations:1\n"
      "BM_string_memcpy/176/1/16/iterations:1\n"
      "BM_string_memcpy/176/1/32/iterations:1\n"
      "BM_string_memcpy/176/2/1/iterations:1\n"
      "BM_string_memcpy/176/2/2/iterations:1\n"
      "BM_string_memcpy/176/2/4/iterations:1\n"
      "BM_string_memcpy/176/2/8/iterations:1\n"
      "BM_string_memcpy/176/2/16/iterations:1\n"
      "BM_string_memcpy/176/2/32/iterations:1\n"
      "BM_string_memcpy/176/4/1/iterations:1\n"
      "BM_string_memcpy/176/4/2/iterations:1\n"
      "BM_string_memcpy/176/4/4/iterations:1\n"
      "BM_string_memcpy/176/4/8/iterations:1\n"
      "BM_string_memcpy/176/4/16/iterations:1\n"
      "BM_string_memcpy/176/4/32/iterations:1\n"
      "BM_string_memcpy/176/8/1/iterations:1\n"
      "BM_string_memcpy/176/8/2/iterations:1\n"
      "BM_string_memcpy/176/8/4/iterations:1\n"
      "BM_string_memcpy/176/8/8/iterations:1\n"
      "BM_string_memcpy/176/8/16/iterations:1\n"
      "BM_string_memcpy/176/8/32/iterations:1\n"
      "BM_string_memcpy/176/16/1/iterations:1\n"
      "BM_string_memcpy/176/16/2/iterations:1\n"
      "BM_string_memcpy/176/16/4/iterations:1\n"
      "BM_string_memcpy/176/16/8/iterations:1\n"
      "BM_string_memcpy/176/16/16/iterations:1\n"
      "BM_string_memcpy/176/16/32/iterations:1\n"
      "BM_string_memcpy/176/32/1/iterations:1\n"
      "BM_string_memcpy/176/32/2/iterations:1\n"
      "BM_string_memcpy/176/32/4/iterations:1\n"
      "BM_string_memcpy/176/32/8/iterations:1\n"
      "BM_string_memcpy/176/32/16/iterations:1\n"
      "BM_string_memcpy/176/32/32/iterations:1\n"
      "BM_string_memcpy/192/0/0/iterations:1\n"
      "BM_string_memcpy/192/1/1/iterations:1\n"
      "BM_string_memcpy/192/1/2/iterations:1\n"
      "BM_string_memcpy/192/1/4/iterations:1\n"
      "BM_string_memcpy/192/1/8/iterations:1\n"
      "BM_string_memcpy/192/1/16/iterations:1\n"
      "BM_string_memcpy/192/1/32/iterations:1\n"
      "BM_string_memcpy/192/2/1/iterations:1\n"
      "BM_string_memcpy/192/2/2/iterations:1\n"
      "BM_string_memcpy/192/2/4/iterations:1\n"
      "BM_string_memcpy/192/2/8/iterations:1\n"
      "BM_string_memcpy/192/2/16/iterations:1\n"
      "BM_string_memcpy/192/2/32/iterations:1\n"
      "BM_string_memcpy/192/4/1/iterations:1\n"
      "BM_string_memcpy/192/4/2/iterations:1\n"
      "BM_string_memcpy/192/4/4/iterations:1\n"
      "BM_string_memcpy/192/4/8/iterations:1\n"
      "BM_string_memcpy/192/4/16/iterations:1\n"
      "BM_string_memcpy/192/4/32/iterations:1\n"
      "BM_string_memcpy/192/8/1/iterations:1\n"
      "BM_string_memcpy/192/8/2/iterations:1\n"
      "BM_string_memcpy/192/8/4/iterations:1\n"
      "BM_string_memcpy/192/8/8/iterations:1\n"
      "BM_string_memcpy/192/8/16/iterations:1\n"
      "BM_string_memcpy/192/8/32/iterations:1\n"
      "BM_string_memcpy/192/16/1/iterations:1\n"
      "BM_string_memcpy/192/16/2/iterations:1\n"
      "BM_string_memcpy/192/16/4/iterations:1\n"
      "BM_string_memcpy/192/16/8/iterations:1\n"
      "BM_string_memcpy/192/16/16/iterations:1\n"
      "BM_string_memcpy/192/16/32/iterations:1\n"
      "BM_string_memcpy/192/32/1/iterations:1\n"
      "BM_string_memcpy/192/32/2/iterations:1\n"
      "BM_string_memcpy/192/32/4/iterations:1\n"
      "BM_string_memcpy/192/32/8/iterations:1\n"
      "BM_string_memcpy/192/32/16/iterations:1\n"
      "BM_string_memcpy/192/32/32/iterations:1\n"
      "BM_string_memcpy/208/0/0/iterations:1\n"
      "BM_string_memcpy/208/1/1/iterations:1\n"
      "BM_string_memcpy/208/1/2/iterations:1\n"
      "BM_string_memcpy/208/1/4/iterations:1\n"
      "BM_string_memcpy/208/1/8/iterations:1\n"
      "BM_string_memcpy/208/1/16/iterations:1\n"
      "BM_string_memcpy/208/1/32/iterations:1\n"
      "BM_string_memcpy/208/2/1/iterations:1\n"
      "BM_string_memcpy/208/2/2/iterations:1\n"
      "BM_string_memcpy/208/2/4/iterations:1\n"
      "BM_string_memcpy/208/2/8/iterations:1\n"
      "BM_string_memcpy/208/2/16/iterations:1\n"
      "BM_string_memcpy/208/2/32/iterations:1\n"
      "BM_string_memcpy/208/4/1/iterations:1\n"
      "BM_string_memcpy/208/4/2/iterations:1\n"
      "BM_string_memcpy/208/4/4/iterations:1\n"
      "BM_string_memcpy/208/4/8/iterations:1\n"
      "BM_string_memcpy/208/4/16/iterations:1\n"
      "BM_string_memcpy/208/4/32/iterations:1\n"
      "BM_string_memcpy/208/8/1/iterations:1\n"
      "BM_string_memcpy/208/8/2/iterations:1\n"
      "BM_string_memcpy/208/8/4/iterations:1\n"
      "BM_string_memcpy/208/8/8/iterations:1\n"
      "BM_string_memcpy/208/8/16/iterations:1\n"
      "BM_string_memcpy/208/8/32/iterations:1\n"
      "BM_string_memcpy/208/16/1/iterations:1\n"
      "BM_string_memcpy/208/16/2/iterations:1\n"
      "BM_string_memcpy/208/16/4/iterations:1\n"
      "BM_string_memcpy/208/16/8/iterations:1\n"
      "BM_string_memcpy/208/16/16/iterations:1\n"
      "BM_string_memcpy/208/16/32/iterations:1\n"
      "BM_string_memcpy/208/32/1/iterations:1\n"
      "BM_string_memcpy/208/32/2/iterations:1\n"
      "BM_string_memcpy/208/32/4/iterations:1\n"
      "BM_string_memcpy/208/32/8/iterations:1\n"
      "BM_string_memcpy/208/32/16/iterations:1\n"
      "BM_string_memcpy/208/32/32/iterations:1\n"
      "BM_string_memcpy/224/0/0/iterations:1\n"
      "BM_string_memcpy/224/1/1/iterations:1\n"
      "BM_string_memcpy/224/1/2/iterations:1\n"
      "BM_string_memcpy/224/1/4/iterations:1\n"
      "BM_string_memcpy/224/1/8/iterations:1\n"
      "BM_string_memcpy/224/1/16/iterations:1\n"
      "BM_string_memcpy/224/1/32/iterations:1\n"
      "BM_string_memcpy/224/2/1/iterations:1\n"
      "BM_string_memcpy/224/2/2/iterations:1\n"
      "BM_string_memcpy/224/2/4/iterations:1\n"
      "BM_string_memcpy/224/2/8/iterations:1\n"
      "BM_string_memcpy/224/2/16/iterations:1\n"
      "BM_string_memcpy/224/2/32/iterations:1\n"
      "BM_string_memcpy/224/4/1/iterations:1\n"
      "BM_string_memcpy/224/4/2/iterations:1\n"
      "BM_string_memcpy/224/4/4/iterations:1\n"
      "BM_string_memcpy/224/4/8/iterations:1\n"
      "BM_string_memcpy/224/4/16/iterations:1\n"
      "BM_string_memcpy/224/4/32/iterations:1\n"
      "BM_string_memcpy/224/8/1/iterations:1\n"
      "BM_string_memcpy/224/8/2/iterations:1\n"
      "BM_string_memcpy/224/8/4/iterations:1\n"
      "BM_string_memcpy/224/8/8/iterations:1\n"
      "BM_string_memcpy/224/8/16/iterations:1\n"
      "BM_string_memcpy/224/8/32/iterations:1\n"
      "BM_string_memcpy/224/16/1/iterations:1\n"
      "BM_string_memcpy/224/16/2/iterations:1\n"
      "BM_string_memcpy/224/16/4/iterations:1\n"
      "BM_string_memcpy/224/16/8/iterations:1\n"
      "BM_string_memcpy/224/16/16/iterations:1\n"
      "BM_string_memcpy/224/16/32/iterations:1\n"
      "BM_string_memcpy/224/32/1/iterations:1\n"
      "BM_string_memcpy/224/32/2/iterations:1\n"
      "BM_string_memcpy/224/32/4/iterations:1\n"
      "BM_string_memcpy/224/32/8/iterations:1\n"
      "BM_string_memcpy/224/32/16/iterations:1\n"
      "BM_string_memcpy/224/32/32/iterations:1\n"
      "BM_string_memcpy/240/0/0/iterations:1\n"
      "BM_string_memcpy/240/1/1/iterations:1\n"
      "BM_string_memcpy/240/1/2/iterations:1\n"
      "BM_string_memcpy/240/1/4/iterations:1\n"
      "BM_string_memcpy/240/1/8/iterations:1\n"
      "BM_string_memcpy/240/1/16/iterations:1\n"
      "BM_string_memcpy/240/1/32/iterations:1\n"
      "BM_string_memcpy/240/2/1/iterations:1\n"
      "BM_string_memcpy/240/2/2/iterations:1\n"
      "BM_string_memcpy/240/2/4/iterations:1\n"
      "BM_string_memcpy/240/2/8/iterations:1\n"
      "BM_string_memcpy/240/2/16/iterations:1\n"
      "BM_string_memcpy/240/2/32/iterations:1\n"
      "BM_string_memcpy/240/4/1/iterations:1\n"
      "BM_string_memcpy/240/4/2/iterations:1\n"
      "BM_string_memcpy/240/4/4/iterations:1\n"
      "BM_string_memcpy/240/4/8/iterations:1\n"
      "BM_string_memcpy/240/4/16/iterations:1\n"
      "BM_string_memcpy/240/4/32/iterations:1\n"
      "BM_string_memcpy/240/8/1/iterations:1\n"
      "BM_string_memcpy/240/8/2/iterations:1\n"
      "BM_string_memcpy/240/8/4/iterations:1\n"
      "BM_string_memcpy/240/8/8/iterations:1\n"
      "BM_string_memcpy/240/8/16/iterations:1\n"
      "BM_string_memcpy/240/8/32/iterations:1\n"
      "BM_string_memcpy/240/16/1/iterations:1\n"
      "BM_string_memcpy/240/16/2/iterations:1\n"
      "BM_string_memcpy/240/16/4/iterations:1\n"
      "BM_string_memcpy/240/16/8/iterations:1\n"
      "BM_string_memcpy/240/16/16/iterations:1\n"
      "BM_string_memcpy/240/16/32/iterations:1\n"
      "BM_string_memcpy/240/32/1/iterations:1\n"
      "BM_string_memcpy/240/32/2/iterations:1\n"
      "BM_string_memcpy/240/32/4/iterations:1\n"
      "BM_string_memcpy/240/32/8/iterations:1\n"
      "BM_string_memcpy/240/32/16/iterations:1\n"
      "BM_string_memcpy/240/32/32/iterations:1\n"
      "BM_string_memcpy/256/0/0/iterations:1\n"
      "BM_string_memcpy/256/1/1/iterations:1\n"
      "BM_string_memcpy/256/1/2/iterations:1\n"
      "BM_string_memcpy/256/1/4/iterations:1\n"
      "BM_string_memcpy/256/1/8/iterations:1\n"
      "BM_string_memcpy/256/1/16/iterations:1\n"
      "BM_string_memcpy/256/1/32/iterations:1\n"
      "BM_string_memcpy/256/2/1/iterations:1\n"
      "BM_string_memcpy/256/2/2/iterations:1\n"
      "BM_string_memcpy/256/2/4/iterations:1\n"
      "BM_string_memcpy/256/2/8/iterations:1\n"
      "BM_string_memcpy/256/2/16/iterations:1\n"
      "BM_string_memcpy/256/2/32/iterations:1\n"
      "BM_string_memcpy/256/4/1/iterations:1\n"
      "BM_string_memcpy/256/4/2/iterations:1\n"
      "BM_string_memcpy/256/4/4/iterations:1\n"
      "BM_string_memcpy/256/4/8/iterations:1\n"
      "BM_string_memcpy/256/4/16/iterations:1\n"
      "BM_string_memcpy/256/4/32/iterations:1\n"
      "BM_string_memcpy/256/8/1/iterations:1\n"
      "BM_string_memcpy/256/8/2/iterations:1\n"
      "BM_string_memcpy/256/8/4/iterations:1\n"
      "BM_string_memcpy/256/8/8/iterations:1\n"
      "BM_string_memcpy/256/8/16/iterations:1\n"
      "BM_string_memcpy/256/8/32/iterations:1\n"
      "BM_string_memcpy/256/16/1/iterations:1\n"
      "BM_string_memcpy/256/16/2/iterations:1\n"
      "BM_string_memcpy/256/16/4/iterations:1\n"
      "BM_string_memcpy/256/16/8/iterations:1\n"
      "BM_string_memcpy/256/16/16/iterations:1\n"
      "BM_string_memcpy/256/16/32/iterations:1\n"
      "BM_string_memcpy/256/32/1/iterations:1\n"
      "BM_string_memcpy/256/32/2/iterations:1\n"
      "BM_string_memcpy/256/32/4/iterations:1\n"
      "BM_string_memcpy/256/32/8/iterations:1\n"
      "BM_string_memcpy/256/32/16/iterations:1\n"
      "BM_string_memcpy/256/32/32/iterations:1\n"
      "BM_string_memcpy/512/0/0/iterations:1\n"
      "BM_string_memcpy/512/1/1/iterations:1\n"
      "BM_string_memcpy/512/1/2/iterations:1\n"
      "BM_string_memcpy/512/1/4/iterations:1\n"
      "BM_string_memcpy/512/1/8/iterations:1\n"
      "BM_string_memcpy/512/1/16/iterations:1\n"
      "BM_string_memcpy/512/1/32/iterations:1\n"
      "BM_string_memcpy/512/2/1/iterations:1\n"
      "BM_string_memcpy/512/2/2/iterations:1\n"
      "BM_string_memcpy/512/2/4/iterations:1\n"
      "BM_string_memcpy/512/2/8/iterations:1\n"
      "BM_string_memcpy/512/2/16/iterations:1\n"
      "BM_string_memcpy/512/2/32/iterations:1\n"
      "BM_string_memcpy/512/4/1/iterations:1\n"
      "BM_string_memcpy/512/4/2/iterations:1\n"
      "BM_string_memcpy/512/4/4/iterations:1\n"
      "BM_string_memcpy/512/4/8/iterations:1\n"
      "BM_string_memcpy/512/4/16/iterations:1\n"
      "BM_string_memcpy/512/4/32/iterations:1\n"
      "BM_string_memcpy/512/8/1/iterations:1\n"
      "BM_string_memcpy/512/8/2/iterations:1\n"
      "BM_string_memcpy/512/8/4/iterations:1\n"
      "BM_string_memcpy/512/8/8/iterations:1\n"
      "BM_string_memcpy/512/8/16/iterations:1\n"
      "BM_string_memcpy/512/8/32/iterations:1\n"
      "BM_string_memcpy/512/16/1/iterations:1\n"
      "BM_string_memcpy/512/16/2/iterations:1\n"
      "BM_string_memcpy/512/16/4/iterations:1\n"
      "BM_string_memcpy/512/16/8/iterations:1\n"
      "BM_string_memcpy/512/16/16/iterations:1\n"
      "BM_string_memcpy/512/16/32/iterations:1\n"
      "BM_string_memcpy/512/32/1/iterations:1\n"
      "BM_string_memcpy/512/32/2/iterations:1\n"
      "BM_string_memcpy/512/32/4/iterations:1\n"
      "BM_string_memcpy/512/32/8/iterations:1\n"
      "BM_string_memcpy/512/32/16/iterations:1\n"
      "BM_string_memcpy/512/32/32/iterations:1\n"
      "BM_string_memcpy/1024/0/0/iterations:1\n"
      "BM_string_memcpy/1024/1/1/iterations:1\n"
      "BM_string_memcpy/1024/1/2/iterations:1\n"
      "BM_string_memcpy/1024/1/4/iterations:1\n"
      "BM_string_memcpy/1024/1/8/iterations:1\n"
      "BM_string_memcpy/1024/1/16/iterations:1\n"
      "BM_string_memcpy/1024/1/32/iterations:1\n"
      "BM_string_memcpy/1024/2/1/iterations:1\n"
      "BM_string_memcpy/1024/2/2/iterations:1\n"
      "BM_string_memcpy/1024/2/4/iterations:1\n"
      "BM_string_memcpy/1024/2/8/iterations:1\n"
      "BM_string_memcpy/1024/2/16/iterations:1\n"
      "BM_string_memcpy/1024/2/32/iterations:1\n"
      "BM_string_memcpy/1024/4/1/iterations:1\n"
      "BM_string_memcpy/1024/4/2/iterations:1\n"
      "BM_string_memcpy/1024/4/4/iterations:1\n"
      "BM_string_memcpy/1024/4/8/iterations:1\n"
      "BM_string_memcpy/1024/4/16/iterations:1\n"
      "BM_string_memcpy/1024/4/32/iterations:1\n"
      "BM_string_memcpy/1024/8/1/iterations:1\n"
      "BM_string_memcpy/1024/8/2/iterations:1\n"
      "BM_string_memcpy/1024/8/4/iterations:1\n"
      "BM_string_memcpy/1024/8/8/iterations:1\n"
      "BM_string_memcpy/1024/8/16/iterations:1\n"
      "BM_string_memcpy/1024/8/32/iterations:1\n"
      "BM_string_memcpy/1024/16/1/iterations:1\n"
      "BM_string_memcpy/1024/16/2/iterations:1\n"
      "BM_string_memcpy/1024/16/4/iterations:1\n"
      "BM_string_memcpy/1024/16/8/iterations:1\n"
      "BM_string_memcpy/1024/16/16/iterations:1\n"
      "BM_string_memcpy/1024/16/32/iterations:1\n"
      "BM_string_memcpy/1024/32/1/iterations:1\n"
      "BM_string_memcpy/1024/32/2/iterations:1\n"
      "BM_string_memcpy/1024/32/4/iterations:1\n"
      "BM_string_memcpy/1024/32/8/iterations:1\n"
      "BM_string_memcpy/1024/32/16/iterations:1\n"
      "BM_string_memcpy/1024/32/32/iterations:1\n"
      "BM_string_memcpy/8192/0/0/iterations:1\n"
      "BM_string_memcpy/8192/1/1/iterations:1\n"
      "BM_string_memcpy/8192/1/2/iterations:1\n"
      "BM_string_memcpy/8192/1/4/iterations:1\n"
      "BM_string_memcpy/8192/1/8/iterations:1\n"
      "BM_string_memcpy/8192/1/16/iterations:1\n"
      "BM_string_memcpy/8192/1/32/iterations:1\n"
      "BM_string_memcpy/8192/2/1/iterations:1\n"
      "BM_string_memcpy/8192/2/2/iterations:1\n"
      "BM_string_memcpy/8192/2/4/iterations:1\n"
      "BM_string_memcpy/8192/2/8/iterations:1\n"
      "BM_string_memcpy/8192/2/16/iterations:1\n"
      "BM_string_memcpy/8192/2/32/iterations:1\n"
      "BM_string_memcpy/8192/4/1/iterations:1\n"
      "BM_string_memcpy/8192/4/2/iterations:1\n"
      "BM_string_memcpy/8192/4/4/iterations:1\n"
      "BM_string_memcpy/8192/4/8/iterations:1\n"
      "BM_string_memcpy/8192/4/16/iterations:1\n"
      "BM_string_memcpy/8192/4/32/iterations:1\n"
      "BM_string_memcpy/8192/8/1/iterations:1\n"
      "BM_string_memcpy/8192/8/2/iterations:1\n"
      "BM_string_memcpy/8192/8/4/iterations:1\n"
      "BM_string_memcpy/8192/8/8/iterations:1\n"
      "BM_string_memcpy/8192/8/16/iterations:1\n"
      "BM_string_memcpy/8192/8/32/iterations:1\n"
      "BM_string_memcpy/8192/16/1/iterations:1\n"
      "BM_string_memcpy/8192/16/2/iterations:1\n"
      "BM_string_memcpy/8192/16/4/iterations:1\n"
      "BM_string_memcpy/8192/16/8/iterations:1\n"
      "BM_string_memcpy/8192/16/16/iterations:1\n"
      "BM_string_memcpy/8192/16/32/iterations:1\n"
      "BM_string_memcpy/8192/32/1/iterations:1\n"
      "BM_string_memcpy/8192/32/2/iterations:1\n"
      "BM_string_memcpy/8192/32/4/iterations:1\n"
      "BM_string_memcpy/8192/32/8/iterations:1\n"
      "BM_string_memcpy/8192/32/16/iterations:1\n"
      "BM_string_memcpy/8192/32/32/iterations:1\n"
      "BM_string_memcpy/16384/0/0/iterations:1\n"
      "BM_string_memcpy/16384/1/1/iterations:1\n"
      "BM_string_memcpy/16384/1/2/iterations:1\n"
      "BM_string_memcpy/16384/1/4/iterations:1\n"
      "BM_string_memcpy/16384/1/8/iterations:1\n"
      "BM_string_memcpy/16384/1/16/iterations:1\n"
      "BM_string_memcpy/16384/1/32/iterations:1\n"
      "BM_string_memcpy/16384/2/1/iterations:1\n"
      "BM_string_memcpy/16384/2/2/iterations:1\n"
      "BM_string_memcpy/16384/2/4/iterations:1\n"
      "BM_string_memcpy/16384/2/8/iterations:1\n"
      "BM_string_memcpy/16384/2/16/iterations:1\n"
      "BM_string_memcpy/16384/2/32/iterations:1\n"
      "BM_string_memcpy/16384/4/1/iterations:1\n"
      "BM_string_memcpy/16384/4/2/iterations:1\n"
      "BM_string_memcpy/16384/4/4/iterations:1\n"
      "BM_string_memcpy/16384/4/8/iterations:1\n"
      "BM_string_memcpy/16384/4/16/iterations:1\n"
      "BM_string_memcpy/16384/4/32/iterations:1\n"
      "BM_string_memcpy/16384/8/1/iterations:1\n"
      "BM_string_memcpy/16384/8/2/iterations:1\n"
      "BM_string_memcpy/16384/8/4/iterations:1\n"
      "BM_string_memcpy/16384/8/8/iterations:1\n"
      "BM_string_memcpy/16384/8/16/iterations:1\n"
      "BM_string_memcpy/16384/8/32/iterations:1\n"
      "BM_string_memcpy/16384/16/1/iterations:1\n"
      "BM_string_memcpy/16384/16/2/iterations:1\n"
      "BM_string_memcpy/16384/16/4/iterations:1\n"
      "BM_string_memcpy/16384/16/8/iterations:1\n"
      "BM_string_memcpy/16384/16/16/iterations:1\n"
      "BM_string_memcpy/16384/16/32/iterations:1\n"
      "BM_string_memcpy/16384/32/1/iterations:1\n"
      "BM_string_memcpy/16384/32/2/iterations:1\n"
      "BM_string_memcpy/16384/32/4/iterations:1\n"
      "BM_string_memcpy/16384/32/8/iterations:1\n"
      "BM_string_memcpy/16384/32/16/iterations:1\n"
      "BM_string_memcpy/16384/32/32/iterations:1\n"
      "BM_string_memcpy/32768/0/0/iterations:1\n"
      "BM_string_memcpy/32768/1/1/iterations:1\n"
      "BM_string_memcpy/32768/1/2/iterations:1\n"
      "BM_string_memcpy/32768/1/4/iterations:1\n"
      "BM_string_memcpy/32768/1/8/iterations:1\n"
      "BM_string_memcpy/32768/1/16/iterations:1\n"
      "BM_string_memcpy/32768/1/32/iterations:1\n"
      "BM_string_memcpy/32768/2/1/iterations:1\n"
      "BM_string_memcpy/32768/2/2/iterations:1\n"
      "BM_string_memcpy/32768/2/4/iterations:1\n"
      "BM_string_memcpy/32768/2/8/iterations:1\n"
      "BM_string_memcpy/32768/2/16/iterations:1\n"
      "BM_string_memcpy/32768/2/32/iterations:1\n"
      "BM_string_memcpy/32768/4/1/iterations:1\n"
      "BM_string_memcpy/32768/4/2/iterations:1\n"
      "BM_string_memcpy/32768/4/4/iterations:1\n"
      "BM_string_memcpy/32768/4/8/iterations:1\n"
      "BM_string_memcpy/32768/4/16/iterations:1\n"
      "BM_string_memcpy/32768/4/32/iterations:1\n"
      "BM_string_memcpy/32768/8/1/iterations:1\n"
      "BM_string_memcpy/32768/8/2/iterations:1\n"
      "BM_string_memcpy/32768/8/4/iterations:1\n"
      "BM_string_memcpy/32768/8/8/iterations:1\n"
      "BM_string_memcpy/32768/8/16/iterations:1\n"
      "BM_string_memcpy/32768/8/32/iterations:1\n"
      "BM_string_memcpy/32768/16/1/iterations:1\n"
      "BM_string_memcpy/32768/16/2/iterations:1\n"
      "BM_string_memcpy/32768/16/4/iterations:1\n"
      "BM_string_memcpy/32768/16/8/iterations:1\n"
      "BM_string_memcpy/32768/16/16/iterations:1\n"
      "BM_string_memcpy/32768/16/32/iterations:1\n"
      "BM_string_memcpy/32768/32/1/iterations:1\n"
      "BM_string_memcpy/32768/32/2/iterations:1\n"
      "BM_string_memcpy/32768/32/4/iterations:1\n"
      "BM_string_memcpy/32768/32/8/iterations:1\n"
      "BM_string_memcpy/32768/32/16/iterations:1\n"
      "BM_string_memcpy/32768/32/32/iterations:1\n"
      "BM_string_memcpy/65536/0/0/iterations:1\n"
      "BM_string_memcpy/65536/1/1/iterations:1\n"
      "BM_string_memcpy/65536/1/2/iterations:1\n"
      "BM_string_memcpy/65536/1/4/iterations:1\n"
      "BM_string_memcpy/65536/1/8/iterations:1\n"
      "BM_string_memcpy/65536/1/16/iterations:1\n"
      "BM_string_memcpy/65536/1/32/iterations:1\n"
      "BM_string_memcpy/65536/2/1/iterations:1\n"
      "BM_string_memcpy/65536/2/2/iterations:1\n"
      "BM_string_memcpy/65536/2/4/iterations:1\n"
      "BM_string_memcpy/65536/2/8/iterations:1\n"
      "BM_string_memcpy/65536/2/16/iterations:1\n"
      "BM_string_memcpy/65536/2/32/iterations:1\n"
      "BM_string_memcpy/65536/4/1/iterations:1\n"
      "BM_string_memcpy/65536/4/2/iterations:1\n"
      "BM_string_memcpy/65536/4/4/iterations:1\n"
      "BM_string_memcpy/65536/4/8/iterations:1\n"
      "BM_string_memcpy/65536/4/16/iterations:1\n"
      "BM_string_memcpy/65536/4/32/iterations:1\n"
      "BM_string_memcpy/65536/8/1/iterations:1\n"
      "BM_string_memcpy/65536/8/2/iterations:1\n"
      "BM_string_memcpy/65536/8/4/iterations:1\n"
      "BM_string_memcpy/65536/8/8/iterations:1\n"
      "BM_string_memcpy/65536/8/16/iterations:1\n"
      "BM_string_memcpy/65536/8/32/iterations:1\n"
      "BM_string_memcpy/65536/16/1/iterations:1\n"
      "BM_string_memcpy/65536/16/2/iterations:1\n"
      "BM_string_memcpy/65536/16/4/iterations:1\n"
      "BM_string_memcpy/65536/16/8/iterations:1\n"
      "BM_string_memcpy/65536/16/16/iterations:1\n"
      "BM_string_memcpy/65536/16/32/iterations:1\n"
      "BM_string_memcpy/65536/32/1/iterations:1\n"
      "BM_string_memcpy/65536/32/2/iterations:1\n"
      "BM_string_memcpy/65536/32/4/iterations:1\n"
      "BM_string_memcpy/65536/32/8/iterations:1\n"
      "BM_string_memcpy/65536/32/16/iterations:1\n"
      "BM_string_memcpy/65536/32/32/iterations:1\n"
      "BM_string_memcpy/131072/0/0/iterations:1\n"
      "BM_string_memcpy/131072/1/1/iterations:1\n"
      "BM_string_memcpy/131072/1/2/iterations:1\n"
      "BM_string_memcpy/131072/1/4/iterations:1\n"
      "BM_string_memcpy/131072/1/8/iterations:1\n"
      "BM_string_memcpy/131072/1/16/iterations:1\n"
      "BM_string_memcpy/131072/1/32/iterations:1\n"
      "BM_string_memcpy/131072/2/1/iterations:1\n"
      "BM_string_memcpy/131072/2/2/iterations:1\n"
      "BM_string_memcpy/131072/2/4/iterations:1\n"
      "BM_string_memcpy/131072/2/8/iterations:1\n"
      "BM_string_memcpy/131072/2/16/iterations:1\n"
      "BM_string_memcpy/131072/2/32/iterations:1\n"
      "BM_string_memcpy/131072/4/1/iterations:1\n"
      "BM_string_memcpy/131072/4/2/iterations:1\n"
      "BM_string_memcpy/131072/4/4/iterations:1\n"
      "BM_string_memcpy/131072/4/8/iterations:1\n"
      "BM_string_memcpy/131072/4/16/iterations:1\n"
      "BM_string_memcpy/131072/4/32/iterations:1\n"
      "BM_string_memcpy/131072/8/1/iterations:1\n"
      "BM_string_memcpy/131072/8/2/iterations:1\n"
      "BM_string_memcpy/131072/8/4/iterations:1\n"
      "BM_string_memcpy/131072/8/8/iterations:1\n"
      "BM_string_memcpy/131072/8/16/iterations:1\n"
      "BM_string_memcpy/131072/8/32/iterations:1\n"
      "BM_string_memcpy/131072/16/1/iterations:1\n"
      "BM_string_memcpy/131072/16/2/iterations:1\n"
      "BM_string_memcpy/131072/16/4/iterations:1\n"
      "BM_string_memcpy/131072/16/8/iterations:1\n"
      "BM_string_memcpy/131072/16/16/iterations:1\n"
      "BM_string_memcpy/131072/16/32/iterations:1\n"
      "BM_string_memcpy/131072/32/1/iterations:1\n"
      "BM_string_memcpy/131072/32/2/iterations:1\n"
      "BM_string_memcpy/131072/32/4/iterations:1\n"
      "BM_string_memcpy/131072/32/8/iterations:1\n"
      "BM_string_memcpy/131072/32/16/iterations:1\n"
      "BM_string_memcpy/131072/32/32/iterations:1\n"
      "BM_string_memcpy/262144/0/0/iterations:1\n"
      "BM_string_memcpy/262144/1/1/iterations:1\n"
      "BM_string_memcpy/262144/1/2/iterations:1\n"
      "BM_string_memcpy/262144/1/4/iterations:1\n"
      "BM_string_memcpy/262144/1/8/iterations:1\n"
      "BM_string_memcpy/262144/1/16/iterations:1\n"
      "BM_string_memcpy/262144/1/32/iterations:1\n"
      "BM_string_memcpy/262144/2/1/iterations:1\n"
      "BM_string_memcpy/262144/2/2/iterations:1\n"
      "BM_string_memcpy/262144/2/4/iterations:1\n"
      "BM_string_memcpy/262144/2/8/iterations:1\n"
      "BM_string_memcpy/262144/2/16/iterations:1\n"
      "BM_string_memcpy/262144/2/32/iterations:1\n"
      "BM_string_memcpy/262144/4/1/iterations:1\n"
      "BM_string_memcpy/262144/4/2/iterations:1\n"
      "BM_string_memcpy/262144/4/4/iterations:1\n"
      "BM_string_memcpy/262144/4/8/iterations:1\n"
      "BM_string_memcpy/262144/4/16/iterations:1\n"
      "BM_string_memcpy/262144/4/32/iterations:1\n"
      "BM_string_memcpy/262144/8/1/iterations:1\n"
      "BM_string_memcpy/262144/8/2/iterations:1\n"
      "BM_string_memcpy/262144/8/4/iterations:1\n"
      "BM_string_memcpy/262144/8/8/iterations:1\n"
      "BM_string_memcpy/262144/8/16/iterations:1\n"
      "BM_string_memcpy/262144/8/32/iterations:1\n"
      "BM_string_memcpy/262144/16/1/iterations:1\n"
      "BM_string_memcpy/262144/16/2/iterations:1\n"
      "BM_string_memcpy/262144/16/4/iterations:1\n"
      "BM_string_memcpy/262144/16/8/iterations:1\n"
      "BM_string_memcpy/262144/16/16/iterations:1\n"
      "BM_string_memcpy/262144/16/32/iterations:1\n"
      "BM_string_memcpy/262144/32/1/iterations:1\n"
      "BM_string_memcpy/262144/32/2/iterations:1\n"
      "BM_string_memcpy/262144/32/4/iterations:1\n"
      "BM_string_memcpy/262144/32/8/iterations:1\n"
      "BM_string_memcpy/262144/32/16/iterations:1\n"
      "BM_string_memcpy/262144/32/32/iterations:1\n"
      "BM_string_memcpy/524288/0/0/iterations:1\n"
      "BM_string_memcpy/524288/1/1/iterations:1\n"
      "BM_string_memcpy/524288/1/2/iterations:1\n"
      "BM_string_memcpy/524288/1/4/iterations:1\n"
      "BM_string_memcpy/524288/1/8/iterations:1\n"
      "BM_string_memcpy/524288/1/16/iterations:1\n"
      "BM_string_memcpy/524288/1/32/iterations:1\n"
      "BM_string_memcpy/524288/2/1/iterations:1\n"
      "BM_string_memcpy/524288/2/2/iterations:1\n"
      "BM_string_memcpy/524288/2/4/iterations:1\n"
      "BM_string_memcpy/524288/2/8/iterations:1\n"
      "BM_string_memcpy/524288/2/16/iterations:1\n"
      "BM_string_memcpy/524288/2/32/iterations:1\n"
      "BM_string_memcpy/524288/4/1/iterations:1\n"
      "BM_string_memcpy/524288/4/2/iterations:1\n"
      "BM_string_memcpy/524288/4/4/iterations:1\n"
      "BM_string_memcpy/524288/4/8/iterations:1\n"
      "BM_string_memcpy/524288/4/16/iterations:1\n"
      "BM_string_memcpy/524288/4/32/iterations:1\n"
      "BM_string_memcpy/524288/8/1/iterations:1\n"
      "BM_string_memcpy/524288/8/2/iterations:1\n"
      "BM_string_memcpy/524288/8/4/iterations:1\n"
      "BM_string_memcpy/524288/8/8/iterations:1\n"
      "BM_string_memcpy/524288/8/16/iterations:1\n"
      "BM_string_memcpy/524288/8/32/iterations:1\n"
      "BM_string_memcpy/524288/16/1/iterations:1\n"
      "BM_string_memcpy/524288/16/2/iterations:1\n"
      "BM_string_memcpy/524288/16/4/iterations:1\n"
      "BM_string_memcpy/524288/16/8/iterations:1\n"
      "BM_string_memcpy/524288/16/16/iterations:1\n"
      "BM_string_memcpy/524288/16/32/iterations:1\n"
      "BM_string_memcpy/524288/32/1/iterations:1\n"
      "BM_string_memcpy/524288/32/2/iterations:1\n"
      "BM_string_memcpy/524288/32/4/iterations:1\n"
      "BM_string_memcpy/524288/32/8/iterations:1\n"
      "BM_string_memcpy/524288/32/16/iterations:1\n"
      "BM_string_memcpy/524288/32/32/iterations:1\n"
      "BM_string_memcpy/1048576/0/0/iterations:1\n"
      "BM_string_memcpy/1048576/1/1/iterations:1\n"
      "BM_string_memcpy/1048576/1/2/iterations:1\n"
      "BM_string_memcpy/1048576/1/4/iterations:1\n"
      "BM_string_memcpy/1048576/1/8/iterations:1\n"
      "BM_string_memcpy/1048576/1/16/iterations:1\n"
      "BM_string_memcpy/1048576/1/32/iterations:1\n"
      "BM_string_memcpy/1048576/2/1/iterations:1\n"
      "BM_string_memcpy/1048576/2/2/iterations:1\n"
      "BM_string_memcpy/1048576/2/4/iterations:1\n"
      "BM_string_memcpy/1048576/2/8/iterations:1\n"
      "BM_string_memcpy/1048576/2/16/iterations:1\n"
      "BM_string_memcpy/1048576/2/32/iterations:1\n"
      "BM_string_memcpy/1048576/4/1/iterations:1\n"
      "BM_string_memcpy/1048576/4/2/iterations:1\n"
      "BM_string_memcpy/1048576/4/4/iterations:1\n"
      "BM_string_memcpy/1048576/4/8/iterations:1\n"
      "BM_string_memcpy/1048576/4/16/iterations:1\n"
      "BM_string_memcpy/1048576/4/32/iterations:1\n"
      "BM_string_memcpy/1048576/8/1/iterations:1\n"
      "BM_string_memcpy/1048576/8/2/iterations:1\n"
      "BM_string_memcpy/1048576/8/4/iterations:1\n"
      "BM_string_memcpy/1048576/8/8/iterations:1\n"
      "BM_string_memcpy/1048576/8/16/iterations:1\n"
      "BM_string_memcpy/1048576/8/32/iterations:1\n"
      "BM_string_memcpy/1048576/16/1/iterations:1\n"
      "BM_string_memcpy/1048576/16/2/iterations:1\n"
      "BM_string_memcpy/1048576/16/4/iterations:1\n"
      "BM_string_memcpy/1048576/16/8/iterations:1\n"
      "BM_string_memcpy/1048576/16/16/iterations:1\n"
      "BM_string_memcpy/1048576/16/32/iterations:1\n"
      "BM_string_memcpy/1048576/32/1/iterations:1\n"
      "BM_string_memcpy/1048576/32/2/iterations:1\n"
      "BM_string_memcpy/1048576/32/4/iterations:1\n"
      "BM_string_memcpy/1048576/32/8/iterations:1\n"
      "BM_string_memcpy/1048576/32/16/iterations:1\n"
      "BM_string_memcpy/1048576/32/32/iterations:1\n"
      "BM_string_memcpy/2097152/0/0/iterations:1\n"
      "BM_string_memcpy/2097152/1/1/iterations:1\n"
      "BM_string_memcpy/2097152/1/2/iterations:1\n"
      "BM_string_memcpy/2097152/1/4/iterations:1\n"
      "BM_string_memcpy/2097152/1/8/iterations:1\n"
      "BM_string_memcpy/2097152/1/16/iterations:1\n"
      "BM_string_memcpy/2097152/1/32/iterations:1\n"
      "BM_string_memcpy/2097152/2/1/iterations:1\n"
      "BM_string_memcpy/2097152/2/2/iterations:1\n"
      "BM_string_memcpy/2097152/2/4/iterations:1\n"
      "BM_string_memcpy/2097152/2/8/iterations:1\n"
      "BM_string_memcpy/2097152/2/16/iterations:1\n"
      "BM_string_memcpy/2097152/2/32/iterations:1\n"
      "BM_string_memcpy/2097152/4/1/iterations:1\n"
      "BM_string_memcpy/2097152/4/2/iterations:1\n"
      "BM_string_memcpy/2097152/4/4/iterations:1\n"
      "BM_string_memcpy/2097152/4/8/iterations:1\n"
      "BM_string_memcpy/2097152/4/16/iterations:1\n"
      "BM_string_memcpy/2097152/4/32/iterations:1\n"
      "BM_string_memcpy/2097152/8/1/iterations:1\n"
      "BM_string_memcpy/2097152/8/2/iterations:1\n"
      "BM_string_memcpy/2097152/8/4/iterations:1\n"
      "BM_string_memcpy/2097152/8/8/iterations:1\n"
      "BM_string_memcpy/2097152/8/16/iterations:1\n"
      "BM_string_memcpy/2097152/8/32/iterations:1\n"
      "BM_string_memcpy/2097152/16/1/iterations:1\n"
      "BM_string_memcpy/2097152/16/2/iterations:1\n"
      "BM_string_memcpy/2097152/16/4/iterations:1\n"
      "BM_string_memcpy/2097152/16/8/iterations:1\n"
      "BM_string_memcpy/2097152/16/16/iterations:1\n"
      "BM_string_memcpy/2097152/16/32/iterations:1\n"
      "BM_string_memcpy/2097152/32/1/iterations:1\n"
      "BM_string_memcpy/2097152/32/2/iterations:1\n"
      "BM_string_memcpy/2097152/32/4/iterations:1\n"
      "BM_string_memcpy/2097152/32/8/iterations:1\n"
      "BM_string_memcpy/2097152/32/16/iterations:1\n"
      "BM_string_memcpy/2097152/32/32/iterations:1\n";

  Verify(expected, 0,
         std::vector<const char*>{GetBionicXmlArg("test_alignment_twobuf.xml").c_str()});
}

TEST_F(SystemTests, alignment_twobuf_manual) {
  std::string expected =
    "BM_string_strcpy/12345/0/0/iterations:1\n"
    "BM_string_memcpy/10248/2/1/iterations:1\n"
    "BM_string_strcpy/17820/4/8/iterations:1\n"
    "BM_string_memcpy/20000/16/32/iterations:1\n"
    "BM_string_strcmp/1/4/0/iterations:1\n"
    "BM_string_strcmp/2/4/0/iterations:1\n"
    "BM_string_strcmp/3/4/0/iterations:1\n"
    "BM_string_strcmp/4/4/0/iterations:1\n"
    "BM_string_strcmp/5/4/0/iterations:1\n"
    "BM_string_strcmp/6/4/0/iterations:1\n"
    "BM_string_strcmp/7/4/0/iterations:1\n"
    "BM_string_strcmp/8/4/0/iterations:1\n"
    "BM_string_strcmp/9/4/0/iterations:1\n"
    "BM_string_strcmp/10/4/0/iterations:1\n"
    "BM_string_strcmp/11/4/0/iterations:1\n"
    "BM_string_strcmp/12/4/0/iterations:1\n"
    "BM_string_strcmp/13/4/0/iterations:1\n"
    "BM_string_strcmp/14/4/0/iterations:1\n"
    "BM_string_strcmp/15/4/0/iterations:1\n"
    "BM_string_strcmp/16/4/0/iterations:1\n"
    "BM_string_strcmp/24/4/0/iterations:1\n"
    "BM_string_strcmp/32/4/0/iterations:1\n"
    "BM_string_strcmp/40/4/0/iterations:1\n"
    "BM_string_strcmp/48/4/0/iterations:1\n"
    "BM_string_strcmp/56/4/0/iterations:1\n"
    "BM_string_strcmp/64/4/0/iterations:1\n"
    "BM_string_strcmp/72/4/0/iterations:1\n"
    "BM_string_strcmp/80/4/0/iterations:1\n"
    "BM_string_strcmp/88/4/0/iterations:1\n"
    "BM_string_strcmp/96/4/0/iterations:1\n"
    "BM_string_strcmp/104/4/0/iterations:1\n"
    "BM_string_strcmp/112/4/0/iterations:1\n"
    "BM_string_strcmp/120/4/0/iterations:1\n"
    "BM_string_strcmp/128/4/0/iterations:1\n"
    "BM_string_strcmp/136/4/0/iterations:1\n"
    "BM_string_strcmp/144/4/0/iterations:1\n"
    "BM_string_strcmp/160/4/0/iterations:1\n"
    "BM_string_strcmp/176/4/0/iterations:1\n"
    "BM_string_strcmp/192/4/0/iterations:1\n"
    "BM_string_strcmp/208/4/0/iterations:1\n"
    "BM_string_strcmp/224/4/0/iterations:1\n"
    "BM_string_strcmp/240/4/0/iterations:1\n"
    "BM_string_strcmp/256/4/0/iterations:1\n"
    "BM_string_strcmp/512/4/4/iterations:1\n"
    "BM_string_strcmp/1024/4/4/iterations:1\n"
    "BM_string_strcmp/8192/4/4/iterations:1\n"
    "BM_string_strcmp/16384/4/4/iterations:1\n"
    "BM_string_strcmp/32768/4/4/iterations:1\n"
    "BM_string_strcmp/65536/4/4/iterations:1\n"
    "BM_string_strcmp/131072/4/4/iterations:1\n";

  Verify(expected, 0,
         std::vector<const char*>{GetBionicXmlArg("test_alignment_twobuf_manual.xml").c_str()});
}
